﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Xml.Linq;
using CommandUtility;

namespace ContentsUploader.Assistants
{
    public enum ContentMetaType
    {
        Application = 0,
        Patch = 1,
        Delta = 2,
        AddOnContent = 3,
    }

    public class NspAccessor
    {
        public const ulong AddOnContentIdOffset = 0x01000;

        public static readonly Dictionary<ContentMetaType, int> ContentPriority;
        static NspAccessor()
        {
            var table = new Dictionary<ContentMetaType, int>();
            table.Add(ContentMetaType.Application, 0);
            table.Add(ContentMetaType.Patch, 1);
            table.Add(ContentMetaType.Delta, 1);
            table.Add(ContentMetaType.AddOnContent, 3);
            ContentPriority = table;
        }

        public struct Content
        {
            public Id64 ContentMetaId { get; }
            public int Version { get; }

            public ContentMetaType Type { get; }
            public bool HasTicket { get; }
            public int Priority { get { return ContentPriority[Type]; } }

            public Content(Id64 contentMetaId, int version, ContentMetaType type, bool ticket)
            {
                ContentMetaId = contentMetaId;
                Version = version;
                Type = type;
                HasTicket = ticket;
            }

            public override string ToString()
            {
                return $"{ContentMetaId},{Version},{Type},{Priority}";
            }

            public static int CompareForRegister(Content lhs, Content rhs)
            {
                return ToolUtility.Compare(lhs, rhs,
                    (l, r) =>
                    {
                        int result = 0;
                        if (false) { } // 単なる位置合わせ
                        else if (ToolUtility.Compare(out result, l.Priority, r.Priority)) { }
                        else if (ToolUtility.Compare(out result, l.Version, r.Version)) { }
                        else if (ToolUtility.Compare(out result, l.ContentMetaId, r.ContentMetaId)) { }
                        return result;
                    });
            }
        }

        public string Path { get; }
        public DateTime LastWriteDate { get; }
        public List<Content> NspInnerContent { get; }

        // Application Id, Type に関しては 1 つの nsp 内で統一されていることを前提とする
        public Id64 ApplicationId { get; }
        public ContentMetaType Type { get; }
        public int Priority { get { return ContentPriority[Type]; } }
        private int Version { get; set; }
        public int AocIndex { get; }

        public bool IsVerbose { get; }
        public bool IsRedirection { get; }  // ContentMetaAccessor 内の標準出力リダイレクトの有効化

        public NspAccessor(string nspPath, bool isVerbose, bool isRedirection = true)
        {
            var nspFile = new System.IO.FileInfo(nspPath);
            Path = nspFile.FullName;
            LastWriteDate = nspFile.LastWriteTime;

            NspInnerContent = new List<Content>();
            IsVerbose = isVerbose;
            IsRedirection = isRedirection;

            using (var tempHolder = new TemporaryFileHolder("ContentsUploader"))
            {
                var xmlDirectory = tempHolder.CreateTemporaryDirectory("xmlDirectory");
                if (!ExtractNsp(xmlDirectory, Path))
                {
                    throw new Exception($"Failed to extract nsp file. Please check the path \"{Path}\".");
                }

                foreach (var xmlPath in System.IO.Directory.GetFiles(xmlDirectory.ToString(), "*.cnmt.xml"))
                {
                    if (IsVerbose)
                    {
                        var xmlFile = new System.IO.FileInfo(xmlPath);
                        var xmlData = File.ReadAllText(xmlPath);
                        WriteLog($"ContentMeta: {Path}, {xmlFile.Name}");
                        WriteLogAsIs(xmlData);
                    }

                    var nspMetaData = XElement.Load(xmlPath);
                    var contentMetaId = new Id64(nspMetaData.Element("Id").Value);
                    var version = int.Parse(nspMetaData.Element("Version").Value);
                    var type = ToolUtility.ToEnumValue<ContentMetaType>(nspMetaData.Element("Type").Value);
                    var tikectFiles = System.IO.Directory.GetFiles(xmlDirectory.FullName, $"{contentMetaId}*.tik");
                    var hasTicket = tikectFiles.Length > 0;

                    //! 1 つの nsp に 1 種類の type のコンテンツしか入らないことを前提として、InnerContent.innerType の以外に Type を実装
                    //! TODO: 前提とできないケースが発生した場合、全て InnerContent.innerType に移行
                    Type = type;
                    Version = version;

                    Content contentMeta = new Content(contentMetaId, version, type, hasTicket);
                    NspInnerContent.Add(contentMeta);

                    var applicationId = contentMetaId;
                    if (Type == ContentMetaType.AddOnContent || Type == ContentMetaType.Patch || Type == ContentMetaType.Delta)
                    {
                        applicationId = new Id64(nspMetaData.Element("ApplicationId").Value);
                    }
                    if (ApplicationId == null)
                    {
                        ApplicationId = applicationId;
                    }
                    else if (ApplicationId != applicationId)
                    {
                        throw new Exception($"Unexpected file. Please check the path \"{Path}\".");
                    }
                }
                if (NspInnerContent.Count > 0)
                {
                    NspInnerContent.Sort(Content.CompareForRegister);

                    //! 1 つの nsp に 1 種類の type のコンテンツしか入らないことを前提として、InnerContent.innerType の以外に Type を実装
                    //! TODO: 前提とできないケースが発生した場合、全て InnerContent.innerType に移行
                    var content = NspInnerContent[0];
                    Type = content.Type;
                    Version = content.Version;
                    AocIndex = (Type == ContentMetaType.AddOnContent) ? (int)(content.ContentMetaId.Value - ApplicationId.Value - AddOnContentIdOffset) : 0;
                }
                else
                {
                    throw new Exception($"Not found cnmt.xml file. Please check the path \"{Path}\".");
                }
            }
        }

        private void WriteLog(string log)
        {
            if (IsRedirection)
            {
                Log.WriteLine(log);
            }
        }

        private void WriteLogAsIs(string log)
        {
            if (IsRedirection)
            {
                Log.WriteLineAsIs(log);
            }
        }

        private bool ExtractNsp(DirectoryInfo extractDir, string nspPath)
        {
            using (var process = new Process())
            {
                var filename = SdkPath.FindToolPath("AuthoringTool", "AuthoringTool/AuthoringTool.exe", "AuthoringTool/AuthoringTool.exe").FullName;
                var arguments = $"extract {nspPath} --targetregex \"(.*\\.cnmt.xml|.*\\.tik)$\" -o {extractDir.FullName}";
                process.StartInfo.FileName = filename;
                process.StartInfo.Arguments = arguments;
                process.StartInfo.CreateNoWindow = true;
                process.StartInfo.UseShellExecute = false;
                process.Start();
                process.WaitForExit();

                if (process.ExitCode != 0)
                {
                    return false;
                }
                return true;
            }
        }

        public bool IsUnregistrable()
        {
            foreach (var content in NspInnerContent)
            {
                if (Id64.Unregistrable.Contains(content.ContentMetaId))
                {
                    return true;
                }
            }
            return false;
        }

        public override string ToString()
        {
            return $"{ApplicationId},{Version},{Type},{Priority}";
        }

        private static int CompareForRegister(NspAccessor lhs, NspAccessor rhs)
        {
            return ToolUtility.Compare(lhs, rhs,
                (l, r) =>
                {
                    int result = 0;
                    if (false) { } // 単なる位置合わせ
                    else if (ToolUtility.Compare(out result, l.Priority, r.Priority)) { }
                    else if (ToolUtility.Compare(out result, l.Version, r.Version)) { }
                    else if (ToolUtility.Compare(out result, l.ApplicationId, r.ApplicationId)) { }
                    else if (ToolUtility.Compare(out result, l.LastWriteDate, r.LastWriteDate)) { }
                    return result;
                });
        }

        private static int CompareForParallelUpload(NspAccessor lhs, NspAccessor rhs)
        {
            return ToolUtility.Compare(lhs, rhs,
                (l, r) =>
                {
                    int result = 0;
                    if (false) { } // 単なる位置合わせ
                    else if (ToolUtility.Compare(out result, l.ApplicationId, r.ApplicationId)) { }
                    else if (ToolUtility.Compare(out result, l.Priority, r.Priority)) { }
                    else if (ToolUtility.Compare(out result, l.Version, r.Version)) { }
                    else if (ToolUtility.Compare(out result, l.LastWriteDate, r.LastWriteDate)) { }
                    return result;
                });
        }

        public static List<NspAccessor> ReadDirectory(string dirPath, bool verboseMode)
        {
            var nspList = new List<NspAccessor>();
            foreach (var nspPath in System.IO.Directory.GetFiles(dirPath, "*.nsp"))
            {
                nspList.Add(new NspAccessor(nspPath, verboseMode));
            }
            nspList.Sort(NspAccessor.CompareForRegister);
            return nspList;
        }

        public static List<NspAccessor> SortForRegister(List<NspAccessor> nspList)
        {
            var dst = new List<NspAccessor>(nspList);
            dst.Sort(NspAccessor.CompareForParallelUpload);
            return dst;
        }

        public static List<List<NspAccessor>> SortForParallelUpload(List<NspAccessor> nspList)
        {
            // 戻り値はリストを内包するリストで、直列リスト< 並列リスト > という形
            // 同一アプリのコンテンツは直列リスト、それ以外は並列リストに分ける
            // 同一アプリでも並列可能なコンテンツはあるが判定が複雑になるので簡易なものにする
            var dst = new List<List<NspAccessor>>();
            var src = new List<NspAccessor>(nspList);
            src.Sort(NspAccessor.CompareForParallelUpload);

            var id = src[0].ApplicationId;
            var phase = 0;
            foreach (var accessor in src)
            {
                if (id != accessor.ApplicationId)
                {
                    id = accessor.ApplicationId;
                    phase = 0;
                }
                else if (dst.Count == phase)
                {
                    dst.Add(new List<NspAccessor>());
                }
                dst[phase++].Add(accessor);
            }
            return dst;
        }
    }
}
