﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <msclr/marshal.h>
#include <nn/nn_Allocator.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fs/fs_FileStorage.h>
#include <nn/fssystem/fs_PartitionFileSystem.h>

#include "../Util/DeclareAlive.h"
#include "FileSystemArchiveReaderImplBase.h"
#include "Sha256PartitionFileSystemArchiveReader.h"
#include "ManagedStreamStorage.h"

namespace Nintendo { namespace Authoring { namespace FileSystemMetaLibrary {

using namespace System;
using namespace System::IO;
using namespace System::Collections;
using namespace System::Collections::Generic;
using namespace System::Runtime::InteropServices;

using namespace nn;
using namespace nn::fs;

    private class DefaultMemoryResource : public nn::MemoryResource
    {
    protected:
        virtual void* do_allocate(std::size_t bytes, std::size_t alignment) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(alignment);
            return std::malloc(bytes);
        }
        virtual void do_deallocate(void* p, std::size_t bytes, std::size_t alignment) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(bytes);
            NN_UNUSED(alignment);
            return std::free(p);
        }
        virtual bool do_is_equal(const nn::MemoryResource& other) const NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(other);
            return false;
        }
    };

    private class Sha256PartitionFileSystemArchiveReaderImpl : public FileSystemArchiveReaderImplBase
    {
    public:
        explicit Sha256PartitionFileSystemArchiveReaderImpl(std::shared_ptr<fs::IStorage> storage, array<Byte>^ hash) :
              FileSystemArchiveReaderImplBase(std::move(storage)),
              m_FileSystem(std::make_shared<Sha256PartitionFileSystem>()),
              m_Allocator(new DefaultMemoryResource())
        {
            {
                pin_ptr<unsigned char> pinHash = &hash[0];
                std::unique_ptr<Sha256PartitionFileSystemMeta> meta(new Sha256PartitionFileSystemMeta());
                auto result = meta->Initialize(m_Storage.get(), m_Allocator, pinHash, hash->Length);
                if(result.IsFailure())
                {
                    throw gcnew ArgumentException(String::Format("Failed to Initialize Sha256PartitionFileSystemMeta 0x{0:X8}.", result.GetInnerValueForDebug()));
                }
            }
            auto result = m_FileSystem->Initialize(m_Storage);
            if(result.IsFailure())
            {
                throw gcnew ArgumentException(String::Format("Failed to Initialize Sha256PartitionFileSystem 0x{0:X8}.", result.GetInnerValueForDebug()));
            }
            FileSystemArchiveReaderImplBase::Initialize(m_FileSystem.get());
        }

        explicit Sha256PartitionFileSystemArchiveReaderImpl(std::shared_ptr<fs::IStorage> storage) :
            FileSystemArchiveReaderImplBase(std::move(storage)),
            m_FileSystem(std::make_shared<Sha256PartitionFileSystem>())
        {
            auto result = m_FileSystem->Initialize(m_Storage);
            if(result.IsFailure())
            {
                throw gcnew ArgumentException(String::Format("Failed to Initialize Sha256PartitionFileSystem 0x{0:X8}.", result.GetInnerValueForDebug()));
            }
            FileSystemArchiveReaderImplBase::Initialize(m_FileSystem.get());
        }

        List<Tuple<Int64, Int64>^>^ GetFileFragmentList(String^ fileName)
        {
            auto name = static_cast<char*>(GetUtf8CharsFromString(fileName).ToPointer());
            NN_UTIL_SCOPE_EXIT{ Marshal::FreeHGlobal(static_cast<IntPtr>(name)); };

            // partitionFs は断片化しない
            int64_t offset;
            auto result = m_FileSystem->GetFileBaseOffset(&offset, (std::string("/") + name).c_str());
            if (result.IsFailure())
            {
                throw gcnew ArgumentException(String::Format("Failed to OpenFileStorage 0x{0:X8}.", result.GetInnerValueForDebug()));
            }

            auto fragmentList = gcnew List<Tuple<Int64, Int64>^>();
            fragmentList->Add(gcnew Tuple<Int64, Int64>(offset, this->GetFileSize(fileName)));
            return fragmentList;
        }

        void OpenFileStorage(std::shared_ptr<fs::IStorage>* outValue, String^ fileName)
        {
            auto name = static_cast<char*>(GetUtf8CharsFromString(fileName).ToPointer());
            NN_UTIL_SCOPE_EXIT{ Marshal::FreeHGlobal(static_cast<IntPtr>(name)); };

            std::shared_ptr<FileStorageBasedFileSystem> fileStorage = std::make_shared<FileStorageBasedFileSystem>();
            auto copyFileSystem = m_FileSystem;
            auto result = fileStorage->Initialize(std::move(copyFileSystem), (std::string("/") + name).c_str(), nn::fs::OpenMode_Read);
            if (result.IsFailure())
            {
                throw gcnew ArgumentException(String::Format("Failed to OpenFileStorage 0x{0:X8}.", result.GetInnerValueForDebug()));
            }

            *outValue = std::move(fileStorage);
        }

    private:
        std::shared_ptr<Sha256PartitionFileSystem> m_FileSystem;
        nn::MemoryResource* m_Allocator;
    };

    Sha256PartitionFileSystemArchiveReader::Sha256PartitionFileSystemArchiveReader(Stream^ stream, array<Byte>^ hash)
    {
        auto storage = std::shared_ptr<fs::IStorage>(new ManagedStreamStorage(stream));
        m_Impl = new Sha256PartitionFileSystemArchiveReaderImpl(storage, hash);
        GC::KeepAlive(this);
    }

    Sha256PartitionFileSystemArchiveReader::Sha256PartitionFileSystemArchiveReader(Stream^ stream)
    {
        auto storage = std::shared_ptr<fs::IStorage>(new ManagedStreamStorage(stream));
        m_Impl = new Sha256PartitionFileSystemArchiveReaderImpl(storage);
        GC::KeepAlive(this);
    }

    Sha256PartitionFileSystemArchiveReader::~Sha256PartitionFileSystemArchiveReader()
    {
        this->!Sha256PartitionFileSystemArchiveReader();
    }

    Sha256PartitionFileSystemArchiveReader::!Sha256PartitionFileSystemArchiveReader()
    {
        delete m_Impl;
    }

    List<Tuple<String^, Int64>^>^ Sha256PartitionFileSystemArchiveReader::ListFileInfo()
    {
        return Util::ReturnAndDeclareAlive(this, m_Impl->ListFileInfo());
    }

    List<Tuple<String^, Int64>^>^ Sha256PartitionFileSystemArchiveReader::ListFileInfo(String^ rootPath)
    {
        return Util::ReturnAndDeclareAlive(this, m_Impl->ListFileInfo(rootPath));
    }

    List<Tuple<Int64, Int64>^>^ Sha256PartitionFileSystemArchiveReader::GetFileFragmentList(String^ fileName)
    {
        return Util::ReturnAndDeclareAlive(this, m_Impl->GetFileFragmentList(fileName));
    }

    int64_t Sha256PartitionFileSystemArchiveReader::GetFileSize(String^ fileName)
    {
        return Util::ReturnAndDeclareAlive(this, m_Impl->GetFileSize(fileName));
    }

    array<byte>^ Sha256PartitionFileSystemArchiveReader::ReadFile(String^ fileName, int64_t offset, int64_t size)
    {
        return Util::ReturnAndDeclareAlive(this, m_Impl->ReadFile(fileName, offset, size));
    }

    int64_t Sha256PartitionFileSystemArchiveReader::GetBaseSize()
    {
        return Util::ReturnAndDeclareAlive(this, m_Impl->GetBaseSize());
    }

    array<byte>^ Sha256PartitionFileSystemArchiveReader::ReadBase(int64_t offset, int64_t size)
    {
        return Util::ReturnAndDeclareAlive(this, m_Impl->ReadBase(offset, size));
    }

    Sha256PartitionFileSystemArchiveReader::Sha256PartitionFileSystemArchiveReader(std::shared_ptr<fs::IStorage> storage)
    {
        m_Impl = new Sha256PartitionFileSystemArchiveReaderImpl(storage);
        GC::KeepAlive(this);
    }

    void Sha256PartitionFileSystemArchiveReader::OpenFileStorage(std::shared_ptr<fs::IStorage>* outValue, String^ fileName)
    {
        m_Impl->OpenFileStorage(outValue, fileName);
        GC::KeepAlive(this);
    }

}}}
