﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/detail/nn_Version-spec.NX.h>
#include <msclr/marshal.h>
#include <vector>

#include "../Util/DeclareAlive.h"
#include "NintendoContentFileSystemMeta.h"

namespace Nintendo { namespace Authoring { namespace FileSystemMetaLibrary {

using namespace System;
using namespace System::Collections;
using namespace System::Collections::Generic;
using namespace System::Runtime::InteropServices;
using namespace msclr::interop;
using namespace nn::fssystem;
using namespace nn::util;

    NintendoContentFileSystemInfo::NintendoContentFileSystemInfo()
    {
        existentFsIndices = gcnew List<Int32>();
        fsEntries = gcnew List<EntryInfo>();
        encryptedKey = gcnew array<Byte>(NcaHeader::EncryptedKeyAreaSize);
        isProdEncryption = false;
        distributionType = static_cast<Byte>(NcaHeader::DistributionType::Download);
        GC::KeepAlive(this);
    }

    void NintendoContentFileSystemInfo::GenerateExistentFsIndicesFromFsEntries()
    {
        existentFsIndices = gcnew List<Int32>(Linq::Enumerable::Select(fsEntries, gcnew Func<EntryInfo, Int32>(SelectFsIndex)));
    }

    Byte NintendoContentFileSystemInfo::GetRepresentProgramIdOffset()
    {
        if (fsEntries->Count == 0)
        {
            throw gcnew InvalidOperationException("No entry.");
        }

        Nullable<byte> programIdOffset;
        for (int i = 0; i < fsEntries->Count; i++)
        {
            if (fsEntries[i].encryptionType != (byte)NcaFsHeader::EncryptionType::AesCtr &&
                fsEntries[i].encryptionType != (byte)NcaFsHeader::EncryptionType::AesCtrEx)
            {
                continue;
            }
            auto tmpProgramIdOffset = EntryInfo::ConvertSecureValueToIdOffset(fsEntries[i].secureValue);
            System::Diagnostics::Trace::Assert(!programIdOffset.HasValue || programIdOffset.Value == tmpProgramIdOffset);
            programIdOffset = tmpProgramIdOffset;
        }
        if (!programIdOffset.HasValue)
        {
            return (byte)0;
        }
        return programIdOffset.Value;
    }

    Int32 NintendoContentFileSystemInfo::SelectFsIndex(EntryInfo entryInfo)
    {
        return entryInfo.partitionIndex;
    }

    NcaFsHeader::FsType NintendoContentFileSystemMeta::ConvertFormatTypeToFsType(String^ formatType)
    {
        if (String::Compare(formatType, NintendoContentFileSystemMetaConstant::FormatTypeRomFs, false) == 0 ||
            String::Compare(formatType, NintendoContentFileSystemMetaConstant::FormatTypePatchableRomFs, false) == 0)
        {
            return NcaFsHeader::FsType::RomFs;
        }
        else if (String::Compare(formatType, NintendoContentFileSystemMetaConstant::FormatTypePartitionFs, false) == 0 ||
                 String::Compare(formatType, NintendoContentFileSystemMetaConstant::FormatTypePatchablePartitionFs, false) == 0)
        {
            return NcaFsHeader::FsType::PartitionFs;
        }
        else
        {
            throw gcnew ArgumentException("Invalid format type.");
        }
    }

    Int32 NintendoContentFileSystemMeta::GetHashBlockSize(Int64 targetSize)
    {
        const Int32 sizeMin = 4 * 1024;
        const Int32 sizeMax = 1024 * 1024;

        Int32 size = sizeMin;
        const int HashSize = 32;

        while( true )
        {
            if( static_cast<Int64>(size) * size / HashSize >= targetSize || size > sizeMax )
            {
                break;
            }
            size *= 2;
        }

        if( size > sizeMax )
        {
            throw gcnew ArgumentException("Total size of input files is too big.");
        }

        return size;
    }

    array<Byte>^ NintendoContentFileSystemMeta::CreateNcaHeader(NintendoContentFileSystemInfo^ fileSystemInfo)
    {
        std::unique_ptr<NcaHeader> m_pNcaHeader(new NcaHeader());
        std::memset(m_pNcaHeader.get(), 0, sizeof(NcaHeader));

        m_pNcaHeader->signature = NcaHeader::Signature;
        m_pNcaHeader->distributionType = static_cast<NcaHeader::DistributionType>(fileSystemInfo->distributionType);
        m_pNcaHeader->contentType = static_cast<NcaHeader::ContentType>(fileSystemInfo->contentType);
        // W/A: SIGLO-67301
        if (fileSystemInfo->keyGeneration > 2)
        {
            m_pNcaHeader->keyGeneration = 2;
            m_pNcaHeader->keyGeneration2 = fileSystemInfo->keyGeneration;
        }
        else
        {
            m_pNcaHeader->keyGeneration = fileSystemInfo->keyGeneration;
            m_pNcaHeader->keyGeneration2 = 0;
        }

        m_pNcaHeader->keyIndex = fileSystemInfo->keyAreaEncryptionKeyIndex;

        m_pNcaHeader->contentSize = fileSystemInfo->contentSize;
        m_pNcaHeader->programId = fileSystemInfo->programId;
        m_pNcaHeader->contentIndex = fileSystemInfo->contentIndex;
        if (fileSystemInfo->sdkAddonVersion != 0)
        {
            m_pNcaHeader->sdkAddonVersion = fileSystemInfo->sdkAddonVersion;
        }
        else
        {
            m_pNcaHeader->sdkAddonVersion = NN_NX_ADDON_CURRENT_VERSION_NUMBER;
        }

        for (int i = 0; i < fileSystemInfo->fsEntries->Count; i++)
        {
            auto fsEntry = fileSystemInfo->fsEntries[i];
            int index = fsEntry.partitionIndex;
            System::Diagnostics::Debug::Assert(index < NcaHeader::FsCountMax);
            System::Diagnostics::Debug::Assert(m_pNcaHeader->fsInfo[index].endSector == 0);

            m_pNcaHeader->fsInfo[index].startSector = NcaHeader::ByteToSector(fsEntry.startOffset);
            m_pNcaHeader->fsInfo[index].endSector = NcaHeader::ByteToSector(fsEntry.endOffset);
            m_pNcaHeader->fsInfo[index].hashTargetSize = NcaHeader::ByteToSector(FsHeaderSize);
        }

        {
            pin_ptr<unsigned char> ptr = &fileSystemInfo->encryptedKey[0];
            memcpy(m_pNcaHeader->encryptedKey, ptr, NcaHeader::EncryptedKeyAreaSize);
            ptr = nullptr;
        }

        if (fileSystemInfo->rightsId != nullptr)
        {
            System::Diagnostics::Debug::Assert(fileSystemInfo->rightsId->Length == NcaHeader::RightIdSize);
            pin_ptr<unsigned char> ptr = &fileSystemInfo->rightsId[0];
            memcpy(m_pNcaHeader->rightsId, ptr, NcaHeader::RightIdSize);
            ptr = nullptr;
        }

        array<unsigned char>^ buf = gcnew array<unsigned char>(m_pNcaHeader->Size);
        {
            pin_ptr<unsigned char> ptr = &buf[0];
            memcpy(ptr, m_pNcaHeader.get(), m_pNcaHeader->Size);
            ptr = nullptr;
        }

        return buf;
    }

    array<Byte>^ NintendoContentFileSystemMeta::CreateNcaFsHeader(NintendoContentFileSystemInfo::EntryInfo^ fsEntry)
    {
        std::unique_ptr<NcaFsHeader> m_pNcaFsHeader(new NcaFsHeader());
        std::memset(m_pNcaFsHeader.get(), 0, sizeof(NcaFsHeader));

        m_pNcaFsHeader->version = fsEntry->version;
        m_pNcaFsHeader->fsType = ConvertFormatTypeToFsType(fsEntry->formatType);
        m_pNcaFsHeader->encryptionType = static_cast<NcaFsHeader::EncryptionType>(fsEntry->encryptionType);
        m_pNcaFsHeader->hashType = static_cast<NcaFsHeader::HashType>(fsEntry->hashType);
        if (m_pNcaFsHeader->hashType == NcaFsHeader::HashType::HierarchicalIntegrityHash)
        {
            System::Diagnostics::Debug::Assert(m_pNcaFsHeader->fsType == NcaFsHeader::FsType::RomFs);
            auto hashInfo = (NintendoContentFileSystemInfo::HierarchicalIntegrityHashInfo^)fsEntry->hashInfo;
            auto tmpMetaInfo = hashInfo->metaInfo;
            pin_ptr<unsigned char> metaInfoPtr = &tmpMetaInfo[0];
            NcaFsHeader::HashData::IntegrityMetaInfo* meta = (NcaFsHeader::HashData::IntegrityMetaInfo*)metaInfoPtr;
            m_pNcaFsHeader->hashData.integrityMetaInfo.magic = meta->magic;
            m_pNcaFsHeader->hashData.integrityMetaInfo.version = meta->version;
            m_pNcaFsHeader->hashData.integrityMetaInfo.sizeMasterHash = meta->sizeMasterHash;

            m_pNcaFsHeader->hashData.integrityMetaInfo.infoLevelHash.maxLayers = meta->infoLevelHash.maxLayers;
            for (auto i = 0; i < NcaFsHeader::HashData::IntegrityMetaInfo::InfoLevelHash::HierarchicalIntegrityVerificationLevelInformation::IntegrityMaxLayerCount; ++i)
            {
                m_pNcaFsHeader->hashData.integrityMetaInfo.infoLevelHash.info[i] = meta->infoLevelHash.info[i];
            }
            m_pNcaFsHeader->hashData.integrityMetaInfo.infoLevelHash.seed = meta->infoLevelHash.seed;

            metaInfoPtr = nullptr;
        }
        else if (m_pNcaFsHeader->hashType == NcaFsHeader::HashType::HierarchicalSha256Hash)
        {
            auto hashInfo = (NintendoContentFileSystemInfo::HierarchicalSha256HashInfo^)fsEntry->hashInfo;
            m_pNcaFsHeader->hashData.hierarchicalSha256Data.hashBlockSize = GetHashBlockSize(hashInfo->hashTargetSize);
            m_pNcaFsHeader->hashData.hierarchicalSha256Data.hashLayerCount = 2;
            m_pNcaFsHeader->hashData.hierarchicalSha256Data.hashLayerRegion[0].offset = 0;
            m_pNcaFsHeader->hashData.hierarchicalSha256Data.hashLayerRegion[0].size = hashInfo->hashDataSize;
            m_pNcaFsHeader->hashData.hierarchicalSha256Data.hashLayerRegion[1].offset = hashInfo->hashTargetOffset;
            m_pNcaFsHeader->hashData.hierarchicalSha256Data.hashLayerRegion[1].size = hashInfo->hashTargetSize;
        }
        else
        {
            throw gcnew ArgumentException();
        }
        m_pNcaFsHeader->aesCtrUpperIv.part.secureValue = fsEntry->secureValue;
        m_pNcaFsHeader->aesCtrUpperIv.part.generation = fsEntry->generation;
        array<unsigned char>^ buf = gcnew array<unsigned char>(m_pNcaFsHeader->Size);
        {
            pin_ptr<unsigned char> ptr = &buf[0];
            memcpy(ptr, m_pNcaFsHeader.get(), m_pNcaFsHeader->Size);
            ptr = nullptr;
        }

        return buf;
    }

    array<Byte>^ NintendoContentFileSystemMeta::UpdatePatchInfo(array<Byte>^ header, NintendoContentArchivePatchInfo^ patchInfo, UInt32 generation, UInt32 secureValue)
    {
        if( header->GetLength(0) != sizeof(NcaFsHeader) )
        {
            throw gcnew ArgumentException();
        }

        std::unique_ptr<NcaFsHeader> pNcaFsHeader(new NcaFsHeader);
        Marshal::Copy(header, 0, IntPtr(pNcaFsHeader.get()), NcaFsHeader::Size);

        // 暗号方式の変更
        pNcaFsHeader->encryptionType = NcaFsHeader::EncryptionType::AesCtrEx;

        // パッチ情報の変更
        {
            NcaPatchInfo src;
            std::memset(&src, 0, sizeof(src));

            src.indirectOffset = patchInfo->indirectOffset;
            src.indirectSize = patchInfo->indirectSize;
            if( patchInfo->indirectHeader != nullptr )
            {
                Marshal::Copy(patchInfo->indirectHeader, 0, IntPtr(src.indirectHeader), NcaBucketInfo::HeaderSize);
            }

            src.aesCtrExOffset = patchInfo->aesCtrExOffset;
            src.aesCtrExSize = patchInfo->aesCtrExSize;
            if( patchInfo->aesCtrExHeader != nullptr )
            {
                Marshal::Copy(patchInfo->aesCtrExHeader, 0, IntPtr(src.aesCtrExHeader), NcaBucketInfo::HeaderSize);
            }

            nn::util::BytePtr dst(pNcaFsHeader.get());
            dst.Advance(NcaPatchInfo::Offset);

            std::memcpy(dst.Get(), &src, NcaPatchInfo::Size);
        }

        // W/A: Secure Value の注入 TODO: 削除
        pNcaFsHeader->aesCtrUpperIv.part.secureValue = secureValue;

        // 世代番号の変更
        pNcaFsHeader->aesCtrUpperIv.part.generation = generation;

        array<Byte>^ buffer = gcnew array<Byte>(header->GetLength(0));
        Marshal::Copy(IntPtr(pNcaFsHeader.get()), buffer, 0, NcaFsHeader::Size);

        return buffer;
    }

}}}
