﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <string>
#include <msclr/marshal.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/fs/fs_Directory.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fssystem/fs_PartitionFileSystem.h>
#include <nn/fssystem/fs_NcaFileSystemDriver.h>
#include "NintendoContentArchiveReader.h"

namespace Nintendo { namespace Authoring { namespace FileSystemMetaLibrary {

using namespace System;
using namespace System::IO;
using namespace System::Collections;
using namespace System::Collections::Generic;
using namespace System::Runtime::InteropServices;
using namespace System::Text;

using namespace nn;
using namespace nn::fssystem;

    private class FileSystemArchiveReaderImplBase
    {
    public:
        FileSystemArchiveReaderImplBase(std::shared_ptr<fs::IStorage> storage)
            : m_Storage(std::move(storage))
        {
        }

        void Initialize(fs::fsa::IFileSystem* fileSystem)
        {
            m_FileSystemAbstract = fileSystem;
        }

        List<Tuple<String^, Int64>^>^ ListFileInfo()
        {
            return ListFileInfo(String::Empty);
        }

        List<Tuple<String^, Int64>^>^ ListFileInfo(String^ rootPath)
        {
            auto name = static_cast<char*>(GetUtf8CharsFromString(rootPath).ToPointer());
            NN_UTIL_SCOPE_EXIT{ Marshal::FreeHGlobal(static_cast<IntPtr>(name)); };

            auto fileInfoList = gcnew List<Tuple<String^, Int64>^>();

            fs::DirectoryEntry entry = {{0}};
            char pathBuffer[nn::fs::EntryNameLengthMax];
            auto root = std::string("/") + name;
            strncpy_s(pathBuffer, root.c_str(), root.length());
            ListFileInfoRecursive(fileInfoList, &entry, pathBuffer, nn::fs::EntryNameLengthMax);

            return fileInfoList;
        }

        int64_t GetFileSize(String^ fileName)
        {
            auto name = static_cast<char*>(GetUtf8CharsFromString(fileName).ToPointer());
            NN_UTIL_SCOPE_EXIT{ Marshal::FreeHGlobal(static_cast<IntPtr>(name)); };

            std::unique_ptr<fs::fsa::IFile> file;
            auto result = m_FileSystemAbstract->OpenFile(&file, (std::string("/") + name).c_str(), fs::OpenMode_Read);
            if (result.IsFailure())
            {
                throw gcnew ArgumentException(String::Format("Failed to OpenFile {0} 0x{1:X8}.", fileName, result.GetInnerValueForDebug()));
            }

            int64_t fileSize = 0;
            result = file->GetSize(&fileSize);
            if (result.IsFailure())
            {
                throw gcnew ArgumentException(String::Format("Failed to GetSize {0} 0x{1:X8}.", fileName, result.GetInnerValueForDebug()));
            }
            return fileSize;
        }

        array<byte>^ ReadFile(String^ fileName, int64_t offset, int64_t size)
        {
            if (size >= 0x7FFFFFFF)
            {
                throw gcnew InvalidOperationException("Input size is too large.");
            }

            if (size < 0)
            {
                throw gcnew InvalidOperationException("Input size must be positive.");
            }

            auto name = static_cast<char*>(GetUtf8CharsFromString(fileName).ToPointer());
            NN_UTIL_SCOPE_EXIT{ Marshal::FreeHGlobal(static_cast<IntPtr>(name)); };

            std::unique_ptr<fs::fsa::IFile> file;
            auto result = m_FileSystemAbstract->OpenFile(&file, (std::string("/") + name).c_str(), fs::OpenMode_Read);
            if (result.IsFailure())
            {
                throw gcnew ArgumentException(String::Format("Failed to OpenFile {0} 0x{1:X8}.", fileName, result.GetInnerValueForDebug()));
            }

            size_t size32Bit = (size_t)size;

            array<Byte>^ bytes = gcnew array<Byte>(size32Bit);

            if (size == 0)
            {
                return bytes;
            }

            size_t readSize = 0;
            {
                pin_ptr<unsigned char> ptr = &bytes[0];
                result = file->Read(&readSize, offset, ptr, size32Bit, fs::ReadOption::MakeValue(0));
                ptr = nullptr;
            }
            if (result.IsFailure())
            {
                throw gcnew ArgumentException(String::Format("Failed to ReadFile {0} 0x{1:X8}.", fileName, result.GetInnerValueForDebug()));
            }

            if (size32Bit > readSize)
            {
                Array::Resize(bytes, readSize);
            }

            return bytes;
        }

        int64_t GetBaseSize()
        {
            int64_t storageSize = 0;
            auto result = m_Storage->GetSize(&storageSize);
            if (result.IsFailure())
            {
                throw gcnew ArgumentException(String::Format("Failed to GetBaseSize 0x{0:X8}.", result.GetInnerValueForDebug()));
            }
            return storageSize;
        }

        array<byte>^ ReadBase(int64_t offset, int64_t size)
        {
            if (size >= 0x7FFFFFFF)
            {
                throw gcnew InvalidOperationException("Input size is too large.");
            }

            if (size < 0)
            {
                throw gcnew InvalidOperationException("Input size must be positive.");
            }

            size_t size32Bit = (size_t)size;

            array<Byte>^ bytes = gcnew array<Byte>(size32Bit);

            if (size == 0)
            {
                return bytes;
            }

            Result result;
            {
                pin_ptr<unsigned char> ptr = &bytes[0];
                std::memset(ptr, 0, size32Bit);
                result = m_Storage->Read(offset, ptr, size32Bit);
                ptr = nullptr;
            }
            if (result.IsFailure())
            {
                throw gcnew ArgumentException(String::Format("Failed to ReadBase 0x{0:X8}.", result.GetInnerValueForDebug()));
            }

            return bytes;
        }

    protected:
        std::shared_ptr<fs::IStorage> m_Storage;
        fs::fsa::IFileSystem* m_FileSystemAbstract;

        static IntPtr GetUtf8CharsFromString(String^ str)
        {
            auto bytes = Encoding::UTF8->GetBytes(str);
            auto outValue = Marshal::AllocHGlobal(bytes->Length + 1);
            Marshal::Copy(bytes, 0, outValue, bytes->Length);
            Marshal::WriteByte(outValue, bytes->Length, (Byte)0);
            return outValue;
        }

        static String^ GetStringFromUtf8Chars(char* pChar, size_t size)
        {
            auto bytes = gcnew array<Byte>(size);
            pin_ptr<unsigned char> ptr = &bytes[0];
            std::memcpy(ptr, pChar, size);
            ptr = nullptr;
            return Encoding::UTF8->GetString(bytes);
        }

    private:
        void ListFileInfoRecursive(List<Tuple<String^, Int64>^>^ list, fs::DirectoryEntry* entry, char* path, size_t pathSize)
        {
            std::unique_ptr<fs::fsa::IDirectory> directory;
            auto result = m_FileSystemAbstract->OpenDirectory(&directory, path, fs::OpenDirectoryMode_All);
            if (nn::fs::ResultPathNotFound::Includes(result))
            {
                return;
            }
            else if (result.IsFailure())
            {
                throw gcnew ArgumentException(String::Format("Failed to OpenDirectory 0x{0:X8}.", result.GetInnerValueForDebug()));
            }
            for (;;)
            {
                int64_t count = 0;
                result = directory->Read(&count, entry, 1);
                if (result.IsFailure())
                {
                    throw gcnew ArgumentException(String::Format("Failed to ReadDirectory 0x{0:X8}.", result.GetInnerValueForDebug()));
                }
                if (count == 0)
                {
                    break;
                }

                auto parentPathSize = strnlen_s(path, pathSize);
                strncat_s(path, pathSize, entry->name, pathSize - parentPathSize);

                if (entry->directoryEntryType == fs::DirectoryEntryType_Directory)
                {
                    strncat_s(path, pathSize, "/", pathSize - parentPathSize - strnlen_s(entry->name, pathSize));
                    ListFileInfoRecursive(list, entry, path, pathSize);
                }
                else
                {
                    auto str = GetStringFromUtf8Chars(path, parentPathSize + strnlen_s(entry->name, pathSize));
                    list->Add(Tuple::Create(str->TrimStart('/'), entry->fileSize));
                }
                path[parentPathSize] = '\0';
            }
        }

    };

}}}
