﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/crypto/crypto_RsaPssSha256Signer.h>
#include <nn/crypto/crypto_RsaPssSha256Verifier.h>
#include "Rsa2048PssSha256SignCryptoDriver.h"

namespace Nintendo { namespace Authoring { namespace CryptoLibrary {

using namespace System;

    Rsa2048PssSha256SignCryptoDriver::Rsa2048PssSha256SignCryptoDriver(array<Byte>^ modulus, array<Byte>^ publicExponent, array<Byte>^ privateExponent)
    {
        m_KeyModulus = modulus;
        m_KeyPublicExponent  = publicExponent;
        m_KeyPrivateExponent = privateExponent;
        GC::KeepAlive(this);
    }

    Rsa2048PssSha256SignCryptoDriver::Rsa2048PssSha256SignCryptoDriver(Rsa2048PssSha256KeyIndex keyIndex)
    {
        switch (keyIndex)
        {
        case Rsa2048PssSha256KeyIndex::NcaHeader1:
            m_KeyModulus = NcaHeader1SignKeyModulus;
            m_KeyPublicExponent  = NcaHeader1SignKeyPublicExponent;
            m_KeyPrivateExponent = NcaHeader1SignKeyPrivateExponent;
            break;
        case Rsa2048PssSha256KeyIndex::Acid:
            m_KeyModulus = AcidSignKeyModulus;
            m_KeyPublicExponent  = AcidSignKeyPublicExponent;
            m_KeyPrivateExponent = nullptr;
            break;
        case Rsa2048PssSha256KeyIndex::NrrCertificate:
            m_KeyModulus = NrrCertificateSignKeyModulus;
            m_KeyPublicExponent  = NrrCertificateSignKeyPublicExponent;
            m_KeyPrivateExponent = NrrCertificateSignKeyPrivateExponent;
            break;
        default:
            throw gcnew ArgumentException("Invalid KeyIndex.");
        }
    }

    Rsa2048PssSha256SignCryptoDriver::~Rsa2048PssSha256SignCryptoDriver()
    {
        this->!Rsa2048PssSha256SignCryptoDriver();
    }

    Rsa2048PssSha256SignCryptoDriver::!Rsa2048PssSha256SignCryptoDriver()
    {
    }

    array<Byte>^ Rsa2048PssSha256SignCryptoDriver::SignBlock(array<Byte>^ src, int srcOffset, int srcSize)
    {
        array<unsigned char>^ dst = gcnew array<unsigned char>(KeySize);

        pin_ptr<unsigned char> pinSrc = &src[srcOffset];
        pin_ptr<unsigned char> pinDst = &dst[0];
        pin_ptr<unsigned char> pinKeyModulus = &m_KeyModulus[0];
        pin_ptr<unsigned char> pinKeyPrivateExponent = &m_KeyPrivateExponent[0];
        array<unsigned char>^ salt = gcnew array<unsigned char>(nn::crypto::Rsa2048PssSha256Signer::DefaultSaltSize);
        pin_ptr<unsigned char> pinSalt = &salt[0]; // 開発版では salt を 0 に固定

        nn::crypto::SignRsa2048PssSha256(pinDst, KeySize,
                                         pinKeyModulus, m_KeyModulus->Length,
                                         pinKeyPrivateExponent, m_KeyPrivateExponent->Length,
                                         pinSrc, srcSize,
                                         pinSalt, salt->Length);

        pinKeyModulus = nullptr;
        pinKeyPrivateExponent = nullptr;
        pinSalt = nullptr;
        pinSrc = nullptr;
        pinDst = nullptr;

        GC::KeepAlive(this);
        return dst;
    }

    bool Rsa2048PssSha256SignCryptoDriver::VerifyBlock(array<Byte>^ src, int srcOffset, int srcSize, array<Byte>^ signature)
    {
        bool result;
        pin_ptr<unsigned char> pinSrc = &src[srcOffset];
        pin_ptr<unsigned char> pinSignature = &signature[0];

        pin_ptr<unsigned char> pinKeyModulus = &m_KeyModulus[0];
        pin_ptr<unsigned char> pinKeyPublicExponent = &m_KeyPublicExponent[0];

        result = nn::crypto::VerifyRsa2048PssSha256(pinSignature, KeySize,
                                                    pinKeyModulus, m_KeyModulus->Length,
                                                    pinKeyPublicExponent, m_KeyPublicExponent->Length,
                                                    pinSrc, srcSize);

        pinKeyModulus = nullptr;
        pinKeyPublicExponent = nullptr;
        pinSrc = nullptr;
        pinSignature = nullptr;

        GC::KeepAlive(this);
        return result;
    }

    array<Byte>^ Rsa2048PssSha256SignCryptoDriver::GetKeyModulus()
    {
        return m_KeyModulus;
    }

    array<Byte>^ Rsa2048PssSha256SignCryptoDriver::GetKeyPublicExponent()
    {
        return m_KeyPublicExponent;
    }

}}}
