﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <msclr/lock.h>
#include <nn/util/util_ScopeExit.h>

#include "../Util/DeclareAlive.h"
#include "HsmInterfaceManager.h"

namespace Nintendo { namespace Authoring { namespace CryptoLibrary {

using namespace System;
using namespace System::IO;
using namespace System::Reflection;
using namespace System::Runtime::CompilerServices;
using namespace System::Text;

#define CK_USER_TYPE_CRYPTO_OFFICER (static_cast<::CK_USER_TYPE>(1))
#define CK_EXCEPTION_THROW_UNLESS(condition, ...) \
{ \
    if (!(condition)) \
    { \
        throw gcnew InvalidOperationException(String::Format("{0} -> {1:x}", ##__VA_ARGS__)); \
    } \
}

    namespace {
        const ::CK_FLAGS SessionFlags = CKF_SERIAL_SESSION | CKF_RW_SESSION;
        const ::CK_USER_TYPE LoginType = CK_USER_TYPE_CRYPTO_OFFICER;

        const ::CK_ULONG Rsa2048PssSha256SaltSize = 32; // HashSize (Byte)

        ::std::shared_ptr<HsmInterfaceManager> g_InterfaceManager;
    }

    HsmInterfaceManager::HsmInterfaceManager()
    {
        Initialize();
    }

    ::std::shared_ptr<HsmInterfaceManager> HsmInterfaceManager::GetInstance()
    {
        if (!g_InterfaceManager)
        {
            g_InterfaceManager = ::std::shared_ptr<HsmInterfaceManager>(new HsmInterfaceManager());
        }
        return g_InterfaceManager;
    }

    void HsmInterfaceManager::Initialize()
    {
        m_LibHandle = NULL;
        m_P11Functions = NULL;
        m_SlotIdForProd = CK_INVALID_HANDLE;
        m_SlotIdForDev = CK_INVALID_HANDLE;
        m_UseDev = false;

        LoadModule();
        InitializeCryptoki();
        SetSlotId();
        SetPin();
    }

    void HsmInterfaceManager::LoadModule()
    {
        auto cryptokiPath = Environment::GetEnvironmentVariable("ChrystokiConfigurationPath");
        if (cryptokiPath == nullptr)
        {
            throw gcnew ArgumentException("Luna client must be installed.");
        }
        pin_ptr<wchar_t> pinCryptokiPath = &((cryptokiPath + "\\win32\\cryptoki.dll")->ToCharArray())[0];
        m_LibHandle = ::LoadLibrary(pinCryptokiPath);
        pinCryptokiPath = nullptr;
        CK_EXCEPTION_THROW_UNLESS(m_LibHandle != NULL, "Failed LoadLibrary()", reinterpret_cast<int>(m_LibHandle));
    }

    void HsmInterfaceManager::UnloadModule()
    {
        if (m_LibHandle != NULL)
        {
            ::FreeLibrary(m_LibHandle);
            m_LibHandle = NULL;
        }
    }

    void HsmInterfaceManager::InitializeCryptoki()
    {
        ::CK_C_GetFunctionList C_GetFunctionList = (::CK_C_GetFunctionList)::GetProcAddress(m_LibHandle, "C_GetFunctionList");
        CK_EXCEPTION_THROW_UNLESS(C_GetFunctionList != NULL, "Failed GetProcAddress()", reinterpret_cast<uintptr_t>(C_GetFunctionList));

        ::CK_FUNCTION_LIST* p11Functions;

        ::CK_RV rv = CKR_TOKEN_NOT_PRESENT;
        rv = C_GetFunctionList(&p11Functions);
        CK_EXCEPTION_THROW_UNLESS(rv == CKR_OK && p11Functions != nullptr, "Failed C_GetFunctionList()", rv)

        m_P11Functions = p11Functions;

        ::CK_ULONG usStatus = m_P11Functions->C_Initialize(NULL);
        CK_EXCEPTION_THROW_UNLESS(usStatus == CKR_OK, "Failed C_Initialize()", usStatus);
    }

    void HsmInterfaceManager::FinalizeCryptoki()
    {
        if (m_P11Functions != NULL)
        {
            m_P11Functions->C_Finalize(NULL);
            m_P11Functions = NULL;
        }
    }

    void HsmInterfaceManager::SetSlotId()
    {
        ::CK_ULONG usStatus = CKR_OK;
        ::CK_ULONG usCount = 0;
        usStatus = m_P11Functions->C_GetSlotList(FALSE, NULL, &usCount);
        CK_EXCEPTION_THROW_UNLESS(usStatus == CKR_OK, "Failed C_GetSlotList()", usStatus);

        std::unique_ptr<::CK_SLOT_ID[]> slots(new ::CK_SLOT_ID[usCount]);
        usStatus = m_P11Functions->C_GetSlotList(FALSE, slots.get(), &usCount);
        CK_EXCEPTION_THROW_UNLESS(usStatus == CKR_OK, "Failed C_GetSlotList()", usStatus);

        for (int i = 0; i < static_cast<int>(usCount); ++i)
        {
            ::CK_SLOT_INFO info;
            usStatus = m_P11Functions->C_GetSlotInfo(slots[i], &info);
            CK_EXCEPTION_THROW_UNLESS(usStatus == CKR_OK, "Failed C_GetSlotInfo()", usStatus);

            if (info.flags & CKF_TOKEN_PRESENT)
            {
                // Prod: TokenInfo.label が cloudhsm-prod であるものを選択
                // Dev: TokenInfo.label が devel-D4C であるものを選択
                static const char ProdSlotLabel[] = "cloudhsm-prod                   ";
                static const char DevSlotLabel[]  = "devel-D4C                       ";

                ::CK_TOKEN_INFO tokenInfo;
                usStatus = m_P11Functions->C_GetTokenInfo(slots[i], &tokenInfo);
                CK_EXCEPTION_THROW_UNLESS(usStatus == CKR_OK, "Failed C_GetTokenInfo()", usStatus);

                if (strncmp(reinterpret_cast<char*>(tokenInfo.label), ProdSlotLabel, sizeof(tokenInfo.label)) == 0)
                {
                    m_SlotIdForProd = slots[i];
                }
                else if (strncmp(reinterpret_cast<char*>(tokenInfo.label), DevSlotLabel, sizeof(tokenInfo.label)) == 0)
                {
                    m_SlotIdForDev = slots[i];
                }
            }
        }

        CK_EXCEPTION_THROW_UNLESS((m_SlotIdForProd != CK_INVALID_HANDLE) || (m_SlotIdForDev != CK_INVALID_HANDLE), "No valid slot ID.", m_SlotIdForProd);
    }

    array<Byte>^ HsmInterfaceManager::GetPinFromEnv(String^ env)
    {
        auto pinStr = Environment::GetEnvironmentVariable(env);
        if (pinStr == nullptr)
        {
            return nullptr;
        }

        try
        {
            auto sr = gcnew StreamReader(pinStr);
            return Encoding::UTF8->GetBytes(sr->ReadToEnd()->TrimEnd());
        }
        catch (Exception^)
        {
            return nullptr;
        }
    }

    void HsmInterfaceManager::SetPin()
    {
        array<unsigned char>^ pinUtf8StrBytes;

        {
            pinUtf8StrBytes = GetPinFromEnv("NN_NINTENDO_AUTHORING_TOOL_DEV_PIN_PATH");
            if (pinUtf8StrBytes == nullptr)
            {
                pinUtf8StrBytes = GetPinFromEnv("NN_NINTENDO_AUTHORING_TOOL_PROD_PIN_PATH");
                if (pinUtf8StrBytes == nullptr)
                {
                    throw gcnew ArgumentException("Please set env \"NN_NINTENDO_AUTHORING_TOOL_PROD_PIN_PATH\" properly.");
                }
            }
            else
            {
                m_UseDev = true;
                Console::WriteLine("[Warning] Dev hsm server will be used for prod-encryption.");
            }
        }

        m_PinLength = pinUtf8StrBytes->Length;
        m_Pin.reset(new ::CK_UTF8CHAR[m_PinLength]);

        pin_ptr<::CK_UTF8CHAR> pin = &pinUtf8StrBytes[0];
        ::std::memcpy(m_Pin.get(), pin, m_PinLength);
        pin = nullptr;
    }

    HsmInterfaceImpl::HsmInterfaceImpl(::std::shared_ptr<HsmInterfaceManager> pManager)
        : m_P11Functions(pManager->GetFunctionList()),
          m_Manager(::std::move(pManager)),
          m_SessionHandle(CK_INVALID_HANDLE),
          m_Pin(NULL),
          m_PinLength(0)
    {
        ::CK_ULONG usStatus = m_P11Functions->C_OpenSession(m_Manager->GetSlotId(), SessionFlags, reinterpret_cast<void *>("Application"), 0, &m_SessionHandle);
        CK_EXCEPTION_THROW_UNLESS(usStatus == CKR_OK, "Failed C_OpenSession()", usStatus);
        m_Pin = m_Manager->GetPin();
        m_PinLength = m_Manager->GetPinLength();
    }

    HsmInterfaceImpl::~HsmInterfaceImpl()
    {
        m_P11Functions->C_CloseSession(m_SessionHandle);
    }

    HsmInterfaceImpl::ScopedLogin::ScopedLogin(::CK_SESSION_HANDLE sessionHandle, ::CK_FUNCTION_LIST* pP11Functions, ::CK_UTF8CHAR_PTR pPin, ::CK_ULONG pinLength)
        : m_SessionHandle(sessionHandle),
          m_P11Functions(pP11Functions)
    {
        ::CK_ULONG usStatus = m_P11Functions->C_Login(m_SessionHandle, LoginType, pPin, pinLength);
        CK_EXCEPTION_THROW_UNLESS(usStatus == CKR_OK, "Failed C_Login()", usStatus);
    }

    HsmInterfaceImpl::ScopedLogin::~ScopedLogin()
    {
        m_P11Functions->C_Logout(m_SessionHandle);
    }

    void HsmInterfaceImpl::SignRsa2048PssSha256(::CK_OBJECT_HANDLE objectHandle, void* pOutSignature, size_t signatureSize, void* pRawMessage, size_t rawMessageSize)
    {
        ScopedLogin logged(m_SessionHandle, m_P11Functions, m_Pin, m_PinLength);

        ::CK_RSA_PKCS_PSS_PARAMS pssParams = { CKM_SHA256, CKG_MGF1_SHA256, Rsa2048PssSha256SaltSize };
        ::CK_MECHANISM mech = { CKM_SHA256_RSA_PKCS_PSS, &pssParams, sizeof(pssParams) };

        ::CK_RV ret = m_P11Functions->C_SignInit(m_SessionHandle, &mech, objectHandle);
        CK_EXCEPTION_THROW_UNLESS(ret == CKR_OK, "Failed C_SignInit()", ret);

        ::CK_ULONG usSignLen = static_cast<::CK_ULONG>(signatureSize);
        ret = m_P11Functions->C_Sign(m_SessionHandle, reinterpret_cast<::CK_BYTE_PTR>(pRawMessage), rawMessageSize, reinterpret_cast<::CK_BYTE_PTR>(pOutSignature), &usSignLen);
        CK_EXCEPTION_THROW_UNLESS(ret == CKR_OK && usSignLen == static_cast<::CK_ULONG>(signatureSize), "Failed C_Sign()", ret);

        return;
    }

    void HsmInterfaceImpl::SignRsa2048Pkcs1Sha256(::CK_OBJECT_HANDLE objectHandle, void* pOutSignature, size_t signatureSize, void* pRawMessage, size_t rawMessageSize)
    {
        ScopedLogin logged(m_SessionHandle, m_P11Functions, m_Pin, m_PinLength);

        ::CK_MECHANISM mech = { CKM_SHA256_RSA_PKCS, 0, 0 };

        ::CK_RV ret = m_P11Functions->C_SignInit(m_SessionHandle, &mech, objectHandle);
        CK_EXCEPTION_THROW_UNLESS(ret == CKR_OK, "Failed C_SignInit()", ret);

        ::CK_ULONG usSignLen = static_cast<::CK_ULONG>(signatureSize);
        ret = m_P11Functions->C_Sign(m_SessionHandle, reinterpret_cast<::CK_BYTE_PTR>(pRawMessage), rawMessageSize, reinterpret_cast<::CK_BYTE_PTR>(pOutSignature), &usSignLen);
        CK_EXCEPTION_THROW_UNLESS(ret == CKR_OK && usSignLen == static_cast<::CK_ULONG>(signatureSize), "Failed C_Sign()", ret);

        return;
    }

    bool HsmInterfaceImpl::VerifyRsa2048PssSha256(::CK_OBJECT_HANDLE objectHandle, void* pSignature, size_t signatureSize, void* pRawMessage, size_t rawMessageSize)
    {
        ScopedLogin logged(m_SessionHandle, m_P11Functions, m_Pin, m_PinLength);

        ::CK_RSA_PKCS_PSS_PARAMS pssParams = { CKM_SHA256, CKG_MGF1_SHA256, Rsa2048PssSha256SaltSize };
        ::CK_MECHANISM mech = { CKM_SHA256_RSA_PKCS_PSS, &pssParams, sizeof(pssParams) };

        ::CK_RV ret = m_P11Functions->C_VerifyInit(m_SessionHandle, &mech, objectHandle);
        CK_EXCEPTION_THROW_UNLESS(ret == CKR_OK, "Failed C_VerifyInit()", ret);

        ret = m_P11Functions->C_Verify(m_SessionHandle, reinterpret_cast<::CK_BYTE_PTR>(pRawMessage), rawMessageSize, reinterpret_cast<::CK_BYTE_PTR>(pSignature), signatureSize);

        return ret == CKR_OK;
    }

    bool HsmInterfaceImpl::VerifyRsa2048Pkcs1Sha256(::CK_OBJECT_HANDLE objectHandle, void* pSignature, size_t signatureSize, void* pRawMessage, size_t rawMessageSize)
    {
        ScopedLogin logged(m_SessionHandle, m_P11Functions, m_Pin, m_PinLength);

        ::CK_MECHANISM mech = { CKM_SHA256_RSA_PKCS, 0, 0 };

        ::CK_RV ret = m_P11Functions->C_VerifyInit(m_SessionHandle, &mech, objectHandle);
        CK_EXCEPTION_THROW_UNLESS(ret == CKR_OK, "Failed C_VerifyInit()", ret);

        ret = m_P11Functions->C_Verify(m_SessionHandle, reinterpret_cast<::CK_BYTE_PTR>(pRawMessage), rawMessageSize, reinterpret_cast<::CK_BYTE_PTR>(pSignature), signatureSize);

        return ret == CKR_OK;
    }

    void HsmInterfaceImpl::GetRsa2048PssSha256PublicKey(::CK_OBJECT_HANDLE objectHandle, void* pOutModulus, size_t modulusSize, void* pOutPublicExponent, size_t publicExponentSize)
    {
        ScopedLogin logged(m_SessionHandle, m_P11Functions, m_Pin, m_PinLength);

        ::CK_ATTRIBUTE attribute[] =
        {
            { CKA_MODULUS, NULL, 0 },
            { CKA_PUBLIC_EXPONENT, NULL, 0 },
        };
        const ::CK_ULONG AttributeCount = 2;

        ::CK_RV ret = m_P11Functions->C_GetAttributeValue(m_SessionHandle, objectHandle, attribute, AttributeCount);
        CK_EXCEPTION_THROW_UNLESS(ret == CKR_OK, "Failed C_GetAttributeValue()", ret);

        CK_EXCEPTION_THROW_UNLESS(modulusSize == attribute[0].ulValueLen && publicExponentSize == attribute[1].ulValueLen, "Invalid Key Size", attribute[0].ulValueLen)

        attribute[0].pValue = pOutModulus;
        attribute[1].pValue = pOutPublicExponent;

        ret = m_P11Functions->C_GetAttributeValue(m_SessionHandle, objectHandle, attribute, AttributeCount);
        CK_EXCEPTION_THROW_UNLESS(ret == CKR_OK, "Failed C_GetAttributeValue()", ret);

        return;
    }

    void HsmInterfaceImpl::GetRsa2048Pkcs1Sha256PublicKey(::CK_OBJECT_HANDLE objectHandle, void* pOutModulus, size_t modulusSize, void* pOutPublicExponent, size_t publicExponentSize)
    {
        ScopedLogin logged(m_SessionHandle, m_P11Functions, m_Pin, m_PinLength);

        ::CK_ATTRIBUTE attribute[] =
        {
            { CKA_MODULUS, NULL, 0 },
            { CKA_PUBLIC_EXPONENT, NULL, 0 },
        };
        const ::CK_ULONG AttributeCount = 2;

        ::CK_RV ret = m_P11Functions->C_GetAttributeValue(m_SessionHandle, objectHandle, attribute, AttributeCount);
        CK_EXCEPTION_THROW_UNLESS(ret == CKR_OK, "Failed C_GetAttributeValue()", ret);

        CK_EXCEPTION_THROW_UNLESS(modulusSize == attribute[0].ulValueLen && publicExponentSize == attribute[1].ulValueLen, "Invalid Key Size", attribute[0].ulValueLen);

        attribute[0].pValue = pOutModulus;
        attribute[1].pValue = pOutPublicExponent;

        ret = m_P11Functions->C_GetAttributeValue(m_SessionHandle, objectHandle, attribute, AttributeCount);
        CK_EXCEPTION_THROW_UNLESS(ret == CKR_OK, "Failed C_GetAttributeValue()", ret);

        return;
    }

    void HsmInterfaceImpl::EncryptAes128(::CK_OBJECT_HANDLE objectHandle, void* pDst, size_t dstSize, void* pSrc, size_t srcSize)
    {
        const size_t BlockSize = 16;
        CK_EXCEPTION_THROW_UNLESS(srcSize == dstSize && srcSize % BlockSize == 0, "Invalid Parameter for EncryptAes128()", srcSize); // TORIAEZU

        ::CK_MECHANISM mech = { CKM_AES_ECB, 0, 0 };

        EncryptAes(objectHandle, mech, pDst, dstSize, pSrc, srcSize);

        return;
    }

    void HsmInterfaceImpl::DecryptAes128(::CK_OBJECT_HANDLE objectHandle, void* pDst, size_t dstSize, void* pSrc, size_t srcSize)
    {
        const size_t BlockSize = 16;
        CK_EXCEPTION_THROW_UNLESS(srcSize == dstSize && srcSize % BlockSize == 0, "Invalid Parameter for DecryptAes128()", srcSize); // TORIAEZU

        ::CK_MECHANISM mech = { CKM_AES_ECB, 0, 0 };

        DecryptAes(objectHandle, mech, pDst, dstSize, pSrc, srcSize);

        return;
    }

    void HsmInterfaceImpl::EncryptAes128Cbc(::CK_OBJECT_HANDLE objectHandle, void * pDst, size_t dstSize, void * pSrc, size_t srcSize, void * pIv, size_t ivSize)
    {
        const size_t BlockSize = 16;
        CK_EXCEPTION_THROW_UNLESS(srcSize == dstSize && srcSize % BlockSize == 0, "Invalid Parameter for EncryptAes128()", srcSize); // TORIAEZU

        ::CK_MECHANISM mech = { CKM_AES_CBC, pIv, ivSize };

        EncryptAes(objectHandle, mech, pDst, dstSize, pSrc, srcSize);

        return;
    }

    void HsmInterfaceImpl::DecryptAes128Cbc(::CK_OBJECT_HANDLE objectHandle, void * pDst, size_t dstSize, void * pSrc, size_t srcSize, void * pIv, size_t ivSize)
    {
        const size_t BlockSize = 16;
        CK_EXCEPTION_THROW_UNLESS(srcSize == dstSize && srcSize % BlockSize == 0, "Invalid Parameter for DecryptAes128()", srcSize); // TORIAEZU

        ::CK_MECHANISM mech = { CKM_AES_CBC, pIv, ivSize };

        DecryptAes(objectHandle, mech, pDst, dstSize, pSrc, srcSize);

        return;
    }

    void HsmInterfaceImpl::GenerateHmacSha256(::CK_OBJECT_HANDLE objectHandle, void* pDst, size_t dstSize, void* pSrc, size_t srcSize)
    {
        ScopedLogin logged(m_SessionHandle, m_P11Functions, m_Pin, m_PinLength);

        CK_MECHANISM mech = { CKM_SHA256_HMAC, NULL_PTR, 0 };

        ::CK_RV ret = m_P11Functions->C_SignInit(m_SessionHandle, &mech, objectHandle);
        CK_EXCEPTION_THROW_UNLESS(ret == CKR_OK, "Failed C_SignInit()", ret);

        ::CK_ULONG outLength = dstSize;
        auto tmpSrc = reinterpret_cast<unsigned char*>(pSrc);
        auto tmpDst = reinterpret_cast<unsigned char*>(pDst);

        ret = m_P11Functions->C_Sign(m_SessionHandle, reinterpret_cast<::CK_BYTE_PTR>(tmpSrc), srcSize, reinterpret_cast<::CK_BYTE_PTR>(tmpDst), &outLength);
        CK_EXCEPTION_THROW_UNLESS(ret == CKR_OK, "Failed C_Sign()", ret);

        return;
    }

    ::CK_OBJECT_HANDLE HsmInterfaceImpl::FindObject(const char* labelName)
    {
        ScopedLogin logged(m_SessionHandle, m_P11Functions, m_Pin, m_PinLength);

        char label[KeyLabelLengthMax];
        strncpy_s(label, labelName, KeyLabelLengthMax);

        ::CK_ATTRIBUTE attribute[] = { CKA_LABEL, label, strnlen_s(labelName, KeyLabelLengthMax) };

        ::CK_RV ret = m_P11Functions->C_FindObjectsInit(m_SessionHandle, attribute, 1);
        CK_EXCEPTION_THROW_UNLESS(ret == CKR_OK, "Failed C_FindObjectInit()", ret);

        ::CK_OBJECT_HANDLE objectHandle[1];
        ::CK_ULONG count = 1;
        ret = m_P11Functions->C_FindObjects(m_SessionHandle, objectHandle, 1, &count);
        CK_EXCEPTION_THROW_UNLESS(ret == CKR_OK && count == 1, "Failed C_FindObjects()", ret);

        ret = m_P11Functions->C_FindObjectsFinal(m_SessionHandle);
        CK_EXCEPTION_THROW_UNLESS(ret == CKR_OK, "Failed C_FindObjectFinal()", ret);

        return objectHandle[0];
    }

    void HsmInterfaceImpl::EncryptAes(::CK_OBJECT_HANDLE objectHandle, const ::CK_MECHANISM& mech, void * pDst, size_t dstSize, void * pSrc, size_t srcSize)
    {
        ScopedLogin logged(m_SessionHandle, m_P11Functions, m_Pin, m_PinLength);

        ::CK_RV ret = m_P11Functions->C_EncryptInit(m_SessionHandle, (::CK_MECHANISM_PTR)&mech, objectHandle);
        CK_EXCEPTION_THROW_UNLESS(ret == CKR_OK, "Failed C_EncryptInit()", ret);

        ::CK_ULONG encrypted = 0;
        void* pLast = NULL;
        ::CK_ULONG lastLength = 0;
        while (encrypted != dstSize)
        {
            ::CK_ULONG outLength = dstSize;
            auto tmpSrc = reinterpret_cast<unsigned char*>(pSrc) + encrypted;
            auto tmpDst = reinterpret_cast<unsigned char*>(pDst) + encrypted;
            ret = m_P11Functions->C_EncryptUpdate(m_SessionHandle, reinterpret_cast<::CK_BYTE_PTR>(tmpSrc), srcSize - encrypted, reinterpret_cast<::CK_BYTE_PTR>(tmpDst), &outLength);
            CK_EXCEPTION_THROW_UNLESS(ret == CKR_OK, "Failed C_EncryptUpdate()", ret);
            encrypted += outLength;

            pLast = pDst;
            lastLength = outLength;
        }

        ret = m_P11Functions->C_EncryptFinal(m_SessionHandle, reinterpret_cast<::CK_BYTE_PTR>(pLast), &lastLength);
        CK_EXCEPTION_THROW_UNLESS(ret == CKR_OK && encrypted == dstSize, "Failed C_EncryptFinal()", ret);

        return;
    }

    void HsmInterfaceImpl::DecryptAes(::CK_OBJECT_HANDLE objectHandle, const ::CK_MECHANISM& mech, void * pDst, size_t dstSize, void * pSrc, size_t srcSize)
    {
        ScopedLogin logged(m_SessionHandle, m_P11Functions, m_Pin, m_PinLength);

        ::CK_RV ret = m_P11Functions->C_DecryptInit(m_SessionHandle, (::CK_MECHANISM_PTR)&mech, objectHandle);
        CK_EXCEPTION_THROW_UNLESS(ret == CKR_OK, "Failed C_DecryptInit()", ret);

        ::CK_ULONG encrypted = 0;
        void* pLast = NULL;
        ::CK_ULONG lastLength = 0;
        while (encrypted != dstSize)
        {
            ::CK_ULONG outLength = dstSize;
            auto tmpSrc = reinterpret_cast<unsigned char*>(pSrc) + encrypted;
            auto tmpDst = reinterpret_cast<unsigned char*>(pDst) + encrypted;
            ret = m_P11Functions->C_DecryptUpdate(m_SessionHandle, reinterpret_cast<::CK_BYTE_PTR>(tmpSrc), srcSize - encrypted, reinterpret_cast<::CK_BYTE_PTR>(tmpDst), &outLength);
            CK_EXCEPTION_THROW_UNLESS(ret == CKR_OK, "Failed C_DecryptUpdate()", ret);
            encrypted += outLength;

            pLast = pDst;
            lastLength = outLength;
        }

        ret = m_P11Functions->C_DecryptFinal(m_SessionHandle, reinterpret_cast<::CK_BYTE_PTR>(pLast), &lastLength);
        CK_EXCEPTION_THROW_UNLESS(ret == CKR_OK && encrypted == dstSize, "Failed C_DecryptFinal()", ret);

        return;
    }

    void HsmInterface::SignRsa2048PssSha256(Rsa2048PssSha256KeyIndex index, void* pOutSignature, size_t signatureSize, void* pRawMessage, size_t rawMessageSize)
    {
        msclr::lock scopedLock(HsmInterface::m_LockObject);
        m_Impl->SignRsa2048PssSha256(GetRsa2048PssSha256KeyObjectHandle(index, true), pOutSignature, signatureSize, pRawMessage, rawMessageSize);
        GC::KeepAlive(this);
    }

    void HsmInterface::SignRsa2048Pkcs1Sha256(Rsa2048Pkcs1Sha256KeyIndex index, void* pOutSignature, size_t signatureSize, void* pRawMessage, size_t rawMessageSize)
    {
        msclr::lock scopedLock(HsmInterface::m_LockObject);
        m_Impl->SignRsa2048Pkcs1Sha256(GetRsa2048Pkcs1Sha256KeyObjectHandle(index, true), pOutSignature, signatureSize, pRawMessage, rawMessageSize);
        GC::KeepAlive(this);
    }

    bool HsmInterface::VerifyRsa2048PssSha256(Rsa2048PssSha256KeyIndex index, void* pSignature, size_t signatureSize, void* pRawMessage, size_t rawMessageSize)
    {
        msclr::lock scopedLock(HsmInterface::m_LockObject);
        return Util::ReturnAndDeclareAlive(this, m_Impl->VerifyRsa2048PssSha256(GetRsa2048PssSha256KeyObjectHandle(index, false), pSignature, signatureSize, pRawMessage, rawMessageSize));
    }

    bool HsmInterface::VerifyRsa2048Pkcs1Sha256(Rsa2048Pkcs1Sha256KeyIndex index, void* pSignature, size_t signatureSize, void* pRawMessage, size_t rawMessageSize)
    {
        msclr::lock scopedLock(HsmInterface::m_LockObject);
        return Util::ReturnAndDeclareAlive(this, m_Impl->VerifyRsa2048Pkcs1Sha256(GetRsa2048Pkcs1Sha256KeyObjectHandle(index, false), pSignature, signatureSize, pRawMessage, rawMessageSize));
    }

    void HsmInterface::GetRsa2048PssSha256PublicKey(Rsa2048PssSha256KeyIndex index, void* pOutModulus, size_t modulusSize, void* pOutPublicExponent, size_t publicExponentSize)
    {
        msclr::lock scopedLock(HsmInterface::m_LockObject);
        m_Impl->GetRsa2048PssSha256PublicKey(GetRsa2048PssSha256KeyObjectHandle(index, false), pOutModulus, modulusSize, pOutPublicExponent, publicExponentSize);
        GC::KeepAlive(this);
    }

    void HsmInterface::GetRsa2048Pkcs1Sha256PublicKey(Rsa2048Pkcs1Sha256KeyIndex index, void* pOutModulus, size_t modulusSize, void* pOutPublicExponent, size_t publicExponentSize)
    {
        msclr::lock scopedLock(HsmInterface::m_LockObject);
        m_Impl->GetRsa2048Pkcs1Sha256PublicKey(GetRsa2048Pkcs1Sha256KeyObjectHandle(index, false), pOutModulus, modulusSize, pOutPublicExponent, publicExponentSize);
        GC::KeepAlive(this);
    }

    void HsmInterface::EncryptAes128(Aes128KeyIndex index, void* pDst, size_t dstSize, void* pSrc, size_t srcSize)
    {
        return EncryptAes128(index, 0, pDst, dstSize, pSrc, srcSize);
    }

    void HsmInterface::EncryptAes128(Aes128KeyIndex index, byte keyGeneration, void* pDst, size_t dstSize, void* pSrc, size_t srcSize)
    {
        msclr::lock scopedLock(HsmInterface::m_LockObject);
        m_Impl->EncryptAes128(GetAes128KeyObjectHandle(index, keyGeneration), pDst, dstSize, pSrc, srcSize);
        GC::KeepAlive(this);
    }

    void HsmInterface::DecryptAes128(Aes128KeyIndex index, void* pDst, size_t dstSize, void* pSrc, size_t srcSize)
    {
        return DecryptAes128(index, 0, pDst, dstSize, pSrc, srcSize);
    }

    void HsmInterface::DecryptAes128(Aes128KeyIndex index, byte keyGeneration, void* pDst, size_t dstSize, void* pSrc, size_t srcSize)
    {
        msclr::lock scopedLock(HsmInterface::m_LockObject);
        m_Impl->DecryptAes128(GetAes128KeyObjectHandle(index, keyGeneration), pDst, dstSize, pSrc, srcSize);
        GC::KeepAlive(this);
    }

    void HsmInterface::EncryptAes128Cbc(Aes128KeyIndex index, void * pDst, size_t dstSize, void * pSrc, size_t srcSize, void * pIv, size_t ivSize)
    {
        return EncryptAes128Cbc(index, 0, pDst, dstSize, pSrc, srcSize, pIv, ivSize);
    }

    void HsmInterface::EncryptAes128Cbc(Aes128KeyIndex index, byte keyGeneration, void * pDst, size_t dstSize, void * pSrc, size_t srcSize, void * pIv, size_t ivSize)
    {
        msclr::lock scopedLock(HsmInterface::m_LockObject);
        m_Impl->EncryptAes128Cbc(GetAes128KeyObjectHandle(index, keyGeneration), pDst, dstSize, pSrc, srcSize, pIv, ivSize);
        GC::KeepAlive(this);
    }

    void HsmInterface::DecryptAes128Cbc(Aes128KeyIndex index, void * pDst, size_t dstSize, void * pSrc, size_t srcSize, void * pIv, size_t ivSize)
    {
        return DecryptAes128Cbc(index, 0, pDst, dstSize, pSrc, srcSize, pIv, ivSize);
    }

    void HsmInterface::DecryptAes128Cbc(Aes128KeyIndex index, byte keyGeneration, void * pDst, size_t dstSize, void * pSrc, size_t srcSize, void * pIv, size_t ivSize)
    {
        msclr::lock scopedLock(HsmInterface::m_LockObject);
        m_Impl->DecryptAes128Cbc(GetAes128KeyObjectHandle(index, keyGeneration), pDst, dstSize, pSrc, srcSize, pIv, ivSize);
        GC::KeepAlive(this);
    }

    void HsmInterface::GenerateHmacSha256(HmacSha256KeyIndex index, void* pDst, size_t dstSize, void* pSrc, size_t srcSize)
    {
        msclr::lock scopedLock(HsmInterface::m_LockObject);
        m_Impl->GenerateHmacSha256(GetHmacSha256ObjectHandle(index), pDst, dstSize, pSrc, srcSize);
        GC::KeepAlive(this);
    }

    ::CK_OBJECT_HANDLE HsmInterface::GetRsa2048PssSha256KeyObjectHandle(Rsa2048PssSha256KeyIndex index, bool isPrivate)
    {
        // TODO: 本番用の鍵に変更
        switch (index)
        {
        case Rsa2048PssSha256KeyIndex::NcaHeader1:
            return isPrivate ? m_Impl->FindObject("K6_NcaHeader1Sign") : m_Impl->FindObject("K6_NcaHeader1Sign.public");
        case Rsa2048PssSha256KeyIndex::Acid:
            return isPrivate ? m_Impl->FindObject("K6_AcidSign") : m_Impl->FindObject("K6_AcidSign.public");
        case Rsa2048PssSha256KeyIndex::NrrCertificate:
            return isPrivate ? m_Impl->FindObject("K6_NrrCertificateSign") : m_Impl->FindObject("K6_NrrCertificateSign.public");
        default:
            throw gcnew ArgumentException("Invalid Key Index.");
        }
    }

    ::CK_OBJECT_HANDLE HsmInterface::GetRsa2048Pkcs1Sha256KeyObjectHandle(Rsa2048Pkcs1Sha256KeyIndex index, bool isPrivate)
    {
        switch (index)
        {
        case Rsa2048Pkcs1Sha256KeyIndex::XciHeader:
            return isPrivate ? m_Impl->FindObject("K6_XciHeaderSign") : m_Impl->FindObject("K6_XciHeaderSign.public");
        case Rsa2048Pkcs1Sha256KeyIndex::ETicket:
        {
            if (m_UseDev)
            {
                return isPrivate ? m_Impl->FindObject("devel-XS00000021") : m_Impl->FindObject("devel-XS00000021.public");
            }
            else
            {
                return isPrivate ? m_Impl->FindObject("prod-XS00000020") : m_Impl->FindObject("prod-XS00000020.public");
            }
        }
        default:
            throw gcnew ArgumentException("Invalid Key Index.");
        }
    }

    ::CK_OBJECT_HANDLE HsmInterface::GetAes128KeyObjectHandle(Aes128KeyIndex index, byte keyGeneration)
    {
        // TODO: 本番用の鍵に変更
        switch (index)
        {
        case Aes128KeyIndex::NcaContentKey:
        {
            if (keyGeneration <= 1)
            {
                return m_Impl->FindObject("K6_NcaContentKeyEncryption");
            }
            else if (keyGeneration == 2)
            {
                return m_Impl->FindObject("K6_NcaContentKeyEncryptionGen2");
            }
            else if (keyGeneration == 3)
            {
                return m_Impl->FindObject("K6_NcaContentKeyEncryptionGen3");
            }
            else if (keyGeneration == 4)
            {
                return m_Impl->FindObject("K6_NcaContentKeyEncryptionGen4");
            }
            else if (keyGeneration == 5)
            {
                return m_Impl->FindObject("K6_NcaContentKeyEncryptionGen5");
            }
            else
            {
                throw gcnew ArgumentException("Invalid Key Generation.");
            }
        }
        case Aes128KeyIndex::NcaHeader:      return m_Impl->FindObject("K6_NcaHeaderKeyEncryption");
        case Aes128KeyIndex::XciInitialData: return m_Impl->FindObject("K6_XciInitialDataEncryption");
        case Aes128KeyIndex::ETicketCommonKey:
        {
            if (keyGeneration <= 1)
            {
                return m_Impl->FindObject("ETicketCommonKey");
            }
            else if (keyGeneration == 2)
            {
                return m_Impl->FindObject("ETicketCommonKeyGen2");
            }
            else if (keyGeneration == 3)
            {
                return m_Impl->FindObject("ETicketCommonKeyGen3");
            }
            else if (keyGeneration == 4)
            {
                return m_Impl->FindObject("ETicketCommonKeyGen4");
            }
            else if (keyGeneration == 5)
            {
                return m_Impl->FindObject("ETicketCommonKeyGen5");
            }
            else
            {
                throw gcnew ArgumentException("Invalid Key Generation.");
            }
        }
        default:
            throw gcnew ArgumentException("Invalid Key Index.");
        }
    }

    ::CK_OBJECT_HANDLE HsmInterface::GetHmacSha256ObjectHandle(HmacSha256KeyIndex index)
    {
        switch (index)
        {
        case HmacSha256KeyIndex::TitleKeyGenarateKey:
            return m_Impl->FindObject("TitleKeyGenerateKey");
        default:
            throw gcnew ArgumentException("Invalid Key Index.");
        }
    }

}}}
