﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Globalization;
using System.IO;
using System.Linq;
using System.Security.Cryptography;
using System.Text;
using System.Text.RegularExpressions;
using System.Xml;
using System.Xml.Linq;
using System.Xml.Serialization;
using System.Xml.XPath;
using Nintendo.ApplicationControlProperty;
using Nintendo.Authoring.FileSystemMetaLibrary;

namespace Nintendo.Authoring.AuthoringLibrary
{
    public class UnpublishableErrorCheckData
    {
        public const string CheckContentTypeForce                = "ForceCheck";
        public const string CheckContentTypeMultiApplicationCard = "MultiApplicationCard";
        public const string LegalInfoIndexHtmlPath               = "ipnotices.htdocs/index.html";

        public UnpublishableErrorCheckData()
        {
            CheckTarget = CheckTargetType.None;
            CheckContentType = null;
            CheckInfo = new CheckItemInfo();
            Nmeta = new NmetaData();
            Nsp = new NspData();
            ApplicationControlProperty = null;
            CardSpec = null;
        }

        // チェック対象
        public enum CheckTargetType
        {
            None,  // 指定なし
            Nmeta, // nmetaファイル
            Nsp,   // nspファイル
        }

        public string CheckContentType { get; set; }
        public CheckTargetType CheckTarget { get; set; }
        public CheckItemInfo CheckInfo { get; set; }
        public NmetaData Nmeta { get; set; }
        public NspData Nsp { get; set; }
        public ApplicationControlPropertyModel ApplicationControlProperty { get; set; }
        public CardSpecModel CardSpec { get; set; }

        // チェック項目情報
        public class CheckItemInfo
        {
            public CheckItemInfo()
            {
                CheckList = null;
                IgnoreList = null;
            }
            public UnpublishableErrorCheck.CheckItemId[] CheckList { get; set; }
            public UnpublishableErrorCheck.CheckItemId[] IgnoreList { get; set; }
        }
        // Nmetaにのみ含まれるデータ
        public class NmetaData
        {
            public NmetaData()
            {
                NmetaFilePath = null;
                Core = null;
                CoreFsAccessControlData = null;
                CoreFsAccessControlData = null;
                AddOnContentList = null;
            }
            public string NmetaFilePath { get; set; }
            public CoreModel Core { get; set; }
            public FsAccessControlData CoreFsAccessControlData { get; set; }
            public AddOnContentModel AddOnContent { get; set; }
            public List<AddOnContentModel> AddOnContentList { get; set; }
        }
        // Nspにのみ含まれるデータ
        public class NspData
        {
            public NspData()
            {
                ProgramInfo = null;
                ContentMeta = null;
                AddonContentMetaList = null;
                LegalInformation = null;
                ContentMetaPropertyList = null;
                HtmlDocumentXml = null;
                MultiApplicationCardInfo = null;
                ApplicationControlPropertyOfProgramIndex0 = null;
                LegalInformationFileList = null;
                RawIconList = null;
                NxIconList = null;
                AllMiddlewareListInNsp = null;
            }
            public ProgramInfoModel ProgramInfo { get; set; }
            public ContentMetaModel ContentMeta { get; set; }
            public List<AddOnContentContentMetaModel> AddonContentMetaList { get; set; }
            public LegalInformationModel LegalInformation { get; set; }
            public NintendoContentMetaPropertyList ContentMetaPropertyList { get; set; }
            public HtmlDocumentXmlModel HtmlDocumentXml { get; set; }
            public MultiApplicationCardInfoModel MultiApplicationCardInfo { get; set; }
            public ApplicationControlPropertyModel ApplicationControlPropertyOfProgramIndex0 { get; set; }
            public List<Tuple<string, byte[]>> LegalInformationFileList { get; set; }
            public List<Tuple<string, byte[]>> RawIconList { get; set; }
            public List<Tuple<string, byte[]>> NxIconList { get; set; }
            public List<string> AllMiddlewareListInNsp { get; set; }
        }
    }
    /// <summary>
    /// UnpublishableErrorのチェックを行うクラスです。
    /// </summary>
    public class UnpublishableErrorCheck
    {
        // Coreのみが存在する場合に実施可能なチェック項目のリスト
        public static readonly CheckItemId[] CoreCheckList =
            new CheckItemId[]
            {
                CheckItemId.SbmChk43,   // 他アプリのセーブデータアクセス
                CheckItemId.SbmChk43b,  // 他のアプリケーションのセーブデータ設定に自分のID設定の禁止
            };
        // ApplicationControlPropertyのみが存在する場合に実施可能なチェック項目のリスト(一部 Core が必要なものあり)
        public static readonly CheckItemId[] ApplicationControlPropertyCheckList =
            new CheckItemId[]
            {
                CheckItemId.SbmChk11,   // タイトル名
                CheckItemId.SbmChk12,   // Publisher名
                CheckItemId.SbmChk12_1, // タイトルの言語
                CheckItemId.SbmChk12_2, // タイトルの言語
                CheckItemId.SbmChk12_3, // タイトルの言語未設定
                CheckItemId.SbmChk12_4, // タイトルの未定義な言語設定
                CheckItemId.SbmChk12_5, // 指定できない文字設定(日米欧フォント)
                CheckItemId.SbmChk12_6, // 指定できない文字設定(ASCII)
                CheckItemId.SbmChk12_7, // 文字数の超過
                CheckItemId.SbmChk13,   // 表示バージョン
                CheckItemId.SbmChk15,   // 対応言語
                CheckItemId.SbmChk1_8,  // 提出ファイルの正当性 (アイコンが存在しない)
                CheckItemId.SbmChk9a,   // 提出ファイルの正当性 (アイコンと言語が不正)
                CheckItemId.SbmChk9c,   // アイコンがSDKアイコンでないか？
                CheckItemId.SbmChk52c,  // BCATセーブデータの確認 (64MiBを超えるBcatDeliveryCacheStorageSizeの禁止)
                CheckItemId.SbmChk52d,  // BCATセーブデータの確認 (不正な設定)
                CheckItemId.SbmChk8a,   // ユーザーアカウントセーブデータのサイズ(特別承認)
                CheckItemId.SbmChk8b,   // ユーザーアカウントセーブデータ保存領域拡張最大サイズ
                CheckItemId.SbmChk8c,   // ユーザーアカウントセーブジャーナル領域拡張最大サイズ
                CheckItemId.SbmChk8d,   // 本体セーブデータ保存領域拡張最大サイズ
                CheckItemId.SbmChk8e,   // 本体セーブジャーナル領域拡張最大サイズ
                CheckItemId.SbmChk8f,   // 一時ストレージサイズ
                CheckItemId.SbmChk8g,   // キャッシュストレージデータ保存領域サイズ
                CheckItemId.SbmChk8h,   // キャッシュストレージジャーナル領域サイズ
                CheckItemId.SbmChk8i,   // 本体セーブデータのデータ保存領域サイズ
                CheckItemId.SbmChk8j,   // 本体セーブデータのジャーナル領域サイズ
                CheckItemId.SbmChk8k,  // 特別承認が必要なnspファイル（複数キャッシュストレージを使用）
                CheckItemId.SbmChk16,   // プレゼンスグループID
                CheckItemId.SbmChk17a,  // ローカル通信識別子 (Application ID と異なる)
                CheckItemId.SbmChk56,   // HDCPの使用禁止
                CheckItemId.SbmChk1_9b, // NXアイコンがベースライン形式でない
                CheckItemId.SbmChk47a,  // アクセス可能URL
                CheckItemId.SbmChk47b,  // アクセス可能URLのhttp禁止
            };
        // AddOnContentのみが存在する場合に実施可能なチェック項目のリスト
        public static readonly CheckItemId[] AddOnContentCheckList =
            new CheckItemId[]
            {
                CheckItemId.SbmChk12_5, // 指定できない文字設定(日米欧フォント)
                CheckItemId.SbmChk39_1, // AOCのタグがあるか？
                CheckItemId.SbmChk39_2, // AOCのタグがユニークであるか？
                CheckItemId.SbmChk61,   // AOCのタグが128文字以内であるか？
                CheckItemId.SbmChk39_3, // AOCのIndexが指定されていない
            };
        // CardSpecのみが存在する場合に実施可能なチェック項目のリスト
        public static readonly CheckItemId[] CardSpecCheckList =
            new CheckItemId[]
            {
                CheckItemId.SbmChk57,   // オートブート設定の使用禁止
                CheckItemId.SbmChk7c, // 提出不可のカード容量
            };
        //  --error-unpublishable の有無に関係なく実施するチェック項目のリスト
        public static readonly CheckItemId[] NintendoSubmissionPackageMandatoryCheckList =
            new CheckItemId[]
            {
                CheckItemId.SbmChk12_2, // タイトルの言語
                CheckItemId.SbmChk12_3, // タイトルの言語未設定
                CheckItemId.SbmChk12_4, // タイトルの未定義な言語設定
                CheckItemId.SbmChk12_5, // 指定できない文字設定(日米欧フォント)
                CheckItemId.SbmChk12_6, // 指定できない文字設定(ASCII)
                CheckItemId.SbmChk12_7, // 文字数の超過
                CheckItemId.SbmChk9a,   // 提出ファイルの正当性 (アイコンと言語が不正)
                CheckItemId.SbmChk52d,  // BCATセーブデータの確認 (不正な設定)
                CheckItemId.SbmChk41b,  // ソフトリーガル情報のHashの確認
                CheckItemId.SbmChk43b,  // 他のアプリケーションのセーブデータ設定に自分のID設定の禁止
                CheckItemId.SbmChk18c,  // 使用されていないアプリのエラーコード
                CheckItemId.SbmChk7c,   // 提出不可のカード容量
                CheckItemId.SbmChk1_10, // プログラムが含まれていないアプリケーション
                CheckItemId.SbmChk1_9b, // NXアイコンがベースライン形式でない
                CheckItemId.SbmChk66_1, // 個別プログラム間のアプリケーション管理データの不一致
                CheckItemId.SbmChk39_1, // AOCのタグがあるか？
                CheckItemId.SbmChk39_2, // AOCのタグがユニークであるか？
                CheckItemId.SbmChk61,   // AOCのタグが128文字以内であるか？
                CheckItemId.SbmChk39_3, // AOCのIndexが指定されていない
            };

        // 全てのUnpublishableErrorチェック項目のリスト
        private static readonly List<CheckSbm> CheckTable =
            new List<CheckSbm>()
            {
                new CheckSbmChk23b  (new CheckSbm.InitializeParameter(CheckItemId.SbmChk23b,  "Issue_11_004", ErrorType.Error, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk23c  (new CheckSbm.InitializeParameter(CheckItemId.SbmChk23c,  "Issue_11_006", ErrorType.Error, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk26a  (new CheckSbm.InitializeParameter(CheckItemId.SbmChk26a,  "Issue_12_001", ErrorType.Error, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk26b  (new CheckSbm.InitializeParameter(CheckItemId.SbmChk26b,  "Issue_12_002", ErrorType.Error, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk28   (new CheckSbm.InitializeParameter(CheckItemId.SbmChk28,   "Issue_12_003", ErrorType.Error, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk41a  (new CheckSbm.InitializeParameter(CheckItemId.SbmChk41a,  "Issue_12_004", ErrorType.Error, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk41b  (new CheckSbm.InitializeParameter(CheckItemId.SbmChk41b,  "Issue_10_812", ErrorType.Error, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk35   (new CheckSbm.InitializeParameter(CheckItemId.SbmChk35,   "Issue_11_008", ErrorType.Error, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk21   (new CheckSbm.InitializeParameter(CheckItemId.SbmChk21,   "Issue_11_002", ErrorType.Error, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk42   (new CheckSbm.InitializeParameter(CheckItemId.SbmChk42,   "Issue_11_009", ErrorType.Error, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk20c  (new CheckSbm.InitializeParameter(CheckItemId.SbmChk20c,  "Issue_11_011", ErrorType.Error, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk33d  (new CheckSbm.InitializeParameter(CheckItemId.SbmChk33d,  "Issue_02_007", ErrorType.Error, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk24a  (new CheckSbm.InitializeParameter(CheckItemId.SbmChk24a,  "Issue_11_007", ErrorType.Error, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk24b  (new CheckSbm.InitializeParameter(CheckItemId.SbmChk24b,  "Issue_11_012", ErrorType.Warning, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk48   (new CheckSbm.InitializeParameter(CheckItemId.SbmChk48,   "Issue_10_031", ErrorType.Error, TargetContent.App | TargetContent.Patch | TargetContent.Aoc)),
                new CheckSbmChk63   (new CheckSbm.InitializeParameter(CheckItemId.SbmChk63,   "Issue_11_016", ErrorType.Error, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk11   (new CheckSbm.InitializeParameter(CheckItemId.SbmChk11,   "Issue_10_010", ErrorType.Error, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk12   (new CheckSbm.InitializeParameter(CheckItemId.SbmChk12,   "Issue_10_011", ErrorType.Error, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk12_1 (new CheckSbm.InitializeParameter(CheckItemId.SbmChk12_1, "Issue_10_801", ErrorType.Error, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk12_2 (new CheckSbm.InitializeParameter(CheckItemId.SbmChk12_2, "Issue_10_802", ErrorType.Error, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk12_3 (new CheckSbm.InitializeParameter(CheckItemId.SbmChk12_3, "Issue_10_806", ErrorType.Error, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk12_4 (new CheckSbm.InitializeParameter(CheckItemId.SbmChk12_4, "Issue_10_807", ErrorType.Error, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk12_5 (new CheckSbm.InitializeParameter(CheckItemId.SbmChk12_5, "Issue_10_808", ErrorType.Error, TargetContent.App | TargetContent.Patch | TargetContent.Aoc)),
                new CheckSbmChk12_6 (new CheckSbm.InitializeParameter(CheckItemId.SbmChk12_6, "Issue_10_809", ErrorType.Error, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk12_7 (new CheckSbm.InitializeParameter(CheckItemId.SbmChk12_7, "Issue_10_810", ErrorType.Error, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk13   (new CheckSbm.InitializeParameter(CheckItemId.SbmChk13,   "Issue_10_012", ErrorType.Error, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk15   (new CheckSbm.InitializeParameter(CheckItemId.SbmChk15,   "Issue_10_014", ErrorType.Error, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk1_8  (new CheckSbm.InitializeParameter(CheckItemId.SbmChk1_8,  "Issue_10_607", ErrorType.Error, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk9a   (new CheckSbm.InitializeParameter(CheckItemId.SbmChk9a,   "Issue_10_608", ErrorType.Error, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk9c   (new CheckSbm.InitializeParameter(CheckItemId.SbmChk9c,   "Issue_10_047", ErrorType.Error, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk52b  (new CheckSbm.InitializeParameter(CheckItemId.SbmChk52b,  "Issue_10_029", ErrorType.Error, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk52c  (new CheckSbm.InitializeParameter(CheckItemId.SbmChk52c,  "Issue_10_030", ErrorType.Warning, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk52d  (new CheckSbm.InitializeParameter(CheckItemId.SbmChk52d,  "Issue_10_803", ErrorType.Error, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk43b  (new CheckSbm.InitializeParameter(CheckItemId.SbmChk43b,  "Issue_10_811", ErrorType.Error, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk18c  (new CheckSbm.InitializeParameter(CheckItemId.SbmChk18c,  "Issue_10_046", ErrorType.Error, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk54   (new CheckSbm.InitializeParameter(CheckItemId.SbmChk54,   "Issue_11_010", ErrorType.Error, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk22   (new CheckSbm.InitializeParameter(CheckItemId.SbmChk22,   "Issue_11_003", ErrorType.Warning, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk8a   (new CheckSbm.InitializeParameter(CheckItemId.SbmChk8a,   "Issue_10_032", ErrorType.Warning, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk8b   (new CheckSbm.InitializeParameter(CheckItemId.SbmChk8b,   "Issue_10_033", ErrorType.Warning, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk8c   (new CheckSbm.InitializeParameter(CheckItemId.SbmChk8c,   "Issue_10_034", ErrorType.Warning, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk8d   (new CheckSbm.InitializeParameter(CheckItemId.SbmChk8d,   "Issue_10_035", ErrorType.Warning, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk8e   (new CheckSbm.InitializeParameter(CheckItemId.SbmChk8e,   "Issue_10_036", ErrorType.Warning, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk8f   (new CheckSbm.InitializeParameter(CheckItemId.SbmChk8f,   "Issue_10_037", ErrorType.Warning, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk8g   (new CheckSbm.InitializeParameter(CheckItemId.SbmChk8g,   "Issue_10_038", ErrorType.Warning, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk8h   (new CheckSbm.InitializeParameter(CheckItemId.SbmChk8h,   "Issue_10_039", ErrorType.Warning, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk8i   (new CheckSbm.InitializeParameter(CheckItemId.SbmChk8i,   "Issue_10_040", ErrorType.Warning, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk8j   (new CheckSbm.InitializeParameter(CheckItemId.SbmChk8j,   "Issue_10_041", ErrorType.Warning, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk8k   (new CheckSbm.InitializeParameter(CheckItemId.SbmChk8k,   "Issue_10_049", ErrorType.Warning, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk16   (new CheckSbm.InitializeParameter(CheckItemId.SbmChk16,   "Issue_10_015", ErrorType.Warning, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk17a  (new CheckSbm.InitializeParameter(CheckItemId.SbmChk17a,  "Issue_10_016", ErrorType.Warning, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk27a  (new CheckSbm.InitializeParameter(CheckItemId.SbmChk27a,  "Issue_20_001", ErrorType.Warning, TargetContent.Patch)),
                new CheckSbmChk29b  (new CheckSbm.InitializeParameter(CheckItemId.SbmChk29b,  "Issue_10_021", ErrorType.Warning, TargetContent.App)),
                new CheckSbmChk43   (new CheckSbm.InitializeParameter(CheckItemId.SbmChk43,   "Issue_10_024", ErrorType.Warning, TargetContent.App)),
                new CheckSbmChk50a  (new CheckSbm.InitializeParameter(CheckItemId.SbmChk50a,  "Issue_10_027", ErrorType.Warning, TargetContent.App)),
                new CheckSbmChk50b  (new CheckSbm.InitializeParameter(CheckItemId.SbmChk50b,  "Issue_10_042", ErrorType.Warning, TargetContent.Patch)),
                new CheckSbmChk55   (new CheckSbm.InitializeParameter(CheckItemId.SbmChk55,   "Issue_11_013", ErrorType.Warning, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk38a  (new CheckSbm.InitializeParameter(CheckItemId.SbmChk38a,  "Issue_10_043", ErrorType.Warning, TargetContent.Aoc)),
                new CheckSbmChk38b  (new CheckSbm.InitializeParameter(CheckItemId.SbmChk38b,  "Issue_10_044", ErrorType.Warning, TargetContent.Aoc)),
                new CheckSbmChk56   (new CheckSbm.InitializeParameter(CheckItemId.SbmChk56,   "Issue_10_045", ErrorType.Warning, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk57   (new CheckSbm.InitializeParameter(CheckItemId.SbmChk57,   "Issue_11_014", ErrorType.Warning, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk58   (new CheckSbm.InitializeParameter(CheckItemId.SbmChk58,   "Issue_11_015", ErrorType.Warning, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk65b  (new CheckSbm.InitializeParameter(CheckItemId.SbmChk65b,  "Issue_10_050", ErrorType.Warning, TargetContent.MultiApplicationCard)),
                new CheckSbmChk7c   (new CheckSbm.InitializeParameter(CheckItemId.SbmChk7c,   "Issue_13_801", ErrorType.Error, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk1_10 (new CheckSbm.InitializeParameter(CheckItemId.SbmChk1_10, "Issue_10_813", ErrorType.Error, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk1_9b (new CheckSbm.InitializeParameter(CheckItemId.SbmChk1_9b, "Issue_10_815", ErrorType.Error, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk66_1 (new CheckSbm.InitializeParameter(CheckItemId.SbmChk66_1, "Issue_10_816", ErrorType.Error, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk1_7  (new CheckSbm.InitializeParameter(CheckItemId.SbmChk1_7,  "Issue_10_614", ErrorType.Error, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk39_1 (new CheckSbm.InitializeParameter(CheckItemId.SbmChk39_1, "Issue_10_611", ErrorType.Error, TargetContent.Aoc)),
                new CheckSbmChk39_2 (new CheckSbm.InitializeParameter(CheckItemId.SbmChk39_2, "Issue_10_612", ErrorType.Error, TargetContent.Aoc)),
                new CheckSbmChk61   (new CheckSbm.InitializeParameter(CheckItemId.SbmChk61,   "Issue_10_613", ErrorType.Error, TargetContent.Aoc)),
                new CheckSbmChk39_3 (new CheckSbm.InitializeParameter(CheckItemId.SbmChk39_3, "Issue_10_814", ErrorType.Error, TargetContent.Aoc)),
                new CheckSbmChk47a  (new CheckSbm.InitializeParameter(CheckItemId.SbmChk47a,  "Issue_10_609", ErrorType.Error, TargetContent.App | TargetContent.Patch)),
                new CheckSbmChk47b  (new CheckSbm.InitializeParameter(CheckItemId.SbmChk47b,  "Issue_10_048", ErrorType.Error, TargetContent.App | TargetContent.Patch)),
            };

        // チェック項目のID
        public enum CheckItemId
        {
            // 開発用APIの検出
            SbmChk23b, // スタティックライブラリ内の Debug API 使用チェック
            SbmChk23c, // DLL内のDebug API 使用チェック

            // ソフトリーガル情報の整合性確認
            SbmChk26a, // ソフトリーガル情報の内容確認(ソフトリーガル情報の表記不足)
            SbmChk26b, // ソフトリーガル情報の内容確認(不要なソフトリーガル情報の表記)
            SbmChk41a, // ソフトリーガル情報のApplicationIDの確認
            SbmChk41b, // ソフトリーガル情報のHashの確認
            SbmChk28,  // ソフトリーガル情報の存在

            // 開発用ライブラリの検出
            SbmChk35,  // Debug版, Develop版のライブラリ

            // ビルドタイプの確認
            SbmChk21,  // ビルドタイプ

            // アプリのSDKバージョンとツールのバージョンの整合性確認
            SbmChk42,  // アプリとツールのSDKバージョンの整合性

            // SDKバージョンの確認
            SbmChk20c, // SDKバージョン(SnapshotSDKの使用禁止)
            SbmChk33d, // AuthoringTool のツールバージョン(Snapshot SDKの使用禁止)

            // DESCの確認
            SbmChk24a, // 特別なDESCの使用(製品化不可能)
            SbmChk24b, // 特別なDESCの使用(使用許可制DESC)

            // バージョンの確認
            SbmChk48,  // Private Version

            // NEX バージョンの確認
            SbmChk63,  // 脆弱性のあるNEXの使用禁止

            // アプリケーション管理データの設定確認
            SbmChk11,   // タイトル名
            SbmChk12,   // Publisher名
            SbmChk12_1, // タイトルの言語
            SbmChk12_2, // タイトルの言語
            SbmChk12_3, // タイトルの言語未設定
            SbmChk12_4, // タイトルの未定義な言語設定
            SbmChk12_5, // 指定できない文字設定(日米欧フォント)
            SbmChk12_6, // 指定できない文字設定(ASCII)
            SbmChk12_7, // 文字数の超過
            SbmChk13,   // 表示バージョン
            SbmChk15,   // 対応言語
            SbmChk1_8,  // 提出ファイルの正当性 (アイコンが存在しない)
            SbmChk9a,   // 提出ファイルの正当性 (アイコンと言語が不正)
            SbmChk9c,   // アイコンがSDKアイコンでないか？
            SbmChk52b,  // BCATセーブデータの確認 (0.x系、1.x系でのBCATセーブデータの使用の禁止)
            SbmChk52c,  // BCATセーブデータの確認 (64MiBを超えるBcatDeliveryCacheStorageSizeの禁止)
            SbmChk52d,  // BCATセーブデータの確認 (不正な設定)
            SbmChk43b,  // 他のアプリケーションのセーブデータ設定に自分のID設定の禁止
            SbmChk18c,  // 使用されていないアプリのエラーコード

            // XpadAPIの禁止
            SbmChk54,   // XpadのAPIの使用禁止

            // PrivateAPIの検出
            SbmChk22,   // 使用許可制APIのチェック(エラー)

            // 特別許可が必要な機能の確認
            SbmChk8a,  // ユーザーアカウントセーブデータのサイズ(特別承認)
            SbmChk8b,  // ユーザーアカウントセーブデータ保存領域拡張最大サイズ
            SbmChk8c,  // ユーザーアカウントセーブジャーナル領域拡張最大サイズ
            SbmChk8d,  // 本体セーブデータ保存領域拡張最大サイズ
            SbmChk8e,  // 本体セーブジャーナル領域拡張最大サイズ
            SbmChk8f,  // 一時ストレージサイズ
            SbmChk8g,  // キャッシュストレージデータ保存領域サイズ
            SbmChk8h,  // キャッシュストレージジャーナル領域サイズ
            SbmChk8i,  // 本体セーブデータのデータ保存領域サイズ
            SbmChk8j,  // 本体セーブデータのジャーナル領域サイズ
            SbmChk8k,  // 特別承認が必要なnspファイル（複数キャッシュストレージを使用）
            SbmChk16,  // プレゼンスグループID
            SbmChk17a, // ローカル通信識別子 (Application ID と異なる)
            SbmChk27a, // パッチのサイズ(特別承認)
            SbmChk29b, // 体験版 等のROMサイズ(エラー)
            SbmChk43,  // 他アプリのセーブデータアクセス
            SbmChk50a, // アプリの最大サイズの確認
            SbmChk50b, // パッチの最大サイズの確認
            SbmChk55,  // PiaChatの使用禁止
            SbmChk38a, // 追加コンテンツの一コンテンツのサイズ（ガイドラインサイズ）
            SbmChk38b, // 追加コンテンツの一コンテンツのサイズ（検証最大サイズ）
            SbmChk56,  // HDCPの使用禁止
            SbmChk57,  // オートブート設定の使用禁止
            SbmChk58,  // ポインティング機能の使用
            SbmChk65b, // マルチアプリケーションカードの確認（規定数以上のアプリケーション数）

            // カード容量の確認
            SbmChk7c, // 提出不可のカード容量

            // AuthoringToolのみでチェックする基本的な内容
            SbmChk1_10, // プログラムが含まれていないアプリケーション
            SbmChk1_9b, // NXアイコンがベースライン形式でない
            SbmChk66_1, // 個別プログラム間のアプリケーション管理データの不一致

            // D4Cでチェックしている内容
            SbmChk1_7, // スマホアイコンとNXアイコンの同一性

            // コンテンツメタの設定確認
            SbmChk39_1, // AOCのタグがあるか？
            SbmChk39_2, // AOCのタグがユニークであるか？
            SbmChk61,   // AOCのタグが128文字以内であるか？
            SbmChk39_3, // AOCのIndexが指定されていない

            // HtmlDocumentの設定確認
            SbmChk47a, // アクセス可能URL
            SbmChk47b, // アクセス可能URLのhttp禁止
        }
        public enum ErrorType
        {
            Warning,
            Error,
        }
        private enum TargetContent : uint
        {
            App = (1 << 0),
            Patch = (1 << 1),
            Aoc = (1 << 2),
            MultiApplicationCard = (1 << 3),
        }
        public static CheckItemId[] GetMandatoryCheckList(CheckItemId[] checkList)
        {
            // --error-unpublishable の有無に関係なく実施するチェック項目を抽出する
            return checkList.Where(entry => NintendoSubmissionPackageMandatoryCheckList.Contains(entry)).ToArray();
        }
        public static CheckItemId[] ReadCheckList(string filePath)
        {
            if (!File.Exists(filePath))
            {
                return null;
            }

            var lines = File.ReadAllLines(filePath);

            // ファイル内の"xx-xxx"を取得して、メッセージIDが一致するチェック項目の項目IDを取得
            var list = new List<CheckItemId>();
            foreach (var line in lines)
            {
                var match = Regex.Match(line, @"^([0-9]{2})-([0-9]{3})$");
                if (!match.Success)
                {
                    continue;
                }
                var id = "Issue_" + match.Groups[1] + "_" + match.Groups[2];
                var item = CheckTable.Find(entry => entry.MessageId == id);
                if (item != null)
                {
                    list.Add(item.Id);
                }
            }

            return list.ToArray();
        }
        public static void ConvertUnpublishableErrorType(UnpublishableErrorModel unpublishableErrorModel, ErrorType before, ErrorType after, CheckItemId[] ignoreList)
        {
            var beforeType = before.ToString();
            var afterType = after.ToString();

            // 変換しないIssue番号リストを作成
            List<string> issueList = new List<string>();
            if (ignoreList != null)
            {
                issueList.AddRange(CheckTable.FindAll(entry => ignoreList.Contains(entry.Id)).Select(entry => entry.GetMessageIdForDisplay()).ToList());
            }

            // ignoreListに該当しないチェック項目のエラー属性を変換する
            unpublishableErrorModel.Contents.ForEach(content => content.ErrorList.ForEach(error =>
            {
                if (error.Message != null && !issueList.Contains(error.Message.Id) && error.Type == beforeType)
                {
                    error.Type = afterType;
                }
            }));

            return;
        }
        public static void DeleteUnpublishableErrorType(UnpublishableErrorModel unpublishableErrorModel, ErrorType type)
        {
            var deleteType = type.ToString();

            // 指定したエラーを削除する
            unpublishableErrorModel.Contents.ForEach(content => content.ErrorList.RemoveAll(error => error.Type == deleteType));

            return;
        }
        public static List<UnpublishableErrorDetailModel> GetUnpublishableErrorCheck(UnpublishableErrorCheckData checkData)
        {
            List<UnpublishableErrorDetailModel> errorList =
                new List<UnpublishableErrorDetailModel>();

            if (checkData.CheckTarget == UnpublishableErrorCheckData.CheckTargetType.None)
            {
                // チェック対象が未指定の場合はエラー
                throw new Exception("Target is not specified.");
            }

            List<CheckSbm> checkTable = CheckTable;
            if (checkData.CheckInfo.CheckList != null)
            {
                checkTable = CheckTable.FindAll(entry => checkData.CheckInfo.CheckList.Contains(entry.Id));
            }
            if (checkData.CheckInfo.IgnoreList != null)
            {
                // 無視するチェック項目を以外を抽出
                checkTable = checkTable.FindAll(entry => !checkData.CheckInfo.IgnoreList.Contains(entry.Id));
            }

            // チェック対象となる項目IDに関してチェックを実施してErrorモデルを取得
            foreach (var checker in checkTable.Where(c => c.IsCheckTarget(checkData)))
            {
                errorList.AddRange(checker.DoCheck(checkData));
            }
            return errorList;
        }

        public static List<Tuple<string, byte[]>> GetLegalInformationFileListForUnpublishableErrorCheck(NintendoSubmissionPackageFileSystemInfo.ContentInfo nca, NcaKeyGenerator key)
        {
            // チェックに必要なリーガル情報内のファイル
            string[] needFiles =
            {
                UnpublishableErrorCheckData.LegalInfoIndexHtmlPath
            };

            var fileList = new List<Tuple<string, byte[]>>();
            using (var ncaStream = new SourceBasedStream(nca.Source))
            using (var ncaReader = new NintendoContentArchiveReader(ncaStream, key))
            {
                foreach (var ncaInfo in ncaReader.ListFsInfo())
                {
                    var fsReader = ncaReader.OpenFileSystemArchiveReader(ncaInfo.Item1);
                    var files = fsReader.ListFileInfo();
                    foreach (var file in files)
                    {
                        if (needFiles.Contains(file.Item1))
                        {
                            fileList.Add(Tuple.Create(file.Item1, fsReader.ReadFile(file.Item1, 0, file.Item2)));
                        }
                    }
                }
            }
            return (fileList.Count() > 0) ? fileList : null;
        }

        private static string GetApplicationId(ContentMetaModel contentMeta)
        {
            var id = contentMeta.Id;

            if (contentMeta.Type == NintendoContentMetaConstant.ContentMetaTypePatch)
            {
                // パッチの場合はパッチコンテンツメタとしてアプリケーションIDを取得する
                var patchContentMeta = contentMeta as PatchContentMetaModel;
                id = patchContentMeta.ApplicationId;
            }

            return id;
        }
        private static bool IsEqualApplicationId(string id1, string id2)
        {
            if (id1 == null || id2 == null)
            {
                return id1 == id2;
            }

            return Convert.ToUInt64(id1, 16) == Convert.ToUInt64(id2, 16);
        }

        private class CheckSbm
        {
            protected const string ErrorItemSeparator = ", ";
            protected readonly List<UnpublishableErrorDetailModel> ResultSuccess = new List<UnpublishableErrorDetailModel> { };

            // サポートするカルチャのリスト
            private static readonly List<string> CultureList = new List<string>() { "ja", "en" };
            private TargetContent checkTargetContent;
            private string errorMessage;

            public CheckSbm(InitializeParameter param)
            {
                Id = param.CheckItemId;
                MessageId = param.MessageId;
                checkTargetContent = param.TargetContent;
                errorMessage = param.ErrorType.ToString();
            }

            public CheckItemId Id { get; }
            public string MessageId { get; }

            public virtual List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData) { return null; }
            public bool IsCheckTarget(UnpublishableErrorCheckData checkData)
            {
                switch (checkData.CheckContentType)
                {
                    case NintendoContentMetaConstant.ContentMetaTypeApplication:
                        return (checkTargetContent & TargetContent.App) != 0;
                    case NintendoContentMetaConstant.ContentMetaTypePatch:
                        return (checkTargetContent & TargetContent.Patch) != 0;
                    case NintendoContentMetaConstant.ContentMetaTypeAddOnContent:
                        return (checkTargetContent & TargetContent.Aoc) != 0;
                    case UnpublishableErrorCheckData.CheckContentTypeMultiApplicationCard:
                        return (checkTargetContent & TargetContent.MultiApplicationCard) != 0;
                    case UnpublishableErrorCheckData.CheckContentTypeForce:
                        return true;
                    default:
                        throw new ArgumentException();
                }
            }
            public string GetMessageIdForDisplay()
            {
                // メッセージIDを表示用に変換
                Regex replaceRegex = new Regex(@"([a-zA-Z0-9]+)_([0-9]+)_([0-9]+)(_.*)?");
                return replaceRegex.Replace(MessageId, @"$1 $2-$3");
            }
            protected Dictionary<string, string> GetMessageResource(string messageId)
            {
                // メッセージIDに対応するリソースを取得
                var messageDictionary = new Dictionary<string, string>();
                foreach (var culture in CultureList)
                {
                    string message =
                            AuthoringTool.Resources.ResourceManager.GetString(
                                messageId, CultureInfo.GetCultureInfo(culture));
                    messageDictionary.Add(culture, message);
                }
                return messageDictionary;
            }
            // エラーメッセージフォーマットに引数が1つ以上必要なエラーを複数作成
            protected List<UnpublishableErrorDetailModel> CreateUnpublishableErrorDetailModel(List<string[]> messageArgsList, string appendMessageForJapanese = null, string appendMessageForEnglish = null)
            {
                // チェック項目に対応するメッセージIDの表示用文字列を取得
                var messageIdDisplay = GetMessageIdForDisplay();

                // チェック項目に対応するリソースを取得
                var resource = GetMessageResource();

                var errorList = new List<UnpublishableErrorDetailModel>();
                foreach (var messageArgs in messageArgsList)
                {
                    // xmlに出力するTitleとDescriptionを取得
                    string japaneseTitle = string.Format(resource["ja"].Item1, messageArgs);
                    string japaneseDescription = string.Format(resource["ja"].Item2, messageArgs);
                    string englishTitle = string.Format(resource["en"].Item1, messageArgs);
                    string englishDescription = string.Format(resource["en"].Item2, messageArgs);

                    // Errorモデルを追加
                    errorList.Add(
                        UnpublishableErrorDetailModel.Create(
                            errorMessage,
                            UnpublishableErrorMessageModel.Create(
                                messageIdDisplay,
                                UnpublishableErrorTitleModel.Create(japaneseTitle, englishTitle),
                                UnpublishableErrorDescriptionModel.Create(japaneseDescription + appendMessageForJapanese, englishDescription + appendMessageForEnglish))));
                }
                return errorList;
            }
            // エラーメッセージフォーマットに引数が1つ必要なエラーを複数作成
            protected List<UnpublishableErrorDetailModel> CreateUnpublishableErrorDetailModel(List<string> messageList, string appendMessageForJapanese = null, string appendMessageForEnglish = null)
            {
                return CreateUnpublishableErrorDetailModel(messageList.Select(entry => { return new string[] { entry }; }).ToList(), appendMessageForJapanese, appendMessageForEnglish);
            }
            // エラーメッセージフォーマットに引数が1つ以上必要なエラーを作成
            protected List<UnpublishableErrorDetailModel> CreateUnpublishableErrorDetailModel(string[] messageArgs, string appendMessageForJapanese = null, string appendMessageForEnglish = null)
            {
                return CreateUnpublishableErrorDetailModel(new List<string[]> { messageArgs }, appendMessageForJapanese, appendMessageForEnglish);
            }
            // エラーメッセージフォーマットに引数が1つ必要なエラーを作成
            protected List<UnpublishableErrorDetailModel> CreateUnpublishableErrorDetailModel(string messageArg, string appendMessageForJapanese = null, string appendMessageForEnglish = null)
            {
                return CreateUnpublishableErrorDetailModel(new List<string[]> { new string[] { messageArg } }, appendMessageForJapanese, appendMessageForEnglish);
            }
            // エラーメッセージフォーマットに引数がないエラーを作成
            protected List<UnpublishableErrorDetailModel> CreateUnpublishableErrorDetailModel()
            {
                return CreateUnpublishableErrorDetailModel(new List<string[]> { new string[] { string.Empty } });
            }
            protected List<string> JoinStringList(List<string> stringList)
            {
                if (stringList.Count() > 0)
                {
                    // 文字列を改行区切りで結合する
                    var name = string.Join(Environment.NewLine, stringList);
                    return new List<string>() { name };
                }
                return stringList;
            }
            private Dictionary<string, Tuple<string, string>> GetMessageResource()
            {
                string titleId = MessageId + "_Title";
                string descriptionId = MessageId + "_Description";

                // メッセージIDに対応するリソースを取得
                var messageDictionary = new Dictionary<string, Tuple<string, string>>();
                foreach (var culture in CultureList)
                {
                    string titleFormat =
                            AuthoringTool.Resources.ResourceManager.GetString(
                                titleId, CultureInfo.GetCultureInfo(culture));
                    string descriptionFormat =
                            AuthoringTool.Resources.ResourceManager.GetString(
                                descriptionId, CultureInfo.GetCultureInfo(culture));
                    Tuple<string, string> messageSet = new Tuple<string, string>(titleFormat, descriptionFormat);
                    messageDictionary.Add(culture, messageSet);
                }
                return messageDictionary;
            }

            public class InitializeParameter
            {
                public InitializeParameter(CheckItemId checkItemId, string messageId, ErrorType errorType, TargetContent targetContent)
                {
                    CheckItemId = checkItemId;
                    MessageId = messageId;
                    ErrorType = errorType;
                    TargetContent = targetContent;
                }
                public CheckItemId CheckItemId { get; set; }
                public string MessageId { get; set; }
                public ErrorType ErrorType { get; set; }
                public TargetContent TargetContent { get; set; }
            }
        }

        // スタティックライブラリ内の Debug API 使用チェック
        private class CheckSbmChk23b : CheckSbm
        {
            private static readonly Dictionary<string, string> ApiNameConvertTable =
                new Dictionary<string, string>()
                {
                    { "NEX-DebugLoginDirect", "nn::nex::NgsFacade::DebugLoginDirect" },
                };

            public CheckSbmChk23b(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                // programinfo.xmlがない場合はチェックをスキップ
                if (checkData.Nsp.ProgramInfo == null)
                {
                    return ResultSuccess;
                }

                var apiList = checkData.Nsp.ProgramInfo.DebugApiListData.GetApiNameList();

                apiList = apiList.Select(entry =>
                {
                    if (ApiNameConvertTable.ContainsKey(entry))
                    {
                        // APIを変換
                        return ApiNameConvertTable[entry];
                    }
                    {
                        return entry;
                    }
                }).ToList();

                apiList = JoinStringList(apiList);

                return CreateUnpublishableErrorDetailModel(apiList);
            }
        }

        // DLL内のDebug API 使用チェック
        private class CheckSbmChk23c : CheckSbm
        {
            public CheckSbmChk23c(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                // programinfo.xmlがない場合はチェックをスキップ
                if (checkData.Nsp.ProgramInfo == null)
                {
                    return ResultSuccess;
                }

                // programinfoに含まれるUnresolvedApiのリストを取得
                var apiList = checkData.Nsp.ProgramInfo.UnresolvedApiListData.GetApiNameList();

                var unpublishableApiList = new List<string>();
                foreach (var manglingApiName in apiList)
                {
                    if (!UnpublishableApiList.IsUnpublishableDebugApi(manglingApiName))
                    {
                        // 該当のAPIが使用禁止でない場合はスキップ
                        continue;
                    }

                    // マングルされた関数名からマングル前の関数を取得
                    unpublishableApiList.Add(UnpublishableApiList.GetDemanglingApiNameForDebugApi(manglingApiName));
                }

                unpublishableApiList = JoinStringList(unpublishableApiList);

                return CreateUnpublishableErrorDetailModel(unpublishableApiList);
            }
        }

        // ソフトリーガル情報の表記不足チェック
        private class CheckSbmChk26a : CheckSbm
        {
            // Application/LogoTypeがLicensedByNintendoの場合にチェック対象から除外するTechnology
            private static readonly string[] NintendoLicensedTechnologies =
            {
                "Havok",
                "UnrealEngine",
            };

            public CheckSbmChk26a(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                // programinfo.xml,legalinfo.xmlがない場合はチェックをスキップ
                if (checkData.Nsp.ProgramInfo == null || checkData.Nsp.LegalInformation == null)
                {
                    return ResultSuccess;
                }

                // programinfoに含まれるModuleNameのリストを取得
                var moduleList = checkData.Nsp.ProgramInfo.MiddlewareListData.GetModuleNameList();

                // legalinfoに含まれるTechnologyNameのリストを取得
                List<string> copyrightTechnologyNames = new List<string>();
                List<string> requiredTechnologyNames = new List<string>();
                if (checkData.Nsp.LegalInformation.CopyrightNotations != null)
                {
                    copyrightTechnologyNames = DeleteSkipTechnology(checkData.Nsp.LegalInformation.CopyrightNotations.TechnologyName, checkData);
                }
                if (checkData.Nsp.LegalInformation.DeclarationRequiredNotations != null)
                {
                    requiredTechnologyNames = DeleteSkipTechnology(checkData.Nsp.LegalInformation.DeclarationRequiredNotations.TechnologyName, checkData);
                }

                var requiredUseList = new List<string>();
                // ModuleNameの中でDeclarationRequiredNotationsのいずれかが前方一致するものを抽出
                foreach (var module in moduleList)
                {
                    requiredUseList.AddRange(requiredTechnologyNames.FindAll(required => module.StartsWith(required)));
                }
                if (requiredUseList.Count() == 0)
                {
                    // DeclarationRequiredNotationsの内、ModuleNameのいずれかで前方一致するものがない
                    return ResultSuccess;
                }

                var requiredOnlyList = new List<string>();
                // 使用しているDeclarationRequiredNotationsの内、CopyrightNotationsに含まれないものを抽出
                foreach (var required in requiredUseList)
                {
                    if (copyrightTechnologyNames.Contains(required))
                    {
                        continue;
                    }
                    requiredOnlyList.Add(required);
                }

                requiredOnlyList = JoinStringList(requiredOnlyList);

                return CreateUnpublishableErrorDetailModel(requiredOnlyList);
            }
            protected List<string> DeleteSkipTechnology(List<string> technologyList, UnpublishableErrorCheckData checkData)
            {
                if (checkData.ApplicationControlProperty.LogoType == "LicensedByNintendo")
                {
                    technologyList = technologyList.FindAll(entry => !NintendoLicensedTechnologies.Contains(entry));
                }

                return technologyList;
            }
        }

        // 不要なソフトリーガル情報の表記チェック
        private class CheckSbmChk26b : CheckSbmChk26a
        {
            public CheckSbmChk26b(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                // programinfo.xml,legalinfo.xmlがない場合はチェックをスキップ
                if (checkData.Nsp.ProgramInfo == null || checkData.Nsp.LegalInformation == null)
                {
                    return ResultSuccess;
                }

                // nspに含まれるModuleNameのリストを取得
                var moduleList = checkData.Nsp.AllMiddlewareListInNsp;

                // legalinfoに含まれるTechnologyNameのリストを取得
                List<string> copyrightTechnologyNames = new List<string>();
                List<string> requiredTechnologyNames = new List<string>();
                if (checkData.Nsp.LegalInformation.CopyrightNotations != null)
                {
                    copyrightTechnologyNames = DeleteSkipTechnology(checkData.Nsp.LegalInformation.CopyrightNotations.TechnologyName, checkData);
                }
                if (checkData.Nsp.LegalInformation.DeclarationRequiredNotations != null)
                {
                    requiredTechnologyNames = DeleteSkipTechnology(checkData.Nsp.LegalInformation.DeclarationRequiredNotations.TechnologyName, checkData);
                }

                var requiredList = new List<string>();
                // CopyrightNotationsの内、DeclarationRequiredNotationsに含まれるものを抽出
                foreach (var copyright in copyrightTechnologyNames)
                {
                    requiredList.AddRange(requiredTechnologyNames.FindAll(required => copyright == required));
                }
                if (requiredList.Count() == 0)
                {
                    // CopyrightNotationsに、DeclarationRequiredNotationsに含まれるものがない
                    return ResultSuccess;
                }

                var unuseModuleList = new List<string>();
                // ModuleNameの中でDeclarationRequiredNotationsのいずれかに前方一致しないものを抽出
                foreach (var required in requiredList)
                {
                    if (moduleList.FindAll(module => module.StartsWith(required)).Count() != 0)
                    {
                        // 前方一致するものが１つ以上存在する
                        continue;
                    }
                    unuseModuleList.Add(required);
                }

                unuseModuleList = JoinStringList(unuseModuleList);

                return CreateUnpublishableErrorDetailModel(unuseModuleList);
            }
        }

        // ソフトリーガル情報の存在
        private class CheckSbmChk28 : CheckSbm
        {
            public CheckSbmChk28(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                // ContentMetaにLegalInformationが存在するかを確認
                if (checkData.Nsp.ContentMeta?.ContentList == null ||
                    checkData.Nsp.ContentMeta.ContentList.FindAll(entry => entry.Type == NintendoContentMetaConstant.ContentTypeLegalInformation).Count() == 0)
                {
                    // ContentMetaにLegalInformationが存在しない場合はエラー
                    return CreateUnpublishableErrorDetailModel();
                }

                return ResultSuccess;
            }
        }

        // ソフトリーガル情報のApplicationIDの確認
        private class CheckSbmChk41a : CheckSbm
        {
            public CheckSbmChk41a(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                // legalinfo.xmlがない場合はチェックをスキップ
                if (checkData.Nsp.LegalInformation == null)
                {
                    return ResultSuccess;
                }

                var id = GetApplicationId(checkData.Nsp.ContentMeta);

                if (Convert.ToUInt64(id, 16) == Convert.ToUInt64(checkData.Nsp.LegalInformation.ApplicationId, 16))
                {
                    return ResultSuccess;
                }

                // ContentMetaとLegalInformationでアプリケーションIDが一致しない
                return CreateUnpublishableErrorDetailModel(new string[] { id, checkData.Nsp.LegalInformation.ApplicationId });
            }
        }

        // ソフトリーガル情報のHashの確認
        private class CheckSbmChk41b : CheckSbm
        {
            public CheckSbmChk41b(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                // legalinfo.xmlがない場合 or フォーマットバージョンが2.x.xより前の場合はチェックをスキップ
                if (checkData.Nsp.LegalInformation == null ||
                    Convert.ToInt32(checkData.Nsp.LegalInformation.FormatVersion.Split('.').First()) < 2)
                {
                    return ResultSuccess;
                }

                // ソフトリーガル情報の index.html を探す
                if (checkData.Nsp.LegalInformationFileList == null ||
                    checkData.Nsp.LegalInformationFileList.Find(entry => entry.Item1 == UnpublishableErrorCheckData.LegalInfoIndexHtmlPath) == null)
                {
                    // ファイルが見つからない
                    return CreateUnpublishableErrorDetailModel();
                }

                if (checkData.Nsp.LegalInformation.DataHash.ToLower() != GetMd5(checkData.Nsp.LegalInformationFileList.Find(entry => entry.Item1 == UnpublishableErrorCheckData.LegalInfoIndexHtmlPath).Item2))
                {
                    // Hash値が一致しない
                    return CreateUnpublishableErrorDetailModel();
                }

                return ResultSuccess;
            }

            private string GetMd5(byte[] srcData)
            {
                var builder = new StringBuilder();
                var md5 = new MD5CryptoServiceProvider();
                var data = md5.ComputeHash(srcData);
                for (int i = 0; i < data.Length; i++)
                {
                    builder.Append(data[i].ToString("x2"));
                }
                return builder.ToString();
            }
        }

        // Debug版, Develop版のライブラリ
        private class CheckSbmChk35 : CheckSbm
        {
            public CheckSbmChk35(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                // programinfo.xmlがない場合はチェックをスキップ
                if (checkData.Nsp.ProgramInfo == null)
                {
                    return ResultSuccess;
                }

                // programinfoに含まれるMiddlewareのリストを取得
                var moduleNameList = checkData.Nsp.ProgramInfo.MiddlewareListData.GetModuleNameList();

                var ngNameList = new List<string>();
                foreach (var name in moduleNameList)
                {
                    if (name.Contains("Debug") == false && name.Contains("Develop") == false)
                    {
                        // "Debug"と"Develop"が含まれていない
                        continue;
                    }

                    // モジュール名をリストに追加
                    ngNameList.Add(name);
                }

                ngNameList = JoinStringList(ngNameList);

                return CreateUnpublishableErrorDetailModel(ngNameList);
            }
        }

        // ビルドタイプの確認
        private class CheckSbmChk21 : CheckSbm
        {
            public CheckSbmChk21(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                // programinfo.xmlがない場合はチェックをスキップ
                if (checkData.Nsp.ProgramInfo == null)
                {
                    return ResultSuccess;
                }

                List<string> buildType = new List<string>();
                // programinfoに含まれるProgramInfo/BuildTypeを取得
                if (checkData.Nsp.ProgramInfo.BuildType != "Release")
                {
                    buildType.Add(checkData.Nsp.ProgramInfo.BuildType);
                }

                return CreateUnpublishableErrorDetailModel(buildType);
            }
        }

        // アプリのSDKバージョンとツールのバージョンの整合性確認
        private class CheckSbmChk42 : CheckSbm
        {
            public CheckSbmChk42(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                // programinfo.xmlがない場合はチェックをスキップ
                if (checkData.Nsp.ProgramInfo == null)
                {
                    return ResultSuccess;
                }

                // programinfoに含まれるProgramInfo/SdkVersionのメジャーバージョンを取得
                var sdkVersion = checkData.Nsp.ProgramInfo.SdkVersion ?? "null";
                var sdkMajorVersion = sdkVersion.Split('_').First();

                // programinfoに含まれるProgramInfo/ToolVersionのメジャーバージョンを取得
                var toolMajorVersion = checkData.Nsp.ProgramInfo.ToolVersion.Split('_').First();

                if (sdkMajorVersion == toolMajorVersion)
                {
                    return ResultSuccess;
                }

                return CreateUnpublishableErrorDetailModel(new string[] { sdkVersion.Replace('_', '.'), checkData.Nsp.ProgramInfo.ToolVersion.Replace('_', '.') });
            }
        }

        // SDKバージョン(Snapshot SDKの使用禁止)
        private class CheckSbmChk20c : CheckSbm
        {
            public CheckSbmChk20c(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                // programinfo.xmlがない場合はチェックをスキップ
                if (checkData.Nsp.ProgramInfo == null)
                {
                    return ResultSuccess;
                }

                // programinfoに含まれるProgramInfo/SdkVersionのバージョンを取得
                var sdkVersion = checkData.Nsp.ProgramInfo.SdkVersion ?? "null";

                if (sdkVersion.EndsWith("_99") == true)
                {
                    // バージョンの末尾が"_99"で終わる場合はエラー
                    return CreateUnpublishableErrorDetailModel(sdkVersion.Replace('_', '.'));
                }

                return ResultSuccess;
            }
        }

        // AuthoringTool のツールバージョン(Snapshot SDKの使用禁止)
        private class CheckSbmChk33d : CheckSbm
        {
            public CheckSbmChk33d(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                // programinfo.xmlがない場合はチェックをスキップ
                if (checkData.Nsp.ProgramInfo == null)
                {
                    return ResultSuccess;
                }

                // programinfoに含まれるProgramInfo/ToolVersionのバージョンを取得
                var toolVersion = checkData.Nsp.ProgramInfo.ToolVersion ?? "null";

                // ToolVersionのバージョンが "X_Y_99:A_B_C" の場合にエラー
                if (Regex.IsMatch(toolVersion, @"[^_]+_[^_]+_99:.*"))
                {
                    // バージョンの末尾が"_99"で終わる場合はエラー
                    return CreateUnpublishableErrorDetailModel(toolVersion.Replace('_', '.'));
                }

                return ResultSuccess;
            }
        }

        // 特別なDESCの使用(製品化不可能)
        private class CheckSbmChk24a : CheckSbm
        {
            private static readonly string[] UnpublishableDesc =
            {
                "rJWdlhkpc5IJ8WocWnan4CD3aQaarfe6DfIfL0YJMMUy+nxqnm7q46208nIdtElX/b0SdB4kQV7rmJDkN5bawI2ESZgLA89FngWxfWswPBXyF7805W6JX3/FbM+DmxaDk2/2GXuuOoChjFa0lNGZcmwewxEeOR+Dzy2p9iRVdy5D3P7D/Rg6i+7k+l5Um9hLYI1957dWfxJWUsAkEsWdsV4SJtiPGH7DJ7OGpVFMOOPW0MB5AGD1HhlUUflLAJcgp1qcaAM2sI1jnZVDZ0cFTVX/E40cPStftx4AAc5SB3HE8NVU4LohLOGva7l1ThmUhuwEvAEWF3KGVY76t1P0heDlRSQ5nSXASENNWdreAwzAkg/rbLCdqeedAy1mXK9DOp/ICuJx3vnOGwBNwkGpYcMhG6CC0my0SDF97N5qM12KLNpQtILgODC5uw3l0fYQjCS/j5phF+N3Fp+IruBtzAyDcK4fGNttGdNtIMJadrBe3vfFwKMorA0TlC6/NZkc70VAJpcP6MFQcqdileF2gYNnzY4t5/oWoJmZS/S4/TxY55G3G/W78hMvlQvgTpHnURDX1qCB7eDdFJtGgkPd1YhZgDkh+LzVkBwBiXqG9pUfKFloO1yzt7NG04x85RrSuHC9/a8VGDCrHrDpTCNiY5vEHe7OqpLdQl1RxHHKn0tBQ0lEpAIAAAAAAAAAAAAAAAABAAAAAAH/////////AUACAAAsAAAAcAIAAAgBAACAAwAAJAAAAAAAAAAAAAAAAQAAAAAAAAAAAABAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAABWFjYzp1MARhb2M6dQJhcG0EYXBtOnAHYXBwbGV0T0UGYXVkaW46dQdhdWRvdXQ6dQdhdWRyZW46dQRic2Q6dQVic2RjZmcEY3NybmcCZmdtB2ZyaWVuZDp1BmZzcC1zcnYCaGlkAmh0YwdodGM6dGVudgNodGNzBWh3b3B1cwJpcnMEbGRuOnUFbGRyOnJvAWxtBG1paTp1A21tOnUHbmZjOm1mOnUHbmZjOnVzZXIHbmZwOnVzZXIFbmlmbTp1BG5zZDp1Am50YwRudmRydgJwY20DcGN0bANwbDp1BnByZXBvOnUCc2V0B3NmZG5zcmVzAnNzbAV0aW1lOnUDdHNwbQN2aTp1AAAAAAAAAAC3cwACz///H+//fzCPAQBADwAAcP9fAAD/PxgA/38AAv//AgA=",
                "XHPzuv3LYKRsmsvegp2i7ulPuXDKYh9+f65jJZRXMyr49RJ/oV4mA/Kei1pWJrL86INPZ3JVmqHYKOs6oqqwx3fj7ISNKGMNGM0UJSyqQV+v9eJSZNE/CSb5A+VoZnf5Dn2Uw6Tvq/nNa0mziUAQBMWjScHusb/vR4jZD1XrHw7DussSCVTVgR+4OMw4cHBJcZQlQUUTFQP5oAw9rI7KYZWkOZbs6VBSRERdR7BVcWfVOepjE0u0XJMayRRqxbhEtvzwCyVsgUu4WXu77gvJHj8ov2P822/hch2Nd6otXmAIz1+Bdd+DIAr1gSc8EQobOY+BdiOG1ojQyImYDa5xwN0GL5RoaSYsRVWw40GET33k1Y0KgJckQQSbl29Wjq1cCTFNDIplVu+qI+Gql8RlOwl0OJC74OvK/z4SnN6l2kgGL4L/+tXP5Gvu7Ym48xYoxgpNlFE/JDFMeBBxxykDfyOfsWwnjpk6oMh3gBdYfRwprASEISEm9ziEpquJmVNp4s1Gt5yYqqsdWizMSJbjmh5Unp/i2bWtflgik65QeYCGhIdeoI+4XaLOSxZpvkiU4K5hqKyVq8EAETm7NghqJsbeD6Wr51M4BMleMl1jy+P5FVxx9HyAjgwFvWWJybGPM93jM2bnIRibyCmfAaydKrWE5pWRIPc7shoHlsF91QFBQ0lEpAIAAAAAAAAAAAAAAAABAAAAAAH/////////AUACAAAsAAAAcAIAAAoBAACAAwAAJAAAAAAAAAAAAAAAAQAAAAAAAAAAAABAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAABWFjYzp1MARhb2M6dQJhcG0EYXBtOnAHYXBwbGV0T0UGYXVkaW46dQdhdWRvdXQ6dQdhdWRyZW46dQViY2F0OnUEYnNkOnUFYnNkY2ZnBGNzcm5nAmZnbQdmcmllbmQ6dQZmc3Atc3J2AmhpZAJodGMHaHRjOnRlbnYDaHRjcwVod29wdXMCaXJzBGxkbjp1BWxkcjpybwFsbQRtaWk6dQNtbTp1B25mYzptZjp1B25mYzp1c2VyB25mcDp1c2VyBW5pZm06dQRuc2Q6dQJudGMEbnZkcnYCcGNtA3BjdGwDcGw6dQZwcmVwbzp1AnNldAdzZmRuc3JlcwJzc2wFdGltZTp1A3ZpOnUAAAAAAAC3cwACz///H+//fzCPAQBADwAAcP9fAAD/PzAA/38AAv//AgA=",
                "QM+XUg5nWnSLjsG02YmG4wI7ln0s+zTaOZ5qH4s12PIoHFMwLdJuKX88DVMXpn/1C+IaOFDRDNm60t3f8JDrO+PRowOV+GZAnV5sYpGgIRD2/XWnG4YAwu4rZhECHGS52M5P7q4vPxclhK7QH0m/3pDW8GleW3OqYVTy/+zPCgRU3DDsPrhlDVHSUXipg9JS66ygvDp7+Xc82aU5pqcAKiZheIc4dxCl9dsH5wqX2p2gW/Motn4zAllwa22q9wLmEWNxothpT0guksdWFVQpi4YeDHEMax4CT7UCieTfqWNi3im7Qx0Uowrlsqb2qxkV7DEt5DUF4JHyg3ZSK0HsRfU43+GWYTbfjJL6eCfraQb50iAUR27gS84AOY7RLWQ7wPavJ9oHvKGK97eB6kLeeyohIZdbENSTBK9KNVvvJ7lJmS5PIQtYY9akh3vE7G9Xrs05fWOyzkgmJToKEGZ7F1mtbyqP4L8s6GqdLdRWw/jT9h3g++TaaoVBkLachPdTNoxnXqeUoAZl2/s8w//Fj1j5lcHGJXc+CC60WcG2NL9juJL3DLOcjIrLoKVkLDvOhH5e6eHVlL5xC2bxKW8uN4QU7WQb4lHoXJkM4850r9C2OQdpKtcwR153Q0SlMpu7qeORAVIbv4A/s7jP+InCDEs0Vd8qtlXK8eklPl8RurVBQ0lEtAIAAAAAAAAAAAAAAMAJAHA8AAEAwAkAcDwAAUACAAAsAAAAcAIAABUBAACQAwAAJAAAAAAAAAAAAAAAAQAAAAAAAAAAAABAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAABWFjYzp1MARhb2M6dQJhcG0EYXBtOnAHYXBwbGV0T0UGYXVkaW46dQdhdWRvdXQ6dQdhdWRyZW46dQViY2F0OnUEYnNkOnUFYnNkY2ZnBWNhcHM6YQZjYXBzOnNzBGNzcm5nB2ZyaWVuZDp1BmZzcC1zcnYCaGlkAmh0YwdodGM6dGVudgNodGNzBWh3b3B1cwJpcnMEbGRuOnUFbGRyOnJvAWxtBG1paTp1A21tOnUHbmZjOm1mOnUHbmZjOnVzZXIHbmZwOnVzZXIFbmlmbTp1BG5zZDp1Am50YwRudmRydgJwY20DcGN0bANwbDp1BnByZXBvOnUCc2V0B3NmZG5zcmVzAnNzbAV0aW1lOnUDdmk6dQAAAAAAAAAAAAAAt3MAAs///x/v/38wjwEAQA8AAHD/XwAA/z8wAP9/AAL//wIA",
                "LjhN33KJNWWC+auJMXBf//SfYEuJqKPoLGguJ5i4QtIkT2Y72VNKVX1ZvqMFVXM3RAPHscHRLtgWh/PI7L91X9Cz1R8Adbl7obiAUesQchkPERVjFfAccOCAqwKjtgCgCjfbecygQbZTsMJ2CpltMnE2ZTINmEkGu+nS4Chg0xiRjBq1tAoEKi/fbN+AgLqZ0658umYLuQJgFifTkdAlyxJtc/QcYO4nI5zC78J6yWX+OqNQPPzZEdhVlZCM1IjLFWqGezHRvyKxZJlYstKGErMxTBO9zWRuWC4Ce55N4OciZTBM/CtuC+YHSuqPiduYeWajLuTf/hcl1d7xKzP+P8VKKD3cL2/sz6M9G82GBPSJPi09b8S9cizWGc0GKQIQjYQrhcLnr8yhEChTSDXoGg7CGU3LXgSN8O6pB97lvhjIUCBliwCPy8EhXBRogHm7OCt8FBUqjbHZC959NUcps6dLvqJxxLOzH33MPCtozQqKGJHPhTVx8yR3200HRh07e2lNPKHwzvl3ippMzdnz5DS438PgGbMKdF/7M/q/lAyxsQU3KZotb6Z8aBt+USs31UWT4Kcu4ORdjGvIGX4mRAM0wMItVUycd3agLbwZrODXIr9qnMaw7GMxPvzYZrxfbuvn+os3i+8eZSHt3FGnrGBKTwp7rrUhMcqY14XWeUVBQ0lEtAIAAAAAAAAAAAAAAAAKAMA7AAEAAAoAwDsAAUACAAAsAAAAcAIAABUBAACQAwAAJAAAAAAAAAAAAAAAAQAAAAAAAAAAAABAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAABWFjYzp1MARhb2M6dQJhcG0EYXBtOnAHYXBwbGV0T0UGYXVkaW46dQdhdWRvdXQ6dQdhdWRyZW46dQViY2F0OnUEYnNkOnUFYnNkY2ZnBWNhcHM6YQZjYXBzOnNzBGNzcm5nB2ZyaWVuZDp1BmZzcC1zcnYCaGlkAmh0YwdodGM6dGVudgNodGNzBWh3b3B1cwJpcnMEbGRuOnUFbGRyOnJvAWxtBG1paTp1A21tOnUHbmZjOm1mOnUHbmZjOnVzZXIHbmZwOnVzZXIFbmlmbTp1BG5zZDp1Am50YwRudmRydgJwY20DcGN0bANwbDp1BnByZXBvOnUCc2V0B3NmZG5zcmVzAnNzbAV0aW1lOnUDdmk6dQAAAAAAAAAAAAAAt3MAAs///x/v/38wjwEAQA8AAHD/XwAA/z8wAP9/AAL//wIA",
                "M5iti02lPVW/MBl0AGd3dtfPwIGOZjoVjX3eIlezC77WfupnOzB+GPyO9pL5/IcWMvsk6UcWul0bFSBgirguyaTXZHmCvNgrFRJ+57a+zYsJeG4tDEb7xj5AT2nC9cpHFgSlZg9B53J3QCqk5x8bQGOZheR/S2S6PCV9aNa40SGVimb8al5H/XK14n2cC1sap2O3kfU9CCuVsFt6puio7s941mikYGUGClfoXI6V0kILwFvZHM2Yr+hgnRJ5M2nMmv8pkath6NFcV9Jc3jSKAfCkOYBGR7LqJfIjwJOOr8Fsbzo3OE8Kx73HYNrvUTwo3e4qwhbo4c9O7t8U3ziMTqvrkusrayaVFZTKrpFY71Bz4Zfaij8A/B7EWbk5jmCp1l4i3hpHH/DPnYcToXIGRJGoDKFLiBJMAkMmJet5V/B6MMh2e7FsSbrZfL3lG6uR+aVnguv0fF/0SNZGJgSFBdGoNWlpHt6PD7dpm2Phbi8O8B68RZCkydeuPVd/92BVXl+2e/uzfL4Vn2bouHIG24FwXrVd/4ieKP30X8D3j5EmSNn35nIXgxaYEwOS6RvdQj7eL3v3X+HWAM+EHkwpxCs1cjodz4Tz3grZi57JIGPVbUeogJNdB5BzZTpFFdE30cCXNzRkSyxQOabgU/9K+YR0ad3swgw/IB1Qg72Tu89BQ0lEtAIAAAAAAAAAAAAAACAKAPD4AAEAIAoA8PgAAUACAAAsAAAAcAIAABUBAACQAwAAJAAAAAAAAAAAAAAAAQAAAAAAAAAAAABAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAABWFjYzp1MARhb2M6dQJhcG0EYXBtOnAHYXBwbGV0T0UGYXVkaW46dQdhdWRvdXQ6dQdhdWRyZW46dQViY2F0OnUEYnNkOnUFYnNkY2ZnBWNhcHM6YQZjYXBzOnNzBGNzcm5nB2ZyaWVuZDp1BmZzcC1zcnYCaGlkAmh0YwdodGM6dGVudgNodGNzBWh3b3B1cwJpcnMEbGRuOnUFbGRyOnJvAWxtBG1paTp1A21tOnUHbmZjOm1mOnUHbmZjOnVzZXIHbmZwOnVzZXIFbmlmbTp1BG5zZDp1Am50YwRudmRydgJwY20DcGN0bANwbDp1BnByZXBvOnUCc2V0B3NmZG5zcmVzAnNzbAV0aW1lOnUDdmk6dQAAAAAAAAAAAAAAt3MAAs///x/v/38wjwEAQA8AAHD/XwAA/z8wAP9/AAL//wIA",
            };

            public CheckSbmChk24a(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                // programinfo.xmlがない場合はチェックをスキップ
                if (checkData.Nsp.ProgramInfo == null)
                {
                    return ResultSuccess;
                }

                if (checkData.Nsp.ProgramInfo.DescFlags != null &&
                    checkData.Nsp.ProgramInfo.DescFlags.Production == false &&
                    UnpublishableDesc.Contains(checkData.Nsp.ProgramInfo.Desc) == true)
                {
                    // ProgramInfo/DescFlags/Production が存在して且つfalseの場合で、Descが特定の値の場合はエラー
                    return CreateUnpublishableErrorDetailModel(checkData.Nsp.ProgramInfo.DescFileName);
                }

                return ResultSuccess;
            }
        }

        // 特別なDESCの使用(使用許可制DESC)
        private class CheckSbmChk24b : CheckSbm
        {
            public CheckSbmChk24b(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                // programinfo.xmlがない場合はチェックをスキップ
                if (checkData.Nsp.ProgramInfo == null)
                {
                    return ResultSuccess;
                }

                if (checkData.Nsp.ProgramInfo.DescFlags != null &&
                    checkData.Nsp.ProgramInfo.DescFlags.UnqualifiedApproval == false)
                {
                    // ProgramInfo/DescFlags/UnqualifiedApprovalがfalseであれば警告
                    return CreateUnpublishableErrorDetailModel(checkData.Nsp.ProgramInfo.DescFileName);
                }

                return ResultSuccess;
            }
        }

        // Private Version
        private class CheckSbmChk48 : CheckSbm
        {
            public CheckSbmChk48(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                if (checkData.ApplicationControlProperty?.Version != null &&
                    (Convert.ToUInt64(checkData.ApplicationControlProperty.Version, 16) & 0x0000FFFF) != 0)
                {
                    // nmetaのVersionの下位16bitが0でなければエラー
                    return CreateUnpublishableErrorDetailModel();
                }

                if (checkData.Nsp.ContentMeta?.Version != null &&
                   (checkData.Nsp.ContentMeta.Version & 0x0000FFFF) != 0)
                {
                    // ContentMeta/Versionの下位16bitが0でなければエラー
                    return CreateUnpublishableErrorDetailModel();
                }

                return ResultSuccess;
            }
        }

        // 脆弱性のあるNEXの使用禁止
        private class CheckSbmChk63 : CheckSbm
        {
            // エラーとするNEXのモジュール名
            private static readonly string[] ErrorModuleNameList =
            {
                "NEX-4_0_0-",
                "NEX-4_0_1-",
                "NEX-4_0_2-",
                "NEX-4_0_3-",
                "NEX-4_1_1-",
                "NEX-4_1_2-",
                "NEX-4_2_0-",
                "NEX-4_2_1-",
                "NEX-4_3_0-",
            };

            public CheckSbmChk63(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                // programinfo.xmlがない場合はチェックをスキップ
                if (checkData.Nsp.ProgramInfo == null)
                {
                    return ResultSuccess;
                }

                // programinfoに含まれるModuleNameのリストをチェック
                foreach (var moduleName in checkData.Nsp.ProgramInfo.MiddlewareListData.GetModuleNameList())
                {
                    if (ErrorModuleNameList.Contains(moduleName))
                    {
                        // エラーとするモジュール名に一致する物があればエラー
                        return CreateUnpublishableErrorDetailModel();
                    }
                }

                return ResultSuccess;
            }
        }

        // タイトル名
        private class CheckSbmChk11 : CheckSbm
        {
            public CheckSbmChk11(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                if (checkData.ApplicationControlProperty.Title == null ||
                    checkData.ApplicationControlProperty.Title.Count() == 0 || // Titleエントリがない
                    checkData.ApplicationControlProperty.Title.FindAll(entry => string.IsNullOrEmpty(entry.Name)).Count() != 0) // Nameエントリが存在しないか空文字のTitleがある
                {
                    return CreateUnpublishableErrorDetailModel();
                }

                return ResultSuccess;
            }
        }

        // Publisher名
        private class CheckSbmChk12 : CheckSbm
        {
            public CheckSbmChk12(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                if (checkData.ApplicationControlProperty.Title == null ||
                    checkData.ApplicationControlProperty.Title.Count() == 0 || // Titleエントリがない
                    checkData.ApplicationControlProperty.Title.FindAll(entry => string.IsNullOrEmpty(entry.Publisher)).Count() != 0) // Publisherエントリが存在しないか空文字のTitleがある
                {
                    return CreateUnpublishableErrorDetailModel();
                }

                return ResultSuccess;
            }
        }

        // タイトルの言語
        private class CheckSbmChk12_1 : CheckSbm
        {
            public CheckSbmChk12_1(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                if (checkData.ApplicationControlProperty.Title == null ||
                    checkData.ApplicationControlProperty.Title.Count() == 0 || // Titleエントリがない
                    checkData.ApplicationControlProperty.Title.FindAll(entry => string.IsNullOrEmpty(entry.Language)).Count() != 0) // Languageエントリが存在しないか空文字のTitleがある
                {
                    return CreateUnpublishableErrorDetailModel();
                }

                return ResultSuccess;
            }
        }

        // タイトルの言語
        private class CheckSbmChk12_2 : CheckSbm
        {
            public CheckSbmChk12_2(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                List<string> nspIconLanguageList = new List<string>();
                List<string> acpIconLanguageList = new List<string>();

                // nsp内にRawアイコンがある場合はnspに含まれるIconリストを取得
                if (checkData.Nsp.RawIconList != null)
                {
                    // iconファイルの言語(*.raw.%Language%.jpgの%Language%)を取得
                    nspIconLanguageList.AddRange(checkData.Nsp.RawIconList.Select(entry => Regex.Match(entry.Item1, @"[^\.]+\.raw\.([^\.]+)\.jpg$").Groups[1].Value).ToList());
                }

                // ApplicationControlProperty内のIconリストを取得
                if (checkData.ApplicationControlProperty.Icon != null &&
                    checkData.ApplicationControlProperty.Icon.Count() > 0)
                {
                    // ApplicationControlProperty内のIconエントリに含まれる言語リストを取得
                    acpIconLanguageList.AddRange(checkData.ApplicationControlProperty.Icon.FindAll(entry => !string.IsNullOrEmpty(entry.Language)).Select(entry => entry.Language).ToList());
                }

                if (checkData.ApplicationControlProperty.Title == null ||
                    checkData.ApplicationControlProperty.Title.Count() == 0)
                {
                    // Titleエントリが無いかリストが空

                    if (acpIconLanguageList.Count() == 0 && nspIconLanguageList.Count() == 0)
                    {
                        // IconとTitleが登録されていない場合はチェックをスキップ(SbmChk11,SbmChk1_8でのエラーとなる)
                        return ResultSuccess;
                    }

                    // 必要な言語が設定されていないTitleエントリがある
                    acpIconLanguageList.AddRange(nspIconLanguageList);
                    acpIconLanguageList.Distinct().ToList(); // 重複削除

                    // エラー表示用文字列の作成
                    string errorString = string.Join(ErrorItemSeparator, acpIconLanguageList);

                    return CreateUnpublishableErrorDetailModel(errorString);
                }

                List<string> invalidLanguage = new List<string>();

                // ApplicationControlProperty内のTitleエントリに含まれる言語リストを取得
                var titleLanguageList = checkData.ApplicationControlProperty.Title.FindAll(entry => !string.IsNullOrEmpty(entry.Language)).Select(entry => entry.Language).ToList();

                // nsp内のIconの言語のうちTitleに含まれない言語のリストを取得
                invalidLanguage.AddRange(nspIconLanguageList.FindAll(entry => !titleLanguageList.Contains(entry)));

                // ApplicationControlPropertyのIconエントリの言語のうちTitleエントリに含まれない言語のリストを取得
                invalidLanguage.AddRange(acpIconLanguageList.FindAll(entry => !titleLanguageList.Contains(entry)));

                if (invalidLanguage.Count() > 0)
                {
                    // 必要な言語が設定されていないTitleエントリがある
                    invalidLanguage = invalidLanguage.Distinct().ToList(); // 重複削除

                    // エラー表示用文字列の作成
                    string errorString = string.Join(ErrorItemSeparator, invalidLanguage);

                    return CreateUnpublishableErrorDetailModel(errorString);
                }

                return ResultSuccess;
            }
        }

        // タイトルの言語未設定
        private class CheckSbmChk12_3 : CheckSbm
        {
            public CheckSbmChk12_3(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                if (checkData.ApplicationControlProperty.Title == null ||
                    checkData.ApplicationControlProperty.Title.Count() == 0)
                {
                    // Titleがない場合はスキップ
                    return ResultSuccess;
                }

                if (checkData.ApplicationControlProperty.Title.FindAll(entry => string.IsNullOrWhiteSpace(entry.Language)).Count() != 0) // Languageエントリが存在しないTitleがある
                {
                    return CreateUnpublishableErrorDetailModel();
                }

                return ResultSuccess;
            }
        }

        // タイトルの未定義な言語設定
        private class CheckSbmChk12_4 : CheckSbm
        {
            public CheckSbmChk12_4(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                if (checkData.ApplicationControlProperty.Title == null ||
                    checkData.ApplicationControlProperty.Title.Count() == 0)
                {
                    // Titleがない場合はスキップ
                    return ResultSuccess;
                }

                var errorLanguagelist = new List<string[]>();
                foreach (var title in checkData.ApplicationControlProperty.Title)
                {
                    if (!string.IsNullOrWhiteSpace(title.Language) && !Enum.IsDefined(typeof(Language), title.Language))
                    {
                        // Languageに設定できない言語がある場合、言語毎にエラーを作成する
                        errorLanguagelist.Add(new string[] { title.Language });
                    }
                }

                return CreateUnpublishableErrorDetailModel(errorLanguagelist);
            }
        }

        // 指定できない文字設定(日米欧フォント)
        private class CheckSbmChk12_5 : CheckSbm
        {
            public CheckSbmChk12_5(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                var errorlist = new List<string[]>();

                if (checkData.ApplicationControlProperty?.Title != null &&
                    checkData.ApplicationControlProperty.Title.Count() != 0)
                {
                    foreach (var title in checkData.ApplicationControlProperty.Title)
                    {
                        // Languageに設定できない文字がある場合、項目毎にエラーを作成する
                        if (!string.IsNullOrWhiteSpace(title.Name) && title.Name.IncludesUnsupportedCharacter())
                        {
                            errorlist.Add(new string[] { "Name", title.Name });
                        }
                        if (!string.IsNullOrWhiteSpace(title.Publisher) && title.Publisher.IncludesUnsupportedCharacter())
                        {
                            errorlist.Add(new string[] { "Publisher", title.Publisher });
                        }
                    }
                }

                if (checkData.Nmeta.AddOnContent != null &&
                    !string.IsNullOrWhiteSpace(checkData.Nmeta.AddOnContent.Tag) &&
                    checkData.Nmeta.AddOnContent.Tag.IncludesUnsupportedCharacter())
                {
                    errorlist.Add(new string[] { "Tag", checkData.Nmeta.AddOnContent.Tag });
                }

                return CreateUnpublishableErrorDetailModel(errorlist);
            }
        }

        // 指定できない文字設定(ASCII)
        private class CheckSbmChk12_6 : CheckSbm
        {
            public CheckSbmChk12_6(InitializeParameter param) : base(param) { }
            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                var errorlist = new List<string[]>();

                // それぞれのタグでASCII以外の文字を使用していないかをチェック
                errorlist.Add(GetErrorString("Isbn", checkData.ApplicationControlProperty.Isbn));
                errorlist.Add(GetErrorString("DisplayVersion", checkData.ApplicationControlProperty.DisplayVersion));
                errorlist.Add(GetErrorString("ApplicationErrorCodeCategory", checkData.ApplicationControlProperty.ApplicationErrorCodeCategory));
                errorlist.Add(GetErrorString("BcatPassphrase", checkData.ApplicationControlProperty.BcatPassphrase));

                // NULLを削除
                errorlist = errorlist.FindAll(entry => entry != null).ToList();

                return CreateUnpublishableErrorDetailModel(errorlist);
            }
            private bool IsAscii(string value)
            {
                Encoding encoding = Encoding.GetEncoding("us-ascii",
                                                    new EncoderExceptionFallback(),
                                                    new DecoderExceptionFallback());

                // Encodingの関数を実行して例外が発生したら、Ascii以外が含まれている。
                try
                {
                    encoding.GetByteCount(value);
                }
                catch (EncoderFallbackException)
                {
                    return false;
                }

                return true;
            }
            private string[] GetErrorString(string name, string value)
            {
                if (!string.IsNullOrWhiteSpace(value) && !IsAscii(value))
                {
                    return new string[] { name, value };
                }
                return null;
            }
        }

        // 文字数の超過
        private class CheckSbmChk12_7 : CheckSbm
        {
            public CheckSbmChk12_7(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                var errorlist = new List<string[]>();

                // それぞれのタグの最大文字数を超えていないかをチェックする
                if (checkData.ApplicationControlProperty.Title != null)
                {
                    foreach (var title in checkData.ApplicationControlProperty.Title)
                    {
                        errorlist.Add(GetErrorString("Name", title.Name, ApplicationTitle.NameStringLength));
                        errorlist.Add(GetErrorString("Publisher", title.Publisher, ApplicationTitle.PublisherStringLength));
                    }
                }

                errorlist.Add(GetErrorString("Isbn", checkData.ApplicationControlProperty.Isbn, ApplicationControlProperty.ApplicationControlProperty.IsbnLength - 1));
                errorlist.Add(GetErrorString("DisplayVersion", checkData.ApplicationControlProperty.DisplayVersion, ApplicationControlProperty.ApplicationControlProperty.DisplayVersionLength - 1));
                errorlist.Add(GetErrorString("ApplicationErrorCodeCategory", checkData.ApplicationControlProperty.ApplicationErrorCodeCategory, ApplicationControlProperty.ApplicationControlProperty.ApplicationErrorCodeCategoryLength - 1));
                errorlist.Add(GetErrorString("BcatPassphrase", checkData.ApplicationControlProperty.BcatPassphrase, ApplicationControlProperty.ApplicationControlProperty.BcatPassphraseLength - 1));

                // NULLを削除
                errorlist = errorlist.FindAll(entry => entry != null).ToList();

                return CreateUnpublishableErrorDetailModel(errorlist);
            }

            private string[] GetErrorString(string name, string value, int maxCharCount)
            {
                if (string.IsNullOrWhiteSpace(value))
                {
                    return null;
                }

                var trimmed = value.Trim();

                var stringInfo = new StringInfo(trimmed);
                if (stringInfo.LengthInTextElements > maxCharCount)
                {
                    return new string[] { name, value, maxCharCount.ToString() };
                }
                return null;
            }
        }

        // 表示バージョンなし
        private class CheckSbmChk13 : CheckSbm
        {
            public CheckSbmChk13(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                if (string.IsNullOrEmpty(checkData.ApplicationControlProperty.DisplayVersion)) // DisplayVersionエントリが存在しないか空文字
                {
                    return CreateUnpublishableErrorDetailModel();
                }

                return ResultSuccess;
            }
        }

        // 対応言語
        private class CheckSbmChk15 : CheckSbm
        {
            public CheckSbmChk15(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                if (checkData.ApplicationControlProperty.SupportedLanguage == null ||
                    checkData.ApplicationControlProperty.SupportedLanguage.Count() == 0 || // SupportedLanguageエントリがない
                    checkData.ApplicationControlProperty.SupportedLanguage.FindAll(entry => string.IsNullOrEmpty(entry)).Count() != 0) // 値が未設定か空文字のSupportedLanguageがある
                {
                    return CreateUnpublishableErrorDetailModel();
                }

                return ResultSuccess;
            }
        }

        // 提出ファイルの正当性 (アイコン)
        private class CheckSbmChk1_8 : CheckSbm
        {
            public CheckSbmChk1_8(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                if (checkData.CheckTarget == UnpublishableErrorCheckData.CheckTargetType.Nsp)
                {
                    // nspが設定されている場合は、nsp内にiconファイルがあることを確認する
                    if (checkData.Nsp.RawIconList == null && checkData.Nsp.NxIconList == null) // iconファイル(*.%Language%.jpg)の数が0
                    {
                        return CreateUnpublishableErrorDetailModel();
                    }
                }
                else
                {
                    // nspが設定されていない場合はnmetaをチェック
                    if (checkData.ApplicationControlProperty.Icon == null ||
                        checkData.ApplicationControlProperty.Icon.Count() == 0 || // iconエントリがない
                        checkData.ApplicationControlProperty.Icon.FindAll(entry => string.IsNullOrEmpty(entry.IconPath.Path) || string.IsNullOrEmpty(entry.Language)).Count() != 0) // 値が未設定のIconがある
                    {
                        return CreateUnpublishableErrorDetailModel();
                    }
                }

                return ResultSuccess;
            }
        }

        // 提出ファイルの正当性 (アイコン)
        private class CheckSbmChk9a : CheckSbm
        {
            public CheckSbmChk9a(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                if (checkData.ApplicationControlProperty.Title == null ||
                    checkData.ApplicationControlProperty.Title.Count() == 0)
                {
                    // タイトルが含まれていない場合はチェックをスキップ
                    return ResultSuccess;
                }

                // タイトルに含まれる言語リストを取得
                var needLanguageList = checkData.ApplicationControlProperty.Title.FindAll(entry => !string.IsNullOrEmpty(entry.Language)).Select(entry => entry.Language).ToList();

                List<string> invalidLanguage = null;
                if (checkData.CheckTarget == UnpublishableErrorCheckData.CheckTargetType.Nsp)
                {
                    // nspが設定されている場合は、nsp内のファイルを確認

                    // タイトルに含まれる言語の内、対応するiconが含まれていないものを取得
                    invalidLanguage = GetNotExistIconLanguageForNsp(checkData, needLanguageList);
                }
                else
                {
                    // nspが設定されている場合は、nmetaのエントリを確認

                    // タイトルに含まれる言語の内、対応するiconが含まれていないものを取得
                    invalidLanguage = GetNotExistIconLanguageForNmeta(checkData, needLanguageList);
                }

                if (invalidLanguage.Count() > 0)
                {
                    // 不正なIconエントリがある
                    invalidLanguage = invalidLanguage.Distinct().ToList(); // 重複削除

                    // エラー表示用文字列の作成
                    string errorString = string.Join(ErrorItemSeparator, invalidLanguage);

                    // nmetaのiconエントリが不正な場合エラー
                    return CreateUnpublishableErrorDetailModel(errorString);
                }

                return ResultSuccess;
            }

            private List<string> GetNotExistIconLanguageForNsp(UnpublishableErrorCheckData checkData, List<string> needLanguageList)
            {
                // nspに含まれるiconファイルを取得
                var iconFileList = new List<string>();
                if (checkData.Nsp.RawIconList != null)
                {
                    iconFileList.AddRange(checkData.Nsp.RawIconList.Select(entry => entry.Item1));
                }
                if (checkData.Nsp.NxIconList != null)
                {
                    iconFileList.AddRange(checkData.Nsp.NxIconList.Select(entry => entry.Item1));
                }

                List<string> invalidLanguage = new List<string>();
                foreach (var language in needLanguageList)
                {
                    if (iconFileList.FindAll(entry => entry.EndsWith(".raw." + language + ".jpg")).Count() == 0 ||
                        iconFileList.FindAll(entry => entry.EndsWith(".nx." + language + ".jpg")).Count() == 0)
                    {
                        // iconファイルが存在しない
                        invalidLanguage.Add(language);
                    }
                }

                return invalidLanguage;
            }

            private List<string> GetNotExistIconLanguageForNmeta(UnpublishableErrorCheckData checkData, List<string> needLanguageList)
            {
                if (checkData.ApplicationControlProperty.Icon == null ||
                    checkData.ApplicationControlProperty.Icon.Count() == 0)
                {
                    // nmetaにiconエントリが存在していない場合は全ての言語がエラー
                    return needLanguageList;
                }

                List<string> invalidLanguage = new List<string>();
                foreach (var language in needLanguageList)
                {
                    if (checkData.ApplicationControlProperty.Icon.FindAll(entry => entry.Language == language).Count() == 0)
                    {
                        // 言語に対応するIconがない
                        invalidLanguage.Add(language);
                        continue;
                    }
                }

                return invalidLanguage;
            }
        }

        // アイコンがSDKアイコンでないか？
        private class CheckSbmChk9c : CheckSbm
        {
            // NintendoSDK_Application.bmpをjpegにした際の*.raw.%Language%.jpgのSHA-256ハッシュ値
            private const string DefaultRawIconHash = "402d395f3ee2eec3ac4d8adfec446a52";
            // NintendoSDK_Application.bmpをjpegにした際の*.nx.%Language%.jpgのSHA-256ハッシュ値
            private const string DefaultNxIconHash = "8866647b2b59decfc0198d05e1e5cb95";

            public CheckSbmChk9c(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                if (checkData.ApplicationControlProperty.Icon == null ||
                    checkData.ApplicationControlProperty.Icon.Count() == 0)
                {
                    // iconエントリが存在していない場合はスキップ
                    return ResultSuccess;
                }

                var languageList = new List<string>();
                foreach (var icon in checkData.ApplicationControlProperty.Icon)
                {
                    // Iconのハッシュ値が一致する言語を取得
                    if (DefaultRawIconHash == icon.RawIconHash ||
                        DefaultNxIconHash == icon.NxIconHash)
                    {
                        languageList.Add(icon.Language);
                    }
                }

                if (languageList.Count() > 0)
                {
                    // ハッシュが一致するアイコンがある
                    languageList = languageList.Distinct().ToList(); // 重複削除

                    // エラー表示用文字列の作成
                    string errorString = string.Join(ErrorItemSeparator, languageList);

                    return CreateUnpublishableErrorDetailModel(errorString);
                }

                return ResultSuccess;
            }
        }

        // BCATセーブデータの確認 (0.x系、1.x系でのBCATセーブデータの使用の禁止)
        private class CheckSbmChk52b : CheckSbm
        {
            public CheckSbmChk52b(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                var bcatDeliveryCacheStorageSize = checkData.ApplicationControlProperty.BcatDeliveryCacheStorageSize;
                var bcatPassphrase = checkData.ApplicationControlProperty.BcatPassphrase;

                if (!IsSettingValueBcatDeliveryCacheStorageSize(checkData.ApplicationControlProperty.BcatDeliveryCacheStorageSize) &&
                    !IsSettingValueBcatPassphrase(checkData.ApplicationControlProperty.BcatPassphrase))
                {
                    // BcatDeliveryCacheStorageSizeとBcatPassphraseのいずれも未設定の場合はスキップ
                    return ResultSuccess;
                }

                if (checkData.Nsp.ProgramInfo == null)
                {
                    // Programinfo.xmlが存在しない場合はスキップ
                    return ResultSuccess;
                }

                var sdkMajorVersion = checkData.Nsp.ProgramInfo.SdkVersion.Split('_').First();
                if (sdkMajorVersion == "0" || sdkMajorVersion == "1")
                {
                    // メジャーバージョンが 0 or 1 の場合はエラー
                    return CreateUnpublishableErrorDetailModel();
                }

                return ResultSuccess;
            }

            protected bool IsSettingValueBcatDeliveryCacheStorageSize(string value)
            {
                if (string.IsNullOrEmpty(value) || Convert.ToUInt64(value, 16) == 0)
                {
                    return false;
                }
                return true;
            }

            protected bool IsSettingValueBcatPassphrase(string value)
            {
                if (string.IsNullOrEmpty(value))
                {
                    return false;
                }
                return true;
            }
        }

        // BCATセーブデータの確認 (64MiBを超えるBcatDeliveryCacheStorageSizeの禁止)
        private class CheckSbmChk52c : CheckSbmChk52b
        {
            public CheckSbmChk52c(InitializeParameter param) : base(param) { }
            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                if (!IsSettingValueBcatDeliveryCacheStorageSize(checkData.ApplicationControlProperty.BcatDeliveryCacheStorageSize))
                {
                    // BcatDeliveryCacheStorageSizeが未設定の場合はスキップ
                    return ResultSuccess;
                }

                if (Convert.ToInt64(checkData.ApplicationControlProperty.BcatDeliveryCacheStorageSize, 16) > 0x4000000)
                {
                    // BcatDeliveryCacheStorageSizeが64MiBより大きければ警告
                    return CreateUnpublishableErrorDetailModel();
                }

                return ResultSuccess;
            }
        }

        // BCATセーブデータの確認 (不正な設定)
        private class CheckSbmChk52d : CheckSbmChk52b
        {
            public CheckSbmChk52d(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                var storageSizeSet = IsSettingValueBcatDeliveryCacheStorageSize(checkData.ApplicationControlProperty.BcatDeliveryCacheStorageSize);
                var passphraseSet = IsSettingValueBcatPassphrase(checkData.ApplicationControlProperty.BcatPassphrase);

                if (storageSizeSet == passphraseSet)
                {
                    // 両方未設定か両方設定済みの場合は成功
                    return ResultSuccess;
                }

                var enableEntry = (storageSizeSet == true) ? "BcatDeliveryCacheStorageSize" : "BcatPassphrase";
                var disableEntry = (storageSizeSet == true) ? "BcatPassphrase" : "BcatDeliveryCacheStorageSize";

                return CreateUnpublishableErrorDetailModel(new string[] { enableEntry, disableEntry });
            }
        }

        // 他のアプリケーションのセーブデータ設定に自分のID設定の禁止
        private class CheckSbmChk43b : CheckSbm
        {
            public CheckSbmChk43b(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                if (checkData.Nmeta.Core?.ApplicationId != null &&
                    checkData.Nmeta.CoreFsAccessControlData?.Entries != null &&
                    checkData.Nmeta.CoreFsAccessControlData.Entries.Where(entry => IsEqualApplicationId(checkData.Nmeta.Core.ApplicationId, entry.Id)).Count() > 0)
                {
                    // Core/FsAccessControlData/SaveDataOwnerIds/Id の中にCore/ApplicationIdと一致するものがある場合はエラー
                    return CreateUnpublishableErrorDetailModel();
                }

                if (checkData.Nsp.ContentMeta == null)
                {
                    return ResultSuccess;
                }
                var applicationId = GetApplicationId(checkData.Nsp.ContentMeta);

                if (checkData.Nsp.ProgramInfo?.FsAccessControlData?.Entries != null &&
                    checkData.Nsp.ProgramInfo.FsAccessControlData.Entries.Where(entry => IsEqualApplicationId(entry.Id, applicationId)).Count() > 0)
                {
                    // ProgramInfo/FsAccessControlData/SaveDataOwnerIds の中にアプリケーションIDと一致するものがある場合はエラー
                    return CreateUnpublishableErrorDetailModel();
                }

                return ResultSuccess;
            }
        }

        // 使用されていないアプリのエラーコード
        private class CheckSbmChk18c : CheckSbm
        {
            private const string NgApi = "_ZN2nn3err20ShowApplicationErrorERKNS0_19ApplicationErrorArgE";

            public CheckSbmChk18c(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                // programinfo.xmlがない場合はチェックをスキップ
                if (checkData.Nsp.ProgramInfo == null ||
                    checkData.ApplicationControlProperty == null)
                {
                    return ResultSuccess;
                }

                if (string.IsNullOrEmpty(checkData.ApplicationControlProperty.ApplicationErrorCodeCategory) &&
                    checkData.Nsp.ProgramInfo.UnresolvedApiListData.GetApiNameList().Contains(NgApi))
                {
                    return CreateUnpublishableErrorDetailModel();
                }

                return ResultSuccess;
            }
        }

        // XpadのAPIの使用禁止
        private class CheckSbmChk54 : CheckSbm
        {
            public CheckSbmChk54(InitializeParameter param) : base(param) { }
            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                // programinfo.xmlがない場合はチェックをスキップ
                if (checkData.Nsp.ProgramInfo == null)
                {
                    return ResultSuccess;
                }

                // programinfoに含まれるUnresolvedApiのリストを取得
                var apiList = checkData.Nsp.ProgramInfo.UnresolvedApiListData.GetApiNameList();

                var unpublishableApiList = new List<string>();
                foreach (var manglingApiName in apiList)
                {
                    if (!UnpublishableApiList.IsUnpublishableXpadApi(manglingApiName))
                    {
                        // 該当のAPIが使用禁止でない場合はスキップ
                        continue;
                    }

                    // マングルされた関数名からマングル前の関数を取得
                    unpublishableApiList.Add(UnpublishableApiList.GetDemanglingApiNameForXpadApi(manglingApiName));
                }

                unpublishableApiList = JoinStringList(unpublishableApiList);

                return CreateUnpublishableErrorDetailModel(unpublishableApiList);
            }
        }

        // 使用許可制APIのチェック(エラー)
        private class CheckSbmChk22 : CheckSbm
        {
            public CheckSbmChk22(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                if (checkData.Nsp.ProgramInfo == null || checkData.Nsp.ProgramInfo.SdkVersion == null)
                {
                    // programinfo.xmlがない or SDKバージョンが未指定の場合はチェックをスキップ
                    return ResultSuccess;
                }

                // programinfoに含まれるSdkVersionのメジャーバージョンを取得
                var sdkMajorVersion = checkData.Nsp.ProgramInfo.SdkVersion.Split('_').First();
                // UnresolvedApiリスト内に指定したバージョン用のリストがない場合はUnresolvedApiリスト内の最新バージョンを取得する
                sdkMajorVersion = UnpublishableApiList.GetLatestSdkVersionForUnpublishablePrivateApiList(sdkMajorVersion);

                // programinfoに含まれるUnresolvedApiのリストを取得
                var apiList = checkData.Nsp.ProgramInfo.UnresolvedApiListData.GetApiNameList();

                var demanglingApiList = new List<string[]>();
                foreach (var manglingApiName in apiList)
                {
                    if (!UnpublishableApiList.IsUnpublishablePrivateApi(sdkMajorVersion, manglingApiName))
                    {
                        // 該当のAPIが使用制限ありのPrivateApiでない場合はスキップ
                        continue;
                    }

                    // マングルされた関数名からマングル前の関数を取得
                    var apiInfo = UnpublishableApiList.GetDemanglingApiNameForPrivateApi(sdkMajorVersion, manglingApiName);
                    var demanglingApiName = apiInfo.Item1;
                    foreach (var title in apiInfo.Item2)
                    {
                        if (!IsUnpublishableErrorSpecificRules(checkData, title))
                        {
                            // 特殊ルール(SbmChk-22b, SbmChk-22c)でエラーとしない場合はスキップする
                            continue;
                        }
                        demanglingApiList.Add(new string[] { demanglingApiName, GetBundleNameString(title) });
                    }
                }

                // 内容が重複するデータを削除する
                demanglingApiList = demanglingApiList.Distinct(new StringArrayComparer()).ToList();

                // Bundle名が同じAPIをまとめる
                List<string[]> unpublishableApiList = new List<string[]>();
                foreach (var api in demanglingApiList)
                {
                    // api[0] == API名、api[1] == Bundle名が格納されている
                    string[] table = unpublishableApiList.Find(entry => entry[1] == api[1]);
                    if (table == null)
                    {
                        // まだ登録されていないBundle名の場合は新規追加
                        unpublishableApiList.Add(new string[] { api[0], api[1] });
                    }
                    else
                    {
                        // 登録済みのBundle名の場合はAPI名の後に新しいAPI名を追加
                        table[0] = table[0] + Environment.NewLine + api[0];
                    }
                }

                return CreateUnpublishableErrorDetailModel(unpublishableApiList);
            }

            private string GetBundleNameString(string title)
            {
                // NX or Genericを取得
                var type = Regex.IsMatch(title, @"_NX(_|$)") ? "NX" : "Generic";

                // titleの最初の バージョン情報 を削除して'_'区切りの配列に変換
                var elements = Regex.Replace(title, @"^[0-9_]+", string.Empty).Split('_');

                // BundleRules、BundleDefinitions、Generic, NX以外の最初の_XXX_YYYのXXX_YYYを抽出(YYYがない場合はXXXを抽出)
                string bundle = string.Empty;
                foreach (var element in elements)
                {
                    if (element == "BundleDefinitions" || element == "BundleRules" || element == "NX" || element == "Generic")
                    {
                        if (bundle == string.Empty)
                        {
                            continue;
                        }
                        break;
                    }
                    if (bundle == string.Empty)
                    {
                        bundle = element;
                        continue;
                    }
                    bundle += "_" + element;
                    break;
                }

                return "Bundle_" + bundle + "_" + type;
            }
            private bool IsUnpublishableErrorSpecificRules(UnpublishableErrorCheckData checkData, string title)
            {
                var moduleList = checkData.Nsp.ProgramInfo.MiddlewareListData.GetModuleNameList();

                // 特殊ルールのチェック
                // SbmChk-22c-BundleRules_NetworkMiddlewareKit_[NX|Generic]_Files_NetworkMiddleware
                // SbmChk-22c-BundleRules_LdnPrivate_NX_Ldn_Private
                if (title.Contains("NetworkMiddleware") || title.Contains("LdnPrivate") || title.Contains("Ldn_Private"))
                {
                    // Bundle名に"NetworkMiddleware","LdnPrivate","Ldn_Private"のいずれかが含まれる

                    if (moduleList.FindAll(entry => entry.Contains("Pia")).Count() > 0 || moduleList.FindAll(entry => entry.Contains("NEX")).Count() > 0)
                    {
                        // "Pia"か"NEX"が含まれている場合はエラーにしない
                        return false;
                    }
                }

                // SbmChk-22b-BundleRules_UnityKit_Generic_Files
                if (title.Contains("Unity"))
                {
                    // Bundle名に"Unity"が含まれる
                    if (moduleList.FindAll(entry => entry.Contains("Unity")).Count() > 0)
                    {
                        // ProgramInfo/MiddlewareList/Middleware/ModuleNameに"Unity"が含まれていればエラーにしない
                        return false;
                    }
                }

                return true;
            }

            // 文字配列の重複データ削除用のEqualityComparerクラス
            private class StringArrayComparer : EqualityComparer<string[]>
            {
                public override bool Equals(string[] array1, string[] array2)
                {
                    return array1.SequenceEqual(array2);
                }
                public override int GetHashCode(string[] str)
                {
                    return string.Join(string.Empty, str).GetHashCode();
                }
            }
        }

        // ユーザーアカウントセーブデータのサイズ(特別承認)
        private class CheckSbmChk8a : CheckSbm
        {
            public CheckSbmChk8a(InitializeParameter param) : base(param) { }
            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                const long LimitSize = 67108864;

                long size = Convert.ToInt64(checkData.ApplicationControlProperty.UserAccountSaveDataSize, 16) + Convert.ToInt64(checkData.ApplicationControlProperty.UserAccountSaveDataJournalSize, 16);

                if (size > LimitSize)
                {
                    size = size / (1024 * 1024);
                    return CreateUnpublishableErrorDetailModel(size.ToString());
                }

                return ResultSuccess;
            }
        }

        private class CheckSbmChkSizeZeroCheckCommon : CheckSbm
        {
            public CheckSbmChkSizeZeroCheckCommon(InitializeParameter param) : base(param) { }
            public List<UnpublishableErrorDetailModel> CheckSize(string size)
            {
                if (Convert.ToInt64(size, 16) != 0)
                {
                    return CreateUnpublishableErrorDetailModel();
                }

                return ResultSuccess;
            }
        }

        // ユーザーアカウントセーブデータ保存領域拡張最大サイズ
        private class CheckSbmChk8b : CheckSbmChkSizeZeroCheckCommon
        {
            public CheckSbmChk8b(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                return CheckSize(checkData.ApplicationControlProperty.UserAccountSaveDataSizeMax);
            }
        }

        // ユーザーアカウントセーブジャーナル領域拡張最大サイズ
        private class CheckSbmChk8c : CheckSbmChkSizeZeroCheckCommon
        {
            public CheckSbmChk8c(InitializeParameter param) : base(param) { }
            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                return CheckSize(checkData.ApplicationControlProperty.UserAccountSaveDataJournalSizeMax);
            }
        }

        // 本体セーブデータ保存領域拡張最大サイズ
        private class CheckSbmChk8d : CheckSbmChkSizeZeroCheckCommon
        {
            public CheckSbmChk8d(InitializeParameter param) : base(param) { }
            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                return CheckSize(checkData.ApplicationControlProperty.DeviceSaveDataSizeMax);
            }
        }

        // 本体セーブジャーナル領域拡張最大サイズ
        private class CheckSbmChk8e : CheckSbmChkSizeZeroCheckCommon
        {
            public CheckSbmChk8e(InitializeParameter param) : base(param) { }
            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                return CheckSize(checkData.ApplicationControlProperty.DeviceSaveDataJournalSizeMax);
            }
        }

        // 一時ストレージサイズ
        private class CheckSbmChk8f : CheckSbmChkSizeZeroCheckCommon
        {
            public CheckSbmChk8f(InitializeParameter param) : base(param) { }
            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                return CheckSize(checkData.ApplicationControlProperty.TemporaryStorageSize);
            }
        }

        // キャッシュストレージデータ保存領域サイズ
        private class CheckSbmChk8g : CheckSbmChkSizeZeroCheckCommon
        {
            public CheckSbmChk8g(InitializeParameter param) : base(param) { }
            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                return CheckSize(checkData.ApplicationControlProperty.CacheStorageSize);
            }
        }

        // キャッシュストレージジャーナル領域サイズ
        private class CheckSbmChk8h : CheckSbmChkSizeZeroCheckCommon
        {
            public CheckSbmChk8h(InitializeParameter param) : base(param) { }
            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                return CheckSize(checkData.ApplicationControlProperty.CacheStorageJournalSize);
            }
        }

        // 本体セーブデータのデータ保存領域サイズ
        private class CheckSbmChk8i : CheckSbmChkSizeZeroCheckCommon
        {
            public CheckSbmChk8i(InitializeParameter param) : base(param) { }
            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                return CheckSize(checkData.ApplicationControlProperty.DeviceSaveDataSize);
            }
        }

        // 本体セーブデータのジャーナル領域サイズ
        private class CheckSbmChk8j : CheckSbmChkSizeZeroCheckCommon
        {
            public CheckSbmChk8j(InitializeParameter param) : base(param) { }
            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                return CheckSize(checkData.ApplicationControlProperty.DeviceSaveDataJournalSize);
            }
        }

        // 特別承認が必要なnspファイル（複数キャッシュストレージを使用）
        private class CheckSbmChk8k : CheckSbm
        {
            public CheckSbmChk8k(InitializeParameter param) : base(param) { }
            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                var sizeMax = checkData.ApplicationControlProperty?.CacheStorageDataAndJournalSizeMax;
                var indexMax = checkData.ApplicationControlProperty?.CacheStorageIndexMax;
                if ((!string.IsNullOrEmpty(sizeMax) && Convert.ToInt64(sizeMax, 16) != 0) ||
                    (!string.IsNullOrEmpty(indexMax) && Convert.ToInt16(indexMax, 16) != 0))
                {
                    // CacheStorageDataAndJournalSizeMax or CacheStorageIndexMax が 存在して 0 ではない場合は警告
                    return CreateUnpublishableErrorDetailModel();
                }

                return ResultSuccess;
            }
        }

        // プレゼンスグループID
        private class CheckSbmChk16 : CheckSbm
        {
            public CheckSbmChk16(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                if (checkData.Nmeta.Core?.ApplicationId != null &&
                    checkData.ApplicationControlProperty?.PresenceGroupId != null &&
                    !IsEqualApplicationId(checkData.Nmeta.Core.ApplicationId, checkData.ApplicationControlProperty.PresenceGroupId))
                {
                    // Core/ApplicationId と ApplicationControlProperty/PresenceGroupId が一致しない場合はエラー
                    return CreateUnpublishableErrorDetailModel(checkData.ApplicationControlProperty.PresenceGroupId);
                }

                if (checkData.Nsp.ContentMeta == null)
                {
                    return ResultSuccess;
                }

                var id = GetApplicationId(checkData.Nsp.ContentMeta);

                if (checkData.ApplicationControlProperty.PresenceGroupId != null &&
                    Convert.ToUInt64(id, 16) != Convert.ToUInt64(checkData.ApplicationControlProperty.PresenceGroupId, 16))
                {
                    // ContentMetaとLegalInformationでアプリケーションIDが一致しない
                    return CreateUnpublishableErrorDetailModel(checkData.ApplicationControlProperty.PresenceGroupId);
                }

                return ResultSuccess;
            }
        }

        // ローカル通信識別子 (Application ID と異なる)
        private class CheckSbmChk17a : CheckSbm
        {
            public CheckSbmChk17a(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                if (checkData.Nmeta.Core?.ApplicationId != null &&
                    checkData.ApplicationControlProperty?.LocalCommunicationId != null)
                {
                    var errorStringList = checkData.ApplicationControlProperty.LocalCommunicationId.Where(entry => !IsEqualApplicationId(checkData.Nmeta.Core.ApplicationId, entry)).ToList();
                    if (errorStringList.Count() > 0)
                    {
                        // ApplicationControlProperty/LocalCommunicationId の中でCore/ApplicationId に一致しないものがある場合はエラー
                        return CreateUnpublishableErrorDetailModel(string.Join(ErrorItemSeparator, errorStringList));
                    }
                }

                if (checkData.Nsp.ContentMeta == null)
                {
                    return ResultSuccess;
                }

                var id = GetApplicationId(checkData.Nsp.ContentMeta);

                if (checkData.ApplicationControlProperty.LocalCommunicationId != null)
                {
                    var errorStringList = checkData.ApplicationControlProperty.LocalCommunicationId.FindAll(entry => Convert.ToUInt64(entry, 16) != Convert.ToUInt64(id, 16)).ToList();
                    if (errorStringList.Count() > 0)
                    {
                        // LocalCommunicationIdの中とApplicationIdで一致しないものがある
                        return CreateUnpublishableErrorDetailModel(string.Join(ErrorItemSeparator, errorStringList));
                    }
                }

                return ResultSuccess;
            }
        }

        // 他アプリのセーブデータアクセス
        private class CheckSbmChk43 : CheckSbm
        {
            public CheckSbmChk43(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                if (checkData.Nmeta.Core?.ApplicationId != null &&
                    checkData.Nmeta.CoreFsAccessControlData?.Entries != null)
                {
                    var errorStringList = checkData.Nmeta.CoreFsAccessControlData.Entries.Where(entry => !IsEqualApplicationId(checkData.Nmeta.Core.ApplicationId, entry.Id) && !IsEqualApplicationId("0", entry.Id)).Select(entry => entry.Id).ToList();
                    if (errorStringList.Count() > 0)
                    {
                        // Core/FsAccessControlData/SaveDataOwnerIds/Id の中で0でもCore/ApplicationIdでも無いものがある場合はエラー
                        return CreateUnpublishableErrorDetailModel(string.Join(ErrorItemSeparator, errorStringList));
                    }
                }

                if (checkData.Nsp.ContentMeta == null)
                {
                    return ResultSuccess;
                }
                var applicationId = GetApplicationId(checkData.Nsp.ContentMeta);

                if (applicationId != null &&
                    checkData.Nsp.ProgramInfo?.FsAccessControlData?.Entries != null)
                {
                    var errorStringList = checkData.Nsp.ProgramInfo.FsAccessControlData.Entries.Where(entry => !IsEqualApplicationId(applicationId, entry.Id) && !IsEqualApplicationId("0", entry.Id)).Select(entry => entry.Id).ToList();
                    if (errorStringList.Count() > 0)
                    {
                        // ProgramInfo/FsAccessControlData/SaveDataOwnerIds の中に0にもアプリケーションIDでも無いものがある場合はエラー
                        return CreateUnpublishableErrorDetailModel(string.Join(ErrorItemSeparator, errorStringList));
                    }
                }

                return ResultSuccess;
            }
        }

        private class CheckSbmChkDownLoadSizeCheckCommon : CheckSbm
        {
            public CheckSbmChkDownLoadSizeCheckCommon(InitializeParameter param) : base(param) { }
            public List<UnpublishableErrorDetailModel> CheckSize(NintendoContentMetaProperty[] properties, long limitSize, long unitSize)
            {
                foreach (var property in properties)
                {
                    if (property.Size.DownLoad > limitSize)
                    {
                        return CreateUnpublishableErrorDetailModel((property.Size.DownLoad / unitSize).ToString());
                    }
                }

                return ResultSuccess;
            }
        }

        // パッチのサイズ(特別承認)
        private class CheckSbmChk27a : CheckSbmChkDownLoadSizeCheckCommon
        {
            public CheckSbmChk27a(InitializeParameter param) : base(param) { }
            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                const long LimitSize = 536870912;
                const long UnitSize = 1024 * 1024;
                return CheckSize(checkData.Nsp.ContentMetaPropertyList.Properties, LimitSize, UnitSize);
            }
        }

        // 体験版 等のROMサイズ(エラー)
        private class CheckSbmChk29b : CheckSbmChkDownLoadSizeCheckCommon
        {
            public CheckSbmChk29b(InitializeParameter param) : base(param) { }
            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                const long LimitSize = 1073741824;
                const long UnitSize = 1024 * 1024;
                if (checkData.ApplicationControlProperty.Attribute != null &&
                    checkData.ApplicationControlProperty.Attribute.FindAll(entry => entry == "Demo").Count() > 0)
                {
                    // デモの場合はサイズチェックを実施する
                    return CheckSize(checkData.Nsp.ContentMetaPropertyList.Properties, LimitSize, UnitSize);
                }
                return ResultSuccess;
            }
        }

        // アプリの最大サイズの確認
        private class CheckSbmChk50a : CheckSbmChkDownLoadSizeCheckCommon
        {
            public CheckSbmChk50a(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                const long LimitSize = 68719476736;
                const long UnitSize = 1024 * 1024 * 1024;
                return CheckSize(checkData.Nsp.ContentMetaPropertyList.Properties, LimitSize, UnitSize);
            }
        }

        // パッチの最大サイズの確認
        private class CheckSbmChk50b : CheckSbmChk50a
        {
            public CheckSbmChk50b(InitializeParameter param) : base(param) { }
        }

        // PiaChatの使用禁止
        private class CheckSbmChk55 : CheckSbm
        {
            public CheckSbmChk55(InitializeParameter param) : base(param) { }
            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                // programinfo.xmlがない場合はチェックをスキップ
                if (checkData.Nsp.ProgramInfo == null)
                {
                    return ResultSuccess;
                }

                // programinfoに含まれるModuleNameのリストを取得
                var moduleList = checkData.Nsp.ProgramInfo.MiddlewareListData.GetModuleNameList();

                if (moduleList.Find(entry => entry.Contains("PiaChat")) != null)
                {
                    // ModuleNameにPiaChatが含まれるものがあれば警告
                    return CreateUnpublishableErrorDetailModel();
                }

                return ResultSuccess;
            }
        }

        // 追加コンテンツの一コンテンツのサイズ（ガイドラインサイズ）
        private class CheckSbmChk38a : CheckSbm
        {
            public CheckSbmChk38a(InitializeParameter param) : base(param) { }
            public List<UnpublishableErrorDetailModel> CheckSize(List<ContentModel> contentList, long limitSize, long unitSize)
            {
                foreach (var content in contentList)
                {
                    if (content.Size > limitSize)
                    {
                        return CreateUnpublishableErrorDetailModel((content.Size / unitSize).ToString());
                    }
                }

                return ResultSuccess;
            }
            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                if (checkData.Nsp.ContentMeta == null)
                {
                    return ResultSuccess;
                }
                const long LimitSize = 10737418240;
                const long UnitSize = 1024 * 1024 * 1024;
                return CheckSize(checkData.Nsp.ContentMeta.ContentList, LimitSize, UnitSize);
            }
        }

        // 追加コンテンツの一コンテンツのサイズ（検証最大サイズ）
        private class CheckSbmChk38b : CheckSbmChk38a
        {
            public CheckSbmChk38b(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                if (checkData.Nsp.ContentMeta == null)
                {
                    return ResultSuccess;
                }
                const long LimitSize = 17179869184;
                const long UnitSize = 1024 * 1024 * 1024;
                return CheckSize(checkData.Nsp.ContentMeta.ContentList, LimitSize, UnitSize);
            }
        }

        // HDCPの使用禁止
        private class CheckSbmChk56 : CheckSbm
        {
            public CheckSbmChk56(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                if (checkData.ApplicationControlProperty.Hdcp != null &&
                    checkData.ApplicationControlProperty.Hdcp != "None")
                {
                    // HDCPが存在していてNone以外ならエラー
                    return CreateUnpublishableErrorDetailModel(checkData.ApplicationControlProperty.Hdcp);
                }

                return ResultSuccess;
            }
        }

        // オートブート設定の使用禁止
        private class CheckSbmChk57 : CheckSbm
        {
            public CheckSbmChk57(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                if (checkData.CardSpec?.LaunchFlags != null &&
                    checkData.CardSpec.LaunchFlags != 0)
                {
                    // CardSpec/LaunchFlagsが存在していて0以外ならエラー
                    return CreateUnpublishableErrorDetailModel(checkData.CardSpec.LaunchFlags.ToString());
                }

                return ResultSuccess;
            }
        }

        // ポインティング機能の使用
        private class CheckSbmChk58 : CheckSbm
        {
            public static readonly List<Tuple<string, string>> PointingApiTable =
                new List<Tuple<string, string>>
                {
                    Tuple.Create("_ZN2nn8irsensor20RunPointingProcessorERKNS0_14IrCameraHandleE", "nn::irsensor::RunPointingProcessor(nn::irsensor::IrCameraHandle const&)"),
                    Tuple.Create("_ZN2nn8irsensor26GetPointingProcessorStatesEPNS0_22PointingProcessorStateEPiiRKNS0_14IrCameraHandleE", "nn::irsensor::GetPointingProcessorStates(nn::irsensor::PointingProcessorState*, int*, int, nn::irsensor::IrCameraHandle const&)"),
                    Tuple.Create("_ZN2nn8irsensor32GetPointingProcessorMarkerStatesEPNS0_28PointingProcessorMarkerStateEPiiRKNS0_14IrCameraHandleE", "nn::irsensor::GetPointingProcessorMarkerStates(nn::irsensor::PointingProcessorMarkerState*, int*, int, nn::irsensor::IrCameraHandle const&)"),
                };

            public CheckSbmChk58(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                // programinfo.xmlがない場合はチェックをスキップ
                if (checkData.Nsp.ProgramInfo == null)
                {
                    return ResultSuccess;
                }

                // programinfoに含まれるUnresolvedApiのリストを取得
                var unresolvedApiList = checkData.Nsp.ProgramInfo.UnresolvedApiListData.GetApiNameList();

                List<string> pointingApiList = new List<string>();
                foreach (var api in PointingApiTable)
                {
                    if (unresolvedApiList.Contains(api.Item1) == true)
                    {
                        pointingApiList.Add(api.Item2);
                    }
                }

                pointingApiList = JoinStringList(pointingApiList);

                return CreateUnpublishableErrorDetailModel(pointingApiList);
            }
        }

        // マルチアプリケーションカードの確認（規定数以上のアプリケーション数）
        private class CheckSbmChk65b : CheckSbm
        {
            public CheckSbmChk65b(InitializeParameter param) : base(param) { }
            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                if (checkData.Nsp.MultiApplicationCardInfo?.ApplicationCount != null &&
                    checkData.Nsp.MultiApplicationCardInfo?.ApplicationCount >= 5)
                {
                    // MultiApplicationCard/ApplicationCount が 存在して 5 以上の場合は警告
                    return CreateUnpublishableErrorDetailModel();
                }

                return ResultSuccess;
            }
        }

        // 提出不可のカード容量
        private class CheckSbmChk7c : CheckSbm
        {
            public CheckSbmChk7c(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                if (checkData.CardSpec?.Size != null &&
                    Convert.ToInt16(checkData.CardSpec.Size) == 1)
                {
                    // CardSpec/Sizeが存在していて1ならエラー
                    return CreateUnpublishableErrorDetailModel();
                }

                return new List<UnpublishableErrorDetailModel>();
            }
        }

        // プログラムが含まれていないアプリケーション
        private class CheckSbmChk1_10 : CheckSbm
        {
            public CheckSbmChk1_10(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                if (checkData.Nsp.ProgramInfo == null)
                {
                    // programinfo.xmlがない
                    return CreateUnpublishableErrorDetailModel();
                }

                if (checkData.Nsp.ContentMeta?.ContentList == null ||
                    checkData.Nsp.ContentMeta.ContentList.Where(entry => entry.Type == NintendoContentMetaConstant.ContentTypeProgram).Count() == 0)
                {
                    // コンテンツメタXMLにProgramが存在しない
                    return CreateUnpublishableErrorDetailModel();
                }

                return ResultSuccess;
            }
        }

        // NXアイコンがベースライン形式でない
        private class CheckSbmChk1_9b : CheckSbm
        {
            private const int MarkerSize = 2;
            private const int LengthDataSize = 2;
            private static readonly byte[] StartMarkerPattern = { 0xff, 0xd8 };
            private static readonly byte[] BaseLinePattern = { 0xff, 0xc0 };
            private static readonly byte[] ScanHeaderPattern = { 0xff, 0xda };

            public CheckSbmChk1_9b(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                // ApplicationControlProperty内のIconリストをチェック
                if (!string.IsNullOrEmpty(checkData.Nmeta.NmetaFilePath) &&
                    checkData.ApplicationControlProperty.Icon != null &&
                    checkData.ApplicationControlProperty.Icon.Count() > 0)
                {
                    var nxIconPathList = checkData.ApplicationControlProperty.Icon.Where(entry => (entry.NxIconPath != null) && !string.IsNullOrEmpty(entry.NxIconPath.Path)).Select(entry => entry.NxIconPath.Path).ToList();
                    foreach (var nxIconPath in nxIconPathList)
                    {
                        if (!IsBaseLineJpeg(File.ReadAllBytes(ApplicationControlPropertyPathInfo.GetFullPath(nxIconPath, checkData.Nmeta.NmetaFilePath))))
                        {
                            // ベースラインではないアイコンが含まれている
                            return CreateUnpublishableErrorDetailModel();
                        }
                    }
                }

                // Nxアイコンリストがある場合は各データをチェック
                if (checkData.Nsp.NxIconList != null && checkData.Nsp.NxIconList.Count() > 0)
                {
                    foreach (var iconData in checkData.Nsp.NxIconList)
                    {
                        if (!IsBaseLineJpeg(iconData.Item2))
                        {
                            // ベースラインではないアイコンが含まれている
                            return CreateUnpublishableErrorDetailModel();
                        }
                    }
                }

                return ResultSuccess;
            }

            private bool IsBaseLineJpeg(byte[] data)
            {
                if (data.Length > 2 && !(data[0] == StartMarkerPattern[0] && data[1] == StartMarkerPattern[1]))
                {
                    // スタートマーカーがない場合はjpegではない
                    return false;
                }

                var offset = MarkerSize; // スタートマーカーの後から開始
                while (offset + (MarkerSize + LengthDataSize) <= data.Length)
                {
                    if (data[offset] == BaseLinePattern[0] && data[offset + 1] == BaseLinePattern[1])
                    {
                        return true;
                    }
                    else if (data[offset] == ScanHeaderPattern[0] && data[offset + 1] == ScanHeaderPattern[1])
                    {
                        // これ以降は画像データ
                        break;
                    }

                    offset += MarkerSize + ((data[offset + 2] << 8) | data[offset + 3]);
                }

                return false;
            }
        }

        // 個別プログラム間のアプリケーション管理データの不一致
        private class CheckSbmChk66_1 : CheckSbm
        {
            // チェック対象から除外する要素のリスト
            private static readonly string[] IgnoreElements =
            {
                "/Application/ProgramIndex",
            };

            public CheckSbmChk66_1(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                // チェック対象が ProgramIndex = 0 の場合 or ProgramIndex = 0 の nacp が指定されていない場合はチェックをスキップ
                if (checkData.ApplicationControlProperty == null ||
                    Convert.ToInt32(checkData.ApplicationControlProperty.ProgramIndex) == 0 ||
                    checkData.Nsp.ApplicationControlPropertyOfProgramIndex0 == null)
                {
                    return new List<UnpublishableErrorDetailModel>();
                }

                // ProgramIndex = 0 の nacp とチェック対象のコンテンツの nacp を比較して差異のある要素名を取得
                var errorElementList = CheckApplicationControlProperty(checkData.Nsp.ApplicationControlPropertyOfProgramIndex0, checkData.ApplicationControlProperty);

                if (errorElementList == null || errorElementList.Count() == 0)
                {
                    return new List<UnpublishableErrorDetailModel>();
                }

                // 「<要素名> が一致していません。」のメッセージを作成
                var resource = GetMessageResource(MessageId + "_Description_Child_1");
                var appendMessageForJapnese = Environment.NewLine + string.Join(Environment.NewLine, errorElementList.Select(x => string.Format(resource["ja"], x)));
                var appendMessageForEnglish = Environment.NewLine + string.Join(Environment.NewLine, errorElementList.Select(x => string.Format(resource["en"], x)));

                return CreateUnpublishableErrorDetailModel(string.Empty, appendMessageForJapnese, appendMessageForEnglish);
            }

            private List<string> CheckApplicationControlProperty(ApplicationControlPropertyModel baseNacp, ApplicationControlPropertyModel targetNacp)
            {
                Action<XDocument> deleteUncheckedElements = (XDocument document) =>
                {
                    // ApplicationControlPropertyModel 内のチェックしない要素を削除する
                    foreach (var ignore in IgnoreElements)
                    {
                        document.XPathSelectElements(ignore).Remove();
                    }
                };

                Func<XDocument, List<Tuple<string, string>>> getElementInfoList = (XDocument document) =>
                {
                    // XDocument のルート直下の小要素の要素名と設定値の組み合わせのリストを取得
                    var elementInfoList = new List<Tuple<string, string>>();

                    foreach (var name in document.XPathSelectElements("*/*"))
                    {
                        elementInfoList.Add(Tuple.Create(name.Name.ToString(), name.Value));
                    }

                    return elementInfoList;
                };

                Func<XDocument, XDocument, List<string>> getDifferentElementNameList = (XDocument baseXml, XDocument targetXml) =>
                {
                    // XDocumet ルートの小要素で値が違う要素名を取得
                    var elementNameList = new List<string>();

                    // 比較元と比較対象のルート直下の要素名と設定値の組み合わせを取得
                    var baseElementInfoList = getElementInfoList(baseXml);
                    var targetElementInfoList = getElementInfoList(targetXml);

                    // 比較元の中で比較対象に含まれない組み合わせを取得
                    foreach (var elementInfo in baseElementInfoList)
                    {
                        if (!targetElementInfoList.Contains(elementInfo))
                        {
                            // 要素名を取得
                            elementNameList.Add(elementInfo.Item1);
                        }
                        else
                        {
                            // 組み合わせが一致したものは比較対象のリストから削除
                            targetElementInfoList.Remove(elementInfo);
                        }
                    }

                    // 比較対象のリストに残った要素は、一致しない要素として追加
                    elementNameList.AddRange(targetElementInfoList.Select(x => x.Item1));

                    return elementNameList.Distinct().ToList();
                };

                XDocument baseXDocument;
                // 比較元の情報を取得
                using (var writer = new StringWriter())
                {
                    var xmlSerializer = new XmlSerializer(typeof(ApplicationControlPropertyModel));
                    xmlSerializer.Serialize(writer, baseNacp);
                    baseXDocument = XDocument.Parse(writer.ToString());
                    deleteUncheckedElements(baseXDocument);
                }

                using (var writer = new StringWriter())
                {
                    var xmlSerializer = new XmlSerializer(typeof(ApplicationControlPropertyModel));
                    xmlSerializer.Serialize(writer, targetNacp);
                    var targetXDocument = XDocument.Parse(writer.ToString());
                    deleteUncheckedElements(targetXDocument);

                    if (baseXDocument.ToString() == targetXDocument.ToString())
                    {
                        // xml の内容が同じならエラーなし
                        return null;
                    }

                    return getDifferentElementNameList(baseXDocument, targetXDocument);
                }
            }
        }

        // スマホアイコンとNXアイコンの同一性
        private class CheckSbmChk1_7 : CheckSbm
        {
            // エラーとするAverageHashのハミング距離の閾値
            private const int ThresholdValue = 10;

            public CheckSbmChk1_7(InitializeParameter param) : base(param) { }

            public static string GetAverageHash(byte[] data)
            {
                if (data == null || data.Length == 0)
                {
                    return string.Empty;
                }

                using (var memoryStream = new MemoryStream(data))
                {
                    var bitmap = new Bitmap(memoryStream);

                    // 8x8にリサイズ
                    var newBitmap = new Bitmap(8, 8);
                    Graphics graphic = Graphics.FromImage(newBitmap);
                    graphic.InterpolationMode = System.Drawing.Drawing2D.InterpolationMode.HighQualityBilinear;
                    graphic.DrawImage(bitmap, 0, 0, newBitmap.Width, newBitmap.Height);

                    int[] pixels = new int[newBitmap.Width * newBitmap.Height];

                    for (int x = 0; x < newBitmap.Width; x++)
                    {
                        for (int y = 0; y < newBitmap.Height; y++)
                        {
                            // グレースケールに変換して、ピクセル値として保存
                            Color pixelColor = newBitmap.GetPixel(x, y);
                            float grayScale8 = (float)((0.299 * pixelColor.R) + (0.587 * pixelColor.G) + (0.114 * pixelColor.B));
                            ushort grayScale16 = (ushort)(grayScale8 * 256);
                            pixels[(x * newBitmap.Width) + y] = grayScale16;
                        }
                    }

                    // ピクセル値の平均を計算
                    double average = pixels.Average();

                    // ピクセル列からAverage Hashを習得
                    string hash = string.Empty;
                    foreach (var pixel in pixels)
                    {
                        if (pixel > average)
                        {
                            hash += "1";
                        }
                        else
                        {
                            hash += "0";
                        }
                    }

                    return hash;
                }
            }

            public static int GetDistance(string hash1, string hash2)
            {
                int distance = 0;
                for (int i = 0; i < hash1.Length; i++)
                {
                    if (hash1[i] != hash2[i])
                    {
                        distance++;
                    }
                }
                return distance;
            }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                if (checkData.Nsp.RawIconList == null || checkData.Nsp.NxIconList == null)
                {
                    // Nsp内にIconが設定されていない場合はスキップ
                    return ResultSuccess;
                }

                // Iconが用意されている言語リストを習得する
                var languageList = checkData.Nsp.RawIconList.Select(entry => Regex.Match(entry.Item1, @"[^\.]+\.raw\.([^\.]+)\.jpg$").Groups[1].Value);

                List<string> hashList = new List<string>();
                List<string> errorLanguageList = new List<string>();
                foreach (var language in languageList)
                {
                    string rawIconAverageHash;
                    string nxIconAverageHash;
                    try
                    {
                        rawIconAverageHash = GetAverageHash(checkData.Nsp.RawIconList.Find(entry => entry.Item1.EndsWith(language + ".jpg")).Item2);
                        nxIconAverageHash = GetAverageHash(checkData.Nsp.NxIconList.Find(entry => entry.Item1.EndsWith(language + ".jpg")).Item2);
                    }
                    catch
                    {
                        // 通常は発生しないが、TestでJpegを壊してreplaceしている物があるので、リサイズなどに失敗した場合はエラーとする
                        errorLanguageList.Add(language);
                        continue;
                    }

                    if (!string.IsNullOrEmpty(rawIconAverageHash) &&
                        !string.IsNullOrEmpty(nxIconAverageHash) &&
                        GetDistance(rawIconAverageHash, nxIconAverageHash) > ThresholdValue)
                    {
                        // ハミング距離が閾値より大きい場合はエラー
                        errorLanguageList.Add(language);
                    }
                }

                if (errorLanguageList.Count() > 0)
                {
                    return CreateUnpublishableErrorDetailModel(string.Join(ErrorItemSeparator, errorLanguageList));
                }

                return ResultSuccess;
            }
        }

        // AOCのタグがあるか？
        private class CheckSbmChk39_1 : CheckSbm
        {
            public CheckSbmChk39_1(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                if (checkData.Nmeta.AddOnContent != null &&
                    string.IsNullOrEmpty(checkData.Nmeta.AddOnContent.Tag))
                {
                    return CreateUnpublishableErrorDetailModel();
                }

                if (checkData.Nsp.ContentMeta != null &&
                    checkData.Nsp.ContentMeta.Type == NintendoContentMetaConstant.ContentMetaTypeAddOnContent &&
                    string.IsNullOrEmpty((checkData.Nsp.ContentMeta as AddOnContentContentMetaModel).Tag))
                {
                    return CreateUnpublishableErrorDetailModel();
                }

                return ResultSuccess;
            }
        }

        // AOCのタグがユニークであるか？
        private class CheckSbmChk39_2 : CheckSbmChk39_1
        {
            public CheckSbmChk39_2(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                if (checkData.Nmeta.AddOnContent != null &&
                    checkData.Nmeta.AddOnContentList != null &&
                    checkData.Nmeta.AddOnContentList.Where(entry => entry.Tag == checkData.Nmeta.AddOnContent.Tag).Count() > 1)
                {
                    return CreateUnpublishableErrorDetailModel(checkData.Nmeta.AddOnContent.Tag);
                }

                if (checkData.Nsp.ContentMeta != null &&
                    checkData.Nsp.ContentMeta.Type == NintendoContentMetaConstant.ContentMetaTypeAddOnContent &&
                    checkData.Nsp.AddonContentMetaList != null &&
                    checkData.Nsp.AddonContentMetaList.Where(entry => entry.Tag == (checkData.Nsp.ContentMeta as AddOnContentContentMetaModel).Tag).Count() > 1)
                {
                    return CreateUnpublishableErrorDetailModel((checkData.Nsp.ContentMeta as AddOnContentContentMetaModel).Tag);
                }
                return ResultSuccess;
            }
        }

        // AOCのタグが128文字以内であるか？
        private class CheckSbmChk61 : CheckSbm
        {
            public CheckSbmChk61(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                if (checkData.Nmeta.AddOnContent?.Tag != null &&
                    IsStringLengthOver(checkData.Nmeta.AddOnContent.Tag))
                {
                    return CreateUnpublishableErrorDetailModel();
                }

                if (checkData.Nsp.ContentMeta != null &&
                    checkData.Nsp.ContentMeta.Type == NintendoContentMetaConstant.ContentMetaTypeAddOnContent &&
                    (checkData.Nsp.ContentMeta as AddOnContentContentMetaModel).Tag != null &&
                    IsStringLengthOver((checkData.Nsp.ContentMeta as AddOnContentContentMetaModel).Tag))
                {
                    return CreateUnpublishableErrorDetailModel();
                }

                return ResultSuccess;
            }

            private bool IsStringLengthOver(string str)
            {
                System.Globalization.StringInfo info = new System.Globalization.StringInfo(str);

                if (info.LengthInTextElements > 128)
                {
                    return true;
                }

                return false;
            }
        }

        // AOCのIndexが指定されていない
        private class CheckSbmChk39_3 : CheckSbm
        {
            public CheckSbmChk39_3(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                if (checkData.Nmeta.AddOnContent != null &&
                    checkData.Nmeta.AddOnContent.Index == null &&
                    string.IsNullOrEmpty(checkData.Nmeta.AddOnContent.Id))
                {
                    return CreateUnpublishableErrorDetailModel();
                }

                return ResultSuccess;
            }
        }

        // アクセス可能URL
        private class CheckSbmChk47a : CheckSbm
        {
            private const string BlackListMark = "---- ";

            public CheckSbmChk47a(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                List<string> ngUrlList = new List<string>
                {
                    "http://localhost",
                    "https://localhost",
                    "http://example.com",
                    "https://www.example.com",
                    "http://example.org",
                    "https://www.example.org",
                    "http://127.0.0.1",
                    "https://127.0.0.1",
                    "http://192.168.0.11",
                    "https://192.168.0.11",
                };

                List<string> urlList = new List<string>();
                if (checkData.Nsp.HtmlDocumentXml?.AccessibleUrls?.Url != null)
                {
                    urlList.AddRange(checkData.Nsp.HtmlDocumentXml.AccessibleUrls.Url);
                }
                else if (checkData.Nmeta.NmetaFilePath != null &&
                         checkData.ApplicationControlProperty?.AccessibleUrlsFilePath?.Path != null)
                {
                    urlList.AddRange(GetUrlList(checkData.ApplicationControlProperty.AccessibleUrlsFilePath, checkData.Nmeta.NmetaFilePath));
                }

                if (urlList.Count() == 0)
                {
                    return ResultSuccess;
                }

                // ホワイトリスト、ブラックリストを取得
                var whiteList = urlList.Where(entry => !entry.StartsWith(BlackListMark)).ToList();
                var blackList = urlList.Where(entry => entry.StartsWith(BlackListMark)).Select(entry => entry.Substring(BlackListMark.Length)).ToList();

                Func<string, string, bool> isMatch = delegate (string target, string pattern)
                {
                    try
                    {
                        return Regex.IsMatch(target, pattern);
                    }
                    catch
                    {
                        // 正規表現パターンが不正などマッチングで失敗した場合には一致しないとして扱う
                        return false;
                    }
                };

                // NGリストからブラックリストに一致するものを削除
                foreach (var pattern in blackList)
                {
                    ngUrlList.RemoveAll(entry => isMatch(entry, pattern));
                }

                // NGリストの中にホワイトリストに一致する物があればエラー
                foreach (var ngUrl in ngUrlList)
                {
                    if (whiteList.Where(pattern => isMatch(ngUrl, pattern)).Count() > 0)
                    {
                        return CreateUnpublishableErrorDetailModel(JoinStringList(urlList));
                    }
                }

                return ResultSuccess;
            }

            protected List<string> GetUrlList(DataPath urlsFilePath, string nmetaFilePath)
            {
                var fullPath = ApplicationControlPropertyPathInfo.GetFullPath(urlsFilePath.Path, nmetaFilePath);
                if (File.Exists(fullPath) == false)
                {
                    return new List<string>();
                }

                var lines = File.ReadAllLines(fullPath);
                return lines.ToList();
            }
        }

        // アクセス可能URLのhttp禁止
        private class CheckSbmChk47b : CheckSbmChk47a
        {
            private const string BlackListMark = "---- ";
            private const string HttpsPattern = "^https://";

            public CheckSbmChk47b(InitializeParameter param) : base(param) { }

            public override List<UnpublishableErrorDetailModel> DoCheck(UnpublishableErrorCheckData checkData)
            {
                List<string> urlList = new List<string>();
                if (checkData.Nsp.HtmlDocumentXml?.AccessibleUrls?.Url != null)
                {
                    urlList.AddRange(checkData.Nsp.HtmlDocumentXml.AccessibleUrls.Url);
                }
                else if (checkData.Nmeta.NmetaFilePath != null &&
                         checkData.ApplicationControlProperty?.AccessibleUrlsFilePath?.Path != null)
                {
                    urlList.AddRange(GetUrlList(checkData.ApplicationControlProperty.AccessibleUrlsFilePath, checkData.Nmeta.NmetaFilePath));
                }

                if (urlList.Count() == 0)
                {
                    return ResultSuccess;
                }

                // 先頭が "^https://" や "---- " で始まらないものを取得
                var ngPatternList = urlList.Where(entry => !entry.StartsWith(HttpsPattern) && !entry.StartsWith(BlackListMark)).ToList();
                if (ngPatternList.Count() > 0)
                {
                    // 一つでも在る場合はURLを結合して表示
                    return CreateUnpublishableErrorDetailModel(JoinStringList(ngPatternList));
                }

                return ResultSuccess;
            }
        }
    }
}
