﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Runtime.InteropServices;
using System.Text.RegularExpressions;

namespace Nintendo.Authoring.AuthoringLibrary
{
    public enum ModuleInfoType
    {
        MiddlewareInfo = (1 << 0),
        DebugApiInfo = (1 << 1),
        PrivateApiInfo = (1 << 2),
        SdkVersionInfo = (1 << 3),
    }

    internal enum ElfMode
    {
        MODE_ELF32 = 0x00000000,
        MODE_ELF64 = 0x00000001
    }

    internal enum SymbolType
    {
        STT_NOTYPE,
        STT_OBJECT,
        STT_FUNC,
        STT_SECTION,
        STT_FILE,
        STT_LOOS = 10,
        STT_HIOS = 12,
        STT_LOPROC = 13,
        STT_HIPROC = 15
    }

    internal enum SpecialSectionIndex
    {
        SHN_UNDEF,
        SHN_LORESERVE = 0xff00,
        SHN_LOPROC = 0xff00,
        SHN_HIPROC = 0xff1f,
        SHN_LOOS = 0xff20,
        SHN_HIOS = 0xff3f,
        SHN_ABS = 0xfff1,
        SHN_COMMON = 0xfff2,
        SHN_HIRESERVE = 0xffff
    }

    public struct ModuleInfo
    {
        public ModuleInfo(ModuleInfoType type, string module, string vender, string file)
        {
            this.Type = type;
            this.ModuleName = module;
            this.VenderName = vender;
            this.FileName = file;
        }
        public ModuleInfoType Type { get; set; }
        public string ModuleName { get; set; }
        public string VenderName { get; set; }
        public string FileName { get; set; }
    }

    public struct UndefSymbolInfo
    {
        public string ApiName;
        public string NsoName;
        public UndefSymbolInfo(string apiName, string nsoName)
        {
            this.ApiName = apiName;
            this.NsoName = nsoName;
        }
    }

    /// <summary>
    /// DynamicSymbol(ELF32)情報
    /// </summary>
    [StructLayout(LayoutKind.Sequential)]
    internal struct DynSym32
    {
        public uint Name;
        public uint Value;
        public uint Size;
        public byte Info;
        public byte Other;
        public ushort Shndx;
    }

    /// <summary>
    /// DynamicSymbol(ELF64)情報
    /// </summary>
    [StructLayout(LayoutKind.Sequential)]
    internal struct DynSym64
    {
        public uint Name;
        public byte Info;
        public byte Other;
        public ushort Shndx;
        public ulong Value;
        public ulong Size;
    }

    public class SymbolExtract
    {
        private List<ModuleInfo> moduleInfoList;
        private List<UndefSymbolInfo> undefSymbolInfoList;

        public SymbolExtract(byte[] data, string filePath, ObjectType type, bool elf32Mode)
        {
            switch (type)
            {
                case ObjectType.NsoObject:
                    {
                        NsoFile nsoFile = new NsoFile();
                        ParseData(nsoFile, data, filePath, elf32Mode, true);
                        break;
                    }
                case ObjectType.NsoLibraryObject:
                    {
                        NsoFile nsoFile = new NsoFile();
                        ParseData(nsoFile, data, filePath, elf32Mode, false);
                        break;
                    }
                case ObjectType.NroObject:
                    {
                        NroFile nroFile = new NroFile();
                        ParseData(nroFile, data, filePath, elf32Mode, true);
                        break;
                    }
                default:
                    throw new NotImplementedException();
            }
        }

        public static bool IsElf32(byte[] metaBinary)
        {
            return ((BitConverter.ToUInt32(metaBinary, 0x0c) & 0x00000001) == (uint)ElfMode.MODE_ELF32);
        }

        public static List<UndefSymbolInfo> GetUndefSymbolInfoList(byte[] dynSymBinary, byte[] dynStrBinary, string nsoFileName, bool elf32Mode)
        {
            List<UndefSymbolInfo> undefSymbolInfoList = new List<UndefSymbolInfo>();

            // 未定義シンボルAPI取得
            if (dynSymBinary != null)
            {
                int dynSymStructSize = (elf32Mode == true ? Marshal.SizeOf(typeof(DynSym32)) : Marshal.SizeOf(typeof(DynSym64)));

                // DynSym情報取得
                var handle = GCHandle.Alloc(dynSymBinary, GCHandleType.Pinned);
                var currentPointer = handle.AddrOfPinnedObject();

                try
                {
                    for (int nextOffset = 0; nextOffset < dynSymBinary.Length; nextOffset += dynSymStructSize)
                    {
                        byte   symType;
                        ushort symShndx;
                        uint   symName;

                        if (elf32Mode == true)  // MODE_ELF32
                        {
                            DynSym32 dynSymDat = (DynSym32)Marshal.PtrToStructure(currentPointer + nextOffset, typeof(DynSym32));

                            symType  = (byte)(dynSymDat.Info & 0x0F);
                            symShndx = dynSymDat.Shndx;
                            symName  = dynSymDat.Name;
                        }
                        else                    // MODE_ELF64
                        {
                            DynSym64 dynSymDat = (DynSym64)Marshal.PtrToStructure(currentPointer + nextOffset, typeof(DynSym64));

                            symType  = (byte)(dynSymDat.Info & 0x0F);
                            symShndx = dynSymDat.Shndx;
                            symName  = dynSymDat.Name;
                        }

                        if ((symType == (byte)SymbolType.STT_FUNC || symType == (byte)SymbolType.STT_NOTYPE) && symShndx == (ushort)SpecialSectionIndex.SHN_UNDEF)
                        {
                            // 未定義シンボルAPI取得
                            int len;
                            for (len = 0; dynStrBinary[symName + len] != '\0'; ++len) { }
                            string undefSymbolString = System.Text.Encoding.GetEncoding("Shift_JIS").GetString(dynStrBinary, (int)symName, len);
                            if (string.IsNullOrEmpty(undefSymbolString))
                            {
                                continue;
                            }
                            UndefSymbolInfo undefInfo = new UndefSymbolInfo();
                            undefInfo.ApiName = undefSymbolString;
                            undefInfo.NsoName = nsoFileName;
                            undefSymbolInfoList.Add(undefInfo);
                        }
                    }
                }
                finally
                {
                    handle.Free();
                }
            }

            return undefSymbolInfoList;
        }

        public UndefSymbolInfo[] GetUndefSymbolInfos()
        {
            var undefSymbolInfos = undefSymbolInfoList.ToArray();
            return undefSymbolInfos;
        }

        public ModuleInfo[] GetDebugApiInfos()
        {
            var debugApiInfos = moduleInfoList.Where(p => p.Type == ModuleInfoType.DebugApiInfo).ToArray();
            return debugApiInfos;
        }

        public ModuleInfo[] GetPrivateApiInfos()
        {
            var privateApiInfos = moduleInfoList.Where(p => p.Type == ModuleInfoType.PrivateApiInfo).ToArray();
            return privateApiInfos;
        }

        public ModuleInfo[] GetMiddlewareInfos()
        {
            var middlewareInfos = moduleInfoList.Where(p => p.Type == ModuleInfoType.MiddlewareInfo).ToArray();
            return middlewareInfos;
        }

        public ModuleInfo[] GetSdkVersionInfos()
        {
            var middlewareInfos = moduleInfoList.Where(p => p.Type == ModuleInfoType.SdkVersionInfo).ToArray();
            return middlewareInfos;
        }

        private void ParseData<T>(T objFile, byte[] data, string filePath, bool elf32Mode, bool needUndefSymbol)
            where T : ObjectFileBase
        {
            moduleInfoList = new List<ModuleInfo>();
            undefSymbolInfoList = new List<UndefSymbolInfo>();

            if (data.Length == 0)
            {
                return;
            }

            objFile.ParseData(data, filePath);

            if (needUndefSymbol)
            {
                undefSymbolInfoList = GetUndefSymbolInfoList(objFile.DynSymBinary, objFile.DynStrBinary, objFile.FileName, elf32Mode);
            }

            string[] apiInfoStrings = objFile.GetApiInfoStrings();
            if (apiInfoStrings == null)
            {
                return;
            }

            foreach (string apiInfoString in apiInfoStrings)
            {
                ModuleInfo moduleInfo = GetModuleInfo(objFile.FileName, apiInfoString);
                moduleInfoList.Add(moduleInfo);
            }

            return;
        }

        private ModuleInfo GetModuleInfo(string path, string infoString)
        {
            ModuleInfo moduleInfo = new ModuleInfo();
            moduleInfo.FileName = path;

            Regex mwRule = new Regex(@"SDK MW\+([^+]*)\+([^+]*)");
            Regex debugApiRule = new Regex(@"SDK Debug\+([^+]*)\+([^+]*)");
            Regex privateApiRule = new Regex(@"SDK Private\+([^+]*)\+([^+]*)");

            if (mwRule.IsMatch(infoString))
            {
                Regex sdkVersionRule = new Regex(@"NintendoSdk_nnSdk-([^-]*)\-([^-]*)");
                if (sdkVersionRule.IsMatch(infoString))
                {
                    moduleInfo.Type = ModuleInfoType.SdkVersionInfo;
                }
                else
                {
                    moduleInfo.Type = ModuleInfoType.MiddlewareInfo;
                }
                moduleInfo.VenderName = mwRule.Match(infoString).Groups[1].Value;
                moduleInfo.ModuleName = mwRule.Match(infoString).Groups[2].Value;
            }
            else if (debugApiRule.IsMatch(infoString))
            {
                moduleInfo.Type = ModuleInfoType.DebugApiInfo;
                moduleInfo.VenderName = debugApiRule.Match(infoString).Groups[1].Value;
                moduleInfo.ModuleName = debugApiRule.Match(infoString).Groups[2].Value;
            }
            else if (privateApiRule.IsMatch(infoString))
            {
                moduleInfo.Type = ModuleInfoType.PrivateApiInfo;
                moduleInfo.VenderName = privateApiRule.Match(infoString).Groups[1].Value;
                moduleInfo.ModuleName = privateApiRule.Match(infoString).Groups[2].Value;
            }
            else
            {
                throw new Exception("This Module Info string is Invalid.\n" + infoString);
            }

            return moduleInfo;
        }
    }
}
