﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Threading;
using System.Threading.Tasks;

namespace Nintendo.Authoring.AuthoringLibrary
{
    public class Connection
    {
        public ISource Source { get; set; }
        public ISink Sink { get; set; }
        public Connection(ISource source, ISink sink)
        {
            Source = source;
            Sink = sink;
        }
    }

    public class ProgressLogTask
    {
        public Task Task;
        public CancellationTokenSource CancelToken;
        private long TotalSize;

        public ProgressLogTask(Task task, CancellationTokenSource cancelToken, long totalSize)
        {
            Task = task;
            CancelToken = cancelToken;
            TotalSize = totalSize;
        }

        public void ShowComplete()
        {
            Log.Progress(string.Format("{0} / {1} (bytes) ... {2} %", TotalSize, TotalSize, 100));
            Log.Progress("Complete.");
        }
    }

    public class SourceSinkDriver
    {
        private List<Connection> m_connectionList;

        public SourceSinkDriver()
        {
            m_connectionList = new List<Connection>();
        }

        public void Add(IConnector connector)
        {
            foreach (var connection in connector.ConnectionList)
            {
                m_connectionList.Add(connection);
            }
        }

        public void Add(ISource source, ISink sink)
        {
            m_connectionList.Add(new Connection(source, sink));
        }

        private ProgressLogTask CreateProgressLogTask(List<Connection> connectionList)
        {
            const int IntervalMsec = 5000;
            var cancelToken = new CancellationTokenSource();

            long totalSize = 0;
            foreach (var connection in connectionList)
            {
                totalSize += connection.Sink.Size;
            }

            var task = Task.Factory.StartNew(
                () => {
                    while (true)
                    {
                        if (cancelToken.Token.IsCancellationRequested)
                        {
                            cancelToken.Token.ThrowIfCancellationRequested();
                        }

                        long filledSize = 0;
                        foreach (var connection in connectionList)
                        {
                            foreach (var range in connection.Sink.QueryStatus().FilledRangeList)
                            {
                                filledSize += range.Size;
                            }
                        }
                        Log.Progress(string.Format("{0} / {1} (bytes) ... {2} %", filledSize, totalSize, (int)((double)filledSize / totalSize * 100)));
                        Thread.Sleep(IntervalMsec);
                    }
                },
                cancelToken.Token
            );

            return new ProgressLogTask(task, cancelToken, totalSize);
        }

        // m_connectionList に追加された Connection の Source から Sink へデータを転送する
        // 登録されたすべての Connection の Sink が QueryStatus().IsFilled = true を満たすと処理を完了する
        public void Run()
        {
            SourceStatus prevSourceStatus = new SourceStatus();
            bool isAllCompleted = false;

            var progressLogTask = CreateProgressLogTask(m_connectionList);

            // Todo: Lock の仕組み + 並列 Task 化
            while (!isAllCompleted)
            {
                bool isProceeded = false;
                foreach (var connection in m_connectionList)
                {
                    SourceStatus sourceStatus = connection.Source.QueryStatus();
                    SinkStatus sinkStatus = connection.Sink.QueryStatus();

                    // 現状 Source と Sink は同じサイズであることを前提としている
                    // Source 側で利用可能なデータ領域を順に Sink に移していく
                    foreach (var range in sourceStatus.AvailableRangeList.GetDuplicatedDeletedList(sinkStatus.FilledRangeList))
                    {
                        long done = 0;
                        int maxBufferSize = 8 * 1024 * 1024;
                        byte[] buffer = new byte[maxBufferSize];
                        long bufferedSize = 0;

                        while (done < range.Size)
                        {
                            int readSize = (int)Math.Min(range.Size - done - bufferedSize, (long)maxBufferSize);
                            ByteData data = connection.Source.PullData(range.Offset + done + bufferedSize, readSize);

                            if (bufferedSize + data.Buffer.Count > maxBufferSize || (data.Buffer.Count == 0 && bufferedSize != 0))
                            {
                                // バッファに残っているデータを Sink に Push する
                                done += connection.Sink.PushData(
                                    new ByteData(new ArraySegment<byte>(buffer, 0, (int)bufferedSize)), range.Offset + done);

                                // Todo: Push ができなかった場合、転送を中止して次のデータ領域に移る。バッファも破棄する。
                                // Todo: Push できた分を Source に記録

                                bufferedSize = 0;
                                if (data.Buffer.Count == 0)
                                {
                                    continue;
                                }
                            }

                            if (data.Buffer.Count == readSize)
                            {
                                if (0 < bufferedSize)
                                {
                                    // ソースから受け取ったデータをバッファに残っているデータと結合して Sink に Push する
                                    Buffer.BlockCopy(data.Buffer.Array, 0, buffer, (int)bufferedSize, data.Buffer.Count);

                                    done += connection.Sink.PushData(
                                        new ByteData(new ArraySegment<byte>(buffer, 0, (int)bufferedSize + data.Buffer.Count)), range.Offset + done);

                                    bufferedSize = 0;
                                }
                                else
                                {
                                    // ソースから受け取ったデータをバッファリングせずに Push する
                                    done += connection.Sink.PushData(data, range.Offset + done);
                                }

                                // Todo: Push ができなかった場合、転送を中止して次のデータ領域に移る。バッファも破棄する。
                                // Todo: Push できた分を Source に記録
                            }
                            else
                            {
                                // ソースから受け取ったデータをバッファリングする
                                Buffer.BlockCopy(data.Buffer.Array, 0, buffer, (int)bufferedSize, data.Buffer.Count);
                                bufferedSize += data.Buffer.Count;
                            }
                        }
                        if (done > 0)
                        {
                            isProceeded = true;
                        }
                    }
                }

                isAllCompleted = true;
                foreach (var connection in m_connectionList)
                {
                    if (!connection.Sink.QueryStatus().IsFilled)
                    {
                        isAllCompleted = false;
                        break;
                    }
                }

                if (!isAllCompleted && !isProceeded)
                {
                    throw new ApplicationException("Failed to resolve dependencies for archiving. Please contact to NintendoSDK support.");
                }
            }

            progressLogTask.CancelToken.Cancel();
            progressLogTask.ShowComplete();
        }
    }
}
