﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;

namespace Nintendo.Authoring.AuthoringLibrary
{
    /// <summary>
    /// ISource を Stream に変換した Stream
    /// </summary>
    public class SourceBasedStream : Stream
    {
        public override bool CanRead { get { return true; } }
        public override bool CanSeek { get { return true; } }
        public override bool CanWrite { get { return false; } }
        public override long Length { get { return m_source.Size; } }
        public override long Position
        {
            get
            {
                return m_offset;
            }
            set
            {
                this.Seek(value, SeekOrigin.Begin);
            }
        }

        private ISource m_source;
        private long m_offset;

        public SourceBasedStream(ISource source)
        {
            m_source = source;
            m_offset = 0;
        }
        public override int Read(byte[] buffer, int offset, int count)
        {
            ByteData data = m_source.PullData(m_offset, count);
            if (data.Buffer.Count == 0)
            {
                return 0;
            }
            Buffer.BlockCopy(data.Buffer.Array, data.Buffer.Offset, buffer, offset, data.Buffer.Count);
            m_offset += data.Buffer.Count;
            return data.Buffer.Count;
        }
        public override void Write(byte[] buffer, int offset, int count)
        {
            throw new NotSupportedException();
        }
        public override long Seek(long offset, SeekOrigin origin)
        {
            long nextPosition = offset;
            if (origin == SeekOrigin.Begin)
            {
                nextPosition += 0;
            }
            else if (origin == SeekOrigin.Current)
            {
                nextPosition += this.Position;
            }
            else if (origin == SeekOrigin.End)
            {
                nextPosition += this.Length;
            }

            if (nextPosition < 0 || this.Length < nextPosition)
            {
                throw new ArgumentOutOfRangeException();
            }

            m_offset = nextPosition;
            return m_offset;
        }
        public override void SetLength(long value)
        {
            throw new NotSupportedException();
        }
        public override void Flush()
        {
        }
        protected override void Dispose(bool disposing)
        {
            base.Dispose(disposing);
        }
    }
}
