﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Security.Cryptography;
using System.Text;
using System.Xml.Serialization;
using Nintendo.ApplicationControlProperty;
using Nintendo.Authoring.FileSystemMetaLibrary;

namespace Nintendo.Authoring.AuthoringLibrary
{
    public struct ResultXmlInfo
    {
        public List<ContentMetaModel> ContentMetaList;
        public List<ProgramInfoModel> ProgramInfoList;
        public List<ApplicationControlPropertyModel> ApplicationControlPropertyList;
        public List<HtmlDocumentXmlModel> HtmlDocumentList;
        public List<LegalInformationModel> LegalInformationList;
        public List<AuthoringToolInfoModel> AuthoringToolInfoList;
        public byte[] CardHeaderHash;
        public CardHeaderModel CardHeader;
        public bool? ApplyDifference;

        public ResultXmlInfo(List<ContentMetaModel> contentMetaList, List<ProgramInfoModel> programInfoList, List<ApplicationControlPropertyModel> applicationControlPropertyList, List<HtmlDocumentXmlModel> htmlDocumentList, List<LegalInformationModel> legalInformationList, List<AuthoringToolInfoModel> authoringToolInfoList, byte[] cardHeaderHash, CardHeaderModel cardHeader, bool? applyDifference)
        {
            ContentMetaList = contentMetaList;
            ProgramInfoList = programInfoList;
            ApplicationControlPropertyList = applicationControlPropertyList;
            HtmlDocumentList = htmlDocumentList;
            LegalInformationList = legalInformationList;
            AuthoringToolInfoList = authoringToolInfoList;
            CardHeaderHash = cardHeaderHash;
            CardHeader = cardHeader;
            ApplyDifference = applyDifference;
        }
    }

    public class ResultXmlGenerator
    {
        private ResultModel m_model;
        private const string CodePass = "Pass";
        private const string CodeFailed = "Failed";

        public ResultXmlGenerator(Stream stream) : this(stream, GetResultXmlInfoFromNsp(stream))
        {
        }

        public ResultXmlGenerator(Stream stream, ResultXmlInfo info)
        {
            var resultModel = new ResultModel();
            resultModel.Code = CodePass;
            resultModel.ErrorMessage = string.Empty;
            {
                resultModel.Size = stream.Length;
                var hashCalculator = new SHA256CryptoServiceProvider();
                stream.Seek(0, SeekOrigin.Begin);
                resultModel.SetHashBytes(hashCalculator.ComputeHash(stream));
            }

            SetCommon(ref resultModel);
            SetInfo(ref resultModel, info);

            m_model = resultModel;
        }

        public ResultXmlGenerator(string errorMessage)
        {
            var resultModel = new ResultModel();
            resultModel.Code = CodeFailed;
            resultModel.ErrorMessage = errorMessage;
            SetCommon(ref resultModel);

            m_model = resultModel;
        }

        static private ResultXmlInfo GetResultXmlInfoFromNsp(Stream nspStream)
        {
            var reader = new NintendoSubmissionPackageReader(nspStream);
            var contentMetaList = new List<ContentMetaModel>();
            foreach (var info in reader.ListFileInfo().Where(x => x.Item1.EndsWith(".cnmt.xml")))
            {
                var model = ArchiveReconstructionUtils.ReadModel(reader, info.Item1, info.Item2);
                contentMetaList.Add(model);
            }
            var programInfoList = new List<ProgramInfoModel>();
            foreach (var info in reader.ListFileInfo().Where(x => x.Item1.EndsWith(".programinfo.xml")))
            {
                var model = ArchiveReconstructionUtils.ReadXml<ProgramInfoModel>(reader, info.Item1, info.Item2);
                programInfoList.Add(model);
            }
            var applicationControlPropertyList = new List<ApplicationControlPropertyModel>();
            foreach (var info in reader.ListFileInfo().Where(x => x.Item1.EndsWith(".nacp.xml")))
            {
                var model = ArchiveReconstructionUtils.ReadXml<ApplicationControlPropertyModel>(reader, info.Item1, info.Item2);
                applicationControlPropertyList.Add(model);
            }
            var htmlDocumentList = new List<HtmlDocumentXmlModel>();
            foreach (var info in reader.ListFileInfo().Where(x => x.Item1.EndsWith(".htmldocument.xml")))
            {
                var model = ArchiveReconstructionUtils.ReadXml<HtmlDocumentXmlModel>(reader, info.Item1, info.Item2);
                htmlDocumentList.Add(model);
            }
            var legalInformationList = new List<LegalInformationModel>();
            foreach (var info in reader.ListFileInfo().Where(x => x.Item1.EndsWith(".legalinfo.xml")))
            {
                var model = ArchiveReconstructionUtils.ReadXml<LegalInformationModel>(reader, info.Item1, info.Item2);
                legalInformationList.Add(model);
            }
            var authoringToolInfoList = new List<AuthoringToolInfoModel>();
            foreach (var info in reader.ListFileInfo().Where(x => x.Item1.EndsWith("authoringtoolinfo.xml")))
            {
                var model = ArchiveReconstructionUtils.ReadXml<AuthoringToolInfoModel>(reader, info.Item1, info.Item2);
                authoringToolInfoList.Add(model);
            }
            return new ResultXmlInfo(contentMetaList, programInfoList, applicationControlPropertyList, htmlDocumentList, legalInformationList, authoringToolInfoList, null, null, ResultModel.IsApplyDifference(contentMetaList));
        }

        private void SetCommon(ref ResultModel model)
        {
            model.Command = string.Join(" ", Environment.GetCommandLineArgs());
            model.ToolVersion = string.Join(",", System.AppDomain.CurrentDomain.GetAssemblies().Select(asm => asm.GetName().Name + ":" + asm.GetName().Version.ToString()).ToList());
        }

        private void SetInfo(ref ResultModel model, ResultXmlInfo info)
        {
            model.ContentMetaList = info.ContentMetaList;
            model.ProgramInfoList = info.ProgramInfoList;
            model.ApplicationControlPropertyList = info.ApplicationControlPropertyList;
            model.HtmlDocumentList = info.HtmlDocumentList;
            model.LegalInformationList = info.LegalInformationList;
            model.AuthoringToolInfoList = info.AuthoringToolInfoList;
            model.SetCardHeaderHashBytes(info.CardHeaderHash);
            model.CardHeader = info.CardHeader;
            model.ApplyDifference = info.ApplyDifference;
        }

        public void WriteToStream(Stream stream)
        {
            m_model.Date = System.DateTimeOffset.Now.ToString();

            var nameSpace = new XmlSerializerNamespaces();
            nameSpace.Add(String.Empty, String.Empty);

            stream.Seek(0, SeekOrigin.Begin);
            var sw = new StreamWriter(stream, Encoding.UTF8);

            var serializer = new XmlSerializer(typeof(ResultModel));
            serializer.Serialize(sw, m_model, nameSpace);
        }
    }

    [XmlRoot("ContentMeta", IsNullable = false)]
    public class ContentMetaListModel
    {
        [XmlElement("ContentMeta")]
        public List<ContentMetaModel> ContentMetaList { get; set; }
    }

    [XmlRoot("Result", IsNullable = false)]
    public class ResultModel
    {
        [XmlElement("Code")]
        public string Code { get; set; }

        [XmlElement("ErrorMessage")]
        public string ErrorMessage { get; set; }

        [XmlElement("Date")]
        public string Date { get; set; }

        [XmlElement("Command")]
        public string Command { get; set; }

        [XmlElement("ToolVersion")]
        public string ToolVersion { get; set; }

        [XmlElement("ContentMeta")]
        public List<ContentMetaModel> ContentMetaList { get; set; }

        [XmlElement("ProgramInfo")]
        public List<ProgramInfoModel> ProgramInfoList { get; set; }

        [XmlElement("Application")]
        public List<ApplicationControlPropertyModel> ApplicationControlPropertyList { get; set; }

        [XmlElement("HtmlDocument")]
        public List<HtmlDocumentXmlModel> HtmlDocumentList { get; set; }

        [XmlElement("SoftwareLegalInformation")]
        public List<LegalInformationModel> LegalInformationList { get; set; }

        [XmlElement("AuthoringToolInfo")]
        public List<AuthoringToolInfoModel> AuthoringToolInfoList { get; set; }

        [XmlElement("Hash")]
        public string Hash { get; set; }

        [XmlElement("Size")]
        public long Size { get; set; }

        [XmlElement("CardHeaderHash")]
        public string CardHeaderHash { get; set; }

        [XmlElement("CardHeader")]
        public CardHeaderModel CardHeader { get; set; }

        [XmlElement("ApplyDifference")]
        public bool? ApplyDifference { get; set; }
        [XmlIgnore]
        public bool ApplyDifferenceSpecified { get { return ApplyDifference != null; } }

        private byte[] GetStringBytes(string bytesString)
        {
            if (bytesString == null)
            {
                return null;
            }
            var str = bytesString.Replace("0x", string.Empty);
            var bytes = new byte[str.Length / 2];
            int j = 0;
            for (int i = 0; i < bytes.Length; i++)
            {
                bytes[i] = Convert.ToByte(str.Substring(j, 2), 16);
                j += 2;
            }
            return bytes;
        }

        private string GetBytesString(byte[] bytes)
        {
            return "0x" + BitConverter.ToString(bytes, 0, 32).Replace("-", string.Empty).ToLower();
        }

        public void SetHashBytes(byte[] bytes)
        {
            Hash = GetBytesString(bytes);
            Debug.Assert(Hash.Length >= 32);
        }

        public void SetCardHeaderHashBytes(byte[] bytes)
        {
            if (bytes == null)
            {
                return;
            }
            CardHeaderHash = GetBytesString(bytes);
            Debug.Assert(CardHeaderHash.Length >= 32);
        }

        public static bool? IsApplyDifference(List<ContentMetaModel> contentMetaList, bool before300 = false)
        {
            // デルタを含むパッチのみ確認
            var patchModels = contentMetaList.Where(x => x.Type == NintendoContentMetaConstant.ContentMetaTypePatch).Select(x => x as PatchContentMetaModel).Where(x => x.DeltaHistoryList != null && x.DeltaHistoryList.Count != 0);
            if (!patchModels.Any())
            {
                return null;
            }

            // TORIAEZU: デルタを含むパッチが複数ある場合、一つでも差分更新があれば true とする
            foreach (var patchModel in patchModels)
            {
                var deltaDownloadSize = patchModel.DeltaHistoryList.FirstOrDefault(x => x.Destination.Version == patchModel.Version)?.DownloadSize;
                if (before300)
                {
                    deltaDownloadSize += patchModel.DeltaHistoryList.FirstOrDefault()?.DownloadSize;
                }
                var directUpdateDownloadSize = (ulong)0;
                foreach (var content in patchModel.ContentList.Where(x => x.Type != NintendoContentMetaConstant.ContentTypeDeltaFragment))
                {
                    directUpdateDownloadSize += (ulong)content.Size;
                }
                // パッチ間差分を使用する == 直接更新のダウンロードサイズの方が大きい
                if (deltaDownloadSize < directUpdateDownloadSize)
                {
                    return true;
                }
            }

            return false;
        }

        public ResultModel()
        {
            Code = string.Empty;
            ErrorMessage = string.Empty;
            Date = string.Empty;
            Command = string.Empty;
            ToolVersion = string.Empty;
            ContentMetaList = new List<ContentMetaModel>();
            ProgramInfoList = new List<ProgramInfoModel>();
            ApplicationControlPropertyList = new List<ApplicationControlPropertyModel>();
            HtmlDocumentList = new List<HtmlDocumentXmlModel>();
            LegalInformationList = new List<LegalInformationModel>();
            AuthoringToolInfoList = new List<AuthoringToolInfoModel>();
            Hash = string.Empty;
            Size = 0;
        }

        public ResultXmlInfo GetInfo()
        {
            return new ResultXmlInfo(ContentMetaList, ProgramInfoList, ApplicationControlPropertyList, HtmlDocumentList, LegalInformationList, AuthoringToolInfoList, GetStringBytes(CardHeaderHash), CardHeader, ApplyDifference);
        }
    }
}
