﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Runtime.InteropServices;
using System.Security.Cryptography;
using System.Text.RegularExpressions;
using Nintendo.Authoring.Lz4Library;

namespace Nintendo.Authoring.AuthoringLibrary
{
    public enum ObjectType
    {
        NsoObject = (1 << 0),
        NsoLibraryObject = (1 << 1),
        NroObject = (1 << 2),
    }

    public enum NsoHeaderFlags
    {
        TextCompress = (1 << 0),
        RoCompress = (1 << 1),
        DataCompress = (1 << 2),
        TextHash = (1 << 3),
        RoHash = (1 << 4),
        DataHash = (1 << 5),
    }

    /// <summary>
    /// Nrrのヘッダ情報
    /// </summary>
    [StructLayout(LayoutKind.Sequential)]
    internal struct NrrHeader
    {
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 4)]
        public byte[] Signature;
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 12)]
        public byte[] Reserved1;
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 0x220)]
        public byte[] Certificate;
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 0x100)]
        public byte[] Subscribe;
        public UInt64 ApplicationId;
        public UInt32 Size;
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 4)]
        public byte[] Reserved2;
        public UInt32 HashOffset;
        public UInt32 HashCount;
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 8)]
        public byte[] Reserved3;
    }

    /// <summary>
    /// NSO オブジェクトのヘッダ情報
    /// </summary>
    [StructLayout(LayoutKind.Sequential)]
    internal struct NsoHeader
    {
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 4)]
        public byte[] Signature;
        public uint Version;
        public uint Reserved1;
        public uint Flags;
        public uint TextFileOffset;
        public uint TextMemoryOffset;
        public uint TextSize;
        public uint ModuleNameOffset;
        public uint RoFileOffset;
        public uint RoMemoryOffset;
        public uint RoSize;
        public uint ModuleNameSize;
        public uint DataFileOffset;
        public uint DataMemoryOffset;
        public uint DataSize;
        public uint BssSize;
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 0x20)]
        public byte[] ModuleId;
        public uint TextFileSize;
        public uint RoFileSize;
        public uint DataFileSize;
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 0x1c)]
        public byte[] Reserved2;
        public uint EmbededOffset;
        public uint EmbededSize;
        public uint DynStrOffset;
        public uint DynStrSize;
        public uint DynSymOffset;
        public uint DynSymSize;
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 0x20)]
        public byte[] TextHash;
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 0x20)]
        public byte[] RoHash;
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 0x20)]
        public byte[] DataHash;
    }

    /// <summary>
    /// NRO オブジェクトのヘッダ情報
    /// </summary>
    [StructLayout(LayoutKind.Sequential)]
    internal struct NroHeader
    {
        public const int RocrtSize = 0x8;
        public const int Reserved1Size = 0x8;
        public const int ModuleIdSize = 0x20;
        public const int Reserved2Size = 0x4;

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = RocrtSize)]
        public byte[] Rocrt;
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = Reserved1Size)]
        public byte[] Reserved1;
        public uint Signature;
        public uint Version;
        public uint Size;
        public uint Flags;
        public uint TextMemoryOffset;
        public uint TextSize;
        public uint RoMemoryOffset;
        public uint RoSize;
        public uint DataMemoryOffset;
        public uint DataSize;
        public uint BssSize;
        public uint Reserved0x3C;
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = ModuleIdSize)]
        public byte[] ModuleId;
        public uint DsoHandleOffset;
        [MarshalAs(UnmanagedType.ByValArray, SizeConst = Reserved2Size)]
        public byte[] Reserved2;
        public uint EmbededOffset;
        public uint EmbededSize;
        public uint DynStrOffset;
        public uint DynStrSize;
        public uint DynSymOffset;
        public uint DynSymSize;
    }

    public class ObjectFileBase
    {
        public virtual string FileName { get; protected set; }
        public virtual byte[] RoBinary { get; protected set; }
        public virtual byte[] ApiInfoBinary { get; protected set; }
        public virtual byte[] DynStrBinary { get; protected set; }
        public virtual byte[] DynSymBinary { get; protected set; }
        public string[] GetApiInfoStrings()
        {
            if (ApiInfoBinary == null)
            {
                return null;
            }
            string sjisString = System.Text.Encoding.GetEncoding("Shift_JIS").GetString(ApiInfoBinary);
            //null終端で分割
            string[] apiInfoStrings = sjisString.Split('\0');
            //空文字列を削除
            List<string> list = new List<string>(apiInfoStrings);
            list.Remove(string.Empty);
            apiInfoStrings = list.ToArray();
            return apiInfoStrings;
        }

        public virtual void ParseData(byte[] fileData, string path) { }
    }

    /// <summary>
    /// NRR  ファイルを表現するクラス
    /// </summary>
    public class NrrFile
    {
        public static List<byte[]> GetUsefulNroHashList(byte[] nrrData)
        {
            const UInt32 NrrHashValueSize = 32;
            List<byte[]> usefulList = new List<byte[]>();

            if (nrrData != null)
            {
                // ヘッダ取得
                int headerSize = Marshal.SizeOf(typeof(NrrHeader));
                var handle = GCHandle.Alloc(nrrData, GCHandleType.Pinned);
                NrrHeader nrrHeader = (NrrHeader)Marshal.PtrToStructure(handle.AddrOfPinnedObject(), typeof(NrrHeader));
                handle.Free();

                if (nrrHeader.HashCount > 0)
                {
                    for (int i = 0; i < nrrHeader.HashCount; ++i)
                    {
                        byte[] hashValue = new byte[NrrHashValueSize];
                        Array.Copy(nrrData, nrrHeader.HashOffset + i * NrrHashValueSize, hashValue, 0, NrrHashValueSize);
                        usefulList.Add(hashValue);
                    }
                }
            }
            return usefulList;
        }

        public static bool GetUsefulNroHashListResult(List<byte[]> usefulHashList)
        {
            foreach (byte[] usefulHash in usefulHashList)
            {
                if (usefulHash != null)
                {
                    return false;
                }
            }
            return true;
        }

        public static bool SetCheckUsefulHashValue(List<byte[]> usefulHashList, byte[] hashValue)
        {
            for (int i = 0; i < usefulHashList.Count; ++i)
            {
                if (usefulHashList[i] != null)
                {
                    if (System.Linq.Enumerable.SequenceEqual(usefulHashList[i], hashValue))
                    {
                        usefulHashList[i] = null;
                        return true;
                    }
                }
            }

            return false;
        }

        public static byte[] GetHashValue(byte[] byteData)
        {
            SHA256Managed sha256 = new SHA256Managed();
            return sha256.ComputeHash(byteData);
        }
    }

    /// <summary>
    /// NSO  ファイルを表現するクラス
    /// </summary>
    internal class NsoFile : ObjectFileBase
    {
        public override string FileName { get; protected set; }
        public override byte[] RoBinary { get; protected set; }
        public override byte[] ApiInfoBinary { get; protected set; }
        public override byte[] DynStrBinary { get; protected set; }
        public override byte[] DynSymBinary { get; protected set; }

        public override void ParseData(byte[] fileData, string path)
        {
            FileName = GetName(path);

            int headerSize = Marshal.SizeOf(typeof(NsoHeader));

            // ヘッダ取得
            byte[] headerData = fileData.Skip(0).Take(headerSize).ToArray();
            var handle = GCHandle.Alloc(headerData, GCHandleType.Pinned);
            NsoHeader nsoHeader = (NsoHeader)Marshal.PtrToStructure(handle.AddrOfPinnedObject(), typeof(NsoHeader));
            handle.Free();

            // ro領域取得
            byte[] data = fileData.Skip((int)nsoHeader.RoFileOffset).Take((int)nsoHeader.RoFileSize).ToArray();

            RoBinary = new byte[nsoHeader.RoSize];
            if ((nsoHeader.Flags & (uint)NsoHeaderFlags.RoCompress) == (uint)NsoHeaderFlags.RoCompress)
            {
                // ro領域がLZ4で圧縮されているので展開
                var size = Lz4.DecompressSafe(data, 0, RoBinary, 0, data.Length, RoBinary.Length);
                if (size != RoBinary.Length)
                {
                    throw new Exception("Extracting with lz4 was failed.\n" + path);
                }
            }
            else
            {
                // ro領域が無圧縮なのでそのままコピー
                data.CopyTo(RoBinary, 0);
            }

            if (nsoHeader.EmbededSize > 0)
            {
                // api_info領域の取得
                ApiInfoBinary = new byte[nsoHeader.EmbededSize];
                Array.Copy(RoBinary, nsoHeader.EmbededOffset, ApiInfoBinary, 0, (long)ApiInfoBinary.Length);
            }

            if (nsoHeader.DynStrOffset != 0 && nsoHeader.DynStrSize > 0)
            {
                // DynStr情報取得
                DynStrBinary = new byte[nsoHeader.DynStrSize];
                Array.Copy(RoBinary, nsoHeader.DynStrOffset, DynStrBinary, 0, (long)DynStrBinary.Length);
            }

            if (nsoHeader.DynSymOffset != 0 && nsoHeader.DynSymSize > 0)
            {
                // DynSym情報取得
                DynSymBinary = new byte[nsoHeader.DynSymSize];
                Array.Copy(RoBinary, nsoHeader.DynSymOffset, DynSymBinary, 0, (long)DynSymBinary.Length);
            }
        }

        private string GetName(string path)
        {
            Regex rule = new Regex(@"([^\\]+)\.nspd");
            string nsoPath;
            if (rule.IsMatch(path))
            {
                nsoPath = rule.Match(path).Groups[1].Value + ".nso";
            }
            else
            {
                nsoPath = Path.GetFileName(path);
            }

            return nsoPath;
        }
    }

    /// <summary>
    /// NRO  ファイルを表現するクラス
    /// </summary>
    internal class NroFile : ObjectFileBase
    {
        public override string FileName { get; protected set; }
        public override byte[] RoBinary { get; protected set; }
        public override byte[] ApiInfoBinary { get; protected set; }
        public override byte[] DynStrBinary { get; protected set; }
        public override byte[] DynSymBinary { get; protected set; }

        public override void ParseData(byte[] fileData, string path)
        {
            FileName = GetName(path);

            int headerSize = Marshal.SizeOf(typeof(NroHeader));

            // ヘッダ取得
            byte[] headerData = fileData.Skip(0).Take(headerSize).ToArray();
            var handle = GCHandle.Alloc(fileData, GCHandleType.Pinned);
            NroHeader nroHeader = (NroHeader)Marshal.PtrToStructure(handle.AddrOfPinnedObject(), typeof(NroHeader));
            handle.Free();

            // シグネチャのチェック("NRO0")
            if (nroHeader.Signature != 0x304F524E)
            {
                return;
            }
            // ro領域取得
            RoBinary = fileData.Skip((int)nroHeader.RoMemoryOffset).Take((int)nroHeader.RoSize).ToArray();

            if (nroHeader.EmbededSize > 0)
            {
                // api_info領域の取得
                ApiInfoBinary = new byte[nroHeader.EmbededSize];
                Array.Copy(RoBinary, nroHeader.EmbededOffset, ApiInfoBinary, 0, (long)ApiInfoBinary.Length);
            }

            if (nroHeader.DynStrOffset != 0 && nroHeader.DynStrSize > 0)
            {
                // DynStr情報取得
                DynStrBinary = new byte[nroHeader.DynStrSize];
                Array.Copy(RoBinary, nroHeader.DynStrOffset, DynStrBinary, 0, (long)DynStrBinary.Length);
            }

            if (nroHeader.DynSymOffset != 0 && nroHeader.DynSymSize > 0)
            {
                // DynSym情報取得
                DynSymBinary = new byte[nroHeader.DynSymSize];
                Array.Copy(RoBinary, nroHeader.DynSymOffset, DynSymBinary, 0, (long)DynSymBinary.Length);
            }
        }

        private string GetName(string path)
        {
            return Path.GetFileName(path);
        }
    }
}
