﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Security.Cryptography;
using System.Text;
using System.Text.RegularExpressions;
using System.Xml.Serialization;
using Nintendo.Authoring.FileSystemMetaLibrary;

namespace Nintendo.Authoring.AuthoringLibrary
{
    internal class MultiProgramApplicationNintendoSubmissionPackageArchive : IConnector
    {
        public List<Connection> ConnectionList { get; private set; }

        public MultiProgramApplicationNintendoSubmissionPackageArchive(IReadableSink outSink, List<NintendoSubmissionPackageReader> nspReaders, int? cardSize, int? cardClockRate, byte? cardLaunchFlags, KeyConfiguration keyConfig)
        {
            var nspInfo = new NintendoSubmissionPackageFileSystemInfo();
            {
                var unitNspInfos = nspReaders.Select(x => ArchiveReconstructionUtils.GetNspInfo(x, keyConfig));

                // ContentMetaId が 2 つ以上含まれているとエラー
                {
                    var distinctCount = unitNspInfos.SelectMany(x => x.Entries).Select(x => x.ContentMetaInfo.Model.Id).Distinct().Count();
                    if (distinctCount != 1)
                    {
                        throw new ArgumentException("Only one Application ID can be used in multi-program application.");
                    }
                }

                // ProgramIdOffset に重複・欠番があるとエラー
                {
                    var idOffsetList = unitNspInfos.SelectMany(x => x.Entries).Select(x => x.GetProgramIdOffset()).Distinct().ToList();
                    if (idOffsetList.Count != unitNspInfos.SelectMany(x => x.Entries).Count())
                    {
                        throw new ArgumentException("ProgramIndex of application must be serial and start at 0 in multi-program application.");
                    }
                    idOffsetList.Sort();
                    for (int i = 0; i < idOffsetList.Count; i++)
                    {
                        if (idOffsetList[i] != i)
                        {
                            throw new ArgumentException("ProgramIndex of application must be serial and start at 0 in multi-program application.");
                        }
                    }
                }

                // いったんマージ
                foreach (var unitNspInfo in unitNspInfos)
                {
                    nspInfo.Entries.AddRange(unitNspInfo.Entries);
                }

                // ProgramIndex 順に並び替え
                nspInfo.Entries.Sort(delegate(NintendoSubmissionPackageFileSystemInfo.EntryInfo x, NintendoSubmissionPackageFileSystemInfo.EntryInfo y)
                {
                    return x.GetProgramIdOffset().CompareTo(y.GetProgramIdOffset());
                });

                // エントリを集約
                {
                    var entry = nspInfo.Entries[0];
                    for (int i = 1; i < nspInfo.Entries.Count; i++)
                    {
                        // マルチプログラムアプリケーションのリーガル情報は idOffset = 0 のもののみを使う
                        entry.Contents.AddRange(nspInfo.Entries[i].Contents.Where(x => x.ContentType != NintendoContentMetaConstant.ContentTypeLegalInformation).ToList());
                        if (nspInfo.Entries[i].ProgramInfo != null)
                        {
                            if (entry.ProgramInfo == null)
                            {
                                entry.ProgramInfo = new NintendoSubmissionPackageFileSystemInfo.ProgramInfo(nspInfo.Entries[i].ProgramInfo.Models);
                            }
                            else
                            {
                                entry.ProgramInfo.Models.AddRange(nspInfo.Entries[i].ProgramInfo.Models);
                            }
                        }
                        if (nspInfo.Entries[i].ApplicationControlPropertyInfo != null)
                        {
                            if (entry.ApplicationControlPropertyInfo == null)
                            {
                                entry.ApplicationControlPropertyInfo = new NintendoSubmissionPackageFileSystemInfo.ApplicationControlPropertyInfo(nspInfo.Entries[i].ApplicationControlPropertyInfo.Models);
                            }
                            else
                            {
                                entry.ApplicationControlPropertyInfo.Models.AddRange(nspInfo.Entries[i].ApplicationControlPropertyInfo.Models);
                            }
                        }
                        if (nspInfo.Entries[i].HtmlDocumentInfo != null)
                        {
                            if (entry.HtmlDocumentInfo == null)
                            {
                                entry.HtmlDocumentInfo = new NintendoSubmissionPackageFileSystemInfo.HtmlDocumentInfo(nspInfo.Entries[i].HtmlDocumentInfo.Models);
                            }
                            else
                            {
                                entry.HtmlDocumentInfo.Models.AddRange(nspInfo.Entries[i].HtmlDocumentInfo.Models);
                            }
                        }
                        entry.ExtraData.AddRange(nspInfo.Entries[i].ExtraData);
                    }
                    nspInfo.Entries[0] = entry;
                    nspInfo.Entries.RemoveRange(1, nspInfo.Entries.Count - 1);
                }

                // カード設定の外部入力
                {
                    nspInfo.CardSize = cardSize ?? 0;
                    nspInfo.CardClockRate = cardClockRate ?? 0;
                    nspInfo.CardLaunchFlags = cardLaunchFlags;
                }
            }

            var nsp = new NintendoSubmissionPackageArchive(outSink, nspInfo, keyConfig);
            ConnectionList = new List<Connection>(nsp.ConnectionList);
        }

        public ISource GetSource()
        {
            throw new NotImplementedException();
        }
    }
}
