﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Collections.Generic;
using System.Security.Cryptography;
using System.Xml.Serialization;
using Nintendo.Authoring.FileSystemMetaLibrary;
using Nintendo.Authoring.CryptoLibrary;

namespace Nintendo.Authoring.AuthoringLibrary
{
    internal class ProdNcaKeyGenerator : IKeyGenerator
    {
        private IEncryptor[] m_bodyEncryptionKeyEncryptors;
        private IEncryptor m_headerEncryptionKeyEncryptor;

        public ProdNcaKeyGenerator(IEncryptor[] bodyEncryptionKeyEncryptors, IEncryptor headerEncryptionKeyEncryptor)
        {
            m_bodyEncryptionKeyEncryptors = bodyEncryptionKeyEncryptors;
            m_headerEncryptionKeyEncryptor = headerEncryptionKeyEncryptor;
        }

        public virtual byte[] Generate(byte[] encryptedKey, int keyType)
        {
            var key = new byte[encryptedKey.Length];

            var keyGenMap = new Dictionary<int, Action<byte[], int, int, byte[], int>>
            {
                { NintendoContentArchiveKeyType.KeyAreaEncryptionKey0, m_bodyEncryptionKeyEncryptors[0].DecryptBlock },
                { NintendoContentArchiveKeyType.KeyAreaEncryptionKey0Gen2, m_bodyEncryptionKeyEncryptors[2].DecryptBlock },
                { NintendoContentArchiveKeyType.KeyAreaEncryptionKey0Gen3, m_bodyEncryptionKeyEncryptors[3].DecryptBlock },
                { NintendoContentArchiveKeyType.KeyAreaEncryptionKey0Gen4, m_bodyEncryptionKeyEncryptors[4].DecryptBlock },
                { NintendoContentArchiveKeyType.KeyAreaEncryptionKey0Gen5, m_bodyEncryptionKeyEncryptors[5].DecryptBlock },
                { NintendoContentArchiveKeyType.HeaderEncryptionKey, m_headerEncryptionKeyEncryptor.DecryptBlock },
            };

            try
            {
                keyGenMap[keyType](encryptedKey, 0, encryptedKey.Length, key, 0);
            }
            catch (KeyNotFoundException)
            {
                // 未知の鍵インデックス・鍵世代で製品化された NCA は完全性検証でエラーにする
                throw new ArgumentException("Invalid keyType is used for prodencryption.");
            }

            return key;
        }

        public virtual bool GetUseDevHsm()
        {
            var hsm = new HsmInterface();
            return hsm.GetUseDev();
        }
    };

    public class IntegrityChecker
    {
        private KeyConfiguration m_keyConfig;
        private ProdNcaKeyGenerator m_keyGenerator;
        private IEncryptor m_initialDataEncryptor;
        private ICbcModeEncryptor[] m_titleKeyEncryptors = new ICbcModeEncryptor[NintendoContentFileSystemMetaConstant.SupportedKeyGenerationMax + 1];

        public IntegrityChecker(KeyConfiguration config)
        {
            m_keyConfig = config;
            IEncryptor[] bodyEncryptionKeyEncryptors = new IEncryptor[NintendoContentFileSystemMetaConstant.SupportedKeyGenerationMax + 1];
            foreach (var generation in new byte[] { 0, 2, 3, 4, 5 })
            {
                if (m_keyConfig.GetProdKeyAreaEncryptionKey(generation) != null)
                {
                    bodyEncryptionKeyEncryptors[generation] = new Aes128CryptoDriver(m_keyConfig.GetProdKeyAreaEncryptionKey(generation).Key);
                }
                else
                {
                    bodyEncryptionKeyEncryptors[generation] = new HsmAes128CryptoDriver(Aes128KeyIndex.NcaContentKey, generation);
                }
            }

            IEncryptor headerEncryptionKeyEncryptor;
            if (m_keyConfig.GetProdNcaHeaderEncryptionKek() != null)
            {
                headerEncryptionKeyEncryptor = new Aes128CryptoDriver(m_keyConfig.GetProdNcaHeaderEncryptionKek().Key);
            }
            else
            {
                headerEncryptionKeyEncryptor = new HsmAes128CryptoDriver(Aes128KeyIndex.NcaHeader);
            }

            m_keyGenerator = new ProdNcaKeyGenerator(bodyEncryptionKeyEncryptors, headerEncryptionKeyEncryptor);

            if (m_keyConfig.GetProdXciInitialDataEncryptionKey() != null)
            {
                m_initialDataEncryptor = new Aes128CryptoDriver(m_keyConfig.GetProdXciInitialDataEncryptionKey().Key);
            }
            else
            {
                m_initialDataEncryptor = new HsmAes128CryptoDriver(Aes128KeyIndex.XciInitialData);
            }

            foreach (var generation in new byte[] { 0, 2, 3, 4, 5 })
            {
                if (m_keyConfig.GetProdETicketCommonKey(generation) != null)
                {
                    m_titleKeyEncryptors[generation] = new Aes128CbcCryptoDriver(m_keyConfig.GetProdETicketCommonKey(generation).Key);
                }
                else
                {
                    m_titleKeyEncryptors[generation] = new HsmAes128CbcCryptoDriver(Aes128KeyIndex.ETicketCommonKey, generation);
                }
            }
        }

        public void CheckProdEncryptedNintendoSubmissionPackageArchive(Stream stream, Stream originalNspStream = null)
        {
            Log.Progress("Checking Integrity ...");

            stream.Seek(0, SeekOrigin.Begin);

            // 各 NCA 読出し
            var reader = new NintendoSubmissionPackageReader(stream);
            var contentInfoList = GetContentInfoList(reader);
            foreach (var nca in reader.ListFileInfo().Where(x => x.Item1.EndsWith(".nca")))
            {
                var contentInfoTuple = contentInfoList.Where(x => nca.Item1.StartsWith(x.Item1.Id)).Single();
                var contentInfo = contentInfoTuple.Item1;
                var toBeEncryptedByExternalKey = contentInfoTuple.Item2;

                NintendoContentArchiveReader ncaReader;
                if (toBeEncryptedByExternalKey)
                {
                    ncaReader = OpenNintendoContentArchiveReaderWithExternalKey(reader, nca.Item1);
                }
                else
                {
                    ncaReader = OpenNintendoContentArchiveReaderWithInternalKey(reader, nca.Item1);
                }

                if (originalNspStream != null) // Patch
                {
                    CheckProdNintendoContentArchive(ncaReader, contentInfo, new NintendoSubmissionPackageReader(originalNspStream));
                }
                else
                {
                    CheckProdNintendoContentArchive(ncaReader);
                }
                Log.Progress(string.Format("{0} done.", nca.Item1));
            }
        }

        private List<ContentMetaModel> ReadContentMetaList(Stream stream)
        {
            var serializer = new XmlSerializer(typeof(ResultModel));
            stream.Seek(0, SeekOrigin.Begin);
            var model = (ResultModel)serializer.Deserialize(stream);
            return model.ContentMetaList;
        }

        // Xcie の復号化はできないので XcieWriter でのみ検証
        public void CheckProdEncryptedXciArchive(Stream stream, Stream resultXmlStream)
        {
            Log.Progress("Checking Integrity ...");

            stream.Seek(0, SeekOrigin.Begin);

            // KeyArea のチェック
            var initialData = new byte[XciMeta.InitialDataSize];
            var titleKeyData = new byte[XciMeta.TitleKey1Size + XciMeta.TitleKey2Size];
            {
                var xciInitialDataStream = new SubStream(stream, 0, XciMeta.InitialDataSize + XciMeta.TitleKey1Size + XciMeta.TitleKey2Size);
                {
                    xciInitialDataStream.Read(initialData, 0, initialData.Length);
                    xciInitialDataStream.Read(titleKeyData, 0, titleKeyData.Length);
                }
            }

            CheckKeyArea(initialData, titleKeyData);

            byte[] rootHash;
            {
                var xciHeaderOffset = (long)XciInfo.CardKeyAreaPageCount * XciInfo.PageSize;
                var xciHeaderStream = new SubStream(stream, xciHeaderOffset, (long)XciInfo.CardHeaderPageCount * XciInfo.PageSize);
                var xciHeaderData = new byte[xciHeaderStream.Length];
                {
                    xciHeaderStream.Read(xciHeaderData, 0, (int)xciHeaderStream.Length);
                }
                rootHash = XciUtils.GetRootPartitionHash(xciHeaderData);
            }

            List<Tuple<string, string>> patchableContentList;
            {
                var contentMetaList = ReadContentMetaList(resultXmlStream);
                var patchContentMetas = contentMetaList.Where(x => x.Type == NintendoContentMetaConstant.ContentMetaTypePatch).ToList();
                var applicationContentMetas = contentMetaList.Where(x => x.Type == NintendoContentMetaConstant.ContentMetaTypeApplication).ToList();
                PatchedNintendoSubmissionPackageReader.SetContentListForXci(out patchableContentList, patchContentMetas, applicationContentMetas);
            }

            var bodyOffset = (long)XciInfo.NormalAreaStartPageAddress * XciInfo.PageSize;
            var xciBodyStream = new SubStream(stream, bodyOffset, stream.Length - bodyOffset);
            var rootReader = new XciReader(xciBodyStream, rootHash);
            {
                foreach (var rootEntry in rootReader.ListFileInfo())
                {
                    // 各 NCA 読出し
                    var reader = rootReader.OpenXciPartitionReader(rootEntry.Item1);
                    {
                        var ncas = reader.ListFileInfo().Where(x => x.Item1.EndsWith(".nca")).ToList();

                        if (rootEntry.Item1 == "update")
                        {
                            // UPP は個別に製品化・完全性検証済みなので、ハッシュ前半の検証のみに簡略化
                            var hashCalculator = new SHA256CryptoServiceProvider();
                            foreach (var nca in ncas)
                            {
                                var contentId = nca.Item1.Replace(".cnmt", string.Empty).Replace(".nca", string.Empty);
                                var ncaSource = new FileSystemArchvieFileSource(reader, nca.Item1);
                                byte[] hash;
                                using (var ncaStream = new SourceBasedStream(ncaSource))
                                {
                                    ncaStream.Seek(0, SeekOrigin.Begin);
                                    hash = hashCalculator.ComputeHash(ncaStream);
                                }
                                var hashStr = BitConverter.ToString(hash, 0, hash.Length).Replace("-", string.Empty).ToLower().Substring(0, hash.Length);
                                if (contentId != hashStr)
                                {
                                    throw new ArgumentException("Hash mismatch detected in update partition on card.");
                                }
                                Log.Progress(string.Format("{0}:/{1} done.", rootEntry.Item1, nca.Item1));
                            }

                            // 誤って開発版・製品版がすり替わっていないように 1 つだけフルチェックしておく
                            if (ncas.Count != 0)
                            {
                                CheckProdNintendoContentArchive(OpenNintendoContentArchiveReaderWithInternalKey(reader, ncas.First().Item1));
                            }

                            continue;
                        }

                        if (rootEntry.Item1 == "secure")
                        {
                            // パッチとパッチがあたる nca を先にチェックし除外する
                            foreach (var patchableNca in patchableContentList)
                            {
                                // カード上のアプリケーションのコンテンツは内部鍵で暗号化されている
                                // パッチ対象先のコンテンツがない場合は null を originalNcaReader に指定
                                var originalNcaReader = patchableNca.Item2 != null ? OpenNintendoContentArchiveReaderWithInternalKey(reader, patchableNca.Item2) : null;
                                // パッチ FS を含むコンテンツは外部鍵（兼チケット存在確認）
                                var patchNcaReader = OpenNintendoContentArchiveReaderWithExternalKey(reader, patchableNca.Item1);
                                CheckProdNintendoContentArchive(patchNcaReader, originalNcaReader);
                                Log.Progress(string.Format("{0}:/{1} done.", rootEntry.Item1, patchableNca.Item1));
                                ncas.RemoveAll(x => x.Item1 == patchableNca.Item1);
                                if (patchableNca.Item2 != null)
                                {
                                    Log.Progress(string.Format("{0}:/{1} done.", rootEntry.Item1, patchableNca.Item2));
                                    ncas.RemoveAll(x => x.Item1 == patchableNca.Item2);
                                }
                            };
                        }

                        foreach (var nca in ncas)
                        {
                            CheckProdNintendoContentArchive(OpenNintendoContentArchiveReaderWithInternalKey(reader, nca.Item1));
                            Log.Progress(string.Format("{0}:/{1} done.", rootEntry.Item1, nca.Item1));
                        }
                    }
                }
            }
        }

        private void CheckKeyArea(byte[] initialData, byte[] titleKeyData)
        {
            var initialDataA = initialData.Take(XciMeta.InitialDataASize).ToArray();
            var initialDataB = initialData.Skip(initialDataA.Length).Take(XciMeta.TitleKey1Size + XciMeta.TitleKey2Size).ToArray();
            var mac = initialData.Skip(initialDataA.Length + initialDataB.Length).Take(XciMeta.InitialDataMacSize).ToArray();
            var nonce = initialData.Skip(initialDataA.Length + initialDataB.Length + mac.Length).Take(XciMeta.InitialDataNonceSize).ToArray();
            var aad = new byte[0];

            var key = new byte[m_initialDataEncryptor.KeySize];
            {
                var hsm = new HsmInterface();
                if (!hsm.GetUseDev())
                {
                    m_initialDataEncryptor.DecryptBlock(initialDataA, 0, initialDataA.Length, key, 0);
                }
            }

            var titleKeyDataCmp = new byte[XciMeta.TitleKey1Size + XciMeta.TitleKey2Size];
            {
                var titleKeyEncryptor = new Aes128CcmCryptoDriver(key);
                titleKeyEncryptor.DecryptBlock(nonce, aad, initialDataB, 0, initialDataB.Length, mac, 0, mac.Length, titleKeyDataCmp, 0);
            }

            if (!titleKeyData.SequenceEqual(titleKeyDataCmp))
            {
                throw new InvalidOperationException("Title key mismatch.");
            }

            Log.Progress(string.Format("KeyArea done."));
        }

        private void CheckProdNintendoContentArchive(NintendoContentArchiveReader reader, NintendoContentArchiveReader originalReader = null)
        {
            // Nca の各ストレージ読出し
            // Patch 対象のコンテンツを予め指定する
            foreach (var i in reader.GetExistentFsIndices())
            {
                var fsInfo = reader.GetFsHeaderInfo(i);
                IFileSystemArchiveReader fsReader;
                if (fsInfo.GetPatchInfo() != null)
                {
                    fsReader = reader.OpenFileSystemArchiveReader(i, originalReader);
                }
                else
                {
                    fsReader = reader.OpenFileSystemArchiveReader(i);
                }
                ReadBaseAll(fsReader);
            }
        }

        private void CheckProdNintendoContentArchive(NintendoContentArchiveReader reader, ContentInfo contentInfo, NintendoSubmissionPackageReader originalNspReader)
        {
            // Nca (Patch 含む) の各ストレージ読出し
            // Patch 対象のコンテンツは originalNspReader の cnmt.xml から探す
            foreach (var i in reader.GetExistentFsIndices())
            {
                var fsInfo = reader.GetFsHeaderInfo(i);
                IFileSystemArchiveReader fsReader;
                if (fsInfo.GetPatchInfo() != null)
                {
                    var contentId = ArchiveReconstructionUtils.GetSpecifiedContentId(originalNspReader, contentInfo.ApplicationId, contentInfo.Type, contentInfo.IdOffset);
                    if (contentId != null)
                    {
                        var originalNcaReader = originalNspReader.OpenNintendoContentArchiveReader(contentId + ".nca", m_keyGenerator);
                        // 内部鍵か外部鍵かは問わない
                        if (TicketUtility.IsValidRightsId(originalNcaReader.GetRightsId()))
                        {
                            var ticketName = TicketUtility.CreateRightsIdText(originalNcaReader.GetRightsId()) + ".tik";
                            var ticketReader = new TicketReader(originalNspReader.ReadFile(ticketName, 0, originalNspReader.GetFileSize(ticketName)), m_titleKeyEncryptors[originalNcaReader.GetKeyGeneration()]);
                            originalNcaReader.SetExternalKey(ticketReader.GetTitleKey());
                        }
                        fsReader = reader.OpenFileSystemArchiveReader(i, originalNcaReader);
                    }
                    else
                    {
                        fsReader = reader.OpenFileSystemArchiveReader(i, null);
                    }
                }
                else
                {
                    fsReader = reader.OpenFileSystemArchiveReader(i);
                }
                ReadBaseAll(fsReader);
            }
        }

        // Item1: ContentInfo, Item2: ToBeEncryptedByExternalKey
        private List<Tuple<ContentInfo, bool>> GetContentInfoList(NintendoSubmissionPackageReader nspReader)
        {
            var contentInfoList = new List<Tuple<ContentInfo, bool>>();
            foreach (var cnmtXml in nspReader.ListFileInfo().Where(x => x.Item1.EndsWith(".cnmt.xml")))
            {
                var cnmtXmlModel = ArchiveReconstructionUtils.ReadXml<ContentMetaModel>(nspReader, cnmtXml.Item1, cnmtXml.Item2);
                String applicationId;
                if (cnmtXmlModel.Type == NintendoContentMetaConstant.ContentMetaTypePatch)
                {
                    var patchCnmtXmlModel = ArchiveReconstructionUtils.ReadXml<PatchContentMetaModel>(nspReader, cnmtXml.Item1, cnmtXml.Item2);
                    applicationId = patchCnmtXmlModel.ApplicationId;
                }
                else
                {
                    applicationId = cnmtXmlModel.Id;
                }
                foreach (var content in cnmtXmlModel.ContentList)
                {
                    var contentInfo = new ContentInfo();
                    contentInfo.ApplicationId = Convert.ToUInt64(applicationId, 16);
                    contentInfo.Type = content.Type;
                    contentInfo.Id = content.Id;
                    contentInfo.IdOffset = content.IdOffset;
                    bool toBeEncryptedByExternalKey = ((cnmtXmlModel.Type == NintendoContentMetaConstant.ContentMetaTypeApplication || cnmtXmlModel.Type == NintendoContentMetaConstant.ContentMetaTypePatch || cnmtXmlModel.Type == NintendoContentMetaConstant.ContentMetaTypeAddOnContent) &&
                                                       (content.Type == NintendoContentMetaConstant.ContentTypeProgram || content.Type == NintendoContentMetaConstant.ContentTypeHtmlDocument || content.Type == NintendoContentMetaConstant.ContentTypePublicData || content.Type == NintendoContentMetaConstant.ContentTypeData));
                    contentInfoList.Add(Tuple.Create(contentInfo, toBeEncryptedByExternalKey));
                }
            }
            return contentInfoList;
        }

        private void ReadBaseAll(IFileSystemArchiveReader reader)
        {
            const long BufferSize = 8 * 1024 * 1024;
            long restSize = reader.GetBaseSize();
            long offset = 0;
            while (restSize > 0)
            {
                long readSize = Math.Min(restSize, BufferSize);
                reader.ReadBase(offset, readSize);
                restSize -= readSize;
                offset += readSize;
            }
        }

        private void CheckInternalKey(NintendoContentArchiveReader ncaReader, string ncaName)
        {
            if (TicketUtility.IsValidRightsId(ncaReader.GetRightsId()))
            {
                throw new ArgumentException(string.Format("{0} has rightsId although it has to be encrypted by internal key.", ncaName));
            }
            if (!ncaReader.HasValidInternalKey())
            {
                throw new ArgumentException(string.Format("{0} doesn't have internal key although it is encrypted by internal key.", ncaName));
            }
        }

        private void CheckAndSetExternalKey(ref NintendoContentArchiveReader ncaReader, string ncaName, IFileSystemArchiveReader parentReader)
        {
            if (!TicketUtility.IsValidRightsId(ncaReader.GetRightsId()))
            {
                throw new ArgumentException(string.Format("{0} doesn't have rightsId although it has to be encrypted by external key.", ncaName));
            }

            var ticketName = TicketUtility.CreateRightsIdText(ncaReader.GetRightsId()) + ".tik";
            var ticketReader = new TicketReader(parentReader.ReadFile(ticketName, 0, parentReader.GetFileSize(ticketName)), m_titleKeyEncryptors[ncaReader.GetKeyGeneration()]);
            ncaReader.SetExternalKey(ticketReader.GetTitleKey());
            if (ncaReader.HasValidInternalKey())
            {
                throw new ArgumentException(string.Format("{0} has internal key although it is encrypted by external key.", ncaName));
            }
        }

        private NintendoContentArchiveReader OpenNintendoContentArchiveReaderWithInternalKey(NintendoSubmissionPackageReader nspReader, string ncaName)
        {
            var ncaReader = nspReader.OpenNintendoContentArchiveReader(ncaName, m_keyGenerator);
            CheckInternalKey(ncaReader, ncaName);
            return ncaReader;
        }

        private NintendoContentArchiveReader OpenNintendoContentArchiveReaderWithInternalKey(XciPartitionReader partitionReader, string ncaName)
        {
            var ncaReader = partitionReader.OpenNintendoContentArchiveReader(ncaName, m_keyGenerator);
            CheckInternalKey(ncaReader, ncaName);
            return ncaReader;
        }

        private NintendoContentArchiveReader OpenNintendoContentArchiveReaderWithExternalKey(NintendoSubmissionPackageReader nspReader, string ncaName)
        {
            var ncaReader = nspReader.OpenNintendoContentArchiveReader(ncaName, m_keyGenerator);
            CheckAndSetExternalKey(ref ncaReader, ncaName, nspReader);
            return ncaReader;
        }

        private NintendoContentArchiveReader OpenNintendoContentArchiveReaderWithExternalKey(XciPartitionReader partitionReader, string ncaName)
        {
            var ncaReader = partitionReader.OpenNintendoContentArchiveReader(ncaName, m_keyGenerator);
            CheckAndSetExternalKey(ref ncaReader, ncaName, partitionReader);
            return ncaReader;
        }

        private struct ContentInfo
        {
            public UInt64 ApplicationId;
            public String Type;
            public String Id;
            public byte IdOffset;
        }
    }
}
