﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Text;
using System.Xml.Serialization;
using Nintendo.Authoring.FileSystemMetaLibrary;
using YamlDotNet.RepresentationModel;

namespace Nintendo.Authoring.AuthoringLibrary
{
    public class DeltaSourceInfo
    {
        public long Size
        {
            get; private set;
        }

        public DeltaSourceInfo(long sourceSize)
        {
            Size = sourceSize;
        }
    }

    public class DeltaCommand
    {
        public long Offset
        {
            get; private set;
        }
        public long Size
        {
            get; set;
        }
        public bool IsAppend
        {
            get; private set;
        }

        public DeltaCommand(long offset, long size, bool append)
        {
            Offset = offset;
            Size = size;
            IsAppend = append;
        }
    }

    static public class DeltaInfo
    {
        private const int WorkBufferSize = 1 * 1024 * 1024;

        private enum Mode
        {
            SearchDelta,
            SearchSimilarity,
        }

        public static DeltaSourceInfo ReadSourceInfo(ISource source)
        {
            if (source == null)
            {
                throw new ArgumentException("parameter null is not allowed.");
            }
            return new DeltaSourceInfo(source.Size);
        }

        public static IList<DeltaCommand> ExtractDeltaCommands(
            ISource source,
            ISource destination,
            long sizeMax,
            int blockSize,
            Action<double> notifyProgress)
        {
            if (source == null || destination == null)
            {
                throw new ArgumentException("parameter null is not allowed.");
            }

            IList<DeltaCommand> deltaCommands = new List<DeltaCommand>();
            long offset = 0;
            long size = 0;
            var mode = Mode.SearchDelta;

            if (notifyProgress != null)
            {
                notifyProgress.Invoke(0.0);
            }

            while (true)
            {
                if (offset + size >= destination.Size)
                {
                    if (mode == Mode.SearchSimilarity)
                    {
                        size = destination.Size - offset;
                        if (size > 0)
                        {
                            AddDeltaCommand(deltaCommands, offset, size, false, sizeMax);
                        }
                    }
                    else
                    {
                        offset = source.Size;
                        size = destination.Size - source.Size;
                        if (size > 0)
                        {
                            AddDeltaCommand(deltaCommands, offset, size, false, sizeMax);
                        }
                    }

                    if (notifyProgress != null)
                    {
                        notifyProgress.Invoke(1.0);
                    }

                    break;
                }
                if (offset + size >= source.Size)
                {
                    if (mode == Mode.SearchSimilarity)
                    {
                        // 末尾まで
                        size = source.Size - offset;
                        if (size > 0)
                        {
                            AddDeltaCommand(deltaCommands, offset, size, true, sizeMax);
                        }
                    }

                    // 末尾以降
                    offset = source.Size;
                    size = destination.Size - offset;
                    if (size > 0)
                    {
                        AddDeltaCommand(deltaCommands, offset, size, true, sizeMax);
                    }

                    if (notifyProgress != null)
                    {
                        notifyProgress.Invoke(1.0);
                    }

                    break;
                }

                var sourceData = source.PullData(offset + size, (int)Math.Min(source.Size - (offset + size), WorkBufferSize));
                var destinationData = destination.PullData(offset + size, (int)Math.Min(destination.Size - (offset + size), WorkBufferSize));
                if (notifyProgress != null)
                {
                    notifyProgress.Invoke((double)offset / (double)destination.Size);
                }

                int arrayOffset = 0;
                while (true)
                {
                    if (sourceData.Buffer.Count <= arrayOffset)
                    {
                        break;
                    }
                    if (destinationData.Buffer.Count <= arrayOffset)
                    {
                        size += destinationData.Buffer.Count - arrayOffset;
                        break;
                    }

                    int index = SearchDeltaInByteArray(
                        mode,
                        sourceData.Buffer,
                        destinationData.Buffer,
                        arrayOffset,
                        blockSize);
                    if (index < 0)
                    {
                        size += destinationData.Buffer.Count - arrayOffset;
                        break;
                    }
                    else
                    {
                        var commandDataTotalSize = size + index - arrayOffset;
                        if (mode == Mode.SearchSimilarity)
                        {
                            if (commandDataTotalSize > 0)
                            {
                                AddDeltaCommand(deltaCommands, offset, commandDataTotalSize, false, sizeMax);
                            }
                            mode = Mode.SearchDelta;
                        }
                        else
                        {
                            mode = Mode.SearchSimilarity;
                        }

                        size = 0;
                        offset += commandDataTotalSize;
                        arrayOffset = index;
                    }
                }
            }

            return deltaCommands;
        }

        public static void CombineNeighbor(IList<DeltaCommand> deltaCommands, long mergeSeekSizeMax, long commandSizeMax)
        {
            long previousOffset = 0;
            int startIndex = 0;
            int currentIndex = 0;
            while (currentIndex < deltaCommands.Count)
            {
                Func<int, long> getDataSizeCurrent = delegate(int endIndex)
                {
                    var deltaCommandStart = deltaCommands[startIndex];
                    var deltaCommandEnd = deltaCommands[endIndex];
                    return (deltaCommandEnd.Offset + deltaCommandEnd.Size - deltaCommandStart.Offset);
                };

                Func<int, long> getCommandSizeCurrent = delegate (int endIndex)
                {
                    var dataSize = getDataSizeCurrent(endIndex);
                    var deltaCommandStart = deltaCommands[startIndex];
                    return DeltaMeta.GetWriteCommandSize(deltaCommandStart.Offset - previousOffset, dataSize) + dataSize;
                };

                if (currentIndex + 1 == deltaCommands.Count
                    || getCommandSizeCurrent(currentIndex + 1) > commandSizeMax
                    || (deltaCommands[currentIndex + 1].Offset - (deltaCommands[currentIndex].Offset + deltaCommands[currentIndex].Size)) > mergeSeekSizeMax)
                {
                    deltaCommands[startIndex].Size = getDataSizeCurrent.Invoke(currentIndex);
                    for (int j = startIndex + 1; j <= currentIndex; ++j)
                    {
                        deltaCommands.RemoveAt(startIndex + 1);
                    }
                    previousOffset = deltaCommands[startIndex].Offset + deltaCommands[startIndex].Size;
                    ++startIndex;
                    currentIndex = startIndex;
                }
                else
                {
                    ++currentIndex;
                }
            }
        }

        private static void AddDeltaCommand(
            IList<DeltaCommand> dstDeltaCommands,
            long commandDataOffset,
            long commandDataSize,
            bool commandIsAppend,
            long sizeMax)
        {
            System.Diagnostics.Debug.Assert(commandDataSize > 0);

            long previousOffset = 0;
            if (dstDeltaCommands.Count > 0)
            {
                var lastEntry = dstDeltaCommands[dstDeltaCommands.Count - 1];
                previousOffset = lastEntry.Offset + lastEntry.Size;
            }

            long offset = 0;

            while (commandDataSize > 0)
            {
                var entrySize = Math.Min(sizeMax - 3, commandDataSize);
                var commandSize = DeltaMeta.GetWriteCommandSize(commandDataOffset - previousOffset, entrySize);
                if (commandSize + entrySize > sizeMax)
                {
                    entrySize = sizeMax - commandSize;
                }

                dstDeltaCommands.Add(new DeltaCommand(commandDataOffset + offset, entrySize, commandIsAppend));

                offset += entrySize;
                commandDataSize -= entrySize;
            }
        }

        private static int SearchDeltaInByteArray(
            Mode mode,
            ArraySegment<byte> source,
            ArraySegment<byte> destionation,
            int offset,
            int sizeBlock)
        {
            var length = Math.Min(source.Count - offset, destionation.Count - offset);
            var blockCount = ((length + sizeBlock - 1) / sizeBlock);
            for (int i = 0; i < blockCount; i++)
            {
                var resultCompare = DeltaMeta.MemoryCompare(
                    source,
                    destionation,
                    offset + i * sizeBlock,
                    Math.Min(sizeBlock, length - i * sizeBlock));
                if (mode == Mode.SearchDelta)
                {
                    resultCompare = !resultCompare;
                }

                if (resultCompare)
                {
                    return offset + i * sizeBlock;
                }
            }
            return -1;
        }
    }

    public class DeltaArchiveSource : ISource
    {
        public long Size { get; private set; }

        private SourceStatus SourceStatus { get; set; }

        private ISource Header { get; set; }

        private IEnumerable<DeltaCommand> DeltaCommands { get; set; }

        private ISource DestinationSource { get; set; }

        public DeltaArchiveSource(
            DeltaSourceInfo sourceInfo,
            ISource destination,
            IEnumerable<DeltaCommand> deltaCommands)
        {
            if (sourceInfo == null || destination == null || deltaCommands == null)
            {
                throw new ArgumentException("parameter null is not allowed.");
            }

            // ForEachDeltaCommand で使用するので先に設定
            DeltaCommands = deltaCommands;
            DestinationSource = destination;

            // ボディ
            long bodySize = 0;
            ForEachDeltaCommand((command, commandSize, commandOffset) =>
            {
                bodySize += commandSize;
                return OnForEachDeltaCommandNextBehavior.Continue;
            });

            // ヘッダー
            {
                var header = DeltaMeta.CreateHeader(
                    sourceInfo.Size,
                    destination.Size,
                    bodySize);
                Header = new MemorySource(header, 0, DeltaMeta.GetHeaderSize());
            }

            Size = Header.Size + bodySize;

            SourceStatus = new SourceStatus();
            SourceStatus.AvailableRangeList.Add(new Range(0, Size));
        }

        public ByteData PullData(long offset, int size)
        {
            long sizeBuffer = Math.Min(size, Size - offset);
            var buffer = new byte[sizeBuffer];
            int offsetBuffer = 0;

            if (0 <= offset && offset < Header.Size)
            {
                int sizeCopyFromHeader = (int)(Math.Min(Header.Size, size) - offset);
                Array.Copy(
                    Header.PullData(offset, (int)sizeCopyFromHeader).Buffer.Array, 0,
                    buffer, 0,
                    sizeCopyFromHeader);
                offsetBuffer += sizeCopyFromHeader;
                offset += sizeCopyFromHeader;
            }

            if (offset >= Header.Size)
            {
                long offsetData = Header.Size;
                long previousOffset = 0;
                foreach (var deltaCommand in DeltaCommands)
                {
                    var header = DeltaMeta.CreateWriteCommand(
                        deltaCommand.Offset - previousOffset,
                        deltaCommand.Size);
                    previousOffset = deltaCommand.Offset + deltaCommand.Size;

                    if (offsetData <= offset && offset < offsetData + header.Length)
                    {
                        int copyLength = (int)Math.Min(offsetData + header.Length - offset, sizeBuffer - offsetBuffer);
                        Array.Copy(
                            header, offset - offsetData,
                            buffer, offsetBuffer,
                            copyLength);
                        offset += copyLength;
                        offsetBuffer += copyLength;
                    }
                    offsetData += header.Length;

                    if (offsetBuffer >= sizeBuffer)
                    {
                        break;
                    }

                    if (offsetData <= offset && offset < offsetData + deltaCommand.Size)
                    {
                        int copyLength = (int)Math.Min(offsetData + deltaCommand.Size - offset, sizeBuffer - offsetBuffer);
                        var data = DestinationSource.PullData(deltaCommand.Offset + (offset - offsetData), copyLength);
                        Array.Copy(
                            data.Buffer.Array, data.Buffer.Offset,
                            buffer, offsetBuffer,
                            copyLength);
                        offset += copyLength;
                        offsetBuffer += copyLength;
                    }
                    offsetData += deltaCommand.Size;

                    if (offsetBuffer >= sizeBuffer)
                    {
                        break;
                    }
                }
            }

            return new ByteData(new ArraySegment<byte>(buffer));
        }

        public SourceStatus QueryStatus()
        {
            return SourceStatus;
        }

        private enum OnForEachDeltaCommandNextBehavior
        {
            Continue,
            Break,
        }

        private void ForEachDeltaCommand(Func<DeltaCommand, long, long, OnForEachDeltaCommandNextBehavior> onForEachDeltaCommand)
        {
            long previousDestinationOffset = 0;

            // 差分コマンドから探索
            foreach (var deltaCommand in DeltaCommands)
            {
                var commandOffset = deltaCommand.Offset - previousDestinationOffset;
                var commandHeaderSize = DeltaMeta.GetWriteCommandSize(
                    commandOffset,
                    deltaCommand.Size);
                previousDestinationOffset = deltaCommand.Offset + deltaCommand.Size;

                var result = onForEachDeltaCommand.Invoke(deltaCommand, commandHeaderSize + deltaCommand.Size, commandOffset);
                if (result == OnForEachDeltaCommandNextBehavior.Break)
                {
                    return;
                }
            }

            // 終端シーク
            {
                var seekSize = DestinationSource.Size - previousDestinationOffset;
                if (seekSize > 0)
                {
                    var seekCommand = new DeltaCommand(seekSize, 0, false);
                    var commandSize = DeltaMeta.GetWriteCommandSize(seekSize, 0);
                    var result = onForEachDeltaCommand.Invoke(seekCommand, commandSize, seekSize);
                    if (result == OnForEachDeltaCommandNextBehavior.Break)
                    {
                        return;
                    }
                }
            }
        }

        public long FindNearestEndOffset(long offset)
        {
            long nearestDeltaOffset = DeltaMeta.GetHeaderSize();

            ForEachDeltaCommand((command, commandSize, commandDataOffset) =>
            {
                if (nearestDeltaOffset + commandSize > offset)
                {
                    return OnForEachDeltaCommandNextBehavior.Break;
                }
                else
                {
                    nearestDeltaOffset += commandSize;
                    return OnForEachDeltaCommandNextBehavior.Continue;
                }
            });

            return nearestDeltaOffset;
        }

        internal void OutputBuildLog(string targetContentType, long offset, long size, NintendoContentArchiveBuildLog buildLog)
        {
            if (buildLog == null)
            {
                return;
            }

            if (offset + size <= DeltaMeta.GetHeaderSize())
            {
                return;
            }

            if (buildLog != null)
            {
                long startOffset = offset < DeltaMeta.GetHeaderSize() ? 0 : offset - DeltaMeta.GetHeaderSize();
                long endOffset = offset + size - DeltaMeta.GetHeaderSize();
                long currentOffset = 0;

                FileStream deltaCommandStream = null;
                try
                {
                    deltaCommandStream = new FileStream(buildLog.GetOutputPath(".delta.csv"), FileMode.Create);
                    using (var sw = new StreamWriter(deltaCommandStream))
                    {
                        sw.WriteLine("TargetContentType," + targetContentType);
                        sw.WriteLine("Command,CommandValue,CommandSize,Offset,Size,Misc");

                        ForEachDeltaCommand((command, commandSize, commandOffset) =>
                        {
                            if (currentOffset + commandSize > endOffset)
                            {
                                return OnForEachDeltaCommandNextBehavior.Break;
                            }
                            else
                            {
                                if (currentOffset >= startOffset)
                                {
                                    sw.WriteLine(string.Format("{0},{1},{2},{3},{4},{5}",
                                        "Write",
                                        DeltaMeta.CommandTypeWrite,
                                        commandSize,
                                        commandOffset,
                                        command.Size,
                                        command.IsAppend ? "Append" : string.Empty));
                                }

                                currentOffset += commandSize;
                                return OnForEachDeltaCommandNextBehavior.Continue;
                            }
                        });
                    }
                }
                finally
                {
                    if (deltaCommandStream != null)
                    {
                        deltaCommandStream.Dispose();
                    }
                }
            }
        }
    }

    public class DeltaArchive : IConnector
    {
        public static readonly long DeltaCommandSizeMax = 8 * 1024 * 1024;
        public static readonly int DeltaCommandCompareSize = 4;

        // ncm ライブラリテストの ApplyDeltaPerformanceTest.CheckApplyDeltaCharacteristic で得られた結果から求めた値
        public static readonly long DeltaCommandMergeSeekSizeMax = 48 * 1024;

        internal DeltaArchive(
            IReadableSink outSink,
            DeltaSourceInfo sourceInfo,
            ISource destination,
            IEnumerable<DeltaCommand> deltaCommands)
        {
            var source = new DeltaArchiveSource(sourceInfo, destination, deltaCommands);
            outSink.SetSize(source.Size);

            ConnectionList = new List<Connection>();
            ConnectionList.Add(new Connection(source, outSink));
        }

        public List<Connection> ConnectionList
        {
            get; private set;
        }

        public ISource GetSource()
        {
            throw new NotImplementedException();
        }
    }
}
