﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Collections.Generic;
using System.Xml.Serialization;
using Nintendo.ApplicationControlProperty;
using Nintendo.Authoring.FileSystemMetaLibrary;
using YamlDotNet.RepresentationModel;

namespace Nintendo.Authoring.AuthoringLibrary
{
    /// <summary>
    /// コンテンツアーカイブのフォーマットです。
    /// </summary>
    public enum ArchiveFormatType
    {
        /// <summary>
        /// Nintendo コンテンツフォーマットです。
        /// </summary>
        NintendoContent = 0,
        /// <summary>
        /// Parition File System フォーマットです。
        /// </summary>
        PartitionFs,
        /// <summary>
        /// RomFs File System フォーマットです。
        /// </summary>
        RomFs,
        /// <summary>
        /// アプリケーション提出形式フォーマットです。
        /// </summary>
        NintendoSubmissionPackage,
        /// <summary>
        /// delta 用アプリケーション提出形式フォーマットです。
        /// </summary>
        NintendoSubmissionPackageForDelta,
        /// <summary>
        /// 無効なフォーマットです。
        /// </summary>
        Invalid = 255,
    }

    /// <summary>
    /// コンテンツアーカイブを作成・編集するクラスです
    /// </summary>
    public class ContentArchiveLibraryInterface
    {
        public static ArchiveFormatType GetArchiveType(string adfPath)
        {
            // フォーマットチェック
            using (var adf = new StreamReader(adfPath, Encoding.UTF8))
            {
                var yamlStream = new YamlStream();
                yamlStream.Load(adf);

                YamlMappingNode rootNode;
                try
                {
                    rootNode = (YamlMappingNode)yamlStream.Documents[0].RootNode;
                }
                catch
                {
                    throw new ArgumentException("invalid format .adf file. no \"entries\"");
                }

                YamlScalarNode formatType;
                try
                {
                    formatType = (YamlScalarNode)rootNode.Children[new YamlScalarNode("formatType")];
                }
                catch
                {
                    throw new ArgumentException("invalid format .adf file. no \"formatType\"\n");
                }
                switch (formatType.Value)
                {
                    case "NintendoContent":
                        return ArchiveFormatType.NintendoContent;
                    case "PartitionFs":
                        return ArchiveFormatType.PartitionFs;
                    case "RomFs":
                        return ArchiveFormatType.RomFs;
                    case "NintendoSubmissionPackage":
                        return ArchiveFormatType.NintendoSubmissionPackage;
                    case "NintendoSubmissionPackageForDelta":
                        return ArchiveFormatType.NintendoSubmissionPackageForDelta;
                    default:
                        throw new ArgumentException("invalid format .adf file. invalid \"formatType\"\n");
                }
            }
        }

        /// <summary>
        /// アーカイブ記述ファイルからコンテンツアーカイブを新規作成します。
        /// </summary>
        /// <param name="stream">アーカイブデータを出力するストリーム</param>
        /// <param name="adfPath">アーカイブ記述ファイルのパス</param>
        public static void CreateArchiveFromAdf(Stream stream, string metaType, string adfPath, AuthoringConfiguration config)
        {
            SourceSinkDriver sourceSinkDriver = new SourceSinkDriver();

            ArchiveFormatType type = GetArchiveType(adfPath);
            switch (type)
            {
                case ArchiveFormatType.PartitionFs:
                    {
                        PartitionFsAdfReader adfReader = new PartitionFsAdfReader(adfPath);
                        ISource source = new PartitionFsArchiveSource(adfReader.GetFileSystemInfo());
                        ISink sink = new StreamSink(stream);
                        sink.SetSize(source.Size);
                        sourceSinkDriver.Add(source, sink);
                        break;
                    }
                case ArchiveFormatType.NintendoContent:
                    {
                        if (string.IsNullOrEmpty(metaType))
                        {
                            throw new ArgumentException("meta type must not be null or empty.");
                        }

                        NintendoContentAdfReader adfReader = new NintendoContentAdfReader(adfPath);
                        ISource source = new NintendoContentArchiveSource(adfReader.GetFileSystemInfo(metaType), config.GetKeyConfiguration());
                        ISink sink = new StreamSink(stream);
                        sink.SetSize(source.Size);
                        sourceSinkDriver.Add(source, sink);
                        break;
                    }
                case ArchiveFormatType.RomFs:
                    {
                        RomFsAdfReader adfReader = new RomFsAdfReader(adfPath);
                        ISource source = new RomFsArchiveSource(adfReader.GetFileSystemInfo());
                        ISink sink = new StreamSink(stream);
                        sink.SetSize(source.Size);
                        sourceSinkDriver.Add(source, sink);
                        break;
                    }
                case ArchiveFormatType.NintendoSubmissionPackage:
                    {
                        NintendoSubmissionPackageAdfReader adfReader = new NintendoSubmissionPackageAdfReader(adfPath);
                        var keyConfig = config.GetKeyConfiguration();
                        IConnector connector = new NintendoSubmissionPackageArchive(new StreamSink(stream), adfReader.GetFileSystemInfo(keyConfig), keyConfig);
                        sourceSinkDriver.Add(connector);
                        break;
                    }
                default:
                    throw new NotImplementedException();
            }
            sourceSinkDriver.Run();
        }

        // TODO: NintendoSubmissionPackage として作れるようにする
        public static void CreateDeltaFromAdf(Stream stream, string adfPath, Dictionary<ulong, byte> patchContentMetaKeyGeneration, AuthoringConfiguration config, bool isProdEncryption)
        {
            SourceSinkDriver sourceSinkDriver = new SourceSinkDriver();

            ArchiveFormatType type = GetArchiveType(adfPath);
            DeltaFragmentContentGenerator generator = null;
            switch (type)
            {
                case ArchiveFormatType.NintendoSubmissionPackageForDelta:
                    {
                        NintendoSubmissionPackageAdfReader adfReader = new NintendoSubmissionPackageAdfReader(adfPath);
                        generator = new DeltaFragmentContentGenerator(config.GetKeyConfiguration(), patchContentMetaKeyGeneration, isProdEncryption);
                        IConnector connector = new NintendoSubmissionPackageArchive(new StreamSink(stream), adfReader.GetFileSystemInfoForDelta(generator), config.GetKeyConfiguration());
                        sourceSinkDriver.Add(connector);
                        break;
                    }
                default:
                    throw new NotImplementedException();
            }
            sourceSinkDriver.Run();
            if (generator != null)
            {
                generator.Dispose();
            }
        }

        /// <summary>
        /// 複数の NintendoSubmissionPackage に含まれるコンテンツを重複なく含む NintendoSubmissionPackage を作成します。
        /// </summary>
        /// <param name="outStream">アーカイブデータを出力するストリーム</param>
        /// <param name="inStreams">入力ストリームのリスト</param>
        public static void MergeNintendoSubmissionPackageArchive(Stream outStream, List<Stream> inStreams)
        {
            SourceSinkDriver sourceSinkDriver = new SourceSinkDriver();

            var readers = inStreams.Select(x => new NintendoSubmissionPackageReader(x)).ToList();
            var mergedNspInfo = ArchiveReconstructionUtils.GetMergedNspInfo(readers);

            ISource source = new PartitionFsArchiveSource(mergedNspInfo);
            ISink sink = new StreamSink(outStream);

            sink.SetSize(source.Size);
            sourceSinkDriver.Add(source, sink);

            sourceSinkDriver.Run();
        }

        /// <summary>
        /// NintendoSubmissionPackage から、コンテンツメタ ID、バージョン毎にコンテンツを抽出した NintendoSubmissionPackage を作成します。
        /// </summary>
        /// <param name="outStream">アーカイブデータを出力するストリーム</param>
        /// <param name="inStreams">入力ストリームのリスト</param>
        public static void SplitNintendoSubmissionPackageArchive(Stream outStream, Stream inStream, ulong contentMetaId, uint version)
        {
            SourceSinkDriver sourceSinkDriver = new SourceSinkDriver();

            var reader = new NintendoSubmissionPackageReader(inStream);
            var splitNspInfo = ArchiveReconstructionUtils.GetSplitNspInfo(reader, contentMetaId, version);

            ISource source = new PartitionFsArchiveSource(splitNspInfo);
            ISink sink = new StreamSink(outStream);

            sink.SetSize(source.Size);
            sourceSinkDriver.Add(source, sink);

            sourceSinkDriver.Run();
        }

        /// <summary>
        /// NintendoSubmissionPackage フォーマットのアーカイブの製品化処理を行います。
        /// </summary>
        /// <param name="outStream">アーカイブデータを出力するストリーム</param>
        /// <param name="outXmlStream">出力結果 XML を出力するストリーム</param>
        /// <param name="inStream">対象のアーカイブデータのストリーム</param>
        public static void ProdEncryptNintendoSubmissionPackageArchive(Stream outStream, Stream outXmlStream, Stream inStream, uint? requiredSystemVersionSubstitute, List<UInt64> excludeList, Dictionary<UInt64, byte> keyGenerationList, byte desiredSafeKeyGeneration, AuthoringConfiguration config)
        {
            ResultXmlGenerator xmlGen;
            try
            {
                SourceSinkDriver sourceSinkDriver = new SourceSinkDriver();
                IConnector connector = new ProdEncryptedNintendoSubmissionPackageArchive(new StreamSink(outStream), new NintendoSubmissionPackageReader(inStream), requiredSystemVersionSubstitute, excludeList, keyGenerationList, desiredSafeKeyGeneration, config.GetKeyConfiguration(), config.DebugConfig);
                sourceSinkDriver.Add(connector);
                sourceSinkDriver.Run();
            }
            catch (Exception ex)
            {
                xmlGen = new ResultXmlGenerator(ex.Message + Environment.NewLine + ex.StackTrace);
                xmlGen.WriteToStream(outXmlStream);
                throw;
            }

            xmlGen = new ResultXmlGenerator(outStream);
            xmlGen.WriteToStream(outXmlStream);
        }

        /// <summary>
        /// 製品化した NintendoSubmissionPackage アーカイブからコモンチケットを削除して出力します。
        /// </summary>
        /// <param name="outStream">出力用ストリーム</param>
        /// <param name="outXmlStream">出力結果 XML を出力するストリーム</param>
        /// <param name="inStream">対象のアーカイブのストリーム</param>
        /// <param name="inXmlStream">対象アーカイブ生成時の ResultXml のストリーム</param>
        public static void RemoveTicket(Stream outStream, Stream outXmlStream, Stream inStream, Stream inXmlStream)
        {
            ResultXmlGenerator xmlGen;
            try
            {
                SourceSinkDriver sourceSinkDriver = new SourceSinkDriver();

                var reader = new NintendoSubmissionPackageReader(inStream);
                var removedNspInfo = ArchiveReconstructionUtils.GetRootEntryRemovedNspInfo(reader, new List<string>() { @"[0-9a-fA-F]*\.tik" } );

                ISource source = new PartitionFsArchiveSource(removedNspInfo);
                ISink sink = new StreamSink(outStream);

                sink.SetSize(source.Size);
                sourceSinkDriver.Add(source, sink);

                sourceSinkDriver.Run();
            }
            catch (Exception ex)
            {
                xmlGen = new ResultXmlGenerator(ex.Message + Environment.NewLine + ex.StackTrace);
                xmlGen.WriteToStream(outXmlStream);
                throw;
            }

            if (inXmlStream != null)
            {
                ResultXmlInfo resultXmlInfo;
                {
                    var serializer = new XmlSerializer(typeof(ResultModel));
                    inXmlStream.Seek(0, SeekOrigin.Begin);
                    var model = (ResultModel)serializer.Deserialize(inXmlStream);
                    resultXmlInfo = model.GetInfo();
                }
                xmlGen = new ResultXmlGenerator(outStream, resultXmlInfo);
            }
            else
            {
                xmlGen = new ResultXmlGenerator(outStream);
            }

            xmlGen.WriteToStream(outXmlStream);
        }

        /// <summary>
        /// NintendoSubmissionPackage フォーマットのアーカイブから製品化処理済みカードイメージを作成します。
        /// </summary>
        /// <param name="outStream">出力用ストリーム</param>
        /// <param name="outXmlStream">出力結果 XML を出力するストリーム</param>
        /// <param name="inStream">対象のアーカイブデータのストリーム</param>
        /// <param name="inUppStream">同梱する製品化処理済みアップデートパーティションのアーカイブデータのストリーム</param>
        /// <param name="inPatchStream">同梱する製品化処理済みパッチのアーカイブデータのストリーム</param>
        public static void ProdEncryptNintendoSubmissionPackageArchiveForXci(Stream outStream, Stream outXmlStream, Stream inStream, Stream inUppStream, Stream inPatchStream, Stream inAocStream, byte launchFlags, bool noPadding, bool keepGeneration, AuthoringConfiguration config, Stream outDebugStream = null)
        {
            ResultXmlGenerator xmlGen;
            var resultXmlInfo = new ResultXmlInfo();
            List<IReadableSink> outContentMetaXmlSinks;

            try
            {
                SourceSinkDriver sourceSinkDriver = new SourceSinkDriver();
                var uppReader = inUppStream != null ? new NintendoSubmissionPackageReader(inUppStream) : null;
                var patchReader = inPatchStream != null ? new NintendoSubmissionPackageReader(inPatchStream) : null;
                var aocReader = inAocStream != null ? new NintendoSubmissionPackageReader(inAocStream) : null;
                IConnector connector = new ProdEncryptedXciArchive(new StreamSink(outStream), out outContentMetaXmlSinks, ref resultXmlInfo, new NintendoSubmissionPackageReader(inStream), uppReader, patchReader, aocReader, 0, 0, launchFlags, noPadding, keepGeneration, config.GetKeyConfiguration());
                sourceSinkDriver.Add(connector);
                sourceSinkDriver.Run();
            }
            catch (Exception ex)
            {
                xmlGen = new ResultXmlGenerator(ex.Message + Environment.NewLine + ex.StackTrace);
                xmlGen.WriteToStream(outXmlStream);
                throw;
            }

            GenerateXciResultXml(outXmlStream, outContentMetaXmlSinks, outDebugStream, outStream, resultXmlInfo, config);
        }

        /// <summary>
        /// NintendoSubmissionPackage フォーマットの複数のアーカイブから製品化処理済みカードイメージを作成します。
        /// </summary>
        /// <param name="outStream">出力用ストリーム</param>
        /// <param name="outXmlStream">出力結果 XML を出力するストリーム</param>
        /// <param name="indicatorStream">マルチアプリケーションカード情報を格納したアーカイブデータのストリーム</param>
        /// <param name="inStreams">対象のアーカイブデータのストリーム</param>
        /// <param name="inUppStream">同梱する製品化処理済みアップデートパーティションのアーカイブデータのストリーム</param>
        public static void ProdEncryptNintendoSubmissionPackageArchiveForXci(Stream outStream, Stream outXmlStream, Stream indicatorStream, List<Stream> inStreams, Stream inUppStream, byte launchFlags, bool noPadding, bool keepGeneration, AuthoringConfiguration config, Stream outDebugStream = null)
        {
            ResultXmlGenerator xmlGen;
            var resultXmlInfo = new ResultXmlInfo();
            List<IReadableSink> outContentMetaXmlSinks;

            try
            {
                SourceSinkDriver sourceSinkDriver = new SourceSinkDriver();
                var uppReader = inUppStream != null ? new NintendoSubmissionPackageReader(inUppStream) : null;
                List<NintendoSubmissionPackageReader> inNspReaders = null;
                List<NintendoSubmissionPackageReader> inPatchNspReaders = null;
                CardSpecModel cardSpec = null;
                {
                    // 整合性チェック & 並び替え
                    var nspReaders = inStreams.Select(x => new NintendoSubmissionPackageReader(x)).ToList();
                    var indicatorNspReader = new NintendoSubmissionPackageReader(indicatorStream);
                    var model = indicatorNspReader.ListFileInfo().Where(x => x.Item1 == "multiapplicationcard.xml").Select(x => ArchiveReconstructionUtils.ReadXml<MultiApplicationCardInfoModel>(indicatorNspReader, x.Item1, x.Item2)).First();
                    model.VerifyAndReorder(out inNspReaders, out inPatchNspReaders, nspReaders);
                    cardSpec = indicatorNspReader.ListFileInfo().Where(x => x.Item1 == "cardspec.xml").Select(x => ArchiveReconstructionUtils.ReadXml<CardSpecModel>(indicatorNspReader, x.Item1, x.Item2)).First();
                }

                var inputLaunchFlags = launchFlags;
                if (cardSpec.LaunchFlags.HasValue)
                {
                    inputLaunchFlags |= cardSpec.LaunchFlags.Value;
                }

                IConnector connector = new ProdEncryptedXciArchive(new StreamSink(outStream), out outContentMetaXmlSinks, ref resultXmlInfo, inNspReaders, uppReader, inPatchNspReaders, null, Convert.ToInt32(cardSpec.Size, 10), Convert.ToInt32(cardSpec.ClockRate, 10), inputLaunchFlags, noPadding, keepGeneration, config.GetKeyConfiguration());
                sourceSinkDriver.Add(connector);
                sourceSinkDriver.Run();
            }
            catch (Exception ex)
            {
                xmlGen = new ResultXmlGenerator(ex.Message + Environment.NewLine + ex.StackTrace);
                xmlGen.WriteToStream(outXmlStream);
                throw;
            }

            GenerateXciResultXml(outXmlStream, outContentMetaXmlSinks, outDebugStream, outStream, resultXmlInfo, config);
        }

        private static void GenerateXciResultXml(Stream outXmlStream, List<IReadableSink> outContentMetaXmlSinks, Stream outDebugStream, Stream xciStream, ResultXmlInfo resultXmlInfo, AuthoringConfiguration config)
        {
            // カードヘッダの読み込み
            {
                var xciHeaderOffset = (long)XciInfo.CardKeyAreaPageCount * XciInfo.PageSize;
                byte[] xciHeaderData;
                {
                    var xciHeaderStream = new SubStream(xciStream, xciHeaderOffset, (long)XciInfo.CardHeaderPageCount * XciInfo.PageSize);
                    xciHeaderData = new byte[xciHeaderStream.Length];
                    xciHeaderStream.Read(xciHeaderData, 0, (int)xciHeaderStream.Length);
                }
                resultXmlInfo.CardHeaderHash = XciUtils.GetCardHeaderHash(xciHeaderData);
                resultXmlInfo.CardHeader = CardHeaderModel.Create(XciUtils.DecryptHeader(xciHeaderData));
            }

            resultXmlInfo.ContentMetaList = new List<ContentMetaModel>();
            foreach (var sink in outContentMetaXmlSinks)
            {
                var data = sink.ToSource().PullData(0, (int)sink.Size).Buffer.Array;
                var model = ArchiveReconstructionUtils.ReadModel(data);
                resultXmlInfo.ContentMetaList.Add(model);
            }

            if (config.DebugConfig.EnableContentMetaBinaryExport && outDebugStream != null)
            {
                var sink = outContentMetaXmlSinks.Last();
                var data = sink.ToSource().PullData(0, (int)sink.Size).Buffer.Array;
                using (var ms = new MemoryStream(data.Length))
                {
                    ms.Write(data, 0, data.Length);
                    ms.Seek(0, SeekOrigin.Begin);
                    ms.CopyTo(outDebugStream);
                }
            }

            var xmlGen = new ResultXmlGenerator(xciStream, resultXmlInfo);
            xmlGen.WriteToStream(outXmlStream);
        }

        /// <summary>
        /// カードイメージから KeyArea 暗号化済みカードイメージを作成します。
        /// </summary>
        /// <param name="outStream">出力用ストリーム</param>
        /// <param name="outXmlStream">出力結果 XML を出力するストリーム</param>
        /// <param name="inStream">対象のカードイメージのストリーム</param>
        /// <param name="inXmlStream">対象のカードイメージ生成時の ResultXml のストリーム</param>
        public static void EncryptXci(Stream outStream, Stream outXmlStream, Stream inStream, Stream inXmlStream, AuthoringConfiguration config)
        {
            ResultXmlGenerator xmlGen;
            try
            {
                SourceSinkDriver sourceSinkDriver = new SourceSinkDriver();
                IConnector connector = new ProdEncryptedXciArchive(new StreamSink(outStream), new StreamSource(inStream, 0, inStream.Length), config.GetKeyConfiguration());
                sourceSinkDriver.Add(connector);
                sourceSinkDriver.Run();
            }
            catch (Exception ex)
            {
                xmlGen = new ResultXmlGenerator(ex.Message + Environment.NewLine + ex.StackTrace);
                xmlGen.WriteToStream(outXmlStream);
                throw;
            }

            if (inXmlStream != null)
            {
                // TORIAEZU: xci が同時に作られていることを前提に xci.result.xml から情報を取得する
                ResultXmlInfo resultXmlInfo;
                {
                    var serializer = new XmlSerializer(typeof(ResultModel));
                    inXmlStream.Seek(0, SeekOrigin.Begin);
                    var model = (ResultModel)serializer.Deserialize(inXmlStream);
                    resultXmlInfo = model.GetInfo();
                }
                xmlGen = new ResultXmlGenerator(outStream, resultXmlInfo);
                xmlGen.WriteToStream(outXmlStream);
            }
        }

        /// <summary>
        /// NintendoSubmissionPackage フォーマットのアーカイブの部分更新処理を行います。
        /// </summary>
        /// <param name="outStream">アーカイブデータを出力するストリーム</param>
        /// <param name="inStream">対象のアーカイブデータのストリーム</param>
        public static void ModifyNintendoSubmissionPackageArchive(Stream outStream, Stream inStream, IEnumerable<Stream> inEntryStreamList, IEnumerable<string> targetEntryPathList, string descFilePath, string nroDirectoryPath, AuthoringConfiguration config)
        {
            SourceSinkDriver sourceSinkDriver = new SourceSinkDriver();
            var inEntryStreamSourceList = new List<Stream>(inEntryStreamList).ConvertAll<StreamSource>(new Converter<Stream, StreamSource>(x => new StreamSource(x, 0, x.Length)));
            IConnector connector = new ModifiableNintendoSubmissionPackageArchive(
                new StreamSink(outStream), new NintendoSubmissionPackageReader(inStream),
                inEntryStreamSourceList, new List<string>(targetEntryPathList), descFilePath, nroDirectoryPath, config.GetKeyConfiguration());
            sourceSinkDriver.Add(connector);
            sourceSinkDriver.Run();
        }

        /// <summary>
        /// NintendoContentArchive フォーマットのアーカイブの部分更新処理を行います。
        /// </summary>
        /// <param name="outStream">アーカイブデータを出力するストリーム</param>
        /// <param name="inStream">対象のアーカイブデータのストリーム</param>
        public static void ModifyNintendoContentArchive(Stream outStream, Stream inStream, IEnumerable<Stream> inEntryStreamList, IEnumerable<string> targetEntryPathList, string descFilePath, string nroDirectoryPath, AuthoringConfiguration config, int alignmentType)
        {
            SourceSinkDriver sourceSinkDriver = new SourceSinkDriver();
            var inEntryStreamSourceList = new List<Stream>(inEntryStreamList).ConvertAll<StreamSource>(new Converter<Stream, StreamSource>(x => new StreamSource(x, 0, x.Length)));
            IConnector connector = new ModifiableNintendoContentArchive(
                new StreamSink(outStream),
                new NintendoContentArchiveReader(inStream, new NcaKeyGenerator(config.GetKeyConfiguration())),
                inEntryStreamSourceList,
                new List<string>(targetEntryPathList),
                descFilePath,
                config.GetKeyConfiguration(),
                alignmentType);
            sourceSinkDriver.Add(connector);
            sourceSinkDriver.Run();
        }

        /// <summary>
        /// NintendoContentArchive フォーマットのパッチ化を行います。
        /// </summary>
        /// <param name="outStream">アーカイブデータを出力するストリーム</param>
        /// <param name="originalStream">パッチ化する前のアーカイブデータのストリーム</param>
        /// <param name="currentStream">パッチ化したいアーカイブデータのストリーム</param>
        public static void MakeNcaPatch(Stream outStream, Stream originalStream, Stream currentStream, string descFilePath, AuthoringConfiguration config)
        {
            var keyGenerator = new NcaKeyGenerator(config.GetKeyConfiguration());
            var originalReader = new NintendoContentArchiveReader(originalStream, keyGenerator);
            var currentReader = new NintendoContentArchiveReader(currentStream, keyGenerator);

            if (originalReader.GetContentType() != currentReader.GetContentType())
            {
                throw new ArgumentException(
                    string.Format("content type dose not match. original:{0} current:{1}", originalReader.GetContentType(), currentReader.GetContentType()));
            }
            if (originalReader.GetProgramId() != currentReader.GetProgramId())
            {
                throw new ArgumentException(
                    string.Format("program id dose not match. original:{0:X16} current:{1:X16}", originalReader.GetProgramId(), currentReader.GetProgramId()));
            }

            SourceSinkDriver sourceSinkDriver = new SourceSinkDriver();
            IConnector connector = new PatchableNintendoContentArchive(
                new StreamSink(outStream), NintendoContentMetaConstant.ContentTypeProgram, originalReader, currentReader, descFilePath, config.GetKeyConfiguration());
            sourceSinkDriver.Add(connector);
            sourceSinkDriver.Run();
        }

        /// <summary>
        /// NintendoSubmissionPackageArchive フォーマットのパッチ化を行います。
        /// </summary>
        /// <param name="outStream">アーカイブデータを出力するストリーム</param>
        /// <param name="originalStream">パッチ化する前のアーカイブデータのストリーム</param>
        /// <param name="currentStream">パッチ化したいアーカイブデータのストリーム</param>
        /// <param name="previousStream">パッチ化した前バージョンのアーカイブデータのストリーム</param>
        public static void MakeNspPatch(Stream outStream, Stream originalStream, Stream currentStream, Stream previousStream, string metaFilePath, string descFilePath, AuthoringConfiguration config)
        {
            var originalReader = new NintendoSubmissionPackageReader(originalStream);
            var currentReader = new NintendoSubmissionPackageReader(currentStream);
            var previousReader = previousStream != null ? new NintendoSubmissionPackageReader(previousStream) : null;

            SourceSinkDriver sourceSinkDriver = new SourceSinkDriver();
            IConnector connector = new PatchableNintendoSubmissionPackageArchive(
                new StreamSink(outStream), originalReader, currentReader, previousReader, metaFilePath, descFilePath, config.GetKeyConfiguration());
            sourceSinkDriver.Add(connector);
            sourceSinkDriver.Run();
        }
        public static void MergeContentMetaForPatch(Stream output, Stream patch, Stream merging, AuthoringConfiguration config)
        {
            SourceSinkDriver sourceSinkDriver = new SourceSinkDriver();
            IConnector connector = new ModifiableNintendoSubmissionPackageArchive(
                    new StreamSink(output), new NintendoSubmissionPackageReader(patch), new NintendoSubmissionPackageReader(merging),
                    config.GetKeyConfiguration(), false);
            sourceSinkDriver.Add(connector);
            sourceSinkDriver.Run();
        }
        public static void MergeDeltaContentIntoPatch(Stream output, Stream patch, Stream merging, AuthoringConfiguration config)
        {
            SourceSinkDriver sourceSinkDriver = new SourceSinkDriver();
            IConnector connector = new ModifiableNintendoSubmissionPackageArchive(
                    new StreamSink(output), new NintendoSubmissionPackageReader(patch), new NintendoSubmissionPackageReader(merging),
                    config.GetKeyConfiguration(), true);
            sourceSinkDriver.Add(connector);
            sourceSinkDriver.Run();
        }

        /// <summary>
        /// NintendoSubmissionPackageArchive フォーマットのIS-Cパッチ化を行います。
        /// </summary>
        /// <param name="outStream">アーカイブデータを出力するストリーム</param>
        /// <param name="previousStream">前のバージョンのパッチデータのストリーム</param>
        /// <param name="currentStream">IS-Cパッチ化したいパッチデータのストリーム</param>
        public static void MakeNspIscPatch(Stream outStream, Stream previousStream, Stream currentStream, string descFilePath, AuthoringConfiguration config)
        {
            var previousReader = new NintendoSubmissionPackageReader(previousStream);
            var currentReader = new NintendoSubmissionPackageReader(currentStream);

            SourceSinkDriver sourceSinkDriver = new SourceSinkDriver();
            IConnector connector = new IscPatchableNintendoSubmissionPackageArchive(
                new StreamSink(outStream), previousReader, currentReader, descFilePath, config.GetKeyConfiguration());
            sourceSinkDriver.Add(connector);
            sourceSinkDriver.Run();
        }

        /// <summary>
        /// バイナリパッチ フォーマットのパッチ化を行います。
        /// </summary>
        /// <param name="outStream">アーカイブデータを出力するストリーム</param>
        /// <param name="destination">パッチ化したいアーカイブデータ</param>
        public static void MakeDelta(
            Stream outStream,
            ISource destination,
            DeltaSourceInfo sourceInfo,
            IEnumerable<DeltaCommand> deltaCommands)
        {
            SourceSinkDriver sourceSinkDriver = new SourceSinkDriver();
            IConnector connector = new DeltaArchive(
                new StreamSink(outStream),
                sourceInfo,
                destination,
                deltaCommands);
            sourceSinkDriver.Add(connector);
            sourceSinkDriver.Run();
        }

        /// <summary>
        /// パッチアーカイブの製品化処理を行います
        /// </summary>
        /// <param name="outStream">出力用ストリーム</param>
        /// <param name="outXmlStream">出力結果 XML を出力するストリーム</param>
        /// <param name="inPatchStream">対象パッチのアーカイブデータのストリーム</param>
        /// <param name="inOriginalStream">対象パッチの適用先の NintendoSubmissionPackage フォーマットのアーカイブデータのストリーム</param>
        /// <param name="inPreviousProdPatchStream">対象パッチの 1 世代前の製品化処理済みパッチのアーカイブデータのストリーム</param>
        public static void ProdEncryptPatch(Stream outStream, Stream outXmlStream, Stream inPatchStream, Stream inOriginalStream, Stream inPreviousProdPatchStream, uint? requiredSystemVersionSubstitute, byte desiredSafeKeyGeneration, long sizeThresholdForKeyGenerationUpdate, AuthoringConfiguration config)
        {
            ResultXmlGenerator xmlGen;
            try
            {
                SourceSinkDriver sourceSinkDriver = new SourceSinkDriver();
                IConnector connector = new ProdEncryptedPatchableNintendoSubmissionPackageArchive(new StreamSink(outStream), new NintendoSubmissionPackageReader(inPatchStream), new NintendoSubmissionPackageReader(inOriginalStream), inPreviousProdPatchStream != null ? new NintendoSubmissionPackageReader(inPreviousProdPatchStream) : null, requiredSystemVersionSubstitute, desiredSafeKeyGeneration, sizeThresholdForKeyGenerationUpdate, config.GetKeyConfiguration(), config.DebugConfig);
                sourceSinkDriver.Add(connector);
                sourceSinkDriver.Run();
            }
            catch (Exception ex)
            {
                xmlGen = new ResultXmlGenerator(ex.Message + Environment.NewLine + ex.StackTrace);
                xmlGen.WriteToStream(outXmlStream);
                throw;
            }

            xmlGen = new ResultXmlGenerator(outStream);
            xmlGen.WriteToStream(outXmlStream);
        }

        /// <summary>
        /// 製品化した パッチアーカイブとパッチ間差分アーカイブをマージします。
        /// </summary>
        /// <param name="outStream">出力用ストリーム</param>
        /// <param name="outXmlStream">出力結果 XML を出力するストリーム</param>
        /// <param name="inPatchStream">対象のパッチアーカイブのストリーム</param>
        /// <param name="inDeltaStream">対象のパッチ間差分アーカイブのストリーム</param>
        public static void MergeDeltaContentIntoPatch(Stream outStream, Stream outXmlStream, Stream inPatchStream, Stream inDeltaStream, AuthoringConfiguration config)
        {
            ResultXmlGenerator xmlGen;
            try
            {
                SourceSinkDriver sourceSinkDriver = new SourceSinkDriver();
                IConnector connector = new ProdEncryptedPatchableNintendoSubmissionPackageArchive(new StreamSink(outStream), new NintendoSubmissionPackageReader(inPatchStream), new NintendoSubmissionPackageReader(inDeltaStream), config.GetKeyConfiguration());
                sourceSinkDriver.Add(connector);
                sourceSinkDriver.Run();
            }
            catch (Exception ex)
            {
                xmlGen = new ResultXmlGenerator(ex.Message + Environment.NewLine + ex.StackTrace);
                xmlGen.WriteToStream(outXmlStream);
                throw;
            }

            xmlGen = new ResultXmlGenerator(outStream);
            xmlGen.WriteToStream(outXmlStream);
        }

        /// <summary>
        /// NintendoSubmissionPackageArchive フォーマットに含まれるXMLモデルをコンソールに出力します。
        /// </summary>
        public static void GetAllXmlModels()
        {
            List<Type> types = new List<Type>()
            {
                typeof(ApplicationControlPropertyModel),
                typeof(AddOnContentModel),
                typeof(AuthoringToolInfoModel),
                typeof(CardSpecModel),
                typeof(ContentMetaModel),
                typeof(HtmlDocumentXmlModel),
                typeof(LegalInformationModel),
                typeof(MultiApplicationCardInfoModel),
                typeof(OnCardAddOnContentInfoModel),
                typeof(ProgramInfoModel),
            };
            var sw = new StreamWriter(Console.OpenStandardOutput());

            XmlModelWriter.XmlModelWrite(sw, types);
        }
        /// <summary>
        /// NintendoSubmissionPackageArchive フォーマットのアーカイブから各種情報を取り出します。
        /// </summary>
        /// <param name="outInfo">結果の出力するストリーム</param>
        /// <param name="Stream">対象のアーカイブデータのストリーム</param>
        public static void GetContentMetaProperty(out NintendoContentMetaPropertyList outPropertyList, Stream inStream, AuthoringConfiguration config)
        {
            var nspReader = new NintendoSubmissionPackageReader(inStream);
            outPropertyList = NintendoContentMetaProperty.GetList(nspReader, config.GetKeyConfiguration());
        }
        /// <summary>
        /// NintendoSubmissionPackageArchive フォーマットのアーカイブからUnpublishableError情報を取り出します。
        /// </summary>
        /// <param name="outUnpublishableError">結果を出力するXMLモデル</param>
        /// <param name="inStream">対象のアーカイブデータのストリーム</param>
        /// <param name="isSpecifiedOption">--error-unpublishable が指定されているかどうかのフラグ</param>
        /// <param name="ignoreListFilePath">無視するチェック項目を記載したファイルパス</param>
        /// <param name="config">オーサリング処理全体に関わる設定</param>
        public static void GetUnpublishableError(out UnpublishableErrorModel outUnpublishableError, Stream inStream, bool isSpecifiedOption, string ignoreListFilePath, AuthoringConfiguration config)
        {
            // 無視するチェック項目をファイルから取得する
            var ignoreList = UnpublishableErrorCheck.ReadCheckList(ignoreListFilePath);

            using (var nspReader = new NintendoSubmissionPackageReader(inStream))
            {
                var propertyList = NintendoContentMetaProperty.GetListImpl(nspReader, config.GetKeyConfiguration());
                var nspInfo = ArchiveReconstructionUtils.GetNspInfo(nspReader, config.GetKeyConfiguration());

                var addonContentMetaList = new List<AddOnContentContentMetaModel>() { };
                outUnpublishableError = null;
                List<UnpublishableErrorCheckData> checkDataList = new List<UnpublishableErrorCheckData>();
                List<string> middlewareNameList = new List<string>();

                foreach (var entry in nspInfo.Entries)
                {
                    foreach (var i in entry.Contents.Select(x => x.IdOffset).Distinct().Select((v, i) => new { Value = v, IdOffset = i }))
                    {
                        UnpublishableErrorCheckData checkData = new UnpublishableErrorCheckData();
                        checkData.CheckTarget = UnpublishableErrorCheckData.CheckTargetType.Nsp;
                        checkData.Nsp.ContentMeta = entry.ContentMetaInfo.Model;

                        if (checkData.Nsp.ContentMeta.Type != NintendoContentMetaConstant.ContentMetaTypeApplication &&
                            checkData.Nsp.ContentMeta.Type != NintendoContentMetaConstant.ContentMetaTypePatch &&
                            checkData.Nsp.ContentMeta.Type != NintendoContentMetaConstant.ContentMetaTypeAddOnContent)
                        {
                            // アプリ・パッチ・AOCのいずれでもない場合はスキップ
                            continue;
                        }

                        {
                            var singlePropertyList = new NintendoContentMetaPropertyList();
                            singlePropertyList.Properties = new NintendoContentMetaProperty[1];
                            singlePropertyList.Properties[0] = propertyList.Where(x => x.Id == checkData.Nsp.ContentMeta.Id).Single();
                            checkData.Nsp.ContentMetaPropertyList = singlePropertyList;
                        }

                        checkData.CheckContentType = checkData.Nsp.ContentMeta.Type;

                        if (checkData.Nsp.ContentMeta.Type != NintendoContentMetaConstant.ContentMetaTypeAddOnContent) // Aocの場合はprogramInfoは存在しない。
                        {
                            // ProgramInfoを取得する
                            checkData.Nsp.ProgramInfo = entry.ProgramInfo?.Models.Skip(i.IdOffset).FirstOrDefault();
                            if (checkData.Nsp.ProgramInfo != null)
                            {
                                // nsp 内の全てのコンテンツの MiddlewareList をためる
                                middlewareNameList.AddRange(checkData.Nsp.ProgramInfo.MiddlewareListData.GetModuleNameList());
                            }

                            // リーガル情報を取得する(コンテンツが複数ある場合でもリーガル情報は1つのみなので先頭のものを使用する)
                            checkData.Nsp.LegalInformation = entry.LegalInformationInfo?.Models.FirstOrDefault();
                            if (checkData.Nsp.LegalInformation != null)
                            {
                                // UnpublishableErrorのチェックに必要なリーガル情報内のファイルを取得する
                                var legalInformationNca = entry.Contents.FindAll(x => x.ContentType == NintendoContentMetaConstant.ContentTypeLegalInformation)?.FirstOrDefault();
                                checkData.Nsp.LegalInformationFileList = UnpublishableErrorCheck.GetLegalInformationFileListForUnpublishableErrorCheck(legalInformationNca, new NcaKeyGenerator(config.GetKeyConfiguration()));
                            }

                            // アプリケーション管理データを取得する
                            checkData.ApplicationControlProperty = entry.ApplicationControlPropertyInfo?.Models.Skip(i.IdOffset).FirstOrDefault();
                            if (checkData.ApplicationControlProperty == null)
                            {
                                // アプリケーション管理データが存在しない場合はエラー
                                throw new ArgumentException("Application control property is not included. Please add application control property before submitting nsp to nintendo.");
                            }

                            // CardSpecXMLが存在する場合は取得する
                            checkData.CardSpec = CardSpecModel.Create(nspInfo.CardSize, nspInfo.CardClockRate, nspInfo.AutoSetSize, nspInfo.AutoSetClockRate, nspInfo.CardLaunchFlags);

                            // HtmlDocumentXMLを取得する
                            checkData.Nsp.HtmlDocumentXml = entry.HtmlDocumentInfo?.Models.Skip(i.IdOffset).FirstOrDefault();

                            // Rawアイコンリストを取得する
                            checkData.Nsp.RawIconList = entry.ExtraData.Where(x => (x.IdOffset == i.Value) && Regex.IsMatch(x.EntryName, @"[^\.]+\.raw\.[^\.]+\.jpg$")).Select(x => Tuple.Create(x.EntryName, x.PullData(0, (int)x.Size).Buffer.Array)).ToList();

                            // Nxアイコンリストを取得する
                            checkData.Nsp.NxIconList = entry.ExtraData.Where(x => (x.IdOffset == i.Value) && Regex.IsMatch(x.EntryName, @"[^\.]+\.nx\.[^\.]+\.jpg$")).Select(x => Tuple.Create(x.EntryName, x.PullData(0, (int)x.Size).Buffer.Array)).ToList();
                        }
                        else  // Aocの場合
                        {
                            // 複数のAocコンテンツの相互チェック用にコンテンツメタをためる
                            addonContentMetaList.Add(checkData.Nsp.ContentMeta as AddOnContentContentMetaModel);
                        }
                        checkDataList.Add(checkData);
                    }
                }

                if (nspInfo.MultiApplicationCardInfo != null)
                {
                    // マルチアプリケーションカードのチェック
                    UnpublishableErrorCheckData checkData = new UnpublishableErrorCheckData();
                    checkData.CheckTarget = UnpublishableErrorCheckData.CheckTargetType.Nsp;
                    checkData.CheckContentType = UnpublishableErrorCheckData.CheckContentTypeMultiApplicationCard;
                    checkData.Nsp.MultiApplicationCardInfo = nspInfo.MultiApplicationCardInfo;
                    checkDataList.Add(checkData);
                }

                // 重複を削除
                middlewareNameList = middlewareNameList.Distinct().ToList();

                // ProgramIndex = 0 の ApplicationControlProperty を取得（存在しない場合は null が取得される）
                var applicationControlPropertyOfProgramIndex0 = checkDataList.Where(x => x.ApplicationControlProperty?.ProgramIndex != null && Convert.ToInt32(x.ApplicationControlProperty.ProgramIndex) == 0)?.FirstOrDefault()?.ApplicationControlProperty;

                foreach (var checkData in checkDataList)
                {
                    checkData.Nsp.AddonContentMetaList = addonContentMetaList;
                    checkData.Nsp.AllMiddlewareListInNsp = middlewareNameList;
                    checkData.Nsp.ApplicationControlPropertyOfProgramIndex0 = applicationControlPropertyOfProgramIndex0;

                    UnpublishableErrorModel errorModel = null;
                    if (!isSpecifiedOption)
                    {
                        checkData.CheckInfo.CheckList = UnpublishableErrorCheck.NintendoSubmissionPackageMandatoryCheckList;
                    }
                    checkData.CheckInfo.IgnoreList = ignoreList;
                    errorModel = UnpublishableError.GetList(checkData);

                    if (outUnpublishableError == null)
                    {
                        outUnpublishableError = errorModel;
                    }
                    else
                    {
                        outUnpublishableError.Contents.AddRange(errorModel.Contents);
                    }
                }
            }
        }

        /// <summary>
        /// 追加コンテンツアーカイブをカード搭載用に変換します。
        /// </summary>
        /// <param name="outStream">出力用ストリーム</param>
        /// <param name="inStreams">対象の追加コンテンツアーカイブのストリームリスト</param>
        /// <param name="inBaseStream">対象の追加コンテンツとともにカードに搭載するパッチ（パッチがなければアプリ）</param>
        public static void ConvertAddOnContentForCard(Stream outStream, List<Stream> inStreams, Stream inBaseStream, int? cardSize, int? cardClockRate, byte? cardLaunchFlags, AuthoringConfiguration config)
        {
            SourceSinkDriver sourceSinkDriver = new SourceSinkDriver();

            var readers = inStreams.Select(x => new NintendoSubmissionPackageReader(x)).ToList();
            var baseReader = new NintendoSubmissionPackageReader(inBaseStream);
            IConnector connector = new OnCardAddOnContentNintendoSubmissionPackageArchive(new StreamSink(outStream), readers, baseReader, cardSize, cardClockRate, cardLaunchFlags, config.GetKeyConfiguration());
            sourceSinkDriver.Add(connector);

            sourceSinkDriver.Run();
        }

        /// <summary>
        /// マルチプログラムアプリケーションのアーカイブを作成します。
        /// </summary>
        /// <param name="outStream">出力用ストリーム</param>
        /// <param name="inStreams">アプリケーションアーカイブのストリームリスト</param>
        public static void CreateMultiProgramApplication(Stream outStream, List<Stream> inStreams, int? cardSize, int? cardClockRate, byte? cardLaunchFlags, AuthoringConfiguration config)
        {
            SourceSinkDriver sourceSinkDriver = new SourceSinkDriver();

            var readers = inStreams.Select(x => new NintendoSubmissionPackageReader(x)).ToList();
            IConnector connector = new MultiProgramApplicationNintendoSubmissionPackageArchive(new StreamSink(outStream), readers, cardSize, cardClockRate, cardLaunchFlags, config.GetKeyConfiguration());
            sourceSinkDriver.Add(connector);

            sourceSinkDriver.Run();
        }

        /// <summary>
        /// マルチアプリケーションカード情報を格納したアーカイブを作成します。
        /// </summary>
        /// <param name="outStream">出力用ストリーム</param>
        /// <param name="inStreams">アプリケーションアーカイブのストリームリスト</param>
        public static void CreateMultiApplicationCardIndicator(Stream outStream, List<Stream> inStreams, ulong id, int? cardSize, int? cardClockRate, byte? cardLaunchFlags, AuthoringConfiguration config)
        {
            SourceSinkDriver sourceSinkDriver = new SourceSinkDriver();

            var readers = inStreams.Select(x => new NintendoSubmissionPackageReader(x)).ToList();
            IConnector connector = new MultiApplicationCardIndicatorNintendoSubmissionPackageArchive(new StreamSink(outStream), readers, id, cardSize, cardClockRate, cardLaunchFlags, config.GetKeyConfiguration());
            sourceSinkDriver.Add(connector);
            sourceSinkDriver.Run();
        }

        /// <summary>
        /// NintendoSubmissionPackageArchive フォーマットのスパース化を行います。
        /// </summary>
        /// <param name="outOriginalStream">アーカイブデータを出力するストリーム</param>
        /// <param name="outPatchStream">パッチ側のアーカイブデータを出力するストリーム</param>
        /// <param name="originalStream">スパース化するアーカイブデータのストリーム</param>
        /// <param name="patchStream">スパース化のヒントに使用するアーカイブデータのストリーム</param>
        /// <param name="previousStream">スパース済みパッチのアーカイブデータのストリーム</param>
        public static void SparsifyNsp(Stream outOriginalStream, Stream outPatchStream, Stream originalStream, Stream patchStream, Stream previousStream, AuthoringConfiguration config)
        {
            using (var patchReader = new NintendoSubmissionPackageReader(patchStream))
            {
                Dictionary<string, NintendoContentArchiveReader> sparseInfos = new Dictionary<string, NintendoContentArchiveReader>();

                // スパース済みオリジナルファイルの出力
                using (var originalReader = new NintendoSubmissionPackageReader(originalStream))
                using (var previousReader = previousStream != null ? new NintendoSubmissionPackageReader(previousStream) : null)
                {
                    SourceSinkDriver sourceSinkDriver = new SourceSinkDriver();
                    IConnector connector = new SparseOriginalNintendoSubmissionPackageArchive(
                        new StreamSink(outOriginalStream), originalReader, patchReader, previousReader, config.GetKeyConfiguration(), sparseInfos);
                    sourceSinkDriver.Add(connector);
                    sourceSinkDriver.Run();
                }

                outOriginalStream.Flush();
                outOriginalStream.Position = 0;

                // スパース情報付きパッチファイルの出力
                using (var sparseReader = new NintendoSubmissionPackageReader(outOriginalStream))
                {
                    SourceSinkDriver sourceSinkDriver = new SourceSinkDriver();
                    IConnector connector = new SparsePatchNintendoSubmissionPackageArchive(
                        new StreamSink(outPatchStream), sparseReader, patchReader, config.GetKeyConfiguration(), sparseInfos);
                    sourceSinkDriver.Add(connector);
                    sourceSinkDriver.Run();
                }
            }
        }

        /// <summary>
        /// Nsp 内のアプリケーション管理データ変更の為の準備を行います。
        /// </summary>
        /// <param name="outReplaceRuleFile">replace コマンド用のルールファイルパスの出力用</param>
        /// <param name="workDirectoryPath">作業用ディレクトリ</param>
        /// <param name="originalNspPath">変更前のnspファイルパス</param>
        /// <param name="newNacpIncludedNspPath">管理情報抽出用のnspファイルパス</param>
        public static void PrepareNspReplaceApplicationControlProperty(out string outReplaceRuleFile, string workDirectoryPath, string originalNspPath, string newNacpIncludedNspPath, AuthoringConfiguration config)
        {
            // replace ルール作成の為の情報 <元のnsp内のパス, 置換するファイル(PC上のパス)>
            var nacpFile = new Tuple<string, string>(null, null);
            var cnmtFile = new Tuple<string, string>(null, null);
            var legalInformationFile = new Tuple<string, string>(null, null);
            var htmlDocumentFile = new Tuple<string, string>(null, null);
            var rootIconFiles = new Tuple<List<string>, List<string>>(new List<string>(), new List<string>());

            uint newVersion = 0;
            uint oldVersion = 0;

            // 元の nsp の置換対象の情報を取得
            using (var nspReader = new NintendoSubmissionPackageReader(originalNspPath))
            {
                foreach (var nspInfo in nspReader.ListFileInfo().Where(x => x.Item1.EndsWith(".xml")))
                {
                    var fileName = nspInfo.Item1;
                    if (fileName.EndsWith(".nacp.xml"))
                    {
                        var hash = fileName.Replace(".nacp.xml", string.Empty);
                        nacpFile = Tuple.Create(hash + ".nca", nacpFile.Item2);
                    }
                    else if (fileName.EndsWith(".cnmt.xml"))
                    {
                        // .cnmt.nca は直接 nca の置換ができないので Application_*.cnmt を置換
                        // Application_*.cnmtに関しては元のnspのリリースバージョンの変更のみ対応するので、元ファイルを出力しておく
                        var hash = fileName.Replace(".cnmt.xml", string.Empty);
                        var ncaReader = nspReader.OpenNintendoContentArchiveReader(hash + ".cnmt.nca", new NcaKeyGenerator(config.GetKeyConfiguration()));
                        foreach (var ncaInfo in ncaReader.ListFsInfo())
                        {
                            var fsReader = ncaReader.OpenFileSystemArchiveReader(ncaInfo.Item1);
                            var file = fsReader.ListFileInfo().Where(x => x.Item1.EndsWith(".cnmt")).FirstOrDefault();
                            if (file != null)
                            {
                                var reader = new NintendoContentMetaReader(fsReader.ReadFile(file.Item1, 0, file.Item2));
                                oldVersion = reader.GetVersion();
                                var localPath = workDirectoryPath + "/" + file.Item1;
                                using (var sw = new StreamWriter(localPath, false))
                                {
                                    var data = fsReader.ReadFile(file.Item1, 0, file.Item2);
                                    sw.BaseStream.Write(data, 0, data.Length);
                                }
                                cnmtFile = Tuple.Create(hash + ".cnmt.nca/fs0/" + file.Item1, localPath);
                                break;
                            }
                        }
                    }
                    else if (fileName.EndsWith(".htmldocument.xml"))
                    {
                        var hash = fileName.Replace(".htmldocument.xml", string.Empty);
                        htmlDocumentFile = Tuple.Create(hash + ".nca", htmlDocumentFile.Item2);
                    }
                    else if (fileName.EndsWith(".legalinfo.xml"))
                    {
                        var hash = fileName.Replace(".legalinfo.xml", string.Empty);
                        legalInformationFile = Tuple.Create(hash + ".nca", legalInformationFile.Item2);
                    }
                }
                // nsp 直下のiconファイルを取得
                var files = nspReader.ListFileInfo().Where(x => x.Item1.EndsWith(".jpg")).Select(x => x.Item1).ToList();
                rootIconFiles.Item1.AddRange(files);
            }

            // 作成した nsp からデータを抽出
            {
                Action<NintendoSubmissionPackageReader, string, string> dumpFile = (nspReader, fileName, outputPath) =>
                {
                    var fileSize = nspReader.ListFileInfo().Where(x => x.Item1 == fileName).Single().Item2;
                    var data = nspReader.ReadFile(fileName, 0, fileSize);
                    using (var sw = new StreamWriter(outputPath, false))
                    {
                        sw.BaseStream.Write(data, 0, data.Length);
                    }
                };

                using (var nspReader = new NintendoSubmissionPackageReader(newNacpIncludedNspPath))
                {
                    foreach (var nspInfo in nspReader.ListFileInfo().Where(x => x.Item1.EndsWith(".xml") || x.Item1.EndsWith(".jpg")))
                    {
                        var fileName = nspInfo.Item1;

                        if (fileName.EndsWith(".nacp.xml"))
                        {
                            var hash = fileName.Replace(".nacp.xml", string.Empty);
                            var localPath = workDirectoryPath + "/" + hash + ".nca";
                            dumpFile(nspReader, hash + ".nca", localPath);
                            nacpFile = Tuple.Create(nacpFile.Item1, localPath);
                        }
                        else if (fileName.EndsWith(".cnmt.xml"))
                        {
                            // .cnmt は元の nsp 内の .cnmt のリリースバージョンだけ変更するのでリリースバージョンだけ取得しておく
                            var hash = fileName.Replace(".cnmt.xml", string.Empty);
                            var ncaReader = nspReader.OpenNintendoContentArchiveReader(hash + ".cnmt.nca", new NcaKeyGenerator(config.GetKeyConfiguration()));
                            foreach (var ncaInfo in ncaReader.ListFsInfo())
                            {
                                var fsReader = ncaReader.OpenFileSystemArchiveReader(ncaInfo.Item1);
                                var file = fsReader.ListFileInfo().Where(x => x.Item1.EndsWith(".cnmt")).FirstOrDefault();
                                if (file != null)
                                {
                                    var reader = new NintendoContentMetaReader(fsReader.ReadFile(file.Item1, 0, file.Item2));
                                    newVersion = reader.GetVersion();
                                    break;
                                }
                            }
                        }
                        else if (fileName.EndsWith(".htmldocument.xml"))
                        {
                            var hash = fileName.Replace(".htmldocument.xml", string.Empty);
                            var localPath = workDirectoryPath + "/" + hash + ".nca";
                            dumpFile(nspReader, hash + ".nca", localPath);
                            htmlDocumentFile = Tuple.Create(htmlDocumentFile.Item1, localPath);
                        }
                        else if (fileName.EndsWith(".legalinfo.xml"))
                        {
                            var hash = fileName.Replace(".legalinfo.xml", string.Empty);
                            var localPath = workDirectoryPath + "/" + hash + ".nca";
                            dumpFile(nspReader, hash + ".nca", localPath);
                            legalInformationFile = Tuple.Create(legalInformationFile.Item1, localPath);
                        }
                        else if (fileName.EndsWith(".jpg"))
                        {
                            var localPath = workDirectoryPath + "/" + fileName;
                            dumpFile(nspReader, fileName, localPath);
                            rootIconFiles.Item2.Add(localPath);
                        }
                    }
                }
            }

            // replace 用ルール作成
            outReplaceRuleFile = workDirectoryPath + "/replace_files.txt";
            {
                using (var sw = new StreamWriter(outReplaceRuleFile, false, Encoding.UTF8))
                {
                    // nacp 置換
                    sw.WriteLine($"{nacpFile.Item1}\t{nacpFile.Item2}");

                    // cnmt 置換
                    // リリースバージョンを変更
                    if (newVersion != oldVersion)
                    {
                        using (var fs = new FileStream(cnmtFile.Item2, FileMode.Open, FileAccess.ReadWrite, FileShare.ReadWrite))
                        using (var writer = new BinaryWriter(fs))
                        {
                            var releaseVersion = (oldVersion & 0xffff) | (newVersion & 0xffff0000);

                            writer.Seek(sizeof(ulong), SeekOrigin.Begin);
                            writer.Write(BitConverter.GetBytes(releaseVersion));
                        }
                        sw.WriteLine($"{cnmtFile.Item1}\t{cnmtFile.Item2}");
                    }

                    // HtmlDocumentFile
                    if (htmlDocumentFile.Item1 != null && htmlDocumentFile.Item2 != null)
                    {
                        // 置換
                        sw.WriteLine($"{htmlDocumentFile.Item1}\t{htmlDocumentFile.Item2}");
                    }
                    else if (htmlDocumentFile.Item1 != null && htmlDocumentFile.Item2 == null)
                    {
                        // 削除
                        sw.WriteLine($"del:{htmlDocumentFile.Item1}\tnull");
                    }
                    else if (htmlDocumentFile.Item1 == null && htmlDocumentFile.Item2 != null)
                    {
                        // 追加
                        sw.WriteLine($"null.nca@{NintendoContentMetaConstant.ContentTypeHtmlDocument}\t{htmlDocumentFile.Item2}");
                    }

                    // LegalInformationFile
                    if (legalInformationFile.Item1 != null && legalInformationFile.Item2 != null)
                    {
                        // 置換
                        sw.WriteLine($"{legalInformationFile.Item1}\t{legalInformationFile.Item2}");
                    }
                    else if (legalInformationFile.Item1 != null && legalInformationFile.Item2 == null)
                    {
                        // 削除
                        sw.WriteLine($"del:{legalInformationFile.Item1}\tnull");
                    }
                    else if (legalInformationFile.Item1 == null && legalInformationFile.Item2 != null)
                    {
                        // 追加
                        sw.WriteLine($"null.nca@{NintendoContentMetaConstant.ContentTypeLegalInformation}\t{legalInformationFile.Item2}");
                    }

                    // root 直下の Icon
                    // 古いのは一旦削除
                    foreach (var icon in rootIconFiles.Item1)
                    {
                        sw.WriteLine($"del:{icon}\tnull");
                    }
                    // 新しい Icon を追加
                    foreach (var icon in rootIconFiles.Item2)
                    {
                        var pattern = Regex.Match(icon, @"[^\.]+\.(raw|nx)\.([^\.]+)\.jpg$");
                        var type = pattern.Groups[1].Value;
                        var language = pattern.Groups[2].Value;
                        sw.WriteLine($"null.icon@{language}@{type}\t{icon}");
                    }
                }
            }
        }
    }
}
