﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Linq;
using Nintendo.Authoring.FileSystemMetaLibrary;

namespace Nintendo.Authoring.AuthoringLibrary
{
    public class SinkStatus
    {
        /// <summary>
        /// ISink のデータのうち、一度でも書き込みが行われた領域を表すリストです。
        /// </summary>
        public RangeList FilledRangeList { get; set; }
        public bool IsFilled { get; set; }
        public SinkStatus()
        {
            FilledRangeList = new RangeList();
            IsFilled = false;
        }
    }

    public class SinkUtil
    {
        public static bool CheckIsFilled(RangeList rangeList, long size)
        {
            if (rangeList.Count != 0)
            {
                if (rangeList.Count == 1 &&
                    rangeList.First().Offset == 0 &&
                    rangeList.First().Size == size)
                {
                    return true;
                }
            }
            return false;
        }
        public static int GetWritableSize(long sinkSize, long offset, int sizeToWrite)
        {
            if (offset < 0)
            {
                return 0;
            }
            if (offset >= sinkSize)
            {
                return 0;
            }
            return (int)Math.Min(sizeToWrite, sinkSize - offset);
        }
    }

    public class FileSink : IReadableSink
    {
        public long Size { get; private set; }

        private string m_filePath;
        private long m_offset;
        private SinkStatus m_status;

        public FileSink(string filePath, long offset, long size)
        {
            m_filePath = filePath;
            m_offset = offset;
            Size = size;
            m_status = new SinkStatus();
        }
        public int PushData(ByteData data, long offset)
        {
            int writeSize = SinkUtil.GetWritableSize(Size, offset, data.Buffer.Count);
            if (writeSize == 0)
            {
                return 0;
            }

            using (Stream stream = new SubStream(new FileStream(m_filePath, FileMode.Open, FileAccess.ReadWrite, FileShare.None, 4096, FileOptions.SequentialScan), m_offset, this.Size))
            {
                stream.Seek(offset, SeekOrigin.Begin);
                stream.Write(data.Buffer.Array, data.Buffer.Offset, writeSize);
            }
            m_status.FilledRangeList.MergingAdd(new Range(offset, writeSize));
            m_status.IsFilled = SinkUtil.CheckIsFilled(m_status.FilledRangeList, Size);
            return writeSize;
        }
        public SinkStatus QueryStatus()
        {
            return m_status;
        }
        public void SetSize(long size)
        {
            throw new NotSupportedException();
        }
        public ISource ToSource()
        {
            FileInfo fi = new FileInfo(m_filePath);
            return new FileSource(m_filePath, m_offset, Size);
        }
    }

    public class MemorySink : IReadableSink
    {
        public long Size { get; private set; }
        private byte[] m_buffer;
        private SinkStatus m_status;

        public MemorySink(int size)
        {
            m_buffer = new byte[size];
            Size = size;
            m_status = new SinkStatus();
        }
        public int PushData(ByteData data, long offset)
        {
            int writeSize = SinkUtil.GetWritableSize(Size, offset, data.Buffer.Count);
            if (writeSize == 0)
            {
                return 0;
            }

            Buffer.BlockCopy(data.Buffer.Array, data.Buffer.Offset, m_buffer, (int)offset, writeSize);
            m_status.FilledRangeList.MergingAdd(new Range(offset, writeSize));
            m_status.IsFilled = SinkUtil.CheckIsFilled(m_status.FilledRangeList, Size);
            return writeSize;
        }
        public SinkStatus QueryStatus()
        {
            return m_status;
        }
        public void SetSize(long size)
        {
            throw new NotSupportedException();
        }
        public ISource ToSource()
        {
            return new UpdatableMemorySource(m_buffer, 0, (int)Size);
        }
    }

    public class SourceThroughSink : IReadableSink
    {
        public long Size { get; private set; }

        private ISource m_source;

        public SourceThroughSink(ISource source)
        {
            m_source = source;
            Size = source.Size;
        }
        public int PushData(ByteData data, long offset)
        {
            return data.Buffer.Count;
        }
        public SinkStatus QueryStatus()
        {
            SinkStatus sinkStatus = new SinkStatus();
            foreach (var range in m_source.QueryStatus().AvailableRangeList)
            {
                sinkStatus.FilledRangeList.MergingAdd(range);
            }
            sinkStatus.IsFilled = SinkUtil.CheckIsFilled(sinkStatus.FilledRangeList, Size);
            return sinkStatus;
        }
        public void SetSize(long size)
        {
            if (size != m_source.Size)
            {
                throw new ArgumentException();
            }
        }
        public ISource ToSource()
        {
            return m_source;
        }
    }

    public class StreamSink : IReadableSink
    {
        public long Size { get; private set; }
        private Stream m_stream;
        private SinkStatus m_status;

        public StreamSink(Stream stream)
        {
            m_stream = stream;
            Size = 0; // コンストラクト時にはサイズは未定とする
            m_status = new SinkStatus();
        }
        public int PushData(ByteData data, long offset)
        {
            if (Size == 0)
            {
                throw new InvalidOperationException();
            }
            int writeSize = SinkUtil.GetWritableSize(Size, offset, data.Buffer.Count);
            if (writeSize == 0)
            {
                return 0;
            }
            m_stream.Seek(offset, SeekOrigin.Begin);
            m_stream.Write(data.Buffer.Array, data.Buffer.Offset, writeSize);
            m_status.FilledRangeList.MergingAdd(new Range(offset, writeSize));
            m_status.IsFilled = SinkUtil.CheckIsFilled(m_status.FilledRangeList, Size);
            return writeSize;
        }
        public SinkStatus QueryStatus()
        {
            return m_status;
        }
        public void SetSize(long size)
        {
            Size = size;
        }
        public ISource ToSource()
        {
            return new StreamSource(m_stream, 0, Size);
        }
    }

    public class SubSink : ISink
    {
        public long Size { get; private set; }
        private ISink m_sink;
        private long m_offset;
        private SinkStatus m_status;

        public SubSink(ISink sink, long offset, long size)
        {
            // ここでは offset, size のチェックはしない
            m_sink = sink;
            m_offset = offset;
            Size = size;
            m_status = new SinkStatus();
        }
        public int PushData(ByteData data, long offset)
        {
            int pushSize = data.Buffer.Count;
            if (offset + pushSize > Size)
            {
                int restSize = (int)(Size - offset);
                pushSize = restSize > 0 ? restSize : 0;
            }
            ByteData tmpData = new ByteData(new ArraySegment<byte>(data.Buffer.Array, data.Buffer.Offset, pushSize));
            int pushed =  m_sink.PushData(tmpData, m_offset + offset);
            m_status.FilledRangeList.MergingAdd(new Range(offset, pushed));
            m_status.IsFilled = SinkUtil.CheckIsFilled(m_status.FilledRangeList, Size);
            return pushed;
        }
        public SinkStatus QueryStatus()
        {
            return m_status;
        }
        public void SetSize(long size)
        {
            throw new NotSupportedException();
        }
    }

    public class ReadableSubSink : IReadableSink
    {
        public long Size { get; private set; }
        private IReadableSink m_sink;
        private long m_offset;
        private SinkStatus m_status;

        public ReadableSubSink(IReadableSink sink, long offset, long size)
        {
            // ここでは offset, size のチェックはしない
            m_sink = sink;
            m_offset = offset;
            Size = size;
            m_status = new SinkStatus();
        }
        public int PushData(ByteData data, long offset)
        {
            int pushSize = data.Buffer.Count;
            if (offset + pushSize > Size)
            {
                int restSize = (int)(Size - offset);
                pushSize = restSize > 0 ? restSize : 0;
            }
            ByteData tmpData = new ByteData(new ArraySegment<byte>(data.Buffer.Array, data.Buffer.Offset, pushSize));
            int pushed =  m_sink.PushData(tmpData, m_offset + offset);
            m_status.FilledRangeList.MergingAdd(new Range(offset, pushed));
            m_status.IsFilled = SinkUtil.CheckIsFilled(m_status.FilledRangeList, Size);
            return pushed;
        }
        public SinkStatus QueryStatus()
        {
            return m_status;
        }
        public void SetSize(long size)
        {
            throw new NotSupportedException();
        }
        public ISource ToSource()
        {
            return new SubSource(m_sink.ToSource(), m_offset, Size);
        }
    }

    public class IntegrityHierarchicalStorageSink : ISink
    {
        public long Size { get; private set; }
        public int MasterHashSize { get { return (int)m_storage.GetMasterHashSize(); } }
        public int LayerHashSize { get { return (int)m_storage.GetLayerHashSize(); } }

        private HierarchicalIntegrityVerificationStorage m_storage;
        private SinkStatus m_status;

        public IntegrityHierarchicalStorageSink(long size) : this(size, null, null)
        {
        }

        public IntegrityHierarchicalStorageSink(long size, byte[] hashData, byte[] masterHashData)
        {
            Size = size;
            m_storage = new HierarchicalIntegrityVerificationStorage();
            m_storage.Initialize(size, hashData, masterHashData);
            m_status = new SinkStatus();
        }
        public int PushData(ByteData data, long offset)
        {
            int writeSize = SinkUtil.GetWritableSize(Size, offset, data.Buffer.Count);
            if (writeSize == 0)
            {
                return 0;
            }
            m_storage.Write(offset, data.Buffer.Array);
            m_storage.Commit();
            m_status.FilledRangeList.MergingAdd(new Range(offset, writeSize));
            m_status.IsFilled = SinkUtil.CheckIsFilled(m_status.FilledRangeList, Size);
            return writeSize;
        }
        public SinkStatus QueryStatus()
        {
            return m_status;
        }
        public void SetSize(long size)
        {
            throw new NotSupportedException();
        }
        public byte[] GetMetaInfo()
        {
            return m_storage.GetMetaInfo();
        }
        public ByteData GetMasterHash(long offset, int size)
        {
            byte[] data = m_storage.GetMasterHash();
            return new ByteData(new ArraySegment<byte>(data, (int)offset, size));
        }
        public ByteData GetLayerHash(long offset, int size)
        {
            byte[] data = m_storage.GetLayerHash(offset, size);
            return new ByteData(new ArraySegment<byte>(data, 0, size));
        }
    }
}
