﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "systemInitializer_FsFileOnMemory.h"
#include "systemInitializer_BuildInBlockStorage.h"
#include "SystemInitializer_InitialImageHash.h"
#include <nn/nn_Abort.h>
#include <nn/util/util_BitUtil.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/fs_Base.h>
#include <nn/init.h>
#include <nn/nn_Log.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <algorithm>
#include <memory>

namespace
{
    // セーフモード領域の開始アドレス
    const int64_t SafeModePartitionOffset = 0x03800000;

    // セーフモード領域のサイズ
    const int64_t SafeModePartitionSize   = 0x04000000;

    // 修理用初期化イメージのフォーマットバージョン
    const int64_t RepairInitialImageFormatVersion   = 1;

    template <typename T>
    T RoundUp(T value, size_t align)
    {
        return (value + (align - 1)) & ~(align - 1);
    }

    void DumpData(const void* data, size_t size)
    {
        const uint8_t* bytes = reinterpret_cast<const uint8_t*>(data);
        NN_UNUSED(bytes);
        for(int i=0; i<size; i++)
        {
            NN_LOG("%02X ",bytes[i]);
            if(i % 16 == 15 || i + 1 == size) NN_LOG("\n");
        }
    }


    bool VerifySha256(const void* pExpectSha256Data, size_t sha256DataSize, const void* pData, size_t dataSize)
    {
        NN_ABORT_UNLESS_NOT_NULL(pExpectSha256Data);
        NN_ABORT_UNLESS_NOT_NULL(pData);
        NN_ABORT_UNLESS(sha256DataSize == nn::crypto::Sha256Generator::HashSize);

        std::unique_ptr<char[]> actualSha256Data(new char[sha256DataSize]);
        nn::crypto::GenerateSha256Hash( actualSha256Data.get(), sha256DataSize,
                                        pData, dataSize);

        NN_LOG("InitialImageData (size:%d)\n", dataSize);
        DumpData(pData, 64);
        NN_LOG("...\n");

        NN_LOG("SHA256 Expect:\n");
        DumpData(pExpectSha256Data, sha256DataSize);
        NN_LOG("SHA256 Actual:\n");
        DumpData(actualSha256Data.get(), sha256DataSize);

        return (std::memcmp(actualSha256Data.get(), pExpectSha256Data, sha256DataSize) == 0);
    }

    bool VerifyInitialImage(const void* pInitialImageData, size_t size)
    {
        return VerifySha256(InitialImageHash, sizeof(InitialImageHash), pInitialImageData, size);
    }
}

nn::Result FsFile::OpenRead(const char *path)
{
    NN_UNUSED(path);

    std::shared_ptr<IBlockStorage> storage = GetBuildInBlockStorage();
    size_t blockSize = storage.get()->GetBlockSize();
    std::unique_ptr<char[]> readBuf(new char[blockSize]);

    // ヘッダの取得（blocksize=512byte 読み込んでヘッダサイズ分だけを取り出す）
    uint64_t safeModePartitionBlockAddress = SafeModePartitionOffset / blockSize;
    NN_RESULT_DO(storage.get()->Read( readBuf.get(), blockSize, safeModePartitionBlockAddress, blockSize));
    std::memcpy(&m_Header, readBuf.get(), sizeof(m_Header));
    readBuf.reset();

    NN_ABORT_UNLESS(m_Header.FormatVersion == RepairInitialImageFormatVersion);
    NN_ABORT_UNLESS(SafeModePartitionSize >= m_Header.ImageSize + sizeof(m_Header));

    int64_t alignedImageWithHeaderSize = RoundUp(m_Header.ImageSize + sizeof(m_Header), blockSize);
    int64_t alignedImageWithHeaderBlockSize = alignedImageWithHeaderSize / blockSize;

    m_FileDataBuffer = new char[alignedImageWithHeaderSize];
    NN_RESULT_DO(storage.get()->Read( m_FileDataBuffer, alignedImageWithHeaderSize, safeModePartitionBlockAddress, alignedImageWithHeaderBlockSize));

    NN_ABORT_UNLESS(VerifyInitialImage(m_FileDataBuffer, m_Header.ImageSize) == true);

    NN_RESULT_SUCCESS;
}

nn::Result FsFile::OpenWrite(const char *path)
{
    NN_UNUSED(path);
    NN_ABORT("not implemented.");
    NN_RESULT_SUCCESS;
}

void FsFile::Close()
{
    delete m_FileDataBuffer;
    std::memset(&m_Header, 0, sizeof(m_Header));
}

nn::Result FsFile::Write(int64_t offset, const void* buffer, size_t size, bool flush)
{
    NN_UNUSED(offset);
    NN_UNUSED(buffer);
    NN_UNUSED(size);
    NN_UNUSED(flush);
    NN_ABORT("not implemented.");
    NN_RESULT_SUCCESS;
}

nn::Result FsFile::Read(size_t* pOut, int64_t offset, void* buffer, size_t size)
{
    NN_ABORT_UNLESS_NOT_NULL(buffer);
    NN_ABORT_UNLESS_NOT_NULL(pOut);
    NN_ABORT_UNLESS(m_Header.ImageSize >= offset);

    auto actualReadSize = std::min(m_Header.ImageSize - offset, static_cast<int64_t>(size));
    std::memcpy(buffer, m_FileDataBuffer + offset + sizeof(OnMemoryInitialImageHeader), actualReadSize);
    *pOut = actualReadSize;

    NN_RESULT_SUCCESS;
}

nn::Result FsFile::GetSize(int64_t* pOut)
{
    NN_ABORT_UNLESS_NOT_NULL(pOut);
    *pOut = m_Header.ImageSize;
    NN_RESULT_SUCCESS;
}

nn::Result FsFile::Flush()
{
    NN_RESULT_SUCCESS;
}


bool FsFile::IsValid()
{
    return (m_Header.ImageSize != 0);
}

nn::Result FsFile::Exists(bool *pOut, const char *name)
{
    NN_UNUSED(name);
    NN_ABORT_UNLESS_NOT_NULL(pOut);
    *pOut = true;
    NN_RESULT_SUCCESS;
}

nn::Result FsFile::Create(const char *path)
{
    NN_UNUSED(path);
    NN_ABORT("not implemented.");
    NN_RESULT_SUCCESS;
}

