﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "systemInitializer_FileSystemUtility.h"
#include <nn/nn_Abort.h>
#include <nn/fs_Base.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

bool IsWordChar(char c)
{
    return ('a' <= c && c <= 'z') || ('A' <= c && c <= 'Z') || ('0' <= c && c <= '9') || ('_' == c);
}

bool IsValidFileSystemEntry(const char *name)
{
    enum State
    {
        State_Start,
        State_ArchiveChar,
        State_ArchiveEnd,
        State_RestChars
    };

    State state = State_Start;

    for (const char* current = name; *current != '\0'; current++)
    {
        switch (state)
        {
        case State_Start:
            {
                if(IsWordChar(*current))
                {
                    state = State_ArchiveChar;
                }
                else
                {
                    return false;
                }
            }
            break;
        case State_ArchiveChar:
            {
                if(*current == ':')
                {
                    state = State_ArchiveEnd;
                }
                else if(!IsWordChar(*current))
                {
                    return false;
                }
                break;
            }
        case State_ArchiveEnd:
            {
                if(*current == '/' || *current == '\\')
                {
                    state = State_RestChars;
                }
                else
                {
                    return false;
                }
                break;
            }
        case State_RestChars:
            {
                break;
            }
        default:
            NN_ABORT("UnexpectedState: %d\n", state);
        }
    }

    return state == State_RestChars;
}

nn::Result FileExists(bool *pOut, const char *name)
{
    nn::fs::FileHandle handle;

    NN_RESULT_TRY(nn::fs::OpenFile(&handle, name, nn::fs::OpenMode_Read))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            *pOut = false;

            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY;
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    *pOut = true;

    NN_RESULT_SUCCESS;
}

nn::Result DirectoryExists(bool *pOut, const char *name)
{
    nn::fs::DirectoryHandle handle;

    NN_RESULT_TRY(nn::fs::OpenDirectory(&handle, name, nn::fs::OpenDirectoryMode_All))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            *pOut = false;

            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY;
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseDirectory(handle);
    };

    *pOut = true;

    NN_RESULT_SUCCESS;
}
