﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs_Base.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_IStorage.h>

#include <nn/utilTool/utilTool_CommandLog.h>
#include <nn/utilTool/utilTool_ResultHandlingUtility.h>

#include "ListExportablePartitions.h"
#include "Utility/systemInitializer_Allocator.h"
#include "Utility/systemInitializer_BuildInBlockStorage.h"
#include "Utility/systemInitializer_FsBisPartition.h"
#include "Partition/systemInitializer_Partition.h"

#include <memory>
#include <tuple>

nn::Result ListExportablePartitions()
{
    std::list<ExportablePartitionInfo> partitions;

    NN_UTILTOOL_RESULT_DO(
        ListExportablePartitions(&partitions));

    for (auto partition : partitions)
    {
        if(partition.GetName() == "")
        {
            NN_UTILTOOL_LOG_MESSAGE("--type %s --size %lld", partition.GetType().c_str(), partition.GetName().c_str(), partition.GetSize());
        }
        else
        {
            NN_UTILTOOL_LOG_MESSAGE("--type %s --name %s --size %lld", partition.GetType().c_str(), partition.GetName().c_str(), partition.GetSize());
        }
    }

    NN_RESULT_SUCCESS;
}

nn::Result ListExportablePartitions(std::list<ExportablePartitionInfo> *pOut)
{
    NN_UTILTOOL_RESULT_DO(
        ListStorages(pOut));

    NN_UTILTOOL_RESULT_DO(
        ListGptHeaders(pOut));

    NN_UTILTOOL_RESULT_DO(
        ListGptPartitions(pOut));

    NN_UTILTOOL_RESULT_DO(
        ListFsPartitions(pOut));

    NN_RESULT_SUCCESS;
}

nn::Result ListStorages(std::list<ExportablePartitionInfo> *pOut)
{
    std::shared_ptr<IBlockStorage> storage = GetBuildInBlockStorage(true);

    pOut->push_back(
        ExportablePartitionInfo(
            "storage", "", storage->GetTotalSize()));

    NN_RESULT_SUCCESS;
}

nn::Result ListGptHeaders(std::list<ExportablePartitionInfo> *pOut)
{
    pOut->push_back(
        ExportablePartitionInfo(
            "gpt-header", "", sizeof(GptHeader)));

    std::shared_ptr<IBlockStorage> storage = GetBuildInBlockStorage(true);

    static GptHeader gptHeader;

    ReadGptFromStorage(&gptHeader, storage.get());

    pOut->push_back(
        ExportablePartitionInfo(
            "gpt-partition-table", "", gptHeader.sizeOfPartitionEntry * gptHeader.numberOfPartitionEntries));

    NN_RESULT_SUCCESS;
}

nn::Result ListGptPartitions(std::list<ExportablePartitionInfo> *pOut)
{
    std::shared_ptr<IBlockStorage> storage = GetBuildInBlockStorage(true);

    static GptHeader gptHeader;
    static GptPartitionEntry gptEntries[GPT_DEFAULT_PARTITION_ENTRIES];

    NN_UTILTOOL_RESULT_DO(
        ReadGptFromStorage(&gptHeader, storage.get()));
    NN_UTILTOOL_RESULT_DO(
        ReadGptPartitionsFromStorage(gptEntries, GPT_DEFAULT_PARTITION_ENTRIES, gptHeader, storage.get()));

    for(int i = 0; i < gptHeader.numberOfPartitionEntries; i++)
    {
        const GptPartitionEntry &gptEntry = gptEntries[i];

        pOut->push_back(
            ExportablePartitionInfo(
            "gpt-partition", GetPartitionName(gptEntry), CalculatePartitionSize(gptEntry, storage.get()) ));
    }

    NN_RESULT_SUCCESS;
}

nn::Result ListFsPartitions(std::list<ExportablePartitionInfo> *pOut)
{
    for( auto partition : ListFsPartitions() )
    {
        auto partitionName = ConvertToBisPartitionName(partition);

        std::unique_ptr<nn::fs::IStorage> pStorage;

        auto result = nn::fs::OpenBisPartition(&pStorage, partition);
        if( result <= nn::fs::ResultPartitionNotFound() )
        {
            NN_UTILTOOL_LOG_DEBUG("FsPartition(id=%s) is not found.", partitionName.c_str());
            continue;
        }

        int64_t size;
        NN_UTILTOOL_RESULT_DO(
            pStorage->GetSize(&size));

        pOut->push_back(
            ExportablePartitionInfo(
            "fs-partition", partitionName, size));
    }

    NN_RESULT_SUCCESS;
}

nn::Result HasExportablePartition(bool *pOut, std::string type, std::string name)
{
    std::list<ExportablePartitionInfo> partitions;

    NN_UTILTOOL_RESULT_DO(ListExportablePartitions(&partitions));

    for(auto partition : partitions)
    {
        if(partition.GetType() == type && partition.GetName() == name)
        {
            *pOut = true;
            NN_RESULT_SUCCESS;
        }
    }

    *pOut = false;

    NN_RESULT_SUCCESS;
}
