﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ExpHeapAllocator.h>
#include <nn/sf/sf_ObjectFactory.h>

#include <nn/nn_SdkLog.h>
#include <nn/nn_Result.h>

#include <nn/spl/spl_Api.h>
#include <nn/spl/impl/spl_ApiImpl.h>
#include <nn/spl/detail/spl_GeneralInterface.h>
#include <nn/spl/detail/spl_RandomInterface.h>
#include <nn/spl/detail/spl_CryptoInterface.h>
#include <nn/spl/detail/spl_DeviceUniqueDataInterface.h>
#include <nn/spl/detail/spl_FsInterface.h>
#include <nn/spl/detail/spl_SslInterface.h>
#include <nn/spl/detail/spl_EsInterface.h>
#include <nn/spl/detail/spl_ManuInterface.h>

#include "spl_SecureMonitorManager.h"
#include "spl_GeneralServer.h"
#include "spl_RandomServer.h"
#include "spl_CryptoServer.h"
#include "spl_FsServer.h"
#include "spl_SslServer.h"
#include "spl_EsServer.h"
#include "spl_ManuServer.h"

namespace
{
    enum PortIndex
    {
        PortIndex_General,
        PortIndex_Random,
        PortIndex_Crypto,
        PortIndex_Fs,
        PortIndex_Ssl,
        PortIndex_Es,
        PortIndex_Manu,
        PortIndex_Count
    };

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    static const int SessionCountForTest = 1;
#else
    static const int SessionCountForTest = 0;
#endif
    static const int GeneralSessionCount = 8 + SessionCountForTest;
    static const int FsSessionCount      = 1 + SessionCountForTest;
    static const int SslSessionCount     = 2;
    static const int RandomSessionCount  = 3;
    static const int CryptoSessionCount  = 6;
    static const int EsSessionCount      = 2;
    static const int ManuSessionCount    = 1;
    static const int TotalSessionCount   = GeneralSessionCount + RandomSessionCount + CryptoSessionCount + FsSessionCount + SslSessionCount + EsSessionCount + ManuSessionCount;

    struct ServerManagerOptions
    {
        static const size_t PointerTransferBufferSize = 2048;
    };

    class ServerManager
        : public nn::sf::HipcSimpleAllInOneServerManager<TotalSessionCount, PortIndex_Count, ServerManagerOptions>
    {
    public:
        ServerManager(
            nn::sf::ExpHeapAllocator* pAllocator,
            nn::spl::SecureMonitorManager* pSecureMonitorManager)
            : m_pAllocator(pAllocator)
            , m_pSecureMonitorManager(pSecureMonitorManager)
            , m_GeneralServer(pSecureMonitorManager)
            , m_GeneralServerObject(&m_GeneralServer)
            , m_RandomServer(pSecureMonitorManager)
            , m_RandomServerObject(&m_RandomServer)
        {
        }

    protected:
        nn::Result OnNeedsToAccept(int portIndex, AcceptTarget* pPort) NN_NOEXCEPT NN_OVERRIDE final;

    private:
        nn::sf::ExpHeapAllocator* m_pAllocator;
        nn::spl::SecureMonitorManager* m_pSecureMonitorManager;
        nn::spl::GeneralServer m_GeneralServer;
        nn::sf::UnmanagedServiceObjectByPointer
            <nn::spl::detail::IGeneralInterface, nn::spl::GeneralServer> m_GeneralServerObject;
        nn::spl::RandomServer m_RandomServer;
        nn::sf::UnmanagedServiceObjectByPointer
            <nn::spl::detail::IRandomInterface, nn::spl::RandomServer> m_RandomServerObject;
    };


    NN_ALIGNAS(64) nn::Bit8         g_DefaultServerAllocatorBuffer[16 * 1024];

    nn::sf::ExpHeapAllocator        g_DefaultServerAllocator;
    nn::spl::SecureMonitorManager   g_SecureMonitorManager;
    ServerManager                   g_ServerManager(&g_DefaultServerAllocator, &g_SecureMonitorManager);

    nn::Result ServerManager::OnNeedsToAccept(int portIndex, AcceptTarget* pPort) NN_NOEXCEPT
    {
        switch (portIndex)
        {
        case PortIndex_General:
            return AcceptImpl(pPort, m_GeneralServerObject.GetShared());
        case PortIndex_Random:
            return AcceptImpl(pPort, m_RandomServerObject.GetShared());
        case PortIndex_Crypto:
            // セッションを破棄するときにリソースを解放するためセッションごとにひとつオブジェクトを割り当てる
            return AcceptImpl(pPort, nn::sf::ObjectFactory<nn::sf::ExpHeapAllocator::Policy>::CreateSharedEmplaced<nn::spl::detail::ICryptoInterface, nn::spl::CryptoServer>(m_pAllocator, m_pSecureMonitorManager));
        case PortIndex_Fs:
            return AcceptImpl(pPort, nn::sf::ObjectFactory<nn::sf::ExpHeapAllocator::Policy>::CreateSharedEmplaced<nn::spl::detail::IFsInterface, nn::spl::FsServer>(m_pAllocator, m_pSecureMonitorManager));
        case PortIndex_Ssl:
            return AcceptImpl(pPort, nn::sf::ObjectFactory<nn::sf::ExpHeapAllocator::Policy>::CreateSharedEmplaced<nn::spl::detail::ISslInterface, nn::spl::SslServer>(m_pAllocator, m_pSecureMonitorManager));
        case PortIndex_Es:
            return AcceptImpl(pPort, nn::sf::ObjectFactory<nn::sf::ExpHeapAllocator::Policy>::CreateSharedEmplaced<nn::spl::detail::IEsInterface, nn::spl::EsServer>(m_pAllocator, m_pSecureMonitorManager));
        case PortIndex_Manu:
            return AcceptImpl(pPort, nn::sf::ObjectFactory<nn::sf::ExpHeapAllocator::Policy>::CreateSharedEmplaced<nn::spl::detail::IManuInterface, nn::spl::ManuServer>(m_pAllocator, m_pSecureMonitorManager));
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

extern "C" void nninitStartup()
{
}

extern "C" void nninitInitializeSdkModule()
{
}

extern "C" void nninitFinalizeSdkModule()
{
}


extern "C" void nnMain() NN_NOEXCEPT
{
    auto heapHandle = nn::lmem::CreateExpHeap(&g_DefaultServerAllocatorBuffer, sizeof(g_DefaultServerAllocatorBuffer), nn::lmem::CreationOption_NoOption);
    g_DefaultServerAllocator.Attach(heapHandle);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        g_ServerManager.InitializePort(
            PortIndex_General, GeneralSessionCount, nn::spl::detail::PortName ));
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        g_ServerManager.InitializePort(
            PortIndex_Random, RandomSessionCount, nn::spl::detail::PortNameRandom ));
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        g_ServerManager.InitializePort(
            PortIndex_Crypto, CryptoSessionCount, nn::spl::detail::PortNameCrypto ));
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        g_ServerManager.InitializePort(
            PortIndex_Fs, FsSessionCount, nn::spl::detail::PortNameFs ));
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        g_ServerManager.InitializePort(
            PortIndex_Ssl, SslSessionCount, nn::spl::detail::PortNameSsl ));
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        g_ServerManager.InitializePort(
            PortIndex_Es, EsSessionCount, nn::spl::detail::PortNameEs ));
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        g_ServerManager.InitializePort(
            PortIndex_Manu, ManuSessionCount, nn::spl::detail::PortNameManu ));

    g_ServerManager.Start();

    g_SecureMonitorManager.Initialize();

    g_ServerManager.LoopAuto();
}
