﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ExpHeapAllocator.h>
#include <nn/sf/sf_ObjectFactory.h>

#include <nn/nn_SdkLog.h>
#include <nn/nn_Result.h>

#include <nn/sm/sm_UserApi.h>
#include <nn/sm/sm_ManagerApi.h>
#include <nn/sm/detail/sm_UserInterface.h>
#include <nn/sm/detail/sm_ManagerInterface.h>

#include <nn/svc/svc_Tcb.h>

#include "sm_UserInterface.h"
#include "sm_ManagerInterface.h"
#include "sm_WaitList.h"

namespace
{
    enum PortIndex
    {
        PortIndex_User,
        PortIndex_Manager,
        PortIndex_Count
    };

    const int UserSessionCountMax       = 64;
    const int ManagerSessionCountMax    =  1;
    const int SessionCountMax           = UserSessionCountMax + ManagerSessionCountMax;

    struct ServerManagerOption
    {
        static const bool CanDeferInvokeRequest = true;
    };

    class ServerManager
        : public nn::sf::HipcSimpleAllInOneServerManager
            <SessionCountMax, PortIndex_Count, ServerManagerOption>
    {
    private:
        virtual nn::Result OnNeedsToAccept(int portIndex, PortObjectImpl* pPort) NN_NOEXCEPT;
    };


    NN_ALIGNAS(64) nn::Bit8 g_DefaultServerAllocatorBuffer[160 * 1024];

    nn::sf::ExpHeapAllocator  g_DefaultServerAllocator;
    ServerManager g_ServerManager;


    nn::Result ServerManager::OnNeedsToAccept(int portIndex, PortObjectImpl* pPort) NN_NOEXCEPT
    {
        switch(portIndex)
        {
        case PortIndex_User:
            {
                nn::sf::SharedPointer<nn::sm::detail::IUserInterface> refInterface;

                refInterface = nn::sf::ObjectFactory<
                    nn::sf::ExpHeapAllocator::Policy
                >::CreateSharedEmplaced<
                    nn::sm::detail::IUserInterface,
                    nn::sm::UserInterface
                >(&g_DefaultServerAllocator);

                auto result = this->AcceptImpl(pPort, refInterface);
                return result;
            }
            break;

        case PortIndex_Manager:
            {
                nn::sf::SharedPointer<nn::sm::detail::IManagerInterface> refInterface;

                refInterface = nn::sf::ObjectFactory<
                    nn::sf::ExpHeapAllocator::Policy
                >::CreateSharedEmplaced<
                    nn::sm::detail::IManagerInterface,
                    nn::sm::ManagerInterface
                >(&g_DefaultServerAllocator);

                auto result = this->AcceptImpl(pPort, refInterface);
                return result;
            }
            break;

        default:
            NN_ABORT();
        }
    }

    nn::Result ResumeImpl(nn::os::MultiWaitHolderType* pSession)
    {
        return g_ServerManager.ProcessInvokeRequest(pSession);
    }








}   // anonymous namespace


namespace nn { namespace sm {

    Result Finalize() NN_NOEXCEPT
    {
        return ResultSuccess();
    }

    Result UnregisterService(const char* pName, size_t nameSize) NN_NOEXCEPT
    {
        NN_UNUSED(pName);
        NN_UNUSED(nameSize);
        return ResultSuccess();
    }

}}  // namespace nn::sm





extern "C" void nninitStartup() NN_NOEXCEPT
{
}

extern "C" void nninitInitializeSdkModule()
{
}

extern "C" void nninitFinalizeSdkModule()
{
}

extern "C" void nnMain() NN_NOEXCEPT
{
    nn::Result result;

    auto heapHandle = nn::lmem::CreateExpHeap(&g_DefaultServerAllocatorBuffer, sizeof(g_DefaultServerAllocatorBuffer), nn::lmem::CreationOption_NoOption);
    g_DefaultServerAllocator.Attach(heapHandle);

    {
        nn::svc::Handle port;

        result = nn::svc::ManageNamedPort(&port, nn::sm::detail::PortName, UserSessionCountMax);
        NN_ABORT_UNLESS(result.IsSuccess());
        g_ServerManager.InitializePort(PortIndex_User, port);

        result = nn::sm::RegisterSelfService(&port, nn::sm::detail::PortNameForManager, ManagerSessionCountMax);
        NN_ABORT_UNLESS(result.IsSuccess());
        g_ServerManager.InitializePort(PortIndex_Manager, port);
    }

    g_ServerManager.Start();

    while (auto p = g_ServerManager.Wait())
    {
        switch (nn::os::GetMultiWaitHolderUserData(p))
        {
        case ServerManager::InvokeTag:
            {
                result = g_ServerManager.ProcessInvokeRequestWithDefer(p);
                if( result <= nn::sf::ResultProcessDeferred() )
                {
                    nn::sm::ProcessRegisterRetry(p);
                    break;
                }
                NN_ABORT_UNLESS(result.IsSuccess(), "result=%08x", result);

                nn::sm::TestAndResume(ResumeImpl);
            }
            break;

        case ServerManager::AcceptTag:
            {
                result = g_ServerManager.ProcessAcceptRequest(p);
                NN_ABORT_UNLESS(result.IsSuccess(), "result=%08x", result);
            }
            break;

        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

}
