﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/util/util_IntrusiveList.h>
#include <nn/lmem/lmem_UnitHeap.h>
#include <nn/ncm/ncm_ProgramLocation.h>
#include <nn/ldr/ldr_Types.h>
#include <nn/svc/svc_Types.h>
#include <nn/os/os_Types.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_MultipleWaitApi.h>

namespace nn { namespace pm {


    class ProcessInfo
    {
    private:
        enum Flags
        {
            Flags_NotifyExit          = (1u << 0),
            Flags_ExceptionOccured    = (1u << 1),
            Flags_ExceptionWaiting    = (1u << 2),
            Flags_NotifyDebug         = (1u << 3),
            Flags_DebugStateChanged   = (1u << 4),
            Flags_Breaked             = (1u << 5),
            Flags_Application         = (1u << 6),
            Flags_NotifyRunning       = (1u << 7),
            Flags_RunningStateChanged = (1u << 8),
        };

    public:
        ProcessInfo( svc::Handle            handle,
                     os::ProcessId          processId,
                     ldr::PinId             pinId,
                     ncm::ProgramLocation   programLocation ) NN_NOEXCEPT;
        ~ProcessInfo() NN_NOEXCEPT;

        os::NativeHandle GetOsHandle() NN_NOEXCEPT
        {
            return m_Handle.operator nnHandle().value;
        }

        void* operator new (size_t) throw();
        void operator delete (void*);

        svc::Handle GetHandle() const NN_NOEXCEPT
        {
            return m_Handle;
        }
        os::ProcessId GetId() const NN_NOEXCEPT
        {
            return m_ProcessId;
        }
        ldr::PinId GetPinId() const NN_NOEXCEPT
        {
            return m_PinId;
        }
        ncm::ProgramId GetProgramId() const NN_NOEXCEPT
        {
            return m_ProgramLocation.programId;
        }
        const ncm::ProgramLocation& GetProgramLocation() const NN_NOEXCEPT
        {
            return m_ProgramLocation;
        }

        void SetState(svc::ProcessState state) NN_NOEXCEPT
        {
            m_State = state;
        }
        svc::ProcessState GetState() const NN_NOEXCEPT
        {
            return m_State;
        }

        void SetJitFlags() NN_NOEXCEPT
        {
            m_Flags |= (Flags_ExceptionOccured | Flags_ExceptionWaiting);
        }

        void ClearExceptionOccured() NN_NOEXCEPT
        {
            m_Flags &= ~Flags_ExceptionOccured;
        }
        void ClearExceptionWaiting() NN_NOEXCEPT
        {
            m_Flags &= ~Flags_ExceptionWaiting;
        }
        void ClearDebugStateChanged() NN_NOEXCEPT
        {
            m_Flags &= ~Flags_DebugStateChanged;
        }
        void ClearRunningStateChanged() NN_NOEXCEPT
        {
            m_Flags &= ~Flags_RunningStateChanged;
        }
        void ClearNotifyRunningRequire() NN_NOEXCEPT
        {
            m_Flags &= ~Flags_NotifyRunning;
        }

        void SetNotifyExitRequire() NN_NOEXCEPT
        {
            m_Flags |= Flags_NotifyExit;
        }
        void SetNotifyDebugRequire() NN_NOEXCEPT
        {
            m_Flags |= Flags_NotifyDebug;
        }
        void SetNotifyRunningRequire() NN_NOEXCEPT
        {
            m_Flags |= Flags_NotifyRunning;
        }
        void SetApplication() NN_NOEXCEPT
        {
            m_Flags |= Flags_Application;
        }
        void SetDebugStateChanged() NN_NOEXCEPT
        {
            m_Flags |= Flags_DebugStateChanged;
        }
        void SetRunningStateChanged() NN_NOEXCEPT
        {
            m_Flags |= Flags_RunningStateChanged;
        }
        void SetDebugState(bool isBreaked) NN_NOEXCEPT
        {
            if( isBreaked )
            {
                m_Flags |= Flags_Breaked;
            }
            else
            {
                m_Flags &= ~Flags_Breaked;
            }
        }

        bool IsStarted() const NN_NOEXCEPT
        {
            return (m_State != svc::ProcessState_Initializing)
                && (m_State != svc::ProcessState_PreAttached);
        }
        bool IsExited() const NN_NOEXCEPT
        {
            return m_State == svc::ProcessState_Terminated;
        }
        bool IsNotifyExitRequired() const NN_NOEXCEPT
        {
            return (m_Flags & Flags_NotifyExit) != 0;
        }
        bool IsNotifyDebugRequired() const NN_NOEXCEPT
        {
            return (m_Flags & Flags_NotifyDebug) != 0;
        }
        bool IsNotifyRunningRequired() const NN_NOEXCEPT
        {
            return (m_Flags & Flags_NotifyRunning) != 0;
        }
        bool IsExceptionWaiting() const NN_NOEXCEPT
        {
            return (m_Flags & Flags_ExceptionWaiting) != 0;
        }
        bool IsExceptionOccured() const NN_NOEXCEPT
        {
            return (m_Flags & Flags_ExceptionOccured) != 0;
        }
        bool IsDebugStateChanged() const NN_NOEXCEPT
        {
            return (m_Flags & Flags_DebugStateChanged) != 0;
        }
        bool IsRunningStateChanged() const NN_NOEXCEPT
        {
            return (m_Flags & Flags_RunningStateChanged) != 0;
        }
        bool IsBreaked() const NN_NOEXCEPT
        {
            return (m_Flags & Flags_Breaked) != 0;
        }
        bool IsApplication() const NN_NOEXCEPT
        {
            return (m_Flags & Flags_Application) != 0;
        }

        void LinkToMultiWait(os::MultiWaitType* pmw) NN_NOEXCEPT
        {
            os::LinkMultiWaitHolder(pmw, &m_MultiWaitHolder);
        }
        void UnlinkFromMultiWait() NN_NOEXCEPT
        {
            os::UnlinkMultiWaitHolder(&m_MultiWaitHolder);
        }

    public:
        static util::IntrusiveListNode& GetNode(ProcessInfo& item) NN_NOEXCEPT
        {
            return item.m_LinkNode;
        }
        static ProcessInfo& GetItem(util::IntrusiveListNode& node) NN_NOEXCEPT
        {
            return *reinterpret_cast<ProcessInfo*>(&node);
        }

        static void InitializeHeap(void* p, size_t size) NN_NOEXCEPT;

    private:
        util::IntrusiveListNode     m_LinkNode;     // 先頭にあることを GetItem が使用している

        const svc::Handle           m_Handle;
        const os::ProcessId         m_ProcessId;
        const ldr::PinId            m_PinId;
        const ncm::ProgramLocation  m_ProgramLocation;
        svc::ProcessState           m_State;
        Bit32                       m_Flags;

        os::MultiWaitHolderType     m_MultiWaitHolder;

    private:
        static nn::lmem::HeapCommonHead g_HeapHead;
        static nn::lmem::HeapHandle     g_HeapHandle;

    };


}}  // namespace nn::pm

