﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ExpHeapAllocator.h>
#include <nn/sf/sf_ObjectFactory.h>

#include <nn/nn_SdkLog.h>
#include <nn/nn_Result.h>

#include <nn/ldr/ldr_ProcessManagerApi.h>
#include <nn/fs/fs_ProgramRegistry.h>

#include <nn/pm/pm_ShellApi.h>
#include <nn/pm/pm_DebugMonitorApi.h>
#include <nn/pm/pm_InformationApi.h>
#include <nn/pm/detail/pm_ShellInterface.h>
#include <nn/pm/detail/pm_DebugMonitorInterface.h>
#include <nn/pm/detail/pm_InformationInterface.h>

#include <nn/spl/spl_Api.h>

#include "pm_BootModeManager.h"
#include "pm_ShellInterfaceServer.h"
#include "pm_DebugMonitorInterfaceServer.h"
#include "pm_InformationInterfaceServer.h"

namespace
{
    enum PortIndex
    {
        PortIndex_Shell,
        PortIndex_DebugMonitor,
        PortIndex_BootMode,
        PortIndex_Information,
        PortIndex_Count
    };

    static const int ShellSessionCount          = 3;
    static const int DebugMonitorSessionCount   = 2; // dmnt0 + tma で2セッション
    static const int BootModeSessionCount       = 5; // ns + nim + am + pctl + sdb で5セッション
    static const int InformationSessionCount    = 1; // fatal で1セッション
    static const int TotalSessionCount =
        ShellSessionCount + DebugMonitorSessionCount + BootModeSessionCount + InformationSessionCount;

    nn::sf::UnmanagedServiceObject<nn::pm::detail::IBootModeInterface, nn::pm::BootModeManager> g_BootModeManager;

    class ServerManager : public nn::sf::HipcSimpleAllInOneServerManager<TotalSessionCount, PortIndex_Count>
    {
    private:
        virtual nn::Result OnNeedsToAccept(int portIndex, PortObjectImpl* pPort) NN_NOEXCEPT;
    };


    NN_ALIGNAS(64) nn::Bit8 g_DefaultServerAllocatorBuffer[16 * 1024];

    nn::sf::ExpHeapAllocator  g_DefaultServerAllocator;
    ServerManager g_ServerManager;


    nn::Result ServerManager::OnNeedsToAccept(int portIndex, PortObjectImpl* pPort) NN_NOEXCEPT
    {
        switch(portIndex)
        {
        case PortIndex_Shell:
            {
                nn::sf::SharedPointer<nn::pm::detail::IShellInterface> refInterface;

                refInterface = nn::sf::ObjectFactory<
                    nn::sf::ExpHeapAllocator::Policy
                >::CreateSharedEmplaced<
                    nn::pm::detail::IShellInterface,
                    nn::pm::ShellInterfaceServer
                >(&g_DefaultServerAllocator);

                return this->AcceptImpl(pPort, refInterface);
            }
            break;

        case PortIndex_DebugMonitor:
            {
                nn::sf::SharedPointer<nn::pm::detail::IDebugMonitorInterface> refInterface;

                refInterface = nn::sf::ObjectFactory<
                    nn::sf::ExpHeapAllocator::Policy
                >::CreateSharedEmplaced<
                    nn::pm::detail::IDebugMonitorInterface,
                    nn::pm::DebugMonitorInterfaceServer
                >(&g_DefaultServerAllocator);

                return this->AcceptImpl(pPort, refInterface);
            }
            break;

        case PortIndex_BootMode:
            {
                return this->AcceptImpl(pPort, g_BootModeManager.GetShared());
            }
            break;

        case PortIndex_Information:
            {
                nn::sf::SharedPointer<nn::pm::detail::IInformationInterface> refInterface;

                refInterface = nn::sf::ObjectFactory<
                    nn::sf::ExpHeapAllocator::Policy
                >::CreateSharedEmplaced<
                    nn::pm::detail::IInformationInterface,
                    nn::pm::InformationInterfaceServer
                >(&g_DefaultServerAllocator);

                return this->AcceptImpl(pPort, refInterface);
            }
            break;

        default:
            NN_ABORT();
        }
    }


}

extern "C" void nninitStartup()
{
}

extern "C" void nninitInitializeSdkModule()
{
}

extern "C" void nninitFinalizeSdkModule()
{
}


extern "C" void nnMain() NN_NOEXCEPT
{
    nn::Result result;

#if defined(NN_DETAIL_PM_USE_SPL)
    nn::spl::Initialize();

    // SecureMonitor に BootConfig の内容を問い合わせて
    // desc の署名検証を無効化する
    {
        const bool isDisableDescVerification = nn::spl::GetConfigBool(nn::spl::ConfigItem_DisableDescVerification);

        if( isDisableDescVerification )
        {
            NN_SDK_LOG("[pm] disabling desc verification.\n");
            nn::fs::SetEnabledProgramVerification(false);
        }
    }
#endif

    nn::pm::ProcessManager::Initialize();

    auto heapHandle = nn::lmem::CreateExpHeap(&g_DefaultServerAllocatorBuffer, sizeof(g_DefaultServerAllocatorBuffer), nn::lmem::CreationOption_NoOption);
    g_DefaultServerAllocator.Attach(heapHandle);

    result = g_ServerManager.InitializePort(PortIndex_Shell, ShellSessionCount, nn::pm::detail::PortNameForShell);
    NN_ABORT_UNLESS(result.IsSuccess(), "result=%08x", result);

    result = g_ServerManager.InitializePort(PortIndex_DebugMonitor, DebugMonitorSessionCount, nn::pm::detail::PortNameForDebugMonitor);
    NN_ABORT_UNLESS(result.IsSuccess(), "result=%08x", result);

    result = g_ServerManager.InitializePort(PortIndex_BootMode, BootModeSessionCount, nn::pm::detail::PortNameForBootMode);
    NN_ABORT_UNLESS(result.IsSuccess(), "result=%08x", result);

    result = g_ServerManager.InitializePort(PortIndex_Information, InformationSessionCount, nn::pm::detail::PortNameForInformation);
    NN_ABORT_UNLESS(result.IsSuccess(), "result=%08x", result);

    g_ServerManager.Start();

    result = nn::ldr::InitializeForProcessManager();
    NN_ABORT_UNLESS(result.IsSuccess(), "result=%08x", result);

    g_ServerManager.LoopAuto();
}
