﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_StaticAssert.h>
#include <nn/ncm/ncm_ProgramId.h>
#include <nn/util/util_BitPack.h>

namespace nn { namespace ldr {

    struct Meta
    {
        static const Bit32 SignatureValue = 0x4154454D;

        enum AddressSpaceSizeType
        {
            AddressSpaceSizeType_32Bit,
            AddressSpaceSizeType_64BitOld,
            AddressSpaceSizeType_32BitNoReserved,
            AddressSpaceSizeType_64Bit
        };

        typedef util::BitPack8::Field<0, 1, bool>    Is64BitInstruction;
        typedef util::BitPack8::Field<1, 3, int>     AddressSpaceSize;

        Bit32       signature;
        NN_PADDING4;
        NN_PADDING4;
        Bit8        flags;
        NN_PADDING1;
        uint8_t     threadPriority;
        uint8_t     idealCoreNumber;

        NN_PADDING4;
        uint32_t    systemResourceSize;
        Bit32       version;
        uint32_t    stackSize;

        char        name[16];
        Bit8        productCode[16];

        Bit8        reserved0x40[0x30];

        uint32_t    aciOffset;
        uint32_t    aciSize;
        uint32_t    acidOffset;
        uint32_t    acidSize;
    };
    NN_STATIC_ASSERT( sizeof(Meta) == 0x80 );

    struct AcidHeaderData
    {
        static const Bit32 SignatureValue = 0x44494341;

        typedef util::BitPack32::Field<0,  1, bool> ProductionFlag;
        typedef util::BitPack32::Field<1,  1, bool> UnqualifiedApprovalFlag;
        typedef util::BitPack32::Field<2,  4, int>  MemoryRegion;
        typedef util::BitPack32::Field<6, 26, int>  ReservedFlags;

        Bit32           signature;
        uint32_t        signTargetSize;
        uint8_t         descVersion;
        NN_PADDING1;
        NN_PADDING1;
        NN_PADDING1;
        uint32_t        flags;

        ncm::ProgramId  programIdMin;
        ncm::ProgramId  programIdMax;

        uint32_t        facdOffset;
        uint32_t        facdSize;
        uint32_t        sacdOffset;
        uint32_t        sacdSize;
        uint32_t        kcdOffset;
        uint32_t        kcdSize;
        NN_PADDING4;
        NN_PADDING4;
    };
    NN_STATIC_ASSERT( sizeof(AcidHeaderData) == 0x40 );

    struct AcidHeader
    {
        Bit8            headerKeySign[0x100];
        Bit8            headerKey[0x100];

        AcidHeaderData  data;
    };
    NN_STATIC_ASSERT( sizeof(AcidHeader) == 0x240 );

    struct AciHeader
    {
        static const Bit32 SignatureValue = 0x30494341;

        Bit32           signature;
        NN_PADDING4;
        NN_PADDING4;
        NN_PADDING4;

        ncm::ProgramId  programId;
        NN_PADDING4;
        NN_PADDING4;

        uint32_t        facOffset;
        uint32_t        facSize;
        uint32_t        sacOffset;
        uint32_t        sacSize;
        uint32_t        kcOffset;
        uint32_t        kcSize;
        NN_PADDING4;
        NN_PADDING4;
    };
    NN_STATIC_ASSERT( sizeof(AciHeader) == 0x40 );

    const size_t NsoHeaderHashSize = 0x20;

    struct NsoHeader
    {
        Bit32           signature;
        Bit32           version;
        NN_PADDING4;
        Bit32           flags;

        uint32_t        textFileOffset;
        uint32_t        textArrangeOffset;
        uint32_t        textSize;
        uint32_t        moduleNameFileOffset;

        uint32_t        rodataFileOffset;
        uint32_t        rodataArrangeOffset;
        uint32_t        rodataSize;
        uint32_t        moduleNameSize;

        uint32_t        dataFileOffset;
        uint32_t        dataArrangeOffset;
        uint32_t        dataSize;
        uint32_t        bssSize;

        Bit8            moduleId[0x20];
        uint32_t        textFileSize;
        uint32_t        rodataFileSize;
        uint32_t        dataFileSize;
        NN_PADDING4;
        Bit8            rserved0x70[0x30];
        Bit8            textHash[NsoHeaderHashSize];
        Bit8            rodataHash[NsoHeaderHashSize];
        Bit8            dataHash[NsoHeaderHashSize];
    };
    NN_STATIC_ASSERT( sizeof(NsoHeader) == 0x100 );

    enum NsoFlags
    {
        NsoFlags_TextCompress   = (1u << 0),
        NsoFlags_RodataCompress = (1u << 1),
        NsoFlags_DataCompress   = (1u << 2),
        NsoFlags_TextHash       = (1u << 3),
        NsoFlags_RodataHash     = (1u << 4),
        NsoFlags_DataHash       = (1u << 5),
    };

}}  // namespace nn::ldr
