﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstring>
#include <nn/nn_Abort.h>
#include <nn/ldr/ldr_Result.h>
#include "ldr_ArgumentStore.h"

namespace nn { namespace ldr {

    const size_t ArgumentStore::ArgumentBufferSize;

    const ncm::ProgramId NotUsed = { 0 };

    int ArgumentStore::FindIndex(ArgumentStore::Entry* pMap, ncm::ProgramId programId) NN_NOEXCEPT
    {
        for( int i = 0; i < ArgumentMapCount; ++i )
        {
            Entry& e = pMap[i];

            if( e.programId == programId )
            {
                return i;
            }
        }

        return -1;
    }

    const ArgumentStore::Entry* ArgumentStore::Get(ncm::ProgramId programId) NN_NOEXCEPT
    {
        Entry* pArgument = NULL;

        // 事前に登録されている引数を探す
        int argumentIndex = FindIndex(m_ArgumentMap, programId);
        if( argumentIndex >= 0 )
        {
            pArgument = &m_ArgumentMap[argumentIndex];
        }

        return pArgument;
    }

    nn::Result ArgumentStore::Set(ncm::ProgramId programId, const void* pArgument, size_t size) NN_NOEXCEPT
    {
        if( size >= ArgumentBufferSize )
        {
            return ResultArgumentOverflow();
        }

        int foundIndex = FindIndex(m_ArgumentMap, programId);
        if( foundIndex < 0 )
        {
            foundIndex = FindIndex(m_ArgumentMap, NotUsed);
        }
        if( foundIndex < 0 )
        {
            return ResultArgumentCountOverflow();
        }

        Entry& e = m_ArgumentMap[foundIndex];

        e.programId = programId;
        e.argumentSize = std::min(ArgumentBufferSize, size);
        std::memcpy(e.argument, pArgument, e.argumentSize);
        return ResultSuccess();
    }

    nn::Result ArgumentStore::Flush() NN_NOEXCEPT
    {
        for( int i = 0; i < ArgumentMapCount; ++i )
        {
            m_ArgumentMap[i].programId = NotUsed;
        }

        return ResultSuccess();
    }

}}  // nn::ldr
