﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/init.h>
#include <nn/fs_Base.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/detail/fs_DfcFileSystemProxyServiceObject.h>
#include <nn/fssrv/fssrv_SaveDataIndexer.h>
#include <nn/fssrv/fssrv_SaveDataIndexerManager.h>
#include <nn/fssrv/fscreator/fssrv_BuiltInStorageCreator.h>
#include <nn/fssrv/fscreator/fssrv_BuiltInStorageFileSystemCreator.h>
#include <nn/fssrv/fscreator/fssrv_FatFileSystemCreator.h>
#include <nn/fssrv/fscreator/fssrv_HostFileSystemCreator.h>
#include <nn/fssrv/fscreator/fssrv_PartitionFileSystemCreator.h>
#include <nn/fssrv/fscreator/fssrv_RomFileSystemCreator.h>
#include <nn/fssrv/fscreator/fssrv_SaveDataFileSystemCreator.h>
#include <nn/fssrv/fscreator/fssrv_SdStorageCreator.h>
#include <nn/fssrv/fscreator/fssrv_StorageOnNcaCreator.h>
#include <nn/fssrv/fscreator/fssrv_SdCardProxyFileSystemCreator.h>
#include <nn/fssrv/fscreator/fssrv_SubDirectoryFileSystemCreator.h>
#include <nn/fssrv/fscreator/fssrv_TargetManagerFileSystemCreator.h>
#include <nn/fssrv/fscreator/fssrv_GameCardStorageCreator.h>
#include <nn/fssrv/fscreator/fssrv_GameCardFileSystemCreator.h>
#include <nn/fssrv/fscreator/fssrv_EncryptedFileSystemCreator.h>
#include <nn/fssrv/fscreator/fssrv_MemoryStorageCreator.h>
#include <nn/fssrv/fssrv_MemoryReport.h>
#include <nn/fssrv/fssrv_MemoryResourceFromExpHeap.h>
#include <nn/fssrv/fssrv_MemoryResourceFromStandardAllocator.h>
#include <nn/fssrv/fssrv_FileSystemProxyServer.h>
#include <nn/fssrv/fssrv_FileSystemProxyServerSessionResourceManager.h>
#include <nn/fssrv/fssrv_IFileSystemCreator.h>
#include <nn/fssrv/fssrv_SaveDataTransferCryptoConfiguration.h>
#include <nn/fssrv/detail/fssrv_FileSystemProxyServiceObject.h>
#include <nn/fssystem/fs_AllocatorUtility.h>
#include <nn/fssystem/fs_AsynchronousAccess.h>
#include <nn/fssystem/fs_ServiceContext.h>
#include <nn/fssystem/buffers/fs_FileSystemBufferManager.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_Thread.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_Optional.h>
#include <nn/spl/spl_Api.h>
#include <nn/spl/spl_Types.h>
#include <nn/svc/svc_Base.h>
#include "fs_CryptoConfiguration.h"

#if NN_BUILD_CONFIG_TOOLCHAIN_CLANG
namespace {

    const int FileSystemProxyServerThreadCount = 5;
    NN_STATIC_ASSERT(nn::fssrv::FileSystemProxyServerActiveSessionCount == FileSystemProxyServerThreadCount);

    const int FileSystemProxyServerThreadStackSize = 32 * 1024;
    const int SaveDataFileSystemCacheCount = 1;

    NN_OS_ALIGNAS_THREAD_STACK char g_FileSystemProxyServerStack[FileSystemProxyServerThreadCount][FileSystemProxyServerThreadStackSize];
    nn::os::ThreadType g_FileSystemProxyServerThread[FileSystemProxyServerThreadCount];

    void FileSystemProxyServerFunction(void* arg) NN_NOEXCEPT
    {
        NN_UNUSED(arg);
        nn::fssystem::ServiceContext context;
        nn::fssystem::RegisterServiceContext(&context);
        nn::fssrv::LoopFileSystemProxyServer();
    }

    const int MaxThreadNameLength = 64;
    char FileSystemProxyServerThreadNames[FileSystemProxyServerThreadCount][MaxThreadNameLength];

    void FileSystemProxyServerMain(bool enableSignedSystemPartitionOnSdCard) NN_NOEXCEPT
    {
        for( auto i = 0; i < FileSystemProxyServerThreadCount; ++i )
        {
            nn::util::SNPrintf(FileSystemProxyServerThreadNames[i], MaxThreadNameLength, "%s%d", NN_SYSTEM_THREAD_NAME(fs, WorkerNormalPriorityAccess), i);

            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&g_FileSystemProxyServerThread[i], &FileSystemProxyServerFunction, nullptr,
                                                                g_FileSystemProxyServerStack[i], FileSystemProxyServerThreadStackSize, NN_SYSTEM_THREAD_PRIORITY(fs, WorkerRealTimeAccess)));
            nn::os::SetThreadNamePointer(&g_FileSystemProxyServerThread[i], FileSystemProxyServerThreadNames[i]);
            nn::os::StartThread(&g_FileSystemProxyServerThread[i]);
        }

        if (!enableSignedSystemPartitionOnSdCard)
        {
            nn::fssrv::InitializeLocationResolverSet();

            // gc 向け settings 対応。
            // この処理は gc 関連のサービス処理待ちを解除するため、
            // サービス受付を一時停止する WaitPcvAndSwitchToPcvClockResetControl() より先に実行する。
            nn::fssrv::WaitSettingsAndPresetGcInternalKeys();
        }

        nn::fssrv::LoopPmEventServer(InvalidateHwAesKey);

        for( auto& thread : g_FileSystemProxyServerThread )
        {
            nn::os::WaitThread(&thread);
            nn::os::DestroyThread(&thread);
        }
        // Todo: 終了処理の実装
    }

    // 製造時期に応じてセーブデータ ver 4 を許すか決定する
    int GetMinimumAcceptableSaveDataVersion()
    {
        nn::Bit64 keyGeneration;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::spl::GetConfig(&keyGeneration, nn::spl::ConfigItem_DeviceUniqueKeyGeneration));

        // この値以下の DeviceUniqueKeyGeneration を返すような製造時期の本体はセーブデータ ver4 が存在しうる
        const int DeviceUniqueKeyGenerationForDeviceIncludingSaveDataVersion4 = 4;

        if (static_cast<uint64_t>(keyGeneration) <= DeviceUniqueKeyGenerationForDeviceIncludingSaveDataVersion4)
        {
            const int MinimumSaveDataVersion = 4;
            return MinimumSaveDataVersion;
        }
        else
        {
            const int MinimumSaveDataVersion = 5;
            return MinimumSaveDataVersion;
        }
    }
}

namespace nn { namespace fssrv { namespace detail {
    // TORIAEZU: 直接参照
    typedef uint32_t SdCardHandle;
    Result GetSdCardHandle(SdCardHandle* pOutValue) NN_NOEXCEPT;
    bool IsSdCardValid(SdCardHandle handle) NN_NOEXCEPT;
}}}

namespace {
    // heap

#ifdef NN_DETAIL_FS_USE_SIGNED_MEMORY_FS
    const int ExpHeapSize = (2 * 1024 + 512) * 1024; // -2MB

    const int BufferPoolSize = 2 * 1024 * 1024; // -4MB

    const auto MaxCacheCount = 1024;
    const auto SizeBlock = 16 * 1024;
    const auto BufferManagerHeapSize = 4 * 1024 * 1024; // -10MB

    const int MemoryUserPartitionSize            = 4 * 1024 * 1024;
    const int MemorySignedSystemPartitionRawSize = 12 * 1024 * 1024;
    const int MemorySystemPartitionSize          = 31 * 1024 * 1024;

    char MemoryUserPartitionBuffer[MemoryUserPartitionSize];
    char MemorySignedSystemPartitionRawBuffer[MemorySignedSystemPartitionRawSize];
    char MemorySystemPartitionBuffer[MemorySystemPartitionSize];

#else
    const int ExpHeapSize = (4 * 1024 + 512) * 1024; // TODO: サイズ削減

    const int BufferPoolSize = 6 * 1024 * 1024; // TODO: メモリ管理改善

    const auto MaxCacheCount = 1024;
    const auto SizeBlock = 16 * 1024;
    const auto BufferManagerHeapSize = 14 * 1024 * 1024;
#endif

    NN_ALIGNAS(4096) char g_ExpHeapStack[ExpHeapSize];
    nn::lmem::HeapHandle g_ExpHeapHandle = nullptr;
    nn::fssrv::PeakCheckableMemoryResourceFromExpHeap g_ExpAllocator(ExpHeapSize);

    void InitializeExpHeap()
    {
        if( g_ExpHeapHandle == nullptr )
        {
            g_ExpHeapHandle = nn::lmem::CreateExpHeap(&g_ExpHeapStack, ExpHeapSize, nn::lmem::CreationOption_DebugFill | nn::lmem::CreationOption_ThreadSafe);
            NN_ABORT_UNLESS(g_ExpHeapHandle);
            g_ExpAllocator.SetHeapHandle(g_ExpHeapHandle);
        }
    }

    void* Allocate(size_t size)
    {
        NN_ABORT_UNLESS(g_ExpHeapHandle);
        auto scopedLock = g_ExpAllocator.GetScopedLock();
        void* buffer = nn::lmem::AllocateFromExpHeap(g_ExpHeapHandle, size);
        g_ExpAllocator.OnAllocate(buffer, size);
        return buffer;
    }

    void Deallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
        NN_ABORT_UNLESS(g_ExpHeapHandle);
        auto scopedLock = g_ExpAllocator.GetScopedLock();
        g_ExpAllocator.OnDeallocate(p, size);
        nn::lmem::FreeToExpHeap(g_ExpHeapHandle, p);
    }

    NN_ALIGNAS(4096) char g_BufferPool[BufferPoolSize];
    nn::mem::StandardAllocator g_BufferAllocator(g_BufferPool, BufferPoolSize);

    const auto PooledThreadCount = 12;
    const auto PooledThreadStackSize = 32 * 1024;
    nn::fssystem::PooledThread g_PooledThreads[PooledThreadCount];
#if 0 // TODO: スタック用のバッファを BufferManager から確保するため静的な確保は見送り。今後調整が必要
    NN_OS_ALIGNAS_THREAD_STACK
        char g_PooledThreadStack[PooledThreadCount * PooledThreadStackSize] = {};
#else
    std::pair<uintptr_t, size_t> g_PooledThreadStack;
#endif

    NN_STATIC_ASSERT(BufferManagerHeapSize % nn::os::MemoryBlockUnitSize == 0);
}
#endif /* NN_BUILD_CONFIG_TOOLCHAIN_CLANG */

uintptr_t g_BufferAddressForMalloc = 0;
extern "C" void nninitStartup()
{
#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX) || defined(NN_BUILD_CONFIG_HARDWARE_JETSONTX2)
    // http://spdlybra.nintendo.co.jp/jira/browse/SIGLO-46575
    // $sdk\Programs\Chris\Sources\Kernel\NX\kern_SystemControl.cpp:186
    // secureSize += 24 * 1024 * 1024; // fs heap size と同じ値にすること
    const size_t MemoryHeapSize = 24 * 1024 * 1024;
    nn::Result result = nn::os::SetMemoryHeapSize(MemoryHeapSize);
    NN_ABORT_UNLESS(result.IsSuccess(), "Cannot set MemoryHeapSize.");

    const size_t BufferSizeForMalloc = 2 * 1024 * 1024;
    result = nn::os::AllocateMemoryBlock( &g_BufferAddressForMalloc, BufferSizeForMalloc );
    NN_ABORT_UNLESS( result.IsSuccess(), "Failed to allocate memory block for InitializeAllocator()." );
    nn::init::InitializeAllocator( reinterpret_cast<void*>(g_BufferAddressForMalloc), BufferSizeForMalloc );
#endif
}

extern "C" void nninitInitializeSdkModule()
{
}

extern "C" void nninitFinalizeSdkModule()
{
}

extern "C" void nnMain()
{
#if NN_BUILD_CONFIG_TOOLCHAIN_CLANG

    using BuiltInStorageCreator           = NN_DETAIL_FS_BUILT_IN_STORAGE_CREATOR_TYPE;
    using BuiltInStorageFileSystemCreator = NN_DETAIL_FS_BUILT_IN_STORAGE_FILESYSTEM_CREATOR_TYPE;


    bool enableSystemPartitionRedirection =
#ifdef NN_DETAIL_FS_REDIRECT_SYSTEM_PARTITION_ID_TO_SAFE_MODE_PARTITION
        true;
#else
        false;
#endif

    bool enableSignedSystemPartitionOnSdCard =
#ifdef NN_DETAIL_FS_USE_SIGNED_SYSTEM_PARTITION_ON_SD_CARD
        true;
#else
        false;
#endif

    bool enablePackage2HashVerification =
#ifdef NN_DETAIL_FS_ENABLE_P2_HASH_VERIFICATION_FOR_SIGNED_SYSTEM_PARTITION
        true;
#else
        false;
#endif


    nn::fssystem::ServiceContext context;
    nn::fssystem::RegisterServiceContext(&context);

    nn::spl::InitializeForFs();
    bool isProd = !(nn::spl::IsDevelopment());
    bool isDebugFunctionEnabled = nn::spl::GetConfigBool(nn::spl::ConfigItem::ConfigItem_IsDevelopmentFunctionEnabled);

    SetUpKekAccessKeys();
    InitializeExpHeap();

    nn::fs::SetAllocator(Allocate, Deallocate);
    nn::fssystem::InitializeAllocator(Allocate, Deallocate);
    nn::fssrv::MemoryResourceFromStandardAllocator allocator(&g_BufferAllocator);

    nn::fssystem::FileSystemBufferManager bufferManager;
    uintptr_t bufferManagerHeapAddress = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::AllocateMemoryBlock(&bufferManagerHeapAddress, BufferManagerHeapSize));
    bufferManager.Initialize(
        MaxCacheCount,
        bufferManagerHeapAddress,
        BufferManagerHeapSize,
        SizeBlock);

    auto bufferManagerMemoryReport = nn::fssrv::MemoryReportCreator::CreateFunctionalMemoryReport(
        [&]() NN_NOEXCEPT -> size_t
        {
            return bufferManager.GetFreeSizePeak();
        },
        [&]() NN_NOEXCEPT -> size_t
        {
            return bufferManager.GetTotalAllocatableSizePeak();
        },
        [&]() NN_NOEXCEPT -> size_t
        {
            return bufferManager.GetRetriedCount();
        },
        [&]() NN_NOEXCEPT
        {
            bufferManager.ClearPeak();
        }
    );
    auto expHeapMemoryReport = nn::fssrv::MemoryReportCreator::CreateFunctionalMemoryReport(
        [&]() NN_NOEXCEPT -> size_t
        {
            return g_ExpAllocator.GetFreeSizePeak();
        },
        [&]() NN_NOEXCEPT -> size_t
        {
            return 0;
        },
        [&]() NN_NOEXCEPT -> size_t
        {
            return 0;
        },
        [&]() NN_NOEXCEPT
        {
            g_ExpAllocator.ClearPeak();
        }
    );
    auto bufferPoolMemoryReport = nn::fssrv::MemoryReportCreator::CreateFunctionalMemoryReport(
        [&]() NN_NOEXCEPT -> size_t
        {
            return allocator.GetFreeSizePeak();
        },
        [&]() NN_NOEXCEPT -> size_t
        {
            return 0;
        },
        [&]() NN_NOEXCEPT -> size_t
        {
            return 0;
        },
        [&]() NN_NOEXCEPT
        {
            allocator.ClearPeak();
        }
    );

    nn::fssrv::MemoryReportCreator memoryReport(&bufferManagerMemoryReport, &expHeapMemoryReport, &bufferPoolMemoryReport);
    nn::fssrv::SetMemoryReportCreator(&memoryReport);

#if 0 // TODO: スタック用のバッファを BufferManager から確保するため静的な確保は見送り。今後調整が必要
    nn::fssystem::ThreadPool threadPool(g_PooledThreads, PooledThreadCount);
    NN_ABORT_UNLESS_RESULT_SUCCESS(threadPool.Initialize(
        g_PooledThreadStack, PooledThreadCount * PooledThreadStackSize));
#else
    g_PooledThreadStack = bufferManager.AllocateBuffer(PooledThreadCount * PooledThreadStackSize);
    nn::fssystem::ThreadPool threadPool(g_PooledThreads, PooledThreadCount);
    NN_ABORT_UNLESS_RESULT_SUCCESS(threadPool.Initialize(
        reinterpret_cast<char*>(g_PooledThreadStack.first),
        PooledThreadCount * PooledThreadStackSize,
        [](char* buffer, size_t bufferSize, void* pArgument) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_EQUAL(reinterpret_cast<uintptr_t>(buffer), g_PooledThreadStack.first);
            NN_UNUSED(buffer);
            NN_UNUSED(bufferSize);
            reinterpret_cast<nn::fssystem::FileSystemBufferManager*>(pArgument)
                ->DeallocateBuffer(g_PooledThreadStack.first, g_PooledThreadStack.second);
        },
        &bufferManager
    ));
#endif
    nn::fssystem::RegisterThreadPool(&threadPool);

    nn::fssrv::fscreator::RomFileSystemCreator romFsCreator(&allocator);
    nn::fssrv::fscreator::PartitionFileSystemCreator partitionFsCreator;
    nn::fssrv::fscreator::SaveDataFileSystemCreator saveDataFsCreator(&allocator, &bufferManager, GenerateDeviceUniqueMacForSaveData, GenerateSeedUniqueMacForSaveData, GenerateRandomForSaveData, GetMinimumAcceptableSaveDataVersion());
    nn::fssrv::fscreator::StorageOnNcaCreator storageOnNcaCreator(&allocator, *GetNcaCryptoConfiguration(isProd), isProd, &bufferManager);
    nn::fssrv::fscreator::FatFileSystemCreator fatFsCreator(&allocator);
    nn::fssrv::fscreator::HostFileSystemCreator hostFsCreator(!isProd);
    nn::fssrv::fscreator::TargetManagerFileSystemCreator targetFsCreator;
    nn::fssrv::fscreator::SubDirectoryFileSystemCreator subDirFsCreator;

    nn::fssrv::fscreator::MemoryStorageCreator memoryStorageCreator;
#ifdef NN_DETAIL_FS_USE_SIGNED_MEMORY_FS
    memoryStorageCreator.RegisterMemory(nn::fssrv::fscreator::IMemoryStorageCreator::MemoryStorageId::UserPartition,            MemoryUserPartitionBuffer,            MemoryUserPartitionSize);
    memoryStorageCreator.RegisterMemory(nn::fssrv::fscreator::IMemoryStorageCreator::MemoryStorageId::SignedSystemPartitionRaw, MemorySignedSystemPartitionRawBuffer, MemorySignedSystemPartitionRawSize);
    memoryStorageCreator.RegisterMemory(nn::fssrv::fscreator::IMemoryStorageCreator::MemoryStorageId::SystemPartition,          MemorySystemPartitionBuffer,          MemorySystemPartitionSize);
#endif


    nn::fssrv::fscreator::BuiltInStorageCreator::Configuration bisCreatorConfig;
    bisCreatorConfig.getKeyFunc = GetBisEncryptionKey;
    bisCreatorConfig.enablePackage2HashVerification   = enablePackage2HashVerification;
    bisCreatorConfig.pMemoryStorageCreator            = &memoryStorageCreator;
    bisCreatorConfig.pSignedSystemPartitionSignKeyPublicModulus = GetSignedSystemPartitionSignKeyPublicModulus(isProd);
#ifdef NN_DETAIL_FS_ENABLE_P2_HASH_VERIFICATION_FOR_SIGNED_SYSTEM_PARTITION
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::spl::GetPackage2Hash(bisCreatorConfig.package2Hash, sizeof(bisCreatorConfig.package2Hash)));
#endif

    nn::fssrv::fscreator::SdStorageCreator sdStorageCreator;
    nn::fssrv::fscreator::SdCardProxyFileSystemCreator sdCardFsCreator(&sdStorageCreator, &fatFsCreator, nn::fssrv::GetFatFileSystemCacheManager());
    nn::fssrv::fscreator::GameCardStorageCreator gcStorageCreator(&allocator);
    nn::fssrv::fscreator::GameCardFileSystemCreator gcFsCreator(&allocator, &gcStorageCreator);

    BuiltInStorageCreator builtInStorageCreator(bisCreatorConfig);
    BuiltInStorageFileSystemCreator bisFsCreator(&builtInStorageCreator, &memoryStorageCreator, &fatFsCreator, &sdCardFsCreator, nn::fssrv::GetFatFileSystemCacheManager());

    nn::fssrv::fscreator::EncryptedFileSystemCreator::Configuration encryptedFsCreatorConfig;
    encryptedFsCreatorConfig.pGetKeyFunction         = GetSdCardEncryptionKey;
    encryptedFsCreatorConfig.pGenerateRandomFunction = GenerateRandomForSdCardEncryption;
    nn::fssrv::fscreator::EncryptedFileSystemCreator encryptedFsCreator(encryptedFsCreatorConfig);

    nn::fssrv::fscreator::FileSystemCreatorInterfaces fsCreatorInterfaces =
    {
        &romFsCreator,
        &partitionFsCreator,
        &storageOnNcaCreator,
        &fatFsCreator,
        &hostFsCreator,
        &targetFsCreator,
        &subDirFsCreator,
        &builtInStorageCreator,
        &sdStorageCreator,
        &saveDataFsCreator,
        &gcStorageCreator,
        &gcFsCreator,
        &encryptedFsCreator,
        &memoryStorageCreator,
        &bisFsCreator,
        &sdCardFsCreator,
    };

    class SdHandleManager : public nn::fssrv::IDeviceHandleManager
    {
    public:
        virtual nn::Result GetHandle(nn::fssrv::DeviceHandle* pOutValue) NN_NOEXCEPT NN_OVERRIDE
        {
            return nn::fssrv::detail::GetSdCardHandle(pOutValue);
        }
        virtual bool IsValid(nn::fssrv::DeviceHandle handle) NN_NOEXCEPT NN_OVERRIDE
        {
            return nn::fssrv::detail::IsSdCardValid(handle);
        }
    };
    SdHandleManager sdHandleManager;

    auto* pSaveDataTransferCryptoConfiguration = GetSaveDataTransferCryptoConfiguration(isProd);

    // http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=109326482
    // システムプロセス、システムデータ、本体機能は、本体保存メモリーのみに配置される想定で速度エミュレーションを行う
    nn::fssrv::InternalProgramIdRangeForSpeedEmulation internalProgramIdRangeForSpeedEmulation = { 0x0100000000000000ULL, 0x0100000000001FFFULL };

    nn::fssrv::InitializeFileSystemProxy(&fsCreatorInterfaces, &bufferManager, isDebugFunctionEnabled, pSaveDataTransferCryptoConfiguration, &internalProgramIdRangeForSpeedEmulation, GenerateRandomForSaveData);

    auto pFileSystemProxy = nn::fssrv::detail::GetFileSystemProxyServiceObject();
    {
        nn::Bit64 currentProcessId = 0;
        nn::svc::GetProcessId(&currentProcessId, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS);
        pFileSystemProxy->SetCurrentProcess(currentProcessId);
    }
    nn::fs::detail::InitializeDfcFileSystemProxyServiceObject(pFileSystemProxy);

    nn::fs::SetEnabledAutoAbort(false);

    nn::fssrv::InitializeSaveDataFileSystemCacheManager(SaveDataFileSystemCacheCount);

    nn::fssrv::InitializeSaveDataIndexerManager(nn::fssrv::IndexerSaveDataId, &g_ExpAllocator, &sdHandleManager, enableSystemPartitionRedirection);

    nn::fssrv::InitializeFileSystemProxyServer(FileSystemProxyServerThreadCount);


    // SignedSystemPartition on SdCard の場合はまだ System パーティションにアクセスできない
    if( !enableSignedSystemPartitionOnSdCard )
    {

        {
            auto result = pFileSystemProxy.GetImpl().CleanUpTemporaryStorage();
            if (result.IsFailure())
            {
                NN_SDK_LOG("[fs] Failed to clean up temporary storage (%x)\n", result.GetInnerValueForDebug());
            }
        }

        {
            auto result = pFileSystemProxy.GetImpl().CleanUpSaveData();
            if( result.IsFailure() )
            {
                NN_SDK_LOG("[fs] Failed to clean up save data (%x)\n", result.GetInnerValueForDebug());
            }
        }

        {
            auto result = pFileSystemProxy.GetImpl().CompleteSaveDataExtension();
            if( result.IsFailure() )
            {
                NN_SDK_LOG("[fs] Failed to complete save data extension (%x)\n", result.GetInnerValueForDebug());
            }
        }

        {
            auto result = pFileSystemProxy.GetImpl().FixSaveData();
            if (result.IsFailure())
            {
                NN_SDK_LOG("[fs] Failed to fix save data (%x)\n", result.GetInnerValueForDebug());
            }
        }
    }

#if defined(NN_DETAIL_FS_ENABLE_EXFAT)
    NN_SDK_LOG("[fs] NN_DETAIL_FS_ENABLE_EXFAT is defined\n");
#endif

    FileSystemProxyServerMain(enableSignedSystemPartitionOnSdCard);

    nn::spl::Finalize();
#endif /* NN_BUILD_CONFIG_TOOLCHAIN_CLANG */
} // NOLINT(impl/function_size)
