﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <functional>               // for std::function
#include <nn/wlan/wlan_SocketApi.h> // for MacAddress
#include <nn/os/os_SharedMemory.h>

// BSD headers
#include <private/thread.h>         // for NetworkThread
#include <siglo/device.h>           // ETHERTYPE_*

#include <nn/wlan/wlan_Cbuf.h>

// set to true to enable WLAN interface,
// also make sure you have "Eris/.../wlan.nca" built/loaded
#define ENABLE_WLAN_INTERFACE true
#define ENABLE_SHARED_MEMORY  true

namespace nn   {
namespace eth  {
namespace wlan {

const int    PacketBufferSize = 2048;
const int    SharedBufferSize = (512 * 1024);
const size_t ThreadStackSize  = (16 * 1024);

class WorkerThread
{

private:
    NN_OS_ALIGNAS_THREAD_STACK uint8_t  m_Stack[ThreadStackSize];
    NetworkThread                       m_Thread;
    std::function<void(void)>           m_Function;

    void Thread()
    NN_NOEXCEPT
    {
        m_Function();
    }

    static void ThreadEntry(void* argument)
    NN_NOEXCEPT
    {
        WorkerThread* p = (WorkerThread*)argument;
        p->Thread();
    }

public:

    nn::Result Initialize(std::function<void(void)>function, int priority, const char* threadName)
    NN_NOEXCEPT
    {
        m_Function = function;
        CreateNetworkThread(&m_Thread, ThreadEntry, this, m_Stack, sizeof(m_Stack), priority, -1);
        SetNetworkThreadNamePointer(&m_Thread, threadName);
        StartNetworkThread(&m_Thread);
        return ResultSuccess();
    }

    void Finalize()
    NN_NOEXCEPT
    {
        WaitNetworkThread(&m_Thread);
        DestroyNetworkThread(&m_Thread);
    }
};

class Wlan
{

private:
    bool                 m_Initialized;

    // Worker threads
    WorkerThread         m_ReceiveThread;
    WorkerThread         m_SendThread;
    nn::os::EventType    m_ReceiveEvent;
    nn::os::EventType    m_SendEvent;
    volatile bool        m_Running;

    void ReceiveFunction() NN_NOEXCEPT;
    void SendFunction() NN_NOEXCEPT;

#if ENABLE_SHARED_MEMORY
    nn::cbuf::Cbuf*                    m_pTxCbuf;
    nn::cbuf::Cbuf*                    m_pRxCbuf;
    nn::os::SharedMemoryType           m_ServerSharedMemory;
    uint8_t*                           m_SharedBuffer;
#else
    uint8_t              m_ReceiveBuffer[PacketBufferSize];
    uint8_t              m_SendBuffer[PacketBufferSize];
#endif

    // WLAN process related
    const uint16_t       PacketTypes[2] = {ETHERTYPE_IP, ETHERTYPE_ARP};
    const uint32_t       NumberOfBuffers = 90;
    uint32_t             m_RxId;
    nn::wlan::MacAddress m_MacAddress;
    uint8_t*             m_pMacAddressData;
    nn::os::SystemEvent  m_TxPacketEvent;
    nn::os::SystemEvent  m_RxPacketEvent;
    nn::os::SystemEvent  m_TxCbufEvent;
    nn::os::SystemEvent  m_RxCbufEvent;

    nn::Result ReadMacAddress() NN_NOEXCEPT;

    // BSD related
    const char*          Name = "wl";
    const uint8_t        Instance = 0;
    struct ifnet*        m_pIfnet;
    uint8_t              m_LinkState;
    bool                 m_AttachedToSocketProcess;

    nn::Result GetFrameFromStack(uint32_t* pLenthOut, uint8_t* pData, uint32_t length) NN_NOEXCEPT;
    nn::Result SendFrameToStack(uint8_t* pData, uint32_t length) NN_NOEXCEPT;

    void Start() NN_NOEXCEPT;
    void Init()  NN_NOEXCEPT;
    int  Ioctl(uint32_t command, char* pData) NN_NOEXCEPT;
    void SetLinkUp() NN_NOEXCEPT;
    void SetLinkDown() NN_NOEXCEPT;

    static void Start(struct ifnet *pIfnet)
    NN_NOEXCEPT
    {
        Wlan* p = (Wlan*)(pIfnet->if_softc);
        NN_ABORT_UNLESS(p != nullptr);
        p->Start();
    }
    static int Ioctl(struct ifnet *pIfnet, unsigned long command, char* pData)
    NN_NOEXCEPT
    {
        Wlan* p = (Wlan*)(pIfnet->if_softc);
        NN_ABORT_UNLESS(p != nullptr);
        return p->Ioctl(command, pData);
    }
    static void Init(void* argument)
    NN_NOEXCEPT
    {
        Wlan* p = (Wlan*)(argument);
        NN_ABORT_UNLESS(p != nullptr);
        p->Init();
    }

public:
    Wlan()
    NN_NOEXCEPT :
        m_Initialized(false),
        m_pMacAddressData(nullptr),
        m_TxPacketEvent(nn::os::EventClearMode_AutoClear, true),
        m_RxPacketEvent(nn::os::EventClearMode_AutoClear, true),
        m_TxCbufEvent(nn::os::EventClearMode_AutoClear, true),
        m_RxCbufEvent(nn::os::EventClearMode_AutoClear, true),
        m_LinkState(LINK_STATE_DOWN),
        m_AttachedToSocketProcess(false)
    {
    }

    void ControlThread() NN_NOEXCEPT;
    nn::Result Initialize() NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;

};

void InitializeWlan() NN_NOEXCEPT;
void FinalizeWlan()   NN_NOEXCEPT;

}}}
