﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstring>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SystemThreadDefinition.h>

#include <nn/os.h>
#include <nn/tsc/tsc_Result.h>
#include <nn/tsc/tsc_Result.private.h>
#include <nn/tsc/tsc_Types.h>
#include "tsc_Util.h"

namespace nn { namespace tsc { namespace detail {

// ------------------------------------------------------------------------------------------------
//
// TscWorker class
//
// ------------------------------------------------------------------------------------------------

// ------------------------------------------------------------------------------------------------
// TscWorker - private interface
// ------------------------------------------------------------------------------------------------
void nn::tsc::detail::TscWorker::ExecuteWorker(void* pTscWorker)
{
    nn::Result* pResult = &(reinterpret_cast<TscWorker*>(pTscWorker)->m_Result);

    NN_DETAIL_TSC_PRINT_DBG("[TcpWorker]Start operation.\n");
    *pResult = reinterpret_cast<TscWorker*>(pTscWorker)->m_pOperation();
    NN_DETAIL_TSC_PRINT_DBG("[TcpWorker]Operation returned :result(Module:%d Desc:%d).\n",
                            pResult->GetModule(), pResult->GetDescription());

    reinterpret_cast<TscWorker*>(pTscWorker)->m_IsOperationInprogress = false;

    // Callback
    if(reinterpret_cast<TscWorker*>(pTscWorker)->m_pDoneCallback)
    {
        NN_DETAIL_TSC_PRINT_DBG("[TcpWorker]Calling callback.\n");
        reinterpret_cast<TscWorker*>(pTscWorker)->m_pDoneCallback();
    }
}

// ------------------------------------------------------------------------------------------------
// TscWorker - public constructor
// ------------------------------------------------------------------------------------------------
nn::tsc::detail::TscWorker::TscWorker() : m_pOperation(nullptr),
                                          m_pDoneCallback(nullptr),
                                          m_IsOperationInprogress(false),
                                          m_IsThreadInit(false),
                                          m_IsInitialized(false),
                                          m_Result(nn::ResultSuccess())
{}

// ------------------------------------------------------------------------------------------------
// TscWorker - public interface
// ------------------------------------------------------------------------------------------------
nn::Result nn::tsc::detail::TscWorker::Initialize(OperationFunction    pInOpFunction,
                                                  DoneCallbackFunction pInDoneCallback)
{
    if(m_IsInitialized)
    {
        return ResultInternalAlreadyInitialized();
    }

    NN_SDK_REQUIRES_NOT_NULL(pInOpFunction);

    m_pOperation = pInOpFunction;

    if(pInDoneCallback)
    {
        m_pDoneCallback = pInDoneCallback;
    }
    else
    {
        m_pDoneCallback = nullptr;
    }

    m_IsOperationInprogress = false;
    m_IsThreadInit = false;
    m_Result = nn::ResultSuccess();
    m_IsInitialized = true;

    return nn::ResultSuccess();
}

nn::Result nn::tsc::detail::TscWorker::Start()
{
    if(m_IsOperationInprogress)
    {
        NN_DETAIL_TSC_PRINT_DBG("[TcpWorker][%s]Operation in progress.\n", __FUNCTION__);
        return ResultInternalInProgress();
    }

    if(m_IsThreadInit)
    {
        // Don't need to wait on thread because we know m_IsOperationInprogress is false
        nn::os::DestroyThread(&m_WorkerThread);
        m_IsThreadInit = false;
    }

    nn::Result result = nn::ResultSuccess();
    result = nn::os::CreateThread(&m_WorkerThread,
                                  nn::tsc::detail::TscWorker::ExecuteWorker,
                                  this,
                                  m_WorkerThreadStack,
                                  nn::tsc::detail::TscWorker::s_WorkerThreadStackSize,
                                  NN_SYSTEM_THREAD_PRIORITY(tsc, Worker) );
    nn::os::SetThreadNamePointer(&m_WorkerThread, NN_SYSTEM_THREAD_NAME(tsc, Worker));
    NN_DETAIL_TSC_RETURN_ON_ERROR(result);

    m_IsThreadInit = true;

    m_IsOperationInprogress = true;
    nn::os::StartThread(&m_WorkerThread);

    return result;
}

nn::Result nn::tsc::detail::TscWorker::Finalize()
{
    if(m_IsOperationInprogress)
    {
        NN_DETAIL_TSC_PRINT_DBG("[TcpWorker][%s]Operation in progress.\n", __FUNCTION__);
        return ResultInternalInProgress();
    }

    m_pOperation    = nullptr;
    m_pDoneCallback = nullptr;

    if(m_IsThreadInit)
    {
        nn::os::WaitThread(&m_WorkerThread);
        nn::os::DestroyThread(&m_WorkerThread);
        m_IsThreadInit = false;
    }

    m_IsInitialized = false;

    return nn::ResultSuccess();
}

nn::Result nn::tsc::detail::TscWorker::GetResult(nn::Result* pOutResult)
{
    NN_SDK_REQUIRES_NOT_NULL(pOutResult);

    if(m_IsOperationInprogress)
    {
        NN_DETAIL_TSC_PRINT_DBG("[TcpWorker][%s]Operation in progress.\n", __FUNCTION__);
        return ResultInternalInProgress();
    }

    NN_DETAIL_TSC_PRINT_DBG("[TcpWorker][%s] Module:%d Desc:%d.\n", __FUNCTION__,
                            m_Result.GetModule(), m_Result.GetDescription());

    *pOutResult = m_Result;

    return nn::ResultSuccess();
}

}}}
