﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "tma_private.h"
#include <nn/tma/tma_FileIO.h>
#include <nn/fs/fs_Directory.h>
#include <nn/nn_SdkLog.h>
#include <nn/fs/fs_Result.h>
#include <nn/util/util_StringUtil.h>
#include "tma_FileManagerByHipc.h"

namespace {

    nn::sf::SharedPointer<nn::tma::IFileManager> g_Manager;

}

//==============================================================================
namespace nn { namespace tma { namespace file_io {
//==============================================================================

//==============================================================================

void Initialize()
{
    NN_SDK_ASSERT(!g_Manager);
    g_Manager = nn::tma::CreateFileManagerByHipc();
}

//==============================================================================

void Finalize()
{
    g_Manager = nullptr;
    nn::tma::DeleteFileManagerByHipc();
}

//==============================================================================

nn::Result OpenFile( FileHandle* pOutHandle, const char* pathName, uint32_t openMode )
{
    NN_SDK_ASSERT(g_Manager);

    nn::tma::Path path;
    nn::util::Strlcpy( path.str, pathName, sizeof(Path::str) );

    nn::sf::SharedPointer<IFileAccessor> handle;
    NN_RESULT_DO( g_Manager->OpenFile( &handle, path, openMode ) );
    *pOutHandle = handle.Detach();

    NN_RESULT_SUCCESS;
}

//==============================================================================

nn::Result GetFileSize( int64_t* pOutValue, FileHandle handle )
{
    NN_SDK_ASSERT(handle);
    return static_cast<IFileAccessor*>(handle)->GetFileSize( nn::sf::Out<int64_t>(pOutValue) );
}

//==============================================================================

nn::Result SetFileSize( FileHandle handle, int64_t Size )
{
    NN_SDK_ASSERT(handle);
    return static_cast<IFileAccessor*>(handle)->SetFileSize( Size );
}

//==============================================================================

nn::Result GetFileTimeStamp( uint64_t* pCreateTime, uint64_t* pAccessTime, uint64_t* pModifyTime, const char* pathName )
{
    NN_SDK_ASSERT( pathName );

    Path path;
    nn::util::Strlcpy( path.str, pathName, sizeof(Path::str) );
    uint64_t CreateTime, AccessTime, ModifyTime;
    nn::Result Res = g_Manager->GetFileTimeStamp( nn::sf::Out<uint64_t>(&CreateTime), nn::sf::Out<uint64_t>(&AccessTime), nn::sf::Out<uint64_t>(&ModifyTime), path );

    if( pCreateTime )
    {
        *pCreateTime = CreateTime;
    }
    if( pAccessTime )
    {
        *pAccessTime = AccessTime;
    }
    if( pModifyTime )
    {
        *pModifyTime = ModifyTime;
    }

    return Res;
}

//==============================================================================
// fs signature
nn::Result ReadFile( size_t* pOutValue, FileHandle handle, int64_t ReadOffset, void* pWriteToBuffer, size_t ReadSize, const nn::fs::ReadOption& option )
{
    NN_SDK_ASSERT(handle);

    int64_t out;
    NN_RESULT_DO( static_cast<IFileAccessor*>(handle)->ReadFile( nn::sf::Out<int64_t>(&out), ReadOffset, nn::sf::OutBuffer(reinterpret_cast<char*>(pWriteToBuffer), ReadSize), option ) );
    *pOutValue = out;

    NN_RESULT_SUCCESS;
}

//==============================================================================
//nn::fs signature
nn::Result WriteFile( FileHandle handle, int64_t WriteOffset, const void* pWriteBuffer, int64_t WriteSize, const nn::fs::WriteOption& Options )
{
    NN_SDK_ASSERT(handle);
    return static_cast<IFileAccessor*>(handle)->WriteFile( WriteOffset, nn::sf::InBuffer(reinterpret_cast<const char*>(pWriteBuffer), WriteSize), Options );
}

//==============================================================================

nn::Result FlushFile( FileHandle handle )
{
    NN_SDK_ASSERT(handle);
    return static_cast<IFileAccessor*>(handle)->FlushFile();
}

//==============================================================================

nn::Result SetPriorityForFile( FileHandle handle, int32_t Priority )
{
    NN_SDK_ASSERT(handle);
    return static_cast<IFileAccessor*>(handle)->SetPriorityForFile( Priority );
}

//==============================================================================

nn::Result GetPriorityForFile( int32_t* pOutValue, FileHandle handle )
{
    NN_SDK_ASSERT(handle);
    return static_cast<IFileAccessor*>(handle)->GetPriorityForFile( nn::sf::Out<int32_t>(pOutValue) );
}

//==============================================================================

void CloseFile( FileHandle handle )
{
    NN_SDK_ASSERT(handle);
    nn::sf::ReleaseSharedObject(static_cast<IFileAccessor*>(handle));
}

//==============================================================================

nn::Result FileExists( bool* pExists, const char* pathName )
{
    NN_SDK_ASSERT(g_Manager);

    Path path;
    nn::util::Strlcpy( path.str, pathName, sizeof(Path::str) );

    return g_Manager->FileExists( nn::sf::Out<bool>(pExists), path );
}

//==============================================================================

nn::Result DeleteFile( const char* pFileName )
{
    NN_SDK_ASSERT(g_Manager);

    Path path;
    nn::util::Strlcpy( path.str, pFileName, sizeof(Path::str) );

    return g_Manager->DeleteFile( path );
}

//==============================================================================

nn::Result RenameFile( const char* pFromName, const char* pToName )
{
    NN_SDK_ASSERT(g_Manager);

    Path FromPath;
    nn::util::Strlcpy( FromPath.str, pFromName, sizeof(Path::str) );
    Path ToPath;
    nn::util::Strlcpy( ToPath.str, pToName, sizeof(Path::str) );

    return g_Manager->RenameFile( FromPath, ToPath );
}

//==============================================================================

nn::Result GetIOType( nn::fs::DirectoryEntryType* pOutType, const char* pPath )
{
    NN_SDK_ASSERT(g_Manager);

    int32_t out;
    Path path;
    nn::util::Strlcpy( path.str, pPath, sizeof(Path::str) );

    NN_RESULT_DO( g_Manager->GetIOType( nn::sf::Out<int32_t>(&out), path ) );
    *pOutType = static_cast<nn::fs::DirectoryEntryType>(out);

    NN_RESULT_SUCCESS;
}

//==============================================================================

nn::Result OpenDirectory( DirectoryHandle* pOutHandle, const char* pathName, fs::OpenDirectoryMode OpenMode )
{
    NN_SDK_ASSERT(g_Manager);

    nn::tma::Path path;
    nn::util::Strlcpy( path.str, pathName, sizeof(Path::str) );

    nn::sf::SharedPointer<IDirectoryAccessor> handle;
    NN_RESULT_DO( g_Manager->OpenDirectory( &handle, path, OpenMode ) );
    *pOutHandle = handle.Detach();

    NN_RESULT_SUCCESS;
}

//==============================================================================

nn::Result GetEntryCount( int64_t* pOutValue, DirectoryHandle handle )
{
    NN_SDK_ASSERT( handle );
    return static_cast<IDirectoryAccessor*>(handle)->GetEntryCount( nn::sf::Out<int64_t>(pOutValue) );
}

//==============================================================================

nn::Result ReadDirectory( int64_t* pOutValue, nn::fs::DirectoryEntry pEntries[], DirectoryHandle handle, int64_t NumEntries )
{
    NN_SDK_ASSERT( handle );
    return static_cast<IDirectoryAccessor*>(handle)->ReadDirectory( nn::sf::Out<int64_t>(pOutValue), nn::sf::OutArray<nn::fs::DirectoryEntry>( pEntries, NumEntries ) );
}

//==============================================================================

nn::Result SetPriorityForDirectory( DirectoryHandle handle, int32_t Priority )
{
    NN_SDK_ASSERT( handle );
    return static_cast<IDirectoryAccessor*>(handle)->SetPriorityForDirectory( Priority );
}

//==============================================================================

nn::Result GetPriorityForDirectory( int32_t* pOutValue, DirectoryHandle handle )
{
    NN_SDK_ASSERT( handle );
    return static_cast<IDirectoryAccessor*>(handle)->GetPriorityForDirectory( nn::sf::Out<int32_t>(pOutValue) );
}

//==============================================================================

void CloseDirectory( DirectoryHandle handle )
{
    NN_SDK_ASSERT( handle );
    nn::sf::ReleaseSharedObject(static_cast<IDirectoryAccessor*>(handle));
}

//==============================================================================

nn::Result DirectoryExists( bool* pOutValue, const char* pathName )
{
    NN_SDK_ASSERT(g_Manager);

    Path path;
    nn::util::Strlcpy( path.str, pathName, sizeof(Path::str) );

    return g_Manager->DirectoryExists( nn::sf::Out<bool>(pOutValue), path );
}

//==============================================================================

nn::Result CreateDirectory( const char* pPathName )
{
    NN_SDK_ASSERT(g_Manager);

    Path path;
    nn::util::Strlcpy( path.str, pPathName, sizeof(Path::str) );

    return g_Manager->CreateDirectory( path );
}

//==============================================================================

nn::Result DeleteDirectory( const char* pPathName, bool Recursively )
{
    NN_SDK_ASSERT(g_Manager);

    Path path;
    nn::util::Strlcpy( path.str, pPathName, sizeof(Path::str) );

    return g_Manager->DeleteDirectory( path, Recursively );
}

//==============================================================================

nn::Result RenameDirectory( const char* pFromName, const char* pToName )
{
    NN_SDK_ASSERT(g_Manager);

    Path FromPath;
    nn::util::Strlcpy( FromPath.str, pFromName, sizeof(Path::str) );
    Path ToPath;
    nn::util::Strlcpy( ToPath.str, pToName, sizeof(Path::str) );

    return g_Manager->RenameDirectory( FromPath, ToPath );
}

//==============================================================================

nn::Result CreateFile( const char* pPath, int64_t size )
{
    NN_SDK_ASSERT(g_Manager);

    Path path;
    nn::util::Strlcpy( path.str, pPath, sizeof(Path::str) );

    return g_Manager->CreateFile( path, size );
}

//==============================================================================
}}} // namespace
//==============================================================================
