﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/socket/socket_Api.h>
#include <nn/nn_Assert.h>
#include <nn/nn_SdkLog.h>
#include <algorithm>
#include <functional>
#include "resolver_AutoContext.h"
#include <nn/socket/socket_TypesPrivate.h>
#include "resolver_Cancel.h"

namespace nn { namespace socket { namespace resolver {

/**
 * @brief The Resolver Auto Context is class that lives on the
 * stack for the duration of a resolver call. It contains information
 * relevant to the current call such as the calling process, the
 * resolver cancel handle, as well as the Resolver Options.
 */
class ResolverAutoContext
{
public:
    /**
     * @brief ctor
     */
    ResolverAutoContext();

    /**
     * Initialize
     */
    bool Initialize(const nn::Bit64 & processID,
                    int & cancelHandleIDInOut,
                    const struct ResolverOption* options,
                    uint32_t optionsCount);

    /**
     * @brief dtor
     */
    ~ResolverAutoContext();

    /**
     * @brief initialization succeed?
     */
    const bool & DidInitializationSucceed() const;

    /**
     * @brief get the ResolverAutoContext on the current stack
     *
     * @detail
     * The AutoContext pointer is stored in the bionic thread local
     * structure state. The pointer is updated each time that you
     * make a service framework call. If for some bizarre reason
     * something bad happens you cannot get the pointer from TLS then
     * this call will assert in Debug and Develop builds; on the other
     * hand if the same thing happens in a release build then the call
     * returns a static reference so that the process does not crash.
     * The resolver call should probably continue to run but in a
     * severely degraded mode.
     */
    static ResolverAutoContext& CurrentContext();

    /**
     * @brief The process ID
     */
    const Bit64 & GetCallingProcessID() const;

    /**
     * @brief the cancel handle ID
     */
    const int & GetCancelHandleID() const;

    /**
     * @brief This function returns a pointer to the resolver
     * options passed in from the caller. It can be NULL. If it its
     * not NULL then @ref ResolverOptionsCount returns the number of
     * @ref ResolverOption structures are present at the pointer.
     */
    const ResolverOption* GetOptionsArray() const;

    /**
     * @brief This returns zero or the number of @ref ResolverOption
     * structures in the @ref array pointed to by @ref Options
     */
    size_t GetOptionsCount() const;

    /**
     * @brief check to see whether or not the slot is cancelled
     * @param wasExpired: it was cancelled because it was expired
     */
    bool IsSlotCancelledOrExpired(bool& wasExpired);

private:
    Bit64 m_ProcessID;

    int m_HandleID;

    bool m_DidSucceed;

    OperationState m_CurrentOperationState;

    const struct ResolverOption* m_Options;

    size_t m_OptionsCount;
};

}}}; //nn::socket::resolver
