﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#if !defined(NN_BUILD_CONFIG_OS_WIN32)
// Forward declare IPv6 structures
// note: peripheral support for ipv6 is disabled, but it is available
// if deemed necessary in the future
extern "C"
{
struct in6_addr
{
    union {
        uint8_t __s6_addr[16];
        uint16_t __s6_addr16[8];
        uint32_t __s6_addr32[4];
    } __in6_union;
};

struct sockaddr_in6
{
    sa_family_t sin6_family;
    in_port_t sin6_port;
    uint32_t sin6_flowinfo;
    struct in6_addr sin6_addr;
    uint32_t sin6_scope_id;
};
};
#endif


#include "serializer.h"
#include <nn/socket/socket_TypesPrivate.h>

namespace nn { namespace socket { namespace resolver { namespace serializer {


// ResolverOption SizeOf: returns bytes for a single serialized option
template <>
size_t DNSSerializer::SizeOf(const ResolverOption& optionIn);

// ResolverOption Array SizeOf: returns bytes for serialized buffer
template <>
size_t DNSSerializer::SizeOf(const ResolverOption* pOptions, size_t count);

// ResolverOption& ToBuffer: serialize a single resolver options to a buffer
template <>
ssize_t DNSSerializer::ToBuffer(uint8_t* const        pBufferOut,
                                size_t                sizeOfBuffer,
                                const ResolverOption& optionIn);

// ResolverOption& FromBuffer: create a single resolver option from a buffer
// unlike other structures the ResolverOption is not a class and therefore
// pointer values are simply copied and the underlying buffer retains pointer bytes
template <>
ssize_t DNSSerializer::FromBuffer(ResolverOption& optionOut,
                                  const uint8_t*  pBufferIn,
                                  size_t          bufferSize);


// ResolverOption* Array ToBuffer: serialize a collection of resolver options to a buffer
template <>
ssize_t DNSSerializer::ToBuffer(uint8_t * const              pBufferOut,
                                size_t                       sizeOfBuffer,
                                const ResolverOption * const pOptionIn,
                                size_t                       optionCount);

template <>
ssize_t DNSSerializer::FromBuffer(ResolverOption * const pArray,
                                  size_t                 arraySize,
                                  const uint8_t * const  pBufferIn,
                                  size_t                 bufferSize,
                                  size_t                 optionCount);

// uint16_t SizeOf
template <>
size_t DNSSerializer::SizeOf(const uint16_t& integer16In);

// uint16_t ToBuffer
template <>
ssize_t DNSSerializer::ToBuffer(uint8_t* const  pBufferOut,
                                size_t          sizeOfBuffer,
                                const uint16_t& integer16In);

// uint32_t FromBuffer
template <>
ssize_t DNSSerializer::FromBuffer(uint32_t&      integer16Out,
                                  const uint8_t* pBufferIn,
                                  size_t         bufferSize);
// uint32_t SizeOf
template <>
size_t DNSSerializer::SizeOf(const uint16_t& integer32In);

// uint32_t ToBuffer
template <>
ssize_t DNSSerializer::ToBuffer(uint8_t* const  pBufferOut,
                                size_t          sizeOfBuffer,
                                const uint32_t& integerIn);

// uint32_t FromBuffer
template <>
ssize_t DNSSerializer::FromBuffer(uint32_t&      integerOut,
                                  const uint8_t* pBufferIn,
                                  size_t         bufferSize);
// char* SizeOf
template <>
size_t DNSSerializer::SizeOf(const char* pString);

// char* (c-string) ToBuffer; includes null terminator
template <>
ssize_t DNSSerializer::ToBuffer(uint8_t* const pBufferOut,
                                size_t         sizeOfBuffer,
                                char*          pString);

// char* (c-string) FromBuffer
template <>
ssize_t DNSSerializer::FromBuffer(char*&         pOut,
                                  const uint8_t* pBufferIn,
                                  size_t         bufferSize);


// char* (c-string) ToBuffer
template <>
ssize_t DNSSerializer::ToBuffer(uint8_t* const pBufferOut,
                                size_t         sizeOfBuffer,
                                uint8_t*       pString);

// char* (c-string) FromBuffer
template <>
ssize_t DNSSerializer::FromBuffer(uint8_t*&      pOut,
                                  const uint8_t* pBufferIn,
                                  size_t         bufferSize);

// char** SizeOf; includes leading length parameter
template <>
size_t DNSSerializer::SizeOf(const char** ppString, uint32_t & countOut);

// char** (c-string array) ToBuffer
template <>
ssize_t DNSSerializer::ToBuffer(uint8_t* const pBufferOut,
                                size_t         sizeOfBuffer,
                                char**         ppStringArray);

// char** (c-string array) FromBuffer
template <>
ssize_t DNSSerializer::FromBuffer(char**&        ppOut,
                                  const uint8_t* pBufferIn,
                                  size_t         bufferSize);

// struct in_addr SizeOf
template <>
size_t DNSSerializer::SizeOf(const struct in_addr& inaddr4);

// struct in_addr** SizeOf; includes leading length parameter
template <>
size_t DNSSerializer::SizeOf(const struct in_addr** ppInaddr, uint32_t & countOut);

// in_addr** ToBuffer
template <>
ssize_t DNSSerializer::ToBuffer(uint8_t* const   pBufferOut,
                                size_t           sizeOfBuffer,
                                struct in_addr** ppInAddrArray);

// in_addr** FromBuffer
template <>
ssize_t DNSSerializer::FromBuffer(struct in_addr**& ppOut,
                                  const uint8_t*    pBufferIn,
                                  size_t            bufferSize);

// sockaddr_in& ToBuffer
template <>
ssize_t DNSSerializer::ToBuffer(uint8_t* const            pBufferOut,
                                size_t                    sizeOfBuffer,
                                const struct sockaddr_in& sockaddr4In);
// struct sockaddr_in SizeOf
template <>
size_t DNSSerializer::SizeOf(const struct sockaddr_in& inaddr4);

// sockaddr_in& FromBuffer
template <>
ssize_t DNSSerializer::FromBuffer(struct sockaddr_in& sockaddrOut,
                                  const uint8_t*      pBufferIn,
                                  size_t              bufferSize);

// struct sockaddr_in6 SizeOf
template <>
size_t DNSSerializer::SizeOf(const struct sockaddr_in6& inaddr6);

// serialize a struct sockaddr_in6 (ipv6); to the buffer
template <>
ssize_t DNSSerializer::ToBuffer(uint8_t* const             pBufferOut,
                                size_t                     sizeOfBuffer,
                                const struct sockaddr_in6& sockaddr6In);

// sockaddr_in6& FromBuffer
template <>
ssize_t DNSSerializer::FromBuffer(struct sockaddr_in6& sockaddr6Out,
                                  const uint8_t*       pBufferIn,
                                  size_t               bufferSize);


// struct hostent SizeOf
template <>
size_t DNSSerializer::SizeOf(const struct hostent& hostentIn);

// struct hostent ToBuffer
template <>
ssize_t DNSSerializer::ToBuffer(uint8_t* const        pBufferOut,
                                size_t                sizeOfBuffer,
                                const struct hostent& hostentIn);

// struct hostent FromBuffer
template <>
ssize_t DNSSerializer::FromBuffer(struct hostent& hostentOut,
                                  const uint8_t * pBufferIn,
                                  size_t          bufferSize);

// free the contents of a struct hostent, allowing a free structure itself
void FreeHostentContents(nn::socket::HostEnt& hostEntry);
void FreeHostentContents(struct hostent& hostEntry);

// struct addrinfo SizeOf
template <>
size_t DNSSerializer::SizeOf(const struct addrinfo& addrinfoIn);

// struct addrinfo ToBuffer
template <>
ssize_t DNSSerializer::ToBuffer(uint8_t* const         pBufferOut,
                                size_t                 sizeOfBuffer,
                                const struct addrinfo& pAddrinfoIn);

// struct addrinfo internal ToBuffer
template <>
ssize_t DNSSerializer::ToBufferInternal(uint8_t* const         pBufferOut,
                                        size_t                 sizeOfBuffer,
                                        const struct addrinfo& pAddrinfoIn);

// struct addrinfo FromBuffer
template <>
ssize_t DNSSerializer::FromBuffer(struct addrinfo& addrinfoOut,
                                  const uint8_t*   pBufferIn,
                                  size_t           bufferSize);

template <>
ssize_t DNSSerializer::FromBufferInternal(struct addrinfo& addrinfoOut,
                                          const uint8_t  * pBufferIn,
                                          size_t           sizeOfBuffer);


// free struct addrinfo contents
void FreeAddrinfoContents(struct addrinfo& addrinfoIn);

}}}}// namespace nn::socket::resolver::serializer
