﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "serializer.h"
#include "serializer_Specializations.h"

/*
struct sockaddr_in6
{
    sa_family_t sin6_family;
    in_port_t sin6_port;
    uint32_t sin6_flowinfo;
    struct in6_addr sin6_addr;
    uint32_t sin6_scope_id;
};
*/

//#define LOG_LEVEL LOG_LEVEL_MAJOR
#define LOG_MODULE_NAME "sockaddr_in6 serializer" // NOLINT(preprocessor/const)
#include <nn/socket/resolver/private/resolver_DebugLog.h>

namespace nn { namespace socket { namespace resolver { namespace serializer {

template <>
size_t DNSSerializer::SizeOf(const struct sockaddr_in6 & sockaddrIn6)
{
    size_t rc = 0;
    rc += sizeof(uint16_t); // sin6_family
    rc += sizeof(uint16_t); // sin6_port
    rc += sizeof(uint32_t); // sin6_flowinfo
    rc += sizeof(sockaddrIn6.sin6_addr); // sin6_addr
    rc += sizeof(uint32_t); // sin6_scope_id
    return rc;
}

// serialize a struct sockaddr_in6 (ipv6) to the buffer
template <>
ssize_t DNSSerializer::ToBuffer(uint8_t * const pBufferOut,
                                size_t     sizeOfBuffer,
                                const struct sockaddr_in6 & pin6)
{
    ssize_t rc = -1;
    uint8_t* pBufferCurrent = pBufferOut;

    uint16_t shortValue;  // use for sin_family, sin_port
    uint32_t longValue;   // use for in_addr.s_addr

    if ( -1 == CheckToBufferArguments(pBufferCurrent,
                                      sizeOfBuffer,
                                      sizeof(struct sockaddr_in6),
                                      __LINE__))
    {
        goto bail;
    };

    shortValue = pin6.sin6_family;
    if (-1 == ( rc = ToBuffer(pBufferCurrent,
                              sizeOfBuffer - (pBufferCurrent - pBufferOut),
                              shortValue)))
    {
        LogDebug("error: sin6_family\n");
        goto bail;
    };
    pBufferCurrent += rc;

    shortValue = pin6.sin6_port;
    if (-1 == ( rc = ToBuffer(pBufferCurrent,
                              sizeOfBuffer - (pBufferCurrent - pBufferOut),
                              shortValue)))
    {
        LogDebug("error: sin6_port\n");
        goto bail;
    };
    pBufferCurrent += rc;

    longValue = pin6.sin6_flowinfo;
    if (-1 == ( rc = ToBuffer(pBufferCurrent,
                              sizeOfBuffer - (pBufferCurrent - pBufferOut),
                              longValue)))
    {
        LogDebug("error: sin6_flowinfo\n");
        goto bail;
    };
    pBufferCurrent += rc;

    memcpy(pBufferCurrent, &pin6.sin6_addr, sizeof(struct in6_addr));
    pBufferCurrent += sizeof(pin6.sin6_addr);

    longValue = pin6.sin6_scope_id;
    if (-1 == ( rc = ToBuffer(pBufferCurrent,
                              sizeOfBuffer - (pBufferCurrent - pBufferOut),
                              longValue)))
    {
        LogDebug("error: sin6_scope_id\n");
        goto bail;
    };

    pBufferCurrent += rc;
    rc = pBufferCurrent - pBufferOut;

bail:
    return rc;
}

// serialize a null-terminated array of strings from a buffer
template <>
ssize_t DNSSerializer::FromBuffer(struct sockaddr_in6 & pin6,
                                  const uint8_t       * pBufferIn,
                                  size_t                bufferSize)
{
    ssize_t rc = -1;
    const uint8_t* pBufferCurrent = pBufferIn;

    uint16_t shortValue;  // use for sin_family, sin_port
    uint32_t longValue;   // use for in_addr.s_addr

    if (-1 == ( rc = FromBuffer(shortValue,
                                pBufferCurrent,
                                bufferSize - (pBufferCurrent - pBufferIn))))
    {
        LogDebug("error: sin6_family\n");
        goto bail;
    };
    pin6.sin6_family = shortValue;
    pBufferCurrent += rc;

    if (-1 == ( rc = FromBuffer(shortValue,
                                pBufferCurrent,
                                bufferSize - (pBufferCurrent - pBufferIn))))
    {
        LogDebug("error: sin6_port\n");
        goto bail;
    };
    pin6.sin6_port = shortValue;
    pBufferCurrent += rc;

    if (-1 == ( rc = FromBuffer(longValue,
                                pBufferCurrent,
                                bufferSize - (pBufferCurrent - pBufferIn))))
    {
        LogDebug("error: sin6_flowinfo\n");
        goto bail;
    };
    pin6.sin6_flowinfo = longValue;
    pBufferCurrent += rc;

    memcpy(&pin6.sin6_addr, pBufferCurrent, sizeof(struct in6_addr));
    pBufferCurrent += sizeof(pin6.sin6_addr);


    if (-1 == ( rc = FromBuffer(longValue,
                                pBufferCurrent,
                                bufferSize - (pBufferCurrent - pBufferIn))))
    {
        LogDebug("error: sin6_scope_id\n");
        goto bail;
    };
    pin6.sin6_scope_id = longValue;
    pBufferCurrent += rc;

    rc = pBufferCurrent - pBufferIn;

bail:
    return rc;
}

}}}}// namespace nn::socket::resolver::serializer
