﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "serializer.h"
#include "serializer_Specializations.h"
#include "../../detail/socket_Allocator.h"

//#define LOG_LEVEL LOG_LEVEL_MAJOR
#define LOG_MODULE_NAME "inaddr serializer" // NOLINT(preprocessor/const)
#include <nn/socket/resolver/private/resolver_DebugLog.h>

namespace nn { namespace socket { namespace resolver { namespace serializer {

template <>
size_t DNSSerializer::SizeOf(const nn::socket::InAddr& inAddrIn)
{
    // the only thing inside a struct in_addr is a 32-bit
    // address, but due to how the structure is created
    // sizeof(in_addr) != sizeof(uint32_t) and so we To/From buffer
    // only the s_addr field
    return sizeof(uint32_t);
};

template <>
size_t DNSSerializer::SizeOf(const struct in_addr& inAddrIn)
{
    // the only thing inside a struct in_addr is a 32-bit
    // address, but due to how the structure is created
    // sizeof(in_addr) != sizeof(uint32_t) and so we To/From buffer
    // only the s_addr field
    return sizeof(uint32_t);
};

template <>
size_t DNSSerializer::SizeOf(const nn::socket::InAddr** ppInaddrIn, uint32_t & countOut)
{
    size_t rc = 0;
    rc += sizeof(uint32_t); //length
    countOut = 0;
    if (nullptr == ppInaddrIn)
    {
        goto bail;
    };

    for (const nn::socket::InAddr** ppInaddr = ppInaddrIn; *ppInaddr != nullptr; ppInaddr++)
    {
        ++countOut;
        rc += sizeof(uint32_t);
    };

bail:
    return rc;
};

template <>
size_t DNSSerializer::SizeOf(const struct in_addr** ppInaddrIn, uint32_t & countOut)
{
    size_t rc = 0;
    rc += sizeof(uint32_t); //length
    countOut = 0;
    if (NULL == ppInaddrIn)
    {
        goto bail;
    };

    for (const struct in_addr** ppInaddr = ppInaddrIn; *ppInaddr != NULL; ppInaddr++)
    {
        ++countOut;
        rc += sizeof(uint32_t);
    };

bail:
    return rc;
};

template <>
ssize_t DNSSerializer::ToBuffer(uint8_t * const pBufferOut,
                                size_t          sizeOfBuffer,
                                const nn::socket::InAddr& inAddrIn)
{
    ssize_t rc = -1;
    uint8_t * pBufferCurrent = pBufferOut;
    uint32_t valueSwapped = InternalHton(inAddrIn.S_addr);

    if ( -1 == (rc = CheckToBufferArguments(pBufferCurrent,
                                            sizeOfBuffer,
                                            sizeof(inAddrIn),
                                            __LINE__)))
    {
        goto bail;
    };

    memcpy(const_cast<uint8_t*>(pBufferCurrent), &valueSwapped, sizeof(valueSwapped));
    rc += sizeof(valueSwapped);

bail:
    return rc;

}

template <>
ssize_t DNSSerializer::ToBuffer(uint8_t * const pBufferOut,
                                size_t          sizeOfBuffer,
                                const struct in_addr& inAddrIn)
{
    ssize_t rc = -1;
    uint8_t * pBufferCurrent = pBufferOut;
    uint32_t valueSwapped = InternalHton(static_cast<const uint32_t>(inAddrIn.s_addr));

    if ( -1 == (rc = CheckToBufferArguments(pBufferCurrent,
                                            sizeOfBuffer,
                                            sizeof(struct in_addr),
                                            __LINE__)))
    {
        goto bail;
    };

    memcpy(const_cast<uint8_t*>(pBufferCurrent), &valueSwapped, sizeof(uint32_t));
    rc += sizeof(uint32_t);

bail:
    return rc;

}


// serialize a null-terminated array of strings from a buffer
template <>
ssize_t DNSSerializer::FromBuffer(nn::socket::InAddr & inAddrOut,
                                  const uint8_t  * pBufferIn,
                                  size_t           bufferSize)
{
    ssize_t rc = -1;
    if ( bufferSize < sizeof(inAddrOut))
    {
        rc = -1;
        goto bail;
    };

    memset(&inAddrOut, '\0', sizeof(inAddrOut));
    inAddrOut.S_addr = static_cast<uint32_t>(InternalNtoh(*(uint32_t*)pBufferIn));
    rc = sizeof(uint32_t);

bail:
    return rc;
}


// serialize a null-terminated array of strings from a buffer
template <>
ssize_t DNSSerializer::FromBuffer(struct in_addr & inAddrOut,
                                  const uint8_t  * pBufferIn,
                                  size_t           bufferSize)
{
    ssize_t rc = -1;
    if ( bufferSize < sizeof(struct in_addr))
    {
        rc = -1;
        goto bail;
    };

    memset(&inAddrOut, '\0', sizeof(struct in_addr));
    inAddrOut.s_addr = (uint32_t)InternalNtoh(*(uint32_t*)pBufferIn);
    rc = sizeof(uint32_t);

bail:
    return rc;
}


// serialize a null-terminated array of in_addrs
template <>
ssize_t DNSSerializer::ToBuffer(uint8_t * const pBufferOut,
                                size_t           sizeOfBuffer,
                                nn::socket::InAddr** ppInAddrArray)
{
    ssize_t rc = -1;
    uint8_t * pBufferCurrent = pBufferOut;
    uint32_t intValue = 0;
    uint32_t numberOfAddresses = 0;
    nn::socket::InAddr inaddrValue = {};

    if (ppInAddrArray == nullptr && sizeOfBuffer < sizeof(uint32_t))
    {
        LogDebug("ppInAddrArray is NULL and no more space");
        goto bail;
    }
    else if (ppInAddrArray == nullptr)
    {
        if (-1 == ( rc = ToBuffer(pBufferCurrent,
                                  sizeOfBuffer - (pBufferCurrent - pBufferOut),
                                  intValue)))
        {
            goto bail;
        };
        pBufferCurrent += rc;

        goto bail;
    }; // end NULL case

    for (nn::socket::InAddr** pinAddr = ppInAddrArray; *pinAddr != nullptr; pinAddr++)
    {
        ++numberOfAddresses;
    };

    if ( -1 == (rc = CheckToBufferArguments(pBufferCurrent,
                                            sizeOfBuffer,
                                            // +1 for the length
                                            (numberOfAddresses + 1) * sizeof(**ppInAddrArray),
                                            __LINE__)))
    {
        goto bail;
    };

    // add length parameter
    if (-1 == ( rc = ToBuffer(pBufferCurrent,
                              sizeOfBuffer - (pBufferCurrent - pBufferOut),
                              numberOfAddresses)))
    {
        goto bail;
    };
    pBufferCurrent += rc;

    rc = 0;
    for(int idx = 0; ppInAddrArray[idx] != nullptr; idx++)
    {
        inaddrValue = *ppInAddrArray[idx];
        LogDebug("h_addr_list[%d] = %p (%s)\n",
                       idx,
                       ppInAddrArray[idx],
                       nn::socket::InetNtoa(*ppInAddrArray[idx]));

        if (-1 == ( rc = ToBuffer(pBufferCurrent,
                                  sizeOfBuffer - (pBufferCurrent - pBufferOut),
                                  inaddrValue)))
        {
            goto bail;
        };

        pBufferCurrent += rc;
        sizeOfBuffer -= rc;
    };

    rc = pBufferCurrent - pBufferOut;

bail:
    return rc;
}


// serialize a null-terminated array of in_addrs
template <>
ssize_t DNSSerializer::ToBuffer(uint8_t * const pBufferOut,
                                size_t           sizeOfBuffer,
                                struct in_addr** ppInAddrArray)
{
    ssize_t rc = -1;
    uint8_t * pBufferCurrent = pBufferOut;
    uint32_t intValue = 0;
    uint32_t numberOfAddresses = 0;
    struct in_addr inaddrValue;

    if (ppInAddrArray == NULL && sizeOfBuffer < sizeof(uint32_t))
    {
        LogDebug("ppInAddrArray is NULL and no more space");
        goto bail;
    }
    else if (ppInAddrArray == NULL)
    {
        if (-1 == ( rc = ToBuffer(pBufferCurrent,
                                  sizeOfBuffer - (pBufferCurrent - pBufferOut),
                                  intValue)))
        {
            goto bail;
        };
        pBufferCurrent += rc;

        goto bail;
    }; // end NULL case

    for (struct in_addr** pinAddr = ppInAddrArray; *pinAddr != NULL; pinAddr++)
    {
        ++numberOfAddresses;
    };

    if ( -1 == (rc = CheckToBufferArguments(pBufferCurrent,
                                            sizeOfBuffer,
                                            // +1 for the length
                                            (numberOfAddresses + 1) * sizeof(struct in_addr),
                                            __LINE__)))
    {
        goto bail;
    };

    // add length parameter
    if (-1 == ( rc = ToBuffer(pBufferCurrent,
                              sizeOfBuffer - (pBufferCurrent - pBufferOut),
                              numberOfAddresses)))
    {
        goto bail;
    };
    pBufferCurrent += rc;

    rc = 0;
    for(int idx = 0; ppInAddrArray[idx] != NULL; idx++)
    {
        inaddrValue = *(struct in_addr*)ppInAddrArray[idx];
        LogDebug("h_addr_list[%d] = %p (%s)\n",
                       idx,
                       ppInAddrArray[idx],
                       nn::socket::InetNtoa(*ppInAddrArray[idx]));

        if (-1 == ( rc = ToBuffer(pBufferCurrent,
                                  sizeOfBuffer - (pBufferCurrent - pBufferOut),
                                  inaddrValue)))
        {
            goto bail;
        };

        pBufferCurrent += rc;
        sizeOfBuffer -= rc;
    };

    rc = pBufferCurrent - pBufferOut;

bail:
    return rc;
}

// serialize a null-terminated array of strings from a buffer
template <>
ssize_t DNSSerializer::FromBuffer(nn::socket::InAddr **& pOut,
                                  const uint8_t  *   pBufferIn,
                                  size_t             bufferSize)
{
    ssize_t rc = -1;
    uint32_t numberOfInaddrs = 0, intValue = 0;
    uint8_t const * pBufferCurrent = pBufferIn;
    pOut = nullptr;

    if ( pBufferIn == nullptr )
    {
        LogDebug("pBufferIn is NULL");
        rc = 0;
        goto bail;
    }
    else if ( bufferSize == 0 )
    {
        LogDebug("bufferSize is zero");
        rc = 0;
        goto bail;
    }
    else if (-1 == (rc = FromBuffer(numberOfInaddrs, pBufferIn, bufferSize)))
    {
        goto bail; // error
    };
    LogDebug("numberOfInaddrs:\t%d\n", numberOfInaddrs);
    pBufferCurrent += rc;

    if ( 0 == numberOfInaddrs )
    {
        goto bail; // no nn::socket::InAddrs
    }
    else
    {
        unsigned int idx = 0;

        // allocate an array that will hold numberOfInaddr + 1 (for the NULL terminator)
        pOut = (nn::socket::InAddr**)socket::detail::Alloc( (numberOfInaddrs + 1) * sizeof(*pOut));
        if ( nullptr == pOut )
        {
            LogDebug("Unable to allocate nn::socket::InAddr array, code: %d\n", __LINE__);
            rc = -1;
            goto bail;
        };
        memset(pOut, '\0', ( (numberOfInaddrs + 1) * sizeof(*pOut)));

        for ( idx=0; idx<numberOfInaddrs; idx++)
        {
            intValue = 0;
            pOut[idx] = (nn::socket::InAddr*)socket::detail::Alloc(sizeof(**pOut));
            if ( nullptr == pOut[idx] )
            {
                LogDebug("Unable to allocate nn::socket::InAddr, code: %d\n", __LINE__);
                rc = -1;
                goto bail;
            };

            if ( -1 == (rc = FromBuffer(intValue,
                                        pBufferCurrent,
                                        bufferSize - (pBufferCurrent - pBufferIn))))
            {
                goto bail;
            };

            pOut[idx]->S_addr = intValue;
            //LogDebug("h_addr_list[%d] = %p (%s)\n", idx, pOut[idx], inet_ntoa(*pOut[idx]));

            pBufferCurrent += rc;
        }// end
        pOut[idx] = nullptr;
    };

    rc = pBufferCurrent - pBufferIn;

bail:
    // free allocated resources on error
    if ( -1 == rc && nullptr != pOut )
    {
        for (int idx = 0;
             pOut[idx] != nullptr;
             ++idx)
        {
            socket::detail::Free(pOut[idx]);
            pOut[idx] = nullptr;
        };

        socket::detail::Free(pOut);
        pOut= nullptr;
    };

    return rc;
}

// serialize a null-terminated array of strings from a buffer
template <>
ssize_t DNSSerializer::FromBuffer(struct in_addr **& pOut,
                                  const uint8_t  *   pBufferIn,
                                  size_t             bufferSize)
{
    ssize_t rc = -1;
    uint32_t numberOfInaddrs = 0, intValue = 0;
    uint8_t const * pBufferCurrent = pBufferIn;
    pOut = NULL;

    if ( pBufferIn == NULL )
    {
        LogDebug("pBufferIn is NULL");
        rc = 0;
        goto bail;
    }
    else if ( bufferSize == 0 )
    {
        LogDebug("bufferSize is zero");
        rc = 0;
        goto bail;
    }
    else if (-1 == (rc = FromBuffer(numberOfInaddrs, pBufferIn, bufferSize)))
    {
        goto bail; // error
    };
    LogDebug("numberOfInaddrs:\t%d\n", numberOfInaddrs);
    pBufferCurrent += rc;

    if ( 0 == numberOfInaddrs )
    {
        goto bail; // no in_addrs
    }
    else
    {
        unsigned int idx = 0;

        // allocate an array that will hold numberOfInaddr + 1 (for the NULL terminator)
        pOut = (struct in_addr**)socket::detail::Alloc( (numberOfInaddrs + 1) * sizeof(struct in_addr*));
        if ( NULL == pOut )
        {
            LogDebug("Unable to allocate in_addr array\n");
            rc = -1;
            goto bail;
        };
        memset(pOut, '\0', ( (numberOfInaddrs + 1) * sizeof(struct in_addr*)));

        for ( idx=0; idx<numberOfInaddrs; idx++)
        {
            intValue = 0;
            pOut[idx] = (struct in_addr*)socket::detail::Alloc(sizeof(in_addr));
            if ( NULL == pOut[idx] )
            {
                LogDebug("Unable to allocate in_addr\n");
                rc = -1;
                goto bail;
            };

            if ( -1 == (rc = FromBuffer(intValue,
                                        pBufferCurrent,
                                        bufferSize - (pBufferCurrent - pBufferIn))))
            {
                goto bail;
            };

            pOut[idx]->s_addr = intValue;
            //LogDebug("h_addr_list[%d] = %p (%s)\n", idx, pOut[idx], inet_ntoa(*pOut[idx]));

            pBufferCurrent += rc;
        }// end
        pOut[idx] = NULL;
    };

    rc = pBufferCurrent - pBufferIn;

bail:
    // free allocated resources on error
    if ( -1 == rc && NULL != pOut )
    {
        for (int idx = 0;
             pOut[idx] != NULL;
             ++idx)
        {
            socket::detail::Free(pOut[idx]);
            pOut[idx] = NULL;
        };

        socket::detail::Free(pOut);
        pOut= NULL;
    };

    return rc;
}

}}}}// namespace nn::socket::resolver::serializer
