﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once
#include <nn/socket/resolver/private/resolver_PrivateApi.h>
#include <nn/socket/resolver/sfdl/resolver.sfdl.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_ThreadLocalStorage.h>
#include <nn/os/os_MutexApi.h>
#include <nn/os.h>

extern "C"
{
#include <nn/socket/netdb.h> // for struct hostent
}

/**
 * @file
 * @brief DNS Resolver server
 *
 * @details provides resolution service
 */
namespace nn { namespace socket { namespace resolver { namespace tls {

/**
 *  @brief allocate / initialize TLS
 * @return success or failure result code
 */
Result Initialize() NN_NOEXCEPT;

/**
 * @brief finalize the client TLS
 * @note this should only be done at exit time
 * @return success or failure result code
 */
Result Finalize() NN_NOEXCEPT;

/**
 * @brief function pointer type to destroy TLS
 */
typedef void (*TLSDestructorFunction)(void* argument);



/**
 * @brief get the SERVER TLS object
 * @note bionic makes use / manages this
 * @param pServerTLSOut an out parameter for the server TLS
 * @return success or failure result code
 */
Result GetServerTLS(void *& pContext) NN_NOEXCEPT;

/**
 * @brief set the SERVER TLS object
 * @note bionic makes use of this
 * @param destructor function called to destroy argument when thread exits
 * @param argument the client tls value
 * @return success or failure result code
 */
Result SetServerTLS(TLSDestructorFunction destructor, void* argument) NN_NOEXCEPT;

/**
 * @brief get the CLIENT TLS object
 * @note  bionic does not make use of this
 * @param pClientTLSOut an out parameter for the client TLS
 * @return success or failure result code
 */
Result GetClientTLS(void *& pClientTLSOut) NN_NOEXCEPT;

/**
 * @brief set the CLIENT TLS object
 * @note bionic does not make use of this
 * @param destructor function called to destroy argument when thread exits
 * @param argument the client tls value
 * @return success or failure result code
 */
Result SetClientTLS(TLSDestructorFunction destructor, void* argument) NN_NOEXCEPT;

/**
 * @brief namespace to hold server TLS functions
 */
namespace Server
{
    /**
     * @brief Initialize the server tls
     * @note bionic DOES make use of this
     * @return success or failure result code
     */
    Result Initialize();

    /**
     * @brief Finalize the server tls
     * @note bionic DOES make use of this
     * @return success or failure result code
     */
    Result Finalize();
};

/**
 * @brief Client TLS namespace/ functions
 * @return success or failure result code
 */
namespace Client
{
    /**
     * @brief initialize the client tls
     * @return success or failure result code
     */
    Result Initialize();

    /**
     * @brief finalize the client tls
     * @return success or failure result code
     */
    Result Finalize();

    /**
     * @brief host error reference from TLS
     * @note readable and writable
     * @return referemce
     */
    int& InternalHostErrno();

    /**
     * @brief get current hostent in TLS
     * @return const struct hostent* from TLS
     */
    const nn::socket::HostEnt* GetCurrentHostent();
    const struct hostent* GetCurrentHostentOld();

    /**
     *  @brief set current hostent structure
     *  @note invalidates previous structures
     *  @note the ClientTLS takes ownership of the pointer
     *        and will free it
     * @brief get current hostent in TLS
     * @return false on out of memory, true otherwise
     */
    bool SetCurrentHostent(nn::socket::HostEnt* pHostentIn);
    bool SetCurrentHostentOld(struct hostent* pHostentIn);

    /**
     * @brief get herror string vaue
     * @return const string
     */
    const char* GetCurrentHErrorString();

    /**
     * @brief set the current herror string reference
     * @param pStringIn the string to set
     * @return NULL if out of memory, new string otherwise
     * @note overrides previous string
     */
    const char* SetCurrentHErrorString(const char* pStringIn);

    /**
     * @brief get the current gai error string
     * @return NULL if out of memory, new string otherwise
     * @note overrides previous string
     */
    const char* GetCurrentGaiErrorString();

    /**
     * @brief set the current gai error string reference
     * @param pStringIn the string to set
     * @return NULL on out of memory, new string otherwise
     * @note overrides previous strin
     */
    const char* SetCurrentGaiErrorString(const char* pStringIn);
};

}}}} // nn::socket::resolver
