﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/socket/resolver/sfdl/resolver.sfdl.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_MutexApi.h>
#include <nn/os.h>

/**
 * @file
 * @brief DNS Resolver server server implementation
 *
 * @details provides resolution service server implementation
 */
namespace nn { namespace socket { namespace resolver
{
/**
 * @brief initialize the interface
 * @return success or failure result code
 */
Result Initialize() NN_NOEXCEPT;

/**
 * @brief finalize the interface
 * @return success or failure result code
 */
Result Finalize() NN_NOEXCEPT;

/**
 * @brief the resolver implementation class
 * @note provided so that Hipc / DFC can resuse impl code
 * @return success or failure result code
 */
class ResolverImpl
{
public:
    /**
     * @brief the resolver implementation constructor
     */
    ResolverImpl();

    /**
     * @brief the resolver implementation destructor
     */
    virtual ~ResolverImpl();

    /**
     * @brief finalize the server implementation
     * @return success or failure code
     */
    virtual Result Finalize() NN_NOEXCEPT;

    //----------------------------------------------------------------
    // Current functions
    //----------------------------------------------------------------

    /**
     * @brief
     * SFDL MethodId(10)
     */
    virtual Result GetHostByNameRequestWithOptions(Bit64 processId,
                                                   const nn::sf::InBuffer& nameParam,
                                                   const nn::sf::OutBuffer& serializedHostEntryOut,
                                                   const nn::sf::Out<uint32_t> pSerializedHostEntrySizeOut,
                                                   uint32_t optionsRequestVersion,
                                                   const nn::sf::InBuffer& serializedOptionsInBuffer,
                                                   uint32_t serializedOptionsInBufferCount,
                                                   nn::sf::Out<int> pOutHostError,
                                                   nn::sf::Out<int> pOutErrno)  NN_NOEXCEPT;


    /**
     * @brief
     * SFDL MethodId(11)
     */
    virtual Result GetHostByAddrRequestWithOptions(Bit64 processId,
                                                   const nn::sf::InBuffer& inAddrParam,
                                                   uint32_t inAddrLengthParam,
                                                   int addressFamilyParam,
                                                   const nn::sf::OutBuffer& serializedHostEntryOut,
                                                   const nn::sf::Out<uint32_t> pSerializedHostEntrySizeOut,
                                                   uint32_t optionsRequestVersion,
                                                   const nn::sf::InBuffer& serializedOptionsInBuffer,
                                                   uint32_t serializedOptionsInBufferCount,
                                                   nn::sf::Out<int> pOutHostError,
                                                   nn::sf::Out<int> pOutErrno)  NN_NOEXCEPT;

    /**
     * @brief
     * SFDL MethodId(12)
     */
    virtual Result GetAddrInfoRequestWithOptions(Bit64 processId,
                                                 const nn::sf::InBuffer& nodeNameParam,
                                                 const nn::sf::InBuffer& serviceNameParam,
                                                 const nn::sf::InBuffer& serializedHintsParam,
                                                 const nn::sf::OutBuffer serializedAddrInfoResultBuffer,
                                                 const nn::sf::Out<uint32_t> pSerializedAddrinfoResultBufferSize,
                                                 const nn::sf::Out<int> pReturnCode,
                                                 uint32_t optionsRequestVersion,
                                                 const nn::sf::InBuffer& serializedOptionsInBuffer,
                                                 uint32_t serializedOptionsInBufferCount,
                                                 nn::sf::Out<int> pOutHostError,
                                                 nn::sf::Out<int> pOutErrno)  NN_NOEXCEPT;
    /**
     * @brief
     * SFDL MethodId(13)
     */
    virtual Result GetNameInfoRequestWithOptions(Bit64 processId,
                                                 const nn::sf::InBuffer& socketAddressPointerIn,
                                                 const nn::sf::OutBuffer hostBufferOutParam,
                                                 const nn::sf::OutBuffer serviceBufferOutParam,
                                                 uint32_t flagsParam,
                                                 const nn::sf::Out<int> pReturnCode,
                                                 uint32_t optionsRequestVersion,
                                                 const nn::sf::InBuffer& serializedOptionsInBuffer,
                                                 uint32_t serializedOptionsInBufferCount,
                                                 nn::sf::Out<int> pOutHostError,
                                                 nn::sf::Out<int> pOutErrno)  NN_NOEXCEPT;

    /**
     * @brief
     * SFDL MethodId(14)
     */
    virtual Result ResolverSetOptionRequest(Bit64 processId,
                                            uint32_t version,
                                            const nn::sf::InBuffer& serializedOptionsInBuffer,
                                            nn::sf::Out<int> pReturnCode,
                                            nn::sf::Out<int> pErrnoOut)  NN_NOEXCEPT;
    /**
     * @brief
     * SFDL MethodId(15)
     */
    virtual Result ResolverGetOptionRequest(Bit64 processId,
                                            uint32_t version,
                                            uint32_t key,
                                            const nn::sf::InBuffer& optionBuffer,
                                            const nn::sf::OutBuffer& serializedOptionBufferOut,
                                            nn::sf::Out<uint32_t> outBufferUsed,
                                            nn::sf::Out<int> pReturnCode,
                                            nn::sf::Out<int> pErrnoOut)  NN_NOEXCEPT;


    //----------------------------------------------------------------
    // Deprecated but necessary functions
    //----------------------------------------------------------------


    /**
     * @brief get a cancel handle to allow a future request canel
     * @param pid The process ID of this function - needed for Cancel API.
     * @param RequestHandle the handle to return
     * @return success or failure result code
     */
    virtual Result CancelRequest(Bit64 pid, uint32_t RequestHandle) NN_NOEXCEPT;

    /**
     * @brief GetHostByName Request
     * @param cancelHandle A handle that can be used to cancel an in-flight DNS request
     * @param pid The process ID of this function - needed for Cancel API.
     * @param useNsdResolve Should we resolve using the NSD module or not.
     * @param pHostNameAsBuffer a c-string representation of a hostname (i.e. "www.foo.com")
     * @param pHostErrorOut the state of h_error after the call
     * @param pErrnoOut the state of errno after the call
     * @param serializedHostentResponse a bufferized version of the hostent response
     * @return success or failure result code
     */
    virtual Result GetHostByNameRequest(uint32_t cancelHandle,
                                        Bit64 pid,
                                        bool useNsdResolve,
                                        const nn::sf::InBuffer& pHostNameAsBuffer,
                                        nn::sf::Out<uint32_t> pHostErrorOut,
                                        nn::sf::Out<uint32_t> pErrnoOut,
                                        const nn::sf::OutBuffer& serializedHostentResponse,
                                        nn::sf::Out<uint32_t> pBufferLengthOut) NN_NOEXCEPT;

    /**
     * @brief GetHostByAddr Request
     * @param cancelHandle A handle that can be used to cancel an in-flight DNS request
     * @param pid The process ID of this function - needed for Cancel API.
     * @param serializedSocketAddressIn a sockaddr that was serialized (i.e. toBuffer)
     * @param socketLengthIn the size of the socklen provided
     * @param addressFamilyIn the address family (I.e. AF_INET / AF_INET6 / AF_UNSPEC / etc.)
     * @param pHostErrorOut the state of h_error after the call
     * @param pErrnoOut the state of errno after the call
     * @return success or failure result code
     */
    virtual Result GetHostByAddrRequest(uint32_t cancelHandle,
                                        Bit64 pid,
                                        const nn::sf::InBuffer& serializedSocketAddressIn,
                                        uint32_t socketLengthIn,
                                        uint32_t addressFamilyIn,
                                        nn::sf::Out<uint32_t> pHostErrorOut,
                                        nn::sf::Out<uint32_t> pErrnoOut,
                                        const nn::sf::OutBuffer& serializedHostentResponse,
                                        nn::sf::Out<uint32_t> pBufferLengthOut) NN_NOEXCEPT;
    /**
     * @brief get host error request string request
     * @param errorRequestIn the integer value corresponding to the error
     * @param stringErrorResponse a char* c-string response corresponding to the error
     * @return success or failure result code
     */
    virtual Result GetHostStringErrorRequest(uint32_t errorRequestIn,
                                             const nn::sf::OutBuffer& stringErrorResponse) NN_NOEXCEPT;

    /**
     * @brief get gai error request string request
     * @param errorRequestIn the integer value corresponding to the error
     * @param stringErrorResponse a char* c-string response corresponding to the error
     * @return success or failure result code
     */
    virtual Result GetGaiStringErrorRequest(uint32_t errorRequestIn,
                                            const nn::sf::OutBuffer& stringErrorResponse) NN_NOEXCEPT;

    /**
     * @brief get addr info request
     * @param cancelHandle A handle that can be used to cancel an in-flight DNS request
     * @param pid The process ID of this function - needed for Cancel API.
     * @param useNsdResolve Should we resolve using the NSD module or not.
     * @param nodeName the name of the node to get the address for
     * @param serviceNameIn the name of the service for bionic_getaddrinfo
     * @param addrInfoHintsIn the hints provided to bionic_getaddrinfo
     * @param  addrInfoArrayOut the outbuffer that will contain a serialized addrinfo(s)
     * @param pErrnoOut the value of errno after the call
     * @param pReturnValueOut the return value from bionic_getaddrinfo
     * @return success or failure result code
     */
    virtual Result GetAddrInfoRequest(uint32_t cancelHandle,
                                      Bit64 pid,
                                      bool useNsdResolve,
                                      const nn::sf::InBuffer& nodeNameIn,
                                      const nn::sf::InBuffer& serviceNameIn,
                                      const nn::sf::InBuffer& addrInfoHintsIn,
                                      const nn::sf::OutBuffer& addrInfoArrayOut,
                                      nn::sf::Out<uint32_t> pErrnoOut,
                                      nn::sf::Out<int32_t> pReturnValueOut,
                                      nn::sf::Out<uint32_t> pBufferLengthOut) NN_NOEXCEPT;

    /**
     * @brief get name info request
     * @param cancelHandle A handle that can be used to cancel an in-flight DNS request
     * @param pid The process ID of this function - needed for Cancel API.
     * @param socketAddressIn the sockaddr_in structure in a buffer-ized form
     * @param hostBufferOut the name of the host returned from bionic_getnameinfo
     * @param serviceBufferOut the service returned from bionic_getnameinfo
     * @param flags flags provided to bionic_getnameinfo
     * @param pErrnoOut the value of errno after the call
     * @param pReturnValueOut the return value from bionic_getaddrinfo
     * @return success or failure result code
     */
    virtual Result GetNameInfoRequest(uint32_t cancelHandle,
                                      Bit64 pid,
                                      const nn::sf::InBuffer& socketAddressIn,
                                      const nn::sf::OutBuffer& hostBufferOut,
                                      const nn::sf::OutBuffer& serviceBufferOut,
                                      uint32_t flags,
                                      nn::sf::Out<uint32_t> pErrnoOut,
                                      nn::sf::Out<int32_t> pReturnValueOut) NN_NOEXCEPT;

    /**
     * @brief get a cancel handle to allow a future request canel
     * @param pid The process ID of this function - needed for Cancel API.
     * @param RequestHandle the handle to return
     * @return success or failure result code
     */
    virtual Result GetCancelHandleRequest(Bit64 pid, nn::sf::Out<uint32_t> RequestHandle) NN_NOEXCEPT;

    //----------------------------------------------------------------
    // Deprecated functions
    //----------------------------------------------------------------

    /**
     * @brief set the DNS server addresses
     * SFDL MethodId(0)
     * @param addresses the addresses in uint32_t (host byte order)
     * @param length the number of addresses (not the size of the array)
     * @note BSD config calls this on new DHCP / configuration
     * @return success or failure result code
     */
    virtual Result SetDnsAddressesPrivateRequest(const nn::sf::InBuffer& sockaddrInBuffer,
                                                 size_t length) NN_NOEXCEPT;

    /**
     * @brief get a cancel handle to allow a future request canel
     * @param pid The process ID of this function - needed for Cancel API.
     * @return success or failure result code
     */
    virtual Result CancelAll(Bit64 pid) NN_NOEXCEPT;


    /**
     * @brief get the current DNS ip address
     * SFDL MethodId(1)
     * @param nIndex the 'n'th index requested by bionic; this could be 0, 1, etc.
     * @note the goal being that bionic will request 0 on the first attempt, 1 on the second attempt, etc.
     * @return success or failure code
     * @note getting beyond the boundary of the array returns 0.0.0.0
     */
    virtual Result GetDnsAddressPrivateRequest(uint32_t nIndex,
                                               const nn::sf::OutBuffer& in) NN_NOEXCEPT;

protected:

    nn::os::Mutex m_AccessLock;
}; // end resolver impl

}}} // namespace nn::socket::resolver
