﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief BSD Configuration
 *
 * @details Top level BSD configuration management
 */

#pragma once

#include <new>
#include <type_traits>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os/os_ThreadApi.h>

#include <nn/sf/sf_HipcServer.h>
#include <nn/socket/private/thread.h>
#include <nn/socket/resolver/resolver_Server.h>
#include <nn/socket/resolver/private/resolver_PrivateApi.h>

#include "resolver_CreateClient.h"
#include "resolver_ServiceName.h"
#include "resolver_Implementation.h"

/**
 * @file
 * @brief resolver server
 *
 * @details resolver server
 */

namespace nn { namespace socket { namespace resolver {
// resolver service object typedef

enum
{
    SessionCountMax = 30,
    NumberOfPortsMax = 1
};


class Server :
public nn::sf::HipcSimpleAllInOneServerManager<SessionCountMax, NumberOfPortsMax>,
public virtual ResolverImpl
{
public:
   /**
    * @brief resolver server constructor
    */
    Server();

    /**
    * @brief resolver server destructor
    */
    ~Server();

   /**
    * @brief resolver server initialization routine
    * @return success or failure result code
    */
    Result Initialize();

    /**
     * @brief resolver server wait routing
    * @return success or failure result code
     */
    Result Wait();

    /**
     * @brief resolver server finalize routine
    * @return success or failure result code
     */
    Result Finalize() NN_NOEXCEPT;

    /**
     * the maximum number of threads that we are running with; the resolver cancellation
     * feature uses N-1 to allocate the total number of slots
     */
    enum
    {
        MAX_THREADS = 8
    };

private:

    NetworkThread  m_Threads[MAX_THREADS];

    // the bionic resolver has some very deep stacks with large buffers
    // Stack analysis as of 8/22/2016 show usage of between 64k and 72k.
#ifdef NN_SDK_BUILD_DEBUG
    NN_ALIGNAS(nn::os::StackRegionAlignment)uint8_t m_ThreadStacks[MAX_THREADS][1024 * 192];
#else
    NN_ALIGNAS(nn::os::StackRegionAlignment)uint8_t m_ThreadStacks[MAX_THREADS][1024 * 96];
#endif

    /**
     * @brief static thread entry point
     */
    static void SessionThreadEntryStatic(void *pThis);
    /**
     * @brief instance thread entry point
     */
    void SessionThreadEntry();
};

/**
 * @brief exposed resolver server initialization routine
 * @return success or failure result code
 */
Result InitializeServer() NN_NOEXCEPT;

/**
 * @brief wait on the server to finish
 * @return success or failure result code
 */
Result WaitServer() NN_NOEXCEPT;

/**
 * @brief finalize the server
 * @return success or failure result code
 */
Result FinalizeServer() NN_NOEXCEPT;

}}} // nn::socket::resolver
