﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/socket/resolver/private/resolver_DebugLog.h>
#include <nn/nn_SdkLog.h>
#include <cstring>
#include <nn/util/util_FormatString.h>
#include <cctype>
#include <algorithm>
extern "C"
{                   // adding nolint because C++ versions do not exist
                    // in rynda and this causes a compilation error
#include <stdio.h>  // NOLINT(others/c_header)
#include <stdlib.h> // NOLINT(others/c_header)
}

namespace nn { namespace socket { namespace resolver {

const char* g_LogLevelTable[] = {
    "Major",
    "Minor",
    "Info",
    "Debug",
    "Hex"
};

const unsigned HexCharactersPerByte = 3;
const unsigned NewlinePerLine = 1;
const unsigned BytesPerLine = 16;
const unsigned NewlineAndNull = 2;

static
const size_t GetHexStringBufferSize(size_t bufferSize)
{
#define LINES(size)  (                                          \
        (size / BytesPerLine) == 0 ?                            \
        1 :                                                     \
        ((size / BytesPerLine) + (size % BytesPerLine)))

#define GET_HEXDUMP_BUFFER_SIZE(size) (                 \
        (HexCharactersPerByte * size)           +       \
        (NewlinePerLine * (LINES(size)))        +       \
        NewlineAndNull                                  \
        )
    size_t sz = GET_HEXDUMP_BUFFER_SIZE(bufferSize);
    return sz;
#undef GET_HEXDUMP_BUFFER_SIZE
#undef LINES
}

// todo
static
void HexFill(bool linebreak, char* string, size_t strSize,
             const uint8_t* pBytes, unsigned byteSize)
{
    char* pHex = string;

    for (unsigned idx=0; idx<byteSize; ++idx)
    {
        pHex += nn::util::SNPrintf(pHex,
                                   strSize - (pHex - string),
                                   "%02x ", *pBytes);
        pBytes += 1;

        if (linebreak == true && (idx == byteSize - 1 || 0 == ((idx + 1) % BytesPerLine)))
        {
            *pHex = '\n';
            ++pHex;
        };

        if (idx == (byteSize - 1))
        {
            *pHex = '\0';
            ++pHex;
        };
    };
};

extern "C" const char* GetFilename(const char* line)
{
    size_t length = 0;
    if (NULL == line)
    {
        goto bail;
    };

    length = strlen(line);
    for (size_t idx=length; idx!=0; --idx)
    {
        if (line[idx] == '/' || line[idx] == '\\')
        {
            return &line[idx] + 1;
        };
    };

bail:
    return line;
};

extern "C" const char* GetLevelString(unsigned level)
{
    if (level > LOG_LEVEL_MAX)
    {
        return "undefined";
    }
    return g_LogLevelTable[level];
};

extern "C" void ResolverLog(bool linebreak,
                            const void* pBuffer,
                            unsigned size,
                            const char* format, ...)
{
    va_list args;
    va_start (args, format);
    size_t hexSize = NULL == pBuffer ? 0 : GetHexStringBufferSize(size);

    if (hexSize == 0)
    {
        NN_SDK_VLOG(format, args);
    }
    else
    {
        int stringSize = vsnprintf(NULL, 0, format, args);
        char* pString = static_cast<char*>(alloca(stringSize + hexSize));
        memset(pString, 0, stringSize + hexSize);
        vsnprintf(pString, stringSize, format, args);
        HexFill(linebreak, pString + strlen(pString), hexSize, reinterpret_cast<const uint8_t*>(pBuffer), size);
        NN_SDK_LOG("%s\n", pString);
    };
    va_end(args);
};

}}}; // nn::socket::resolver
