﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
/*
  Based on:
  -
  Copyright (c) 2001 Charles Mott <cm@linktel.net>
   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:
   1. Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.

   THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
   ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
   FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
   OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
   HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
   LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
   OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
   SUCH DAMAGE.
 */

#include <cstdio>
#include <cstdarg>
#include <cstdlib>
#include <cctype>
#include <nn/os.h>
#include <nn/socket.h>

namespace nn     {
namespace socket {
namespace detail {

int socketAton(const char *addressStringBuffer, InAddr *pOutNetworkAddress)
NN_NOEXCEPT
{
    uint32_t      addressParts[4] = {0};
    uint64_t      value = 0;
    const char*   currentPointer = addressStringBuffer;
    char*         endPointer = nullptr;
    bool          done = false;
    int           addressPartsCounter = 0;

    if (addressStringBuffer == nullptr || pOutNetworkAddress == nullptr)
    {
        SetLastError(Errno::EAfNoSupport);
        return 0;
    }

    while (!done)
    {
        value = std::strtoull(currentPointer, &endPointer, 0);
        if (value > UINT_MAX || currentPointer == endPointer)
        {
            SetLastError(Errno::EAfNoSupport);
            return 0;
        }

        addressParts[addressPartsCounter] = value;

        switch (*(currentPointer = endPointer))
        {
        case '.' :
            if (addressPartsCounter == 3)
            {
                SetLastError(Errno::EAfNoSupport);
                return 0;
            }
            addressPartsCounter++;
            currentPointer++;
            break;

        case '\0':
            done = true;
            break;

        default:
            if (isspace((unsigned char)(*currentPointer)))
            {
                done = true;
                break;
            } else {
                SetLastError(Errno::EAfNoSupport);
                return 0;
            }
        }
    }

    switch (addressPartsCounter)
    {
    case 0:                /* a -- 32 bits */
        /*
         * Nothing is necessary here.  Overflow checking was
         * already done in strtoul().
         */
        break;

    case 1:                /* a.b -- 8.24 bits */
        if (value > 0xffffff || addressParts[0] > 0xff)
        {
            SetLastError(Errno::EAfNoSupport);
            return 0;
        }
        value |= addressParts[0] << 24;
        break;

    case 2:                /* a.b.c -- 8.8.16 bits */
        if (value > 0xffff || addressParts[0] > 0xff || addressParts[1] > 0xff)
        {
            SetLastError(Errno::EAfNoSupport);
            return 0;
        }
        value |= (addressParts[0] << 24) | (addressParts[1] << 16);
        break;

    case 3:                /* a.b.c.d -- 8.8.8.8 bits */
        if (value > 0xff || addressParts[0] > 0xff || addressParts[1] > 0xff || addressParts[2] > 0xff)
        {
            SetLastError(Errno::EAfNoSupport);
            return 0;
        }
        value |= (addressParts[0] << 24) | (addressParts[1] << 16) | (addressParts[2] << 8);
        break;

    default:
        SetLastError(Errno::EAfNoSupport);
        return 0;
    }

    pOutNetworkAddress->S_addr = nn::socket::InetHtonl((in_addr_t)value);

    return 1;
}

int socketAton(const char *addressStringBuffer, in_addr *pOutNetworkAddress)
NN_NOEXCEPT
{
    uint32_t      addressParts[4] = {0};
    uint64_t      value = 0;
    const char*   currentPointer = addressStringBuffer;
    char*         endPointer = nullptr;
    bool          done = false;
    int           addressPartsCounter = 0;

    if (addressStringBuffer == nullptr || pOutNetworkAddress == nullptr)
    {
        SetLastErrno(EAFNOSUPPORT);
        return 0;
    }

    while (!done)
    {
        value = std::strtoull(currentPointer, &endPointer, 0);
        if (value > UINT_MAX || currentPointer == endPointer)
        {
            SetLastErrno(EAFNOSUPPORT);
            return 0;
        }

        addressParts[addressPartsCounter] = value;

        switch (*(currentPointer = endPointer))
        {
        case '.' :
            if (addressPartsCounter == 3)
            {
                SetLastErrno(EAFNOSUPPORT);
                return 0;
            }
            addressPartsCounter++;
            currentPointer++;
            break;

        case '\0':
            done = true;
            break;

        default:
            if (isspace((unsigned char)(*currentPointer)))
            {
                done = true;
                break;
            } else {
                SetLastErrno(EAFNOSUPPORT);
                return 0;
            }
        }
    }

    switch (addressPartsCounter)
    {
    case 0:                /* a -- 32 bits */
        /*
         * Nothing is necessary here.  Overflow checking was
         * already done in strtoul().
         */
        break;

    case 1:                /* a.b -- 8.24 bits */
        if (value > 0xffffff || addressParts[0] > 0xff)
        {
            SetLastErrno(EAFNOSUPPORT);
            return 0;
        }
        value |= addressParts[0] << 24;
        break;

    case 2:                /* a.b.c -- 8.8.16 bits */
        if (value > 0xffff || addressParts[0] > 0xff || addressParts[1] > 0xff)
        {
            SetLastErrno(EAFNOSUPPORT);
            return 0;
        }
        value |= (addressParts[0] << 24) | (addressParts[1] << 16);
        break;

    case 3:                /* a.b.c.d -- 8.8.8.8 bits */
        if (value > 0xff || addressParts[0] > 0xff || addressParts[1] > 0xff || addressParts[2] > 0xff)
        {
            SetLastErrno(EAFNOSUPPORT);
            return 0;
        }
        value |= (addressParts[0] << 24) | (addressParts[1] << 16) | (addressParts[2] << 8);
        break;

    default:
        SetLastErrno(EAFNOSUPPORT);
        return 0;
    }

    if (pOutNetworkAddress != NULL)
    {
        pOutNetworkAddress->s_addr = nn::socket::InetHtonl((in_addr_t)value);
    }

    return 1;
}

}}}
