﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdarg>
#include <mutex>

#include <nn/lmem/lmem_ExpHeap.h>

#include <nn/os.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/os/os_MemoryHeapApi.h>
#include <nn/os/os_MemoryHeapCommon.h>
#include <nn/os/os_Mutex.h>
#include <nn/nn_SdkLog.h>

#include <nn/sf/sf_Types.h>
#include <nn/svc/svc_Result.h>
#include <nn/socket/socket_Result.h>
#include <nn/sf/hipc/sf_HipcResult.h>

#include <nn/socket/socket_Types.h>
#include <nn/socket/socket_Constants.h>
#include <nn/socket/socket_ConstantsPrivate.h>
#include <nn/socket/private/ioctl.h>
#include <nn/socket/private/session.h>
#include <nn/socket/private/socket_PlatformTypesTranslation.h>
#include <nn/socket/socket_Statistics.h>
#include <nn/socket/sfdl/socket.bsd.sfdl.h>
#include <nn/socket/resolver/resolver_Client.h>
#include <cerrno>

#include "socket_Api.h"
#include "socket_Allocator.h"
#include "socket_CreateClientByHipc.h"
#include "../resolver/resolver_LibraryReferenceCountHelper.h"

namespace nn     {
namespace socket {
namespace detail {
namespace        {


uintptr_t g_MemoryPoolAddress       = 0;

nn::sf::SharedPointer<nn::socket::sf::IClient> g_pClientMonitor;
nn::sf::SharedPointer<nn::socket::sf::IClient> g_pClient;
nn::lmem::HeapHandle g_HeapHandle = nullptr;
int g_HeapGeneration = -1;

// Used in the SF function calls, gets overwritten with real pid by the kernel
const int Pid                       = 0;

Errno TranslateResultToBsdError(const nn::Result & result)
{
    if (result.IsSuccess())
    {
        return Errno::ESuccess;
    }
    else if (nn::svc::ResultInvalidCurrentMemory::Includes(result)
          || nn::svc::ResultOutOfAddressSpace::Includes(result))
    {
        return Errno::EFault;
    }
    else if (nn::sf::hipc::ResultCommunicationError::Includes(result))
    {
        // Not documented as there is no way for the developer to recover from this error.
        return Errno::EL3Hlt;
    }
    else if (nn::sf::hipc::ResultOutOfResource::Includes(result))
    {
        return Errno::EAgain;
    }
    else
    {
        NN_SDK_LOG("Unexpected Service Framework Error. Module: %d, Description: %d\n",result.GetModule(),result.GetDescription());
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        return static_cast<Errno>(-1);
    }
}

// Service framework errors take precedence.
#define NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnResult, bsdError, bsdResult)\
do {\
    Errno ret = Errno::ESuccess;\
    if((ret = TranslateResultToBsdError( nnResult )) != Errno::ESuccess)\
    {\
        bsdError = ret;\
        bsdResult = -1;\
    }\
} while(NN_STATIC_CONDITION(false))

} // namespace <unnamed>

void* Alloc(size_t size)
NN_NOEXCEPT
{
    NN_SDK_ASSERT_GREATER_EQUAL(g_HeapGeneration, 0);
    void* ptr = nullptr;

    if (!g_HeapHandle)
    {
        nn::socket::detail::SetLastError(Errno::EOpNotSupp);
    }
    else if (nullptr == (ptr = nn::lmem::AllocateFromExpHeap(g_HeapHandle, size)))
    {
        nn::socket::detail::SetLastError(Errno::ENoMem);
    };

    return ptr;
}

void* AllocAligned(size_t size, size_t alignment)
NN_NOEXCEPT
{
    NN_SDK_ASSERT_GREATER_EQUAL(g_HeapGeneration, 0);
    void* ptr = nullptr;

    if (!g_HeapHandle)
    {
        nn::socket::detail::SetLastError(Errno::EOpNotSupp);
    }
    else if (nullptr == (ptr = nn::lmem::AllocateFromExpHeap(g_HeapHandle, size, alignment)))
    {
        nn::socket::detail::SetLastError(Errno::ENoMem);
    };

    return ptr;
}

void* Calloc(size_t numberOfElements, size_t size)
NN_NOEXCEPT
{
    NN_SDK_ASSERT_GREATER_EQUAL(g_HeapGeneration, 0);
    void* pBuffer = nullptr;

    if (!g_HeapHandle)
    {
        nn::socket::detail::SetLastError(Errno::EOpNotSupp);
    }
    else if (nullptr == (pBuffer = Alloc(numberOfElements * size)))
    {
        nn::socket::detail::SetLastError(Errno::ENoMem);
    }
    else
    {
        std::memset(pBuffer, 0x00, numberOfElements * size);
    }

    return pBuffer;
}

void* Realloc(void* address, size_t size)
NN_NOEXCEPT
{
    NN_SDK_ASSERT_GREATER_EQUAL(g_HeapGeneration, 0);
    void*  newAddress = nullptr;
    size_t oldSize;

    if (!g_HeapHandle)
    {
        nn::socket::detail::SetLastError(Errno::EOpNotSupp);
    }
    else if (size == (oldSize = nn::lmem::GetExpHeapBlockSize(address)))
    {
        // the same
        return address;
    }
    else if (oldSize > size)
    {
        // smaller
        size_t retSize = nn::lmem::ResizeExpHeapBlock(g_HeapHandle, address, size);
        return retSize == size ? address : nullptr;
    }
    //bigger
    else if (nullptr == (newAddress = Alloc(size)))
    {
        //error
        nn::socket::detail::SetLastError(Errno::ENoMem);
    }
    else
    {
        // copy
        std::memcpy(newAddress, address, oldSize);
        Free(address);
    }

    return newAddress;
}

void Free(void* address)
NN_NOEXCEPT
{
    NN_SDK_ASSERT_GREATER_EQUAL(g_HeapGeneration, 0);
    if (!g_HeapHandle)
    {
        nn::socket::detail::SetLastError(Errno::EOpNotSupp);
    }
    else if (nullptr != address)
    {
        nn::lmem::FreeToExpHeap(g_HeapHandle, address);
    }
}

bool HeapIsAvailable(int heapGeneration)
NN_NOEXCEPT
{
    return g_HeapHandle && heapGeneration == g_HeapGeneration;
}

int GetHeapGeneration()
NN_NOEXCEPT
{
    return g_HeapGeneration;
}

Result InitializeCommon(const nn::socket::Config& config)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(!g_pClient, "Socket library has already been initialized.");
    NN_ABORT_UNLESS_NOT_NULL(config.GetMemoryPool());
    NN_ABORT_UNLESS(1 <= config.GetConcurrencyCountMax() && config.GetConcurrencyCountMax() <= ConcurrencyLimitMax);
    NN_ABORT_UNLESS(config.GetAllocatorPoolSize() < config.GetMemoryPoolSize());
    NN_ABORT_UNLESS_ALIGNED(config.GetMemoryPoolSize(), nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_ALIGNED(config.GetAllocatorPoolSize(), nn::os::MemoryPageSize);
    if (config.GetMemoryPoolSize() < nn::socket::MinSocketMemoryPoolSize && config.IsSystemClient() == false)
    {
        NN_SDK_LOG("ERROR: Provided memory pool is less than nn::socket::MinSocketMemoryPoolSize (%d KB).", nn::socket::MinSocketMemoryPoolSize / 1024);
        return nn::socket::ResultInsufficientProvidedMemory();
    }

    g_HeapHandle = nn::lmem::CreateExpHeap(
        static_cast<Bit8*>(config.GetMemoryPool()) + config.GetMemoryPoolSize() - config.GetAllocatorPoolSize(),
        config.GetAllocatorPoolSize(),
        nn::lmem::CreationOption_ThreadSafe);
    NN_ABORT_UNLESS_NOT_NULL(g_HeapHandle);

    const size_t transferMemorySize = config.GetMemoryPoolSize() - config.GetAllocatorPoolSize();
    nn::os::TransferMemoryType transferMemory;
    nn::os::CreateTransferMemory(
        &transferMemory,
        config.GetMemoryPool(),
        transferMemorySize,
        nn::os::MemoryPermission_None);
    nn::os::NativeHandle handle = nn::os::DetachTransferMemory(&transferMemory);
    const char* pServicePath = config.IsSystemClient() ? SocketServiceSystemPortName : SocketServiceUserPortName;
    g_pClientMonitor = CreateClientMonitorByHipc(pServicePath);
    g_pClient = CreateClientServiceByHipc(config.GetConcurrencyCountMax(), pServicePath);

    int ret;
    nn::socket::sf::LibraryConfigData libraryConfigData = {0};
    libraryConfigData.version                     = config.GetVersion();
    libraryConfigData.tcpInitialSendBufferSize    = config.GetTcpInitialSendBufferSize();
    libraryConfigData.tcpInitialReceiveBufferSize = config.GetTcpInitialReceiveBufferSize();
    libraryConfigData.tcpAutoSendBufferSizeMax    = config.GetTcpAutoSendBufferSizeMax();
    libraryConfigData.tcpAutoReceiveBufferSizeMax = config.GetTcpAutoReceiveBufferSizeMax();
    libraryConfigData.udpSendBufferSize           = config.GetUdpSendBufferSize();
    libraryConfigData.udpReceiveBufferSize        = config.GetUdpReceiveBufferSize();
    libraryConfigData.socketBufferEfficiency      = config.GetSocketBufferEfficiency();
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_pClient->RegisterClient(
        &ret,
        Pid,
        nn::sf::NativeHandle(handle, true),
        transferMemorySize,
        libraryConfigData));
    NN_ABORT_UNLESS(ret == 0);
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_pClientMonitor->StartMonitoring(Pid));
    nn::os::DestroyTransferMemory(&transferMemory);

    g_HeapGeneration = (g_HeapGeneration + 1) % MinimumHeapAlignment;

    // This call must come before nn::socket::resolver::EnableResolverCalls()
    NN_ABORT_UNLESS_RESULT_SUCCESS(resolver::InitializeClient());

    nn::socket::resolver::EnableResolverCalls();

    return ResultSuccess();
}

Result Initialize(void* memoryPool, size_t memoryPoolSize, size_t allocatorPoolSize, int concurrencyLimit)
NN_NOEXCEPT
{
    Result result;
    ConfigDefault defaultUserConfig(memoryPool, memoryPoolSize, allocatorPoolSize);
    defaultUserConfig.SetConcurrencyCountMax(concurrencyLimit);
    result = InitializeCommon(defaultUserConfig);
    return result;
}

Result Initialize(const nn::socket::Config& config)
NN_NOEXCEPT
{
    Result result;
    result = InitializeCommon(config);
    return result;
}

Result Finalize()
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");

    nn::socket::resolver::DisableResolverCalls();

    DestroyClientServiceByHipc(&g_pClient);

    DestroyClientMonitorByHipc(&g_pClientMonitor);

    // Block here until all resolver calls are complete.
    nn::socket::resolver::WaitForAllResolverCallsToComplete();

    nn::socket::resolver::FinalizeClient();

    // Set to global to nullptr before taking down heap as
    // the handle is used as a gate for socket layer memory availability.
    nn::lmem::HeapHandle localHandle = g_HeapHandle;
    g_HeapHandle = nullptr;
    nn::lmem::DestroyExpHeap(localHandle);

    if (g_MemoryPoolAddress != 0)
    {
        nn::os::FreeMemoryBlock(g_MemoryPoolAddress, DefaultSocketMemoryPoolSize);
        g_MemoryPoolAddress = 0;
    }

    return ResultSuccess();
}

void SetLastError(Errno error)
NN_NOEXCEPT
{
    if(!g_pClient)
    {
        return;
    }

    errno = static_cast<int>(error);
}

void SetLastErrno(int error)
NN_NOEXCEPT
{
    if(!g_pClient)
    {
        return;
    }

    errno = error;
}

Errno GetLastError()
NN_NOEXCEPT
{
    if(!g_pClient)
    {
        return Errno::EInval;
    }

    return static_cast<Errno>(errno);
}

int GetLastErrno()
NN_NOEXCEPT
{
    if(!g_pClient)
    {
        return EINVAL;
    }

    return errno;
}

ssize_t Recv(int socket, void* outBuffer, size_t outBufferLength, MsgFlag flags)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    int result;
    Errno error = Errno::ESuccess;

    if (outBufferLength == 0)
    {
        return 0;
    }
    // EInval over EFault to be more Cafe-like
    else if (outBuffer == nullptr)
    {
        nn::socket::detail::SetLastError(Errno::EInval);
        return -1;
    }
    else if (outBufferLength > UINT_MAX)
    {
        nn::socket::detail::SetLastError(Errno::EFault);
        return -1;
    }

    nn::sf::OutBuffer data(
        reinterpret_cast<char*>(outBuffer),
        outBufferLength);
    nn::Result nnRes = g_pClient->Recv(&result, reinterpret_cast<int*>(&error), socket, data, static_cast<int>(flags));
    NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes, error, result);
    if (result < 0)
    {
        nn::socket::detail::SetLastError(error);
    }

    return result;
}

ssize_t Recv(int socket, void* outBuffer, size_t outBufferLength, int flags)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    int result;
    Errno error = Errno::ESuccess;

    if (outBufferLength == 0)
    {
        return 0;
    }
    // EInval over EFault to be more Cafe-like
    else if (outBuffer == nullptr)
    {
        nn::socket::detail::SetLastError(Errno::EInval);
        return -1;
    }
    else if (outBufferLength > UINT_MAX)
    {
        nn::socket::detail::SetLastError(Errno::EFault);
        return -1;
    }

    nn::sf::OutBuffer data(
        reinterpret_cast<char*>(outBuffer),
        outBufferLength);
    nn::Result nnRes = g_pClient->Recv(&result, reinterpret_cast<int*>(&error), socket, data, flags);
    NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes, error, result);
    if (result < 0)
    {
        nn::socket::detail::SetLastError(error);
    }

    return result;
}

ssize_t RecvFrom(int socket, void* pOutBuffer, size_t pOutBufferLength, MsgFlag flags, SockAddr* pOutAddress, SockLenT* pOutAddressLength)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    int result;
    Errno error = Errno::ESuccess;

    if (pOutBufferLength == 0)
    {
        return 0;
    }
    // EInval over EFault to be more Cafe-like
    else if (pOutBuffer == nullptr)
    {
        nn::socket::detail::SetLastError(Errno::EInval);
        return -1;
    }
    else if (pOutBufferLength > UINT_MAX)
    {
        nn::socket::detail::SetLastError(Errno::EFault);
        return -1;
    }

    if (pOutAddress == nullptr || pOutAddressLength == nullptr || *pOutAddressLength == 0)
    {
        return detail::Recv(socket, pOutBuffer, pOutBufferLength, flags);
    }

    nn::sf::OutBuffer data(
        reinterpret_cast<char*>(pOutBuffer),
        pOutBufferLength);
    nn::sf::OutBuffer address(
        reinterpret_cast<char*>(pOutAddress),
        *pOutAddressLength);

    unsigned int length;
    nn::Result nnRes = g_pClient->RecvFrom(
        &result, reinterpret_cast<int*>(&error), socket, data, static_cast<int>(flags), address, &length);
    NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes, error, result);

    if (0 <= result)
    {
        *pOutAddressLength = static_cast<SockLenT>(length);
    }
    else
    {
        nn::socket::detail::SetLastError(error);
    }

    return result;
}

ssize_t RecvFrom(int socket, void* pOutBuffer, size_t pOutBufferLength, int flags, sockaddr* pOutAddress, socklen_t* pOutAddressLength)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    int result;
    Errno error = Errno::ESuccess;

    if (pOutBufferLength == 0)
    {
        return 0;
    }
    // EInval over EFault to be more Cafe-like
    else if (pOutBuffer == nullptr)
    {
        nn::socket::detail::SetLastError(Errno::EInval);
        return -1;
    }
    else if (pOutBufferLength > UINT_MAX)
    {
        nn::socket::detail::SetLastError(Errno::EFault);
        return -1;
    }

    if (pOutAddress == nullptr || pOutAddressLength == nullptr || *pOutAddressLength == 0)
    {
        return Recv(socket, pOutBuffer, pOutBufferLength, flags);
    }

    nn::sf::OutBuffer data(
        reinterpret_cast<char*>(pOutBuffer),
        pOutBufferLength);
    nn::sf::OutBuffer address(
        reinterpret_cast<char*>(pOutAddress),
        *pOutAddressLength);

    unsigned int length;
    nn::Result nnRes = g_pClient->RecvFrom(
        &result, reinterpret_cast<int*>(&error), socket, data, flags, address, &length);
    NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes, error, result);

    if (0 <= result)
    {
        *pOutAddressLength = static_cast<socklen_t>(length);
    }
    else
    {
        nn::socket::detail::SetLastError(error);
    }

    return result;
}

ssize_t RecvMsg(int socket, MsgHdr* pOutMessage, MsgFlag flags)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    NN_UNUSED(socket);
    NN_UNUSED(pOutMessage);
    NN_UNUSED(flags);
    nn::socket::detail::SetLastError(Errno::EOpNotSupp);
    return -1;
}

ssize_t RecvMsg(int socket, msghdr* pOutMessage, int flags)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    NN_UNUSED(socket);
    NN_UNUSED(pOutMessage);
    NN_UNUSED(flags);
    nn::socket::detail::SetLastError(Errno::EOpNotSupp);
    return -1;
}

ssize_t RecvMMsg(int socket, MMsgHdr* pMsgvec, size_t vlen, MsgFlag flags, nn::TimeSpan *pTimeout)
NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_pClient, "socket library has not been initialized");
    int result = 0;
    Errno error = Errno::ESuccess;
    const size_t maxMessages = vlen;
    char *pOutBuf = nullptr;

    // otherwise this is capped to 1024 (UIO_MAXIOV) in copyiniov() in uio.cpp
    if ((pMsgvec == nullptr) || (maxMessages < 1) || (maxMessages > nn::socket::MaxNumMessages))
    {
        nn::socket::detail::SetLastError(Errno::EInval);
        return -1;
    }


    do
    {
        // we want all MaxMessageBytes to be used for the payload so add space for writing pMsgvec::msg_len and pMsgvec::msg_hdr::msg_name
        const size_t bufLen = nn::socket::MaxMessagesBytes + nn::socket::MaxNumMessages * (sizeof(sockaddr_in) + sizeof(pMsgvec[0].msg_len));

        pOutBuf = static_cast<char *>(AllocAligned(bufLen, 8));
        if( pOutBuf == nullptr )
        {
            result = -1;
            // SetLastError(Errno::ENoMem) called inside AllocAligned()
            break;
        }

        nn::sf::OutBuffer outBuffer(pOutBuf, bufLen);
        const nn::socket::sf::Timespec ts = {(pTimeout != nullptr) ? pTimeout->GetSeconds() : 0,
                                             (pTimeout != nullptr) ? pTimeout->GetNanoSeconds() % (1000 * 1000 * 1000) : 0};

        nn::Result nnRes = g_pClient->RecvMMsg(&result, reinterpret_cast<int*>(&error), socket, outBuffer, maxMessages, static_cast<int>(flags), ts);
        NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes,error,result);

        if (0 <= result)
        {
            const int numReceivedMessages = result;

            char *pBufWalker = pOutBuf;
            for( int msgidx = 0; msgidx < numReceivedMessages; msgidx++ )
            {
                std::memcpy(&pMsgvec[msgidx].msg_hdr.msg_namelen, pBufWalker, sizeof(pMsgvec[msgidx].msg_hdr.msg_namelen));
                pBufWalker += sizeof(pMsgvec[msgidx].msg_hdr.msg_namelen);
                std::memcpy(pMsgvec[msgidx].msg_hdr.msg_name, pBufWalker, pMsgvec[msgidx].msg_hdr.msg_namelen);
                pBufWalker += pMsgvec[msgidx].msg_hdr.msg_namelen;

                std::memcpy(&pMsgvec[msgidx].msg_len, pBufWalker, sizeof(pMsgvec[msgidx].msg_len));
                pBufWalker += sizeof(pMsgvec[msgidx].msg_len);
                std::memcpy(pMsgvec[msgidx].msg_hdr.msg_iov[0].iov_base, pBufWalker, pMsgvec[msgidx].msg_len);
                pBufWalker += pMsgvec[msgidx].msg_len;
            }
        }
        else
        {
            nn::socket::detail::SetLastError(error);
        }

    } while (0);

    Free(pOutBuf);
    pOutBuf = nullptr;

    return result;
}

ssize_t RecvMMsg(int socket, mmsghdr* pMsgvec, size_t vlen, int flags, nn::TimeSpan *pTimeout)
NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_pClient, "socket library has not been initialized");
    int result = 0;
    Errno error = Errno::ESuccess;
    const size_t maxMessages = vlen;

    // otherwise this is capped to 1024 (UIO_MAXIOV) in copyiniov() in uio.cpp
    if ((pMsgvec == nullptr) || (maxMessages < 1) || (maxMessages > nn::socket::MaxNumMessages))
    {
        nn::socket::detail::SetLastError(Errno::EInval);
        return -1;
    }

    // we want all MaxMessageBytes to be used for the payload so add space for writing pMsgvec::msg_len and pMsgvec::msg_hdr::msg_name
    const size_t bufLen = nn::socket::MaxMessagesBytes + nn::socket::MaxNumMessages * (sizeof(sockaddr_in) + sizeof(pMsgvec[0].msg_len));
    char *pOutBuf = nullptr;

    do
    {
        pOutBuf = static_cast<char *>(AllocAligned(bufLen, 8));
        if( pOutBuf == nullptr )
        {
            result = -1;
            // SetLastError(Errno::ENoMem) called inside AllocAligned()
            break;
        }

        nn::sf::OutBuffer outBuffer(pOutBuf, bufLen);
        const nn::socket::sf::Timespec ts = {(pTimeout != nullptr) ? pTimeout->GetSeconds() : 0,
                                             (pTimeout != nullptr) ? pTimeout->GetNanoSeconds() % (1000 * 1000 * 1000) : 0};

        nn::Result nnRes = g_pClient->RecvMMsg(&result, reinterpret_cast<int*>(&error), socket, outBuffer, maxMessages, flags, ts);
        NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes,error,result);

        if (0 <= result)
        {
            const int numReceivedMessages = result;

            char *pBufWalker = pOutBuf;
            for( int msgidx = 0; msgidx < numReceivedMessages; msgidx++ )
            {
                if ((pMsgvec[msgidx].msg_hdr.msg_name != nullptr) && (pMsgvec[msgidx].msg_hdr.msg_namelen == sizeof(sockaddr_in)))
                {
                    std::memcpy(pMsgvec[msgidx].msg_hdr.msg_name, pBufWalker, pMsgvec[msgidx].msg_hdr.msg_namelen);
                }
                // the server filled in msg_hdr.msg_name so we want to skip past it whether we read it or not
                pBufWalker += sizeof(sockaddr_in);

                std::memcpy(&pMsgvec[msgidx].msg_len, pBufWalker, sizeof(pMsgvec[msgidx].msg_len));
                pBufWalker += sizeof(pMsgvec[msgidx].msg_len);
                std::memcpy(pMsgvec[msgidx].msg_hdr.msg_iov[0].iov_base, pBufWalker, pMsgvec[msgidx].msg_len);
                pBufWalker += pMsgvec[msgidx].msg_len;
            }
        }
        else
        {
            nn::socket::detail::SetLastError(error);
        }

    } while (0);

    Free(pOutBuf);
    pOutBuf = nullptr;

    return result;
}

ssize_t Send(int socket, const void* inBuffer, size_t inBufferLength, MsgFlag flags)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    int result = 0;
    Errno error = Errno::ESuccess;

    if (inBufferLength == 0)
    {
        return 0;
    }
    // EInval over EFault to be more Cafe-like
    else if (inBuffer == nullptr)
    {
        nn::socket::detail::SetLastError(Errno::EInval);
        return -1;
    }
    else if (inBufferLength > UINT_MAX)
    {
        nn::socket::detail::SetLastError(Errno::EFault);
        return -1;
    }

    nn::sf::InBuffer data(reinterpret_cast<const char*>(inBuffer), inBufferLength);
    nn::Result nnRes = g_pClient->Send(&result, reinterpret_cast<int*>(&error), socket, data, static_cast<int>(flags));
    NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes, error, result);

    if (result < 0)
    {
        nn::socket::detail::SetLastError(error);
    }
    return result;
}

ssize_t Send(int socket, const void* inBuffer, size_t inBufferLength, int flags)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    int result = 0;
    Errno error = Errno::ESuccess;

    if (inBufferLength == 0)
    {
        return 0;
    }
    // EInval over EFault to be more Cafe-like
    else if (inBuffer == nullptr)
    {
        nn::socket::detail::SetLastError(Errno::EInval);
        return -1;
    }
    else if (inBufferLength > UINT_MAX)
    {
        nn::socket::detail::SetLastError(Errno::EFault);
        return -1;
    }

    nn::sf::InBuffer data(reinterpret_cast<const char*>(inBuffer), inBufferLength);
    nn::Result nnRes = g_pClient->Send(&result, reinterpret_cast<int*>(&error), socket, data, flags);
    NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes, error, result);

    if (result < 0)
    {
        nn::socket::detail::SetLastError(error);
    }
    return result;
}

ssize_t SendTo(int socket, const void* inBuffer, size_t inBufferLength, MsgFlag flags, const SockAddr* pInAddress, SockLenT inAddressLength)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    int result;
    Errno error = Errno::ESuccess;

    if (pInAddress == nullptr || inAddressLength == 0)
    {
        return detail::Send(socket, inBuffer, inBufferLength, flags);
    }

    if (inBufferLength == 0)
    {
        return 0;
    }
    // EInval over EFault to be more Cafe-like
    else if (inBuffer == nullptr)
    {
        nn::socket::detail::SetLastError(Errno::EInval);
        return -1;
    }
    else if (inBufferLength > UINT_MAX)
    {
        nn::socket::detail::SetLastError(Errno::EFault);
        return -1;
    }

    nn::sf::InBuffer data(reinterpret_cast<const char*>(inBuffer), inBufferLength);
    nn::sf::InBuffer address(reinterpret_cast<const char*>(pInAddress), inAddressLength);

    nn::Result nnRes = g_pClient->SendTo(&result, reinterpret_cast<int*>(&error), socket, data, static_cast<int>(flags), address);
    NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes, error, result);

    if (result < 0)
    {
        nn::socket::detail::SetLastError(error);
    }

    return result;
}

ssize_t SendTo(int socket, const void* inBuffer, size_t inBufferLength, int flags, const sockaddr* inAddress, socklen_t inAddressLength)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    int result;
    Errno error = Errno::ESuccess;

    if (inAddress == nullptr || inAddressLength == 0)
    {
        return Send(socket, inBuffer, inBufferLength, flags);
    }

    if (inBufferLength == 0)
    {
        return 0;
    }
    // EInval over EFault to be more Cafe-like
    else if (inBuffer == nullptr)
    {
        nn::socket::detail::SetLastError(Errno::EInval);
        return -1;
    }
    else if (inBufferLength > UINT_MAX)
    {
        nn::socket::detail::SetLastError(Errno::EFault);
        return -1;
    }

    nn::sf::InBuffer data(reinterpret_cast<const char*>(inBuffer), inBufferLength);
    nn::sf::InBuffer address(reinterpret_cast<const char*>(inAddress), inAddressLength);

    nn::Result nnRes = g_pClient->SendTo(&result, reinterpret_cast<int*>(&error), socket, data, flags, address);
    NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes, error, result);

    if (result < 0)
    {
        nn::socket::detail::SetLastError(error);
    }

    return result;
}

ssize_t SendMsg(int socket, const MsgHdr* pMessage, MsgFlag flags)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    NN_UNUSED(socket);
    NN_UNUSED(pMessage);
    NN_UNUSED(flags);
    nn::socket::detail::SetLastError(Errno::EOpNotSupp);
    return -1;
}

ssize_t SendMsg(int socket, const msghdr* pMessage, int flags)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    NN_UNUSED(socket);
    NN_UNUSED(pMessage);
    NN_UNUSED(flags);
    nn::socket::detail::SetLastError(Errno::EOpNotSupp);
    return -1;
}

ssize_t SendMMsg(int socket, const MMsgHdr* pMsgvec, size_t vlen, MsgFlag flags)
NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_pClient, "socket library has not been initialized");
    int result = 0;
    Errno error = Errno::ESuccess;
    int bufLen = 0;
    nn::socket::sf::MsgHdr *pSfMsghdr = nullptr;
    char *pBuf = nullptr;
    const size_t numMessages = vlen;

    if (pMsgvec == nullptr)
    {
        nn::socket::detail::SetLastError(Errno::EInval);
        return -1;
    }
    // otherwise this is capped to 1024 (UIO_MAXIOV) in copyiniov() in uio.cpp
    else if ((numMessages < 1) || (numMessages > nn::socket::MaxNumMessages))
    {
        nn::socket::detail::SetLastError(Errno::EMsgSize);
        return -1;
    }

    do
    {
        const size_t msghdrArraySize = sizeof(*pSfMsghdr) * numMessages;
        pSfMsghdr = static_cast<nn::socket::sf::MsgHdr *>(Alloc(msghdrArraySize));
        if( pSfMsghdr == nullptr )
        {
            result = -1;
            // SetLastError(Errno::ENoMem) called inside Alloc()
            break;
        }

        for( int msghdridx = 0; msghdridx < numMessages; msghdridx++ )
        {
            const MsgHdr *pRawMsgHdr = &pMsgvec[msghdridx].msg_hdr;

            bufLen += (pRawMsgHdr->msg_namelen == sizeof(nn::socket::SockAddrIn)) ? sizeof(sockaddr_in) : pRawMsgHdr->msg_namelen;
            bufLen += pRawMsgHdr->msg_controllen;

            for( int iovidx = 0; iovidx < pRawMsgHdr->msg_iovlen; iovidx++ )
            {
                bufLen += pRawMsgHdr->msg_iov[iovidx].iov_len;
            }

            if( bufLen > nn::socket::MaxMessagesBytes )
            {
                result = -1;
                nn::socket::detail::SetLastError(Errno::EMsgSize);
                break;
            }
        }

        pBuf = static_cast<char *>(AllocAligned(sizeof(*pBuf) * bufLen, 8));
        if( pBuf == nullptr )
        {
            result = -1;
            // SetLastError(Errno::ENoMem) called inside AllocAligned()
            break;
        }

        char *pBufWalker = pBuf;
        for( int msghdridx = 0; msghdridx < numMessages; msghdridx++ )
        {
            const MsgHdr *pRawMsgHdr = &pMsgvec[msghdridx].msg_hdr;
            sockaddr_in msg_name = {};
            socklen_t msg_namelen = (pRawMsgHdr->msg_namelen == sizeof(nn::socket::SockAddrIn)) ? sizeof(sockaddr_in) : pRawMsgHdr->msg_namelen;
            nn::socket::detail::CopyToPlatform(&msg_name, static_cast<nn::socket::SockAddrIn *>(pRawMsgHdr->msg_name));

            std::memcpy(pBufWalker, &msg_name, msg_namelen);
            pSfMsghdr[msghdridx].msg_name = pBufWalker - pBuf;
            pSfMsghdr[msghdridx].msg_namelen = msg_namelen;
            pBufWalker += msg_namelen;

            std::memcpy(pBufWalker, pRawMsgHdr->msg_control, pRawMsgHdr->msg_controllen);
            pSfMsghdr[msghdridx].msg_control = pBufWalker - pBuf;
            pSfMsghdr[msghdridx].msg_controllen = pRawMsgHdr->msg_controllen;
            pBufWalker += pRawMsgHdr->msg_controllen;

            pSfMsghdr[msghdridx].msg_iovlen = pRawMsgHdr->msg_iovlen;

            for( int iovidx = 0; iovidx < pRawMsgHdr->msg_iovlen; iovidx++ )
            {
                std::memcpy(pBufWalker, pRawMsgHdr->msg_iov[iovidx].iov_base, pRawMsgHdr->msg_iov[iovidx].iov_len);
                pSfMsghdr[msghdridx].msg_iov[iovidx].iov_base = pBufWalker - pBuf;
                pSfMsghdr[msghdridx].msg_iov[iovidx].iov_len = pRawMsgHdr->msg_iov[iovidx].iov_len;
                pBufWalker += pRawMsgHdr->msg_iov[iovidx].iov_len;
            }
        }

        // make sure we didn't write past the end of the buffer
        NN_ABORT_UNLESS(pBufWalker <= pBuf + bufLen);

        auto inArrayMsgHdr = nn::sf::InArray<nn::socket::sf::MsgHdr>(pSfMsghdr, msghdrArraySize);
        auto inBufferPackedMsgHdrData = nn::sf::InBuffer(pBuf, sizeof(*pBuf) * bufLen);

        nn::Result nnRes = (g_pClient->SendMMsg(&result, reinterpret_cast<int*>(&error), socket, inArrayMsgHdr, vlen, inBufferPackedMsgHdrData, static_cast<int>(flags)));
        NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes, error, result);

        if( 0 > result )
        {
            nn::socket::detail::SetLastError(error);
        }

    } while (0);

    Free(pBuf);
    pBuf = nullptr;
    Free(pSfMsghdr);
    pSfMsghdr = nullptr;

    return result;
}

ssize_t SendMMsg(int socket, const mmsghdr* pMsgvec, size_t vlen, int flags)
NN_NOEXCEPT
{
    NN_SDK_REQUIRES(g_pClient, "socket library has not been initialized");
    int result = 0;
    Errno error = Errno::ESuccess;
    int bufLen = 0;
    nn::socket::sf::MsgHdr *pSfMsghdr = nullptr;
    char *pBuf = nullptr;
    const size_t numMessages = vlen;

    if (pMsgvec == nullptr)
    {
        nn::socket::detail::SetLastError(Errno::EInval);
        return -1;
    }
    // otherwise this is capped to 1024 (UIO_MAXIOV) in copyiniov() in uio.cpp
    else if ((numMessages < 1) || (numMessages > nn::socket::MaxNumMessages))
    {
        nn::socket::detail::SetLastError(Errno::EMsgSize);
        return -1;
    }

    do
    {
        const size_t msghdrArraySize = sizeof(*pSfMsghdr) * numMessages;
        pSfMsghdr = static_cast<nn::socket::sf::MsgHdr *>(Alloc(msghdrArraySize));
        if( pSfMsghdr == nullptr )
        {
            result = -1;
            // SetLastError(Errno::ENoMem) called inside Alloc()
            break;
        }

        for( int msghdridx = 0; msghdridx < numMessages; msghdridx++ )
        {
            const msghdr *pRawMsgHdr = &pMsgvec[msghdridx].msg_hdr;

            bufLen += pRawMsgHdr->msg_namelen + pRawMsgHdr->msg_controllen;

            for( int iovidx = 0; iovidx < pRawMsgHdr->msg_iovlen; iovidx++ )
            {
                bufLen += pRawMsgHdr->msg_iov[iovidx].iov_len;
            }

            if( bufLen > nn::socket::MaxMessagesBytes )
            {
                result = -1;
                nn::socket::detail::SetLastError(Errno::EMsgSize);
                break;
            }
        }

        pBuf = static_cast<char *>(AllocAligned(sizeof(*pBuf) * bufLen, 8));
        if( pBuf == nullptr )
        {
            result = -1;
            // SetLastError(Errno::ENoMem) called inside AllocAligned()
            break;
        }

        char *pBufWalker = pBuf;
        for( int msghdridx = 0; msghdridx < numMessages; msghdridx++ )
        {
            const msghdr *pRawMsgHdr = &pMsgvec[msghdridx].msg_hdr;

            std::memcpy(pBufWalker, pRawMsgHdr->msg_name, pRawMsgHdr->msg_namelen);
            pSfMsghdr[msghdridx].msg_name = pBufWalker - pBuf;
            pSfMsghdr[msghdridx].msg_namelen = pRawMsgHdr->msg_namelen;
            pBufWalker += pRawMsgHdr->msg_namelen;

            std::memcpy(pBufWalker, pRawMsgHdr->msg_control, pRawMsgHdr->msg_controllen);
            pSfMsghdr[msghdridx].msg_control = pBufWalker - pBuf;
            pSfMsghdr[msghdridx].msg_controllen = pRawMsgHdr->msg_controllen;
            pBufWalker += pRawMsgHdr->msg_controllen;

            pSfMsghdr[msghdridx].msg_iovlen = pRawMsgHdr->msg_iovlen;

            for( int iovidx = 0; iovidx < pRawMsgHdr->msg_iovlen; iovidx++ )
            {
                std::memcpy(pBufWalker, pRawMsgHdr->msg_iov[iovidx].iov_base, pRawMsgHdr->msg_iov[iovidx].iov_len);
                pSfMsghdr[msghdridx].msg_iov[iovidx].iov_base = pBufWalker - pBuf;
                pSfMsghdr[msghdridx].msg_iov[iovidx].iov_len = pRawMsgHdr->msg_iov[iovidx].iov_len;
                pBufWalker += pRawMsgHdr->msg_iov[iovidx].iov_len;
            }
        }

        // make sure we didn't write past the end of the buffer
        NN_ABORT_UNLESS(pBufWalker <= pBuf + bufLen);

        auto inArrayMsgHdr = nn::sf::InArray<nn::socket::sf::MsgHdr>(pSfMsghdr, msghdrArraySize);
        auto inBufferPackedMsgHdrData = nn::sf::InBuffer(pBuf, sizeof(*pBuf) * bufLen);

        nn::Result nnRes = (g_pClient->SendMMsg(&result, reinterpret_cast<int*>(&error), socket, inArrayMsgHdr, vlen, inBufferPackedMsgHdrData, flags));
        NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes, error, result);

        if( 0 > result )
        {
            nn::socket::detail::SetLastError(error);
        }

    } while (0);

    Free(pBuf);
    pBuf = nullptr;
    Free(pSfMsghdr);
    pSfMsghdr = nullptr;

    return result;
}

int Accept(int socket, SockAddr* pOutAddress, SockLenT* pOutAddressLength)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    int result;
    Errno error = Errno::ESuccess;

    // EInval over EFault to be more Cafe-like
    if (pOutAddress == nullptr && pOutAddressLength != nullptr && *pOutAddressLength != 0)
    {
        nn::socket::detail::SetLastError(Errno::EInval);
        return -1;
    }

    unsigned int addressLength = static_cast<unsigned int>((pOutAddress && pOutAddressLength) ? *pOutAddressLength : 0);

    nn::sf::OutBuffer address(reinterpret_cast<char*>(pOutAddress), addressLength);
    nn::Result nnRes = g_pClient->Accept(&result,
                                         reinterpret_cast<int*>(&error),
                                         socket,
                                         address,
                                         &addressLength);
    NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes, error, result);

    if (0 <= result)
    {
        if (pOutAddressLength != nullptr)
        {
            *pOutAddressLength = addressLength;
        }
    }
    else
    {
        nn::socket::detail::SetLastError(error);
    }

    return result;
}

int Accept(int socket, sockaddr* outAddress, socklen_t* outAddressLength)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    int result;
    Errno error = Errno::ESuccess;

    // EInval over EFault to be more Cafe-like
    if (outAddress == nullptr && outAddressLength != nullptr && *outAddressLength != 0)
    {
        nn::socket::detail::SetLastError(Errno::EInval);
        return -1;
    }

    unsigned int addressLength = static_cast<unsigned int>(
        (outAddress && outAddressLength) ? *outAddressLength : 0);

    nn::sf::OutBuffer address(
        reinterpret_cast<char*>(outAddress),
        addressLength);

    nn::Result nnRes = g_pClient->Accept(
        &result, reinterpret_cast<int*>(&error), socket, address, &addressLength);
    NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes, error, result);

    if (0 <= result)
    {
        if (outAddressLength != nullptr)
        {
            *outAddressLength = addressLength;
        }
    }
    else
    {
        nn::socket::detail::SetLastError(error);
    }

    return result;
}

int Bind(int socket, const SockAddr* pInAddress, SockLenT inAddressLength)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    int result;
    Errno error = Errno::ESuccess;

    // EInval over EFault to be more Cafe-like
    if (pInAddress == nullptr || inAddressLength == 0)
    {
        NN_SDK_LOG("BIND passed in null or 0\n");
        nn::socket::detail::SetLastError(Errno::EInval);
        return -1;
    }

    if( inAddressLength == sizeof(SockAddr) )
    {
        inAddressLength = sizeof(sockaddr);
    }

    nn::sf::InBuffer address(reinterpret_cast<const char*>(pInAddress), inAddressLength);

    nn::Result nnRes = g_pClient->Bind(&result, reinterpret_cast<int*>(&error), socket, address);
    NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes, error, result);

    if (result < 0)
    {
        NN_SDK_LOG("BIND failed with %d\n", error);
        nn::socket::detail::SetLastError(error);
    }

    return result;
}

int Bind(int socket, const sockaddr* inAddress, socklen_t inAddressLength)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    int result;
    Errno error = Errno::ESuccess;
    // EInval over EFault to be more Cafe-like
    if (inAddress == nullptr)
    {
        nn::socket::detail::SetLastError(Errno::EInval);
        return -1;
    }

    nn::sf::InBuffer address(
        reinterpret_cast<const char*>(inAddress),
        inAddressLength);

    nn::Result nnRes = g_pClient->Bind(&result, reinterpret_cast<int*>(&error), socket, address);
    NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes, error, result);

    if (result < 0)
    {
        nn::socket::detail::SetLastError(error);
    }

    return result;
}

int Connect(int socket, const SockAddr* pInAddress, SockLenT inAddressLength)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    int result;
    Errno error = Errno::ESuccess;

    // EInval for both to be more Cafe-like
    if (pInAddress == nullptr || inAddressLength == 0)
    {
        nn::socket::detail::SetLastError(Errno::EInval);
        return -1;
    }

    if( inAddressLength == sizeof(SockAddr) )
    {
        inAddressLength = sizeof(sockaddr);
    }

    nn::sf::InBuffer address(reinterpret_cast<const char*>(pInAddress), inAddressLength);

    nn::Result nnRes = g_pClient->Connect(&result, reinterpret_cast<int*>(&error), socket, address);
    NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes, error, result);

    if (result < 0)
    {
        nn::socket::detail::SetLastError(error);
    }

    return result;
}

int Connect(int socket, const sockaddr* inAddress, socklen_t inAddressLength)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    int result;
    Errno error = Errno::ESuccess;

    // EInval for both to be more Cafe-like
    if (inAddress == nullptr)
    {
        nn::socket::detail::SetLastError(Errno::EInval);
        return -1;
    }

    nn::sf::InBuffer address(
        reinterpret_cast<const char*>(inAddress),
        inAddressLength);

    nn::Result nnRes = g_pClient->Connect(&result, reinterpret_cast<int*>(&error), socket, address);
    NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes, error, result);

    if (result < 0)
    {
        nn::socket::detail::SetLastError(error);
    }

    return result;
}

int GetPeerName(int socket, SockAddr* pOutAddress, SockLenT* pOutAddressLength)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    int result;
    Errno error = Errno::ESuccess;
    // EInval over EFault to be more Cafe-like
    if (pOutAddress == nullptr ||
        pOutAddressLength == nullptr ||
        *pOutAddressLength == 0)
    {
        nn::socket::detail::SetLastError(Errno::EInval);
        return -1;
    }

    nn::sf::OutBuffer address(reinterpret_cast<char*>(pOutAddress), *pOutAddressLength);

    unsigned int addressLength;
    nn::Result nnRes = g_pClient->GetPeerName(&result, reinterpret_cast<int*>(&error), socket, address, &addressLength);
    NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes, error, result);

    if (0 <= result)
    {
        *pOutAddressLength = (addressLength == sizeof(sockaddr))? sizeof(SockAddr) : static_cast<SockLenT>(addressLength);
    }
    else
    {
        nn::socket::detail::SetLastError(error);
    }

    return result;
}

int GetPeerName(int socket, sockaddr* outAddress, socklen_t* outAddressLength)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    int result;
    Errno error = Errno::ESuccess;
    // EInval over EFault to be more Cafe-like
    if (outAddress == nullptr ||
        outAddressLength == nullptr ||
        *outAddressLength == 0)
    {
        nn::socket::detail::SetLastError(Errno::EInval);
        return -1;
    }

    nn::sf::OutBuffer address(
        reinterpret_cast<char*>(outAddress),
        *outAddressLength);

    unsigned int addressLength;
    nn::Result nnRes = g_pClient->GetPeerName(&result, reinterpret_cast<int*>(&error), socket, address, &addressLength);
    NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes, error, result);

    if (0 <= result)
    {
        *outAddressLength = static_cast<socklen_t>(addressLength);
    }
    else
    {
        nn::socket::detail::SetLastError(error);
    }

    return result;
}

int GetSockName(int socket, SockAddr* pOutAddress, SockLenT* pOutAddressLength)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    int result;
    Errno error = Errno::ESuccess;
    // EInval over EFault to be more Cafe-like
    if (pOutAddress == nullptr ||
        pOutAddressLength == nullptr ||
        *pOutAddressLength == 0)
    {
        nn::socket::detail::SetLastError(Errno::EInval);
        return -1;
    }

    if( *pOutAddressLength == sizeof(SockAddr) )
    {
        *pOutAddressLength = sizeof(sockaddr);
    }

    nn::sf::OutBuffer address(reinterpret_cast<char*>(pOutAddress), *pOutAddressLength);

    unsigned int addressLength;
    nn::Result nnRes = g_pClient->GetSockName(&result, reinterpret_cast<int*>(&error), socket, address, &addressLength);
    NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes, error, result);

    if (0 <= result)
    {
        *pOutAddressLength = (addressLength == sizeof(sockaddr))? sizeof(SockAddr) : static_cast<SockLenT>(addressLength);
    }
    else
    {
        nn::socket::detail::SetLastError(error);
    }

    return result;
}

int GetSockName(int socket, sockaddr* outAddress, socklen_t* outAddressLength)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    int result;
    Errno error = Errno::ESuccess;
    // EInval over EFault to be more Cafe-like
    if (outAddress == nullptr ||
        outAddressLength == nullptr ||
        *outAddressLength == 0)
    {
        nn::socket::detail::SetLastError(Errno::EInval);
        return -1;
    }

    nn::sf::OutBuffer address(
        reinterpret_cast<char*>(outAddress),
        *outAddressLength);

    unsigned int addressLength;
    nn::Result nnRes = g_pClient->GetSockName(&result, reinterpret_cast<int*>(&error), socket, address, &addressLength);
    NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes, error, result);

    if (0 <= result)
    {
        *outAddressLength = static_cast<socklen_t>(addressLength);
    }
    else
    {
        nn::socket::detail::SetLastError(error);
    }

    return result;
}

int GetSockOpt(int socket, Level level, Option optionName, void* pOutOptionValue, SockLenT* pOutOptionLength)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");

    int                 result              = -1;
    Errno               error               = Errno::ESuccess;

    // EInval over EFault to be more Cafe-like
    if (pOutOptionValue == nullptr ||
        pOutOptionLength == nullptr ||
        *pOutOptionLength == 0)
    {
        nn::socket::detail::SetLastError(Errno::EInval);
        return -1;
    }

    {
        unsigned int optionLength = 0;
        nn::sf::OutBuffer optionBuffer(reinterpret_cast<char*>(pOutOptionValue),
                                       *pOutOptionLength);
        nn::Result nnRes = g_pClient->GetSockOpt(&result,
                                                 reinterpret_cast<int*>(&error),
                                                 socket,
                                                 static_cast<int>(level),
                                                 static_cast<int>(optionName),
                                                 optionBuffer,
                                                 &optionLength);
        NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes, error, result);

        if (0 <= result)
        {
            *pOutOptionLength = static_cast<SockLenT>(optionLength);
        }
        else
        {
            nn::socket::detail::SetLastError(error);
        }
    }

    return result;
}

int GetSockOpt(int socket, int level, int optionName, void* outOptionValue, socklen_t* outOptionLength)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    int result;
    Errno error = Errno::ESuccess;
    // EInval over EFault to be more Cafe-like
    if (outOptionValue == nullptr ||
        outOptionLength == nullptr ||
        *outOptionLength == 0)
    {
        nn::socket::detail::SetLastError(Errno::EInval);
        return -1;
    }


    nn::sf::OutBuffer optionBuffer(
        reinterpret_cast<char*>(outOptionValue),
        *outOptionLength);

    unsigned int optionLength;
    nn::Result nnRes = g_pClient->GetSockOpt(
        &result, reinterpret_cast<int*>(&error), socket, level, optionName, optionBuffer, &optionLength);
    NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes, error, result);

    if (0 <= result)
    {
        *outOptionLength = static_cast<socklen_t>(optionLength);
    }
    else
    {
        nn::socket::detail::SetLastError(error);
    }

    return result;
}

int Listen(int socket, int backlog)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    int result;
    Errno error = Errno::ESuccess;

    nn::Result nnRes = g_pClient->Listen(&result, reinterpret_cast<int*>(&error), socket, backlog);
    NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes, error, result);

    if (result < 0)
    {
        nn::socket::detail::SetLastError(error);
    }

    return result;
}

int Fcntl(int socket, FcntlCommand command, ...)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    int result;
    Errno error = Errno::ESuccess;
    int fcntlArg;
    va_list args;

    va_start(args, command);
    fcntlArg = va_arg(args, int);
    va_end(args);

    nn::Result nnRes = g_pClient->Fcntl(&result, reinterpret_cast<int*>(&error), socket, static_cast<int>(command), fcntlArg);
    NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes, error, result);

    if (result < 0)
    {
        nn::socket::detail::SetLastError(error);
    }

    return result;
}

int Fcntl(int socket, int command, ...)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    int result;
    Errno error = Errno::ESuccess;
    int fcntlCommand;
    va_list args;

    va_start(args, command);
    fcntlCommand = va_arg(args, int);
    va_end(args);

    nn::Result nnRes = g_pClient->Fcntl(&result, reinterpret_cast<int*>(&error), socket, command, fcntlCommand);
    NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes, error, result);

    if (result < 0)
    {
        nn::socket::detail::SetLastError(error);
    }

    return result;
}

int SetSockOpt(int socket, Level level, Option optionName, const void* pInOptionValue, SockLenT inOptionLength)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");

    int                 result              = -1;
    Errno               error               = Errno::ESuccess;

    // EInval over EFault to be more Cafe-like
    if (pInOptionValue == nullptr)
    {
        nn::socket::detail::SetLastError(Errno::EInval);
        result = -1;
        goto exit;
    }

    {
        nn::sf::InBuffer optionValue(reinterpret_cast<const char*>(pInOptionValue),
                                     inOptionLength);
        nn::Result nnRes = g_pClient->SetSockOpt(&result,
                                                 reinterpret_cast<int*>(&error),
                                                 socket,
                                                 static_cast<int>(level),
                                                 static_cast<int>(optionName),
                                                 optionValue);
        NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes, error, result);

        if( result < 0 )
        {
            nn::socket::detail::SetLastError(error);
        }
    }

exit:
    return result;
}

int SetSockOpt(int socket, int level, int optionName, const void* inOptionValue, socklen_t inOptionLength)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    int result;
    Errno error = Errno::ESuccess;
    // EInval over EFault to be more Cafe-like
    if (inOptionValue == nullptr)
    {
        nn::socket::detail::SetLastError(Errno::EInval);
        return -1;
    }

    nn::sf::InBuffer optionValue(
        reinterpret_cast<const char*>(inOptionValue),
        inOptionLength);
    nn::Result nnRes = g_pClient->SetSockOpt(
        &result, reinterpret_cast<int*>(&error), socket, level, optionName, optionValue);
    NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes, error, result);

    if (result < 0)
    {
        nn::socket::detail::SetLastError(error);
    }

    return result;
}

int Shutdown(int socket, ShutdownMethod how)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    int result;
    Errno error = Errno::ESuccess;

    nn::Result nnRes = g_pClient->Shutdown(&result, reinterpret_cast<int*>(&error), socket, static_cast<int>(how));
    NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes, error, result);

    if (result < 0)
    {
        nn::socket::detail::SetLastError(error);
    }

    return result;
}

int Shutdown(int socket, int how)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    int result;
    Errno error = Errno::ESuccess;

    nn::Result nnRes = g_pClient->Shutdown(&result, reinterpret_cast<int*>(&error), socket, how);
    NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes, error, result);

    if (result < 0)
    {
        nn::socket::detail::SetLastError(error);
    }

    return result;
}

int ShutdownAllSockets(bool forced)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    int result;
    Errno error = Errno::ESuccess;

    nn::Result nnRes = g_pClient->ShutdownAllSockets(&result, reinterpret_cast<int*>(&error), forced);
    NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes, error, result);

    if (result < 0)
    {
        nn::socket::detail::SetLastError(error);
    }

    return result;
}

int Write(int socket, const void* inBuffer, size_t inBufferLength)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    int result;
    Errno error = Errno::ESuccess;

    if (inBufferLength == 0)
    {
        return 0;
    }
    // EInval over EFault to be more Cafe-like
    else if (inBuffer == nullptr)
    {
        nn::socket::detail::SetLastError(Errno::EInval);
        return -1;
    }
    else if (inBufferLength > UINT_MAX)
    {
        nn::socket::detail::SetLastError(Errno::EFault);
        return -1;
    }

    nn::sf::InBuffer data(
        reinterpret_cast<const char*>(inBuffer),
        inBufferLength);
    nn::Result nnRes = g_pClient->Write(&result, reinterpret_cast<int*>(&error), socket, data);
    NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes, error, result);

    if (result < 0)
    {
        nn::socket::detail::SetLastError(error);
    }

    return result;
}

int Read(int socket, void* outBuffer, size_t outBufferLength)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    int result;
    Errno error = Errno::ESuccess;

    if (outBufferLength == 0)
    {
        return 0;
    }
    // EInval over EFault to be more Cafe-like
    else if (outBuffer == nullptr)
    {
        nn::socket::detail::SetLastError(Errno::EInval);
        return -1;
    }
    else if (outBufferLength > UINT_MAX)
    {
        nn::socket::detail::SetLastError(Errno::EFault);
        return -1;
    }
    nn::sf::OutBuffer data(
        reinterpret_cast<char*>(outBuffer),
        outBufferLength);
    nn::Result nnRes = g_pClient->Read(&result, reinterpret_cast<int*>(&error), socket, data);
    NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes, error, result);

    if (result < 0)
    {
        nn::socket::detail::SetLastError(error);
    }

    return result;
}

int SockAtMark(int socket)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    int atMark;
    return (Ioctl(socket, SIOCATMARK, &atMark, sizeof(atMark)) == -1) ? -1 : atMark;
}

int Select(int numberOfDescriptors, FdSet* pReadDescriptors, FdSet* pWriteDescriptors, FdSet* pExceptDescriptors, TimeVal* pTimeout)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    int result;
    Errno error = Errno::ESuccess;
    nn::socket::sf::SelectTimeval selTimeval = { 0, 0, true };

    if (pReadDescriptors == nullptr
        && pWriteDescriptors == nullptr
        && pExceptDescriptors == nullptr
        && pTimeout == nullptr)
    {
        nn::socket::detail::SetLastError(Errno::EInval);
        result = -1;
    }

    nn::sf::InBuffer  readIn(
        reinterpret_cast<const char*>(pReadDescriptors),
        pReadDescriptors ? sizeof(*pReadDescriptors) : 0);

    nn::sf::InBuffer  writeIn(
        reinterpret_cast<const char*>(pWriteDescriptors),
        pWriteDescriptors ? sizeof(*pWriteDescriptors) : 0);

    nn::sf::InBuffer  exceptIn(
        reinterpret_cast<const char*>(pExceptDescriptors),
        pExceptDescriptors ? sizeof(*pExceptDescriptors) : 0);

    nn::sf::OutBuffer readOut(
        reinterpret_cast<char*>(pReadDescriptors),
        pReadDescriptors ? sizeof(*pReadDescriptors) : 0);

    nn::sf::OutBuffer writeOut(
        reinterpret_cast<char*>(pWriteDescriptors),
        pWriteDescriptors ? sizeof(*pWriteDescriptors) : 0);

    nn::sf::OutBuffer exceptOut(
        reinterpret_cast<char*>(pExceptDescriptors),
        pExceptDescriptors ? sizeof(*pExceptDescriptors) : 0);

    if ( nullptr != pTimeout )
    {
        selTimeval.sec = static_cast<int64_t>(pTimeout->tv_sec);
        selTimeval.usec = static_cast<int64_t>(pTimeout->tv_usec);
        selTimeval.wasNull = false;
    }

    nn::Result nnRes = g_pClient->Select(&result, reinterpret_cast<int*>(&error), numberOfDescriptors,
                                         readIn, writeIn, exceptIn,
                                         readOut, writeOut, exceptOut, selTimeval);
    NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes, error, result);

    if (result < 0)
    {
        nn::socket::detail::SetLastError(error);
    }

    return result;
}

int Select(int numberOfDescriptors, fd_set* pReadDescriptors, fd_set* pWriteDescriptors, fd_set* pExceptDescriptors, timeval* pTimeout)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    int result;
    Errno error = Errno::ESuccess;
    nn::socket::sf::SelectTimeval selTimeval = { 0, 0, true };

    if (pReadDescriptors == nullptr
        && pWriteDescriptors == nullptr
        && pExceptDescriptors == nullptr
        && pTimeout == nullptr)
    {
        nn::socket::detail::SetLastError(Errno::EInval);
        result = -1;
    }

    nn::sf::InBuffer  readIn(
        reinterpret_cast<const char*>(pReadDescriptors),
        pReadDescriptors ? sizeof(fd_set) : 0);

    nn::sf::InBuffer  writeIn(
        reinterpret_cast<const char*>(pWriteDescriptors),
        pWriteDescriptors ? sizeof(fd_set) : 0);

    nn::sf::InBuffer  exceptIn(
        reinterpret_cast<const char*>(pExceptDescriptors),
        pExceptDescriptors ? sizeof(fd_set) : 0);

    nn::sf::OutBuffer readOut(
        reinterpret_cast<char*>(pReadDescriptors),
        pReadDescriptors ? sizeof(fd_set) : 0);

    nn::sf::OutBuffer writeOut(
        reinterpret_cast<char*>(pWriteDescriptors),
        pWriteDescriptors ? sizeof(fd_set) : 0);

    nn::sf::OutBuffer exceptOut(
        reinterpret_cast<char*>(pExceptDescriptors),
        pExceptDescriptors ? sizeof(fd_set) : 0);

    if ( nullptr != pTimeout )
    {
        selTimeval.sec = static_cast<int64_t>(pTimeout->tv_sec);
        selTimeval.usec = static_cast<int64_t>(pTimeout->tv_usec);
        selTimeval.wasNull = false;
    };

    nn::Result nnRes = g_pClient->Select(
        &result, reinterpret_cast<int*>(&error), numberOfDescriptors,
        readIn, writeIn, exceptIn,
        readOut, writeOut, exceptOut, selTimeval);
    NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes, error, result);

    if (result < 0)
    {
        nn::socket::detail::SetLastError(error);
    }

    return result;
}

int Ioctl(int socket, IoctlCommand command, void* data, size_t dataLength)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    int     result;
    Errno   error = Errno::ESuccess;
    int     embeddedSegmentCount;
    void*   embeddedBuffers[IoctlEmbeddedSegmentCount] = {0};
    size_t  embeddedBufferLengths[IoctlEmbeddedSegmentCount] = {0};

    // EInval over EFault to be more Cafe-like
    if (data == nullptr)
    {
        nn::socket::detail::SetLastError(Errno::EInval);
        return -1;
    }
    else if (dataLength > UINT_MAX)
    {
        nn::socket::detail::SetLastError(Errno::EFault);
        return -1;
    }
    // Few of the ioctl's use embedded pointers.
    // Extract pointer/sizes from the command block and pass them as exchange buffers.
    // Total of four segments can be passed over to the server.

    embeddedSegmentCount = ExtractIoctlEmbeddedPointers(static_cast<uint32_t>(command), data, dataLength, embeddedBuffers, embeddedBufferLengths);

    if (embeddedSegmentCount < 1 || embeddedSegmentCount > IoctlEmbeddedSegmentCount)
    {
        nn::socket::detail::SetLastError(Errno::EInval);
        return -1;
    }

    nn::sf::InBuffer  dataInSlot0(
        reinterpret_cast<const char*>(embeddedBuffers[0]),
        embeddedBufferLengths[0]);

    nn::sf::InBuffer  dataInSlot1(
        reinterpret_cast<const char*>(embeddedBuffers[1]),
        embeddedBufferLengths[1]);

    nn::sf::InBuffer  dataInSlot2(
        reinterpret_cast<const char*>(embeddedBuffers[2]),
        embeddedBufferLengths[2]);

    nn::sf::InBuffer  dataInSlot3(
        reinterpret_cast<const char*>(embeddedBuffers[3]),
        embeddedBufferLengths[3]);

    nn::sf::OutBuffer dataOutSlot0(
        reinterpret_cast<char*>(embeddedBuffers[0]),
        embeddedBufferLengths[0]);

    nn::sf::OutBuffer dataOutSlot1(
        reinterpret_cast<char*>(embeddedBuffers[1]),
        embeddedBufferLengths[1]);

    nn::sf::OutBuffer dataOutSlot2(
        reinterpret_cast<char*>(embeddedBuffers[2]),
        embeddedBufferLengths[2]);

    nn::sf::OutBuffer dataOutSlot3(
        reinterpret_cast<char*>(embeddedBuffers[3]),
        embeddedBufferLengths[3]);

    nn::Result nnRes = g_pClient->Ioctl(
        &result, reinterpret_cast<int*>(&error), socket, static_cast<uint32_t>(command),
        dataInSlot0, dataInSlot1, dataInSlot2, dataInSlot3,
        dataOutSlot0, dataOutSlot1, dataOutSlot2, dataOutSlot3,
        embeddedSegmentCount);
    NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes, error, result);

    // Restore user's command block
    RestoreIoctlEmbeddedPointers(static_cast<uint32_t>(command), embeddedBuffers, embeddedBufferLengths, embeddedSegmentCount);

    if (result < 0)
    {
        nn::socket::detail::SetLastError(error);
    }

    return result;
}

int Ioctl(int socket, uint32_t command, void* data, size_t dataLength)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    int     result;
    Errno   error = Errno::ESuccess;
    int     embeddedSegmentCount;
    void*   embeddedBuffers[IoctlEmbeddedSegmentCount] = {0};
    size_t  embeddedBufferLengths[IoctlEmbeddedSegmentCount] = {0};

    // EInval over EFault to be more Cafe-like
    if (data == nullptr)
    {
        nn::socket::detail::SetLastError(Errno::EInval);
        return -1;
    }
    else if (dataLength > UINT_MAX)
    {
        nn::socket::detail::SetLastError(Errno::EFault);
        return -1;
    }
    // Few of the ioctl's use embedded pointers.
    // Extract pointer/sizes from the command block and pass them as exchange buffers.
    // Total of four segments can be passed over to the server.

    embeddedSegmentCount = ExtractIoctlEmbeddedPointers(command, data, dataLength, embeddedBuffers, embeddedBufferLengths);

    if (embeddedSegmentCount < 1 || embeddedSegmentCount > IoctlEmbeddedSegmentCount)
    {
        nn::socket::detail::SetLastError(Errno::EInval);
        return -1;
    }

    nn::sf::InBuffer  dataInSlot0(
        reinterpret_cast<const char*>(embeddedBuffers[0]),
        embeddedBufferLengths[0]);

    nn::sf::InBuffer  dataInSlot1(
        reinterpret_cast<const char*>(embeddedBuffers[1]),
        embeddedBufferLengths[1]);

    nn::sf::InBuffer  dataInSlot2(
        reinterpret_cast<const char*>(embeddedBuffers[2]),
        embeddedBufferLengths[2]);

    nn::sf::InBuffer  dataInSlot3(
        reinterpret_cast<const char*>(embeddedBuffers[3]),
        embeddedBufferLengths[3]);

    nn::sf::OutBuffer dataOutSlot0(
        reinterpret_cast<char*>(embeddedBuffers[0]),
        embeddedBufferLengths[0]);

    nn::sf::OutBuffer dataOutSlot1(
        reinterpret_cast<char*>(embeddedBuffers[1]),
        embeddedBufferLengths[1]);

    nn::sf::OutBuffer dataOutSlot2(
        reinterpret_cast<char*>(embeddedBuffers[2]),
        embeddedBufferLengths[2]);

    nn::sf::OutBuffer dataOutSlot3(
        reinterpret_cast<char*>(embeddedBuffers[3]),
        embeddedBufferLengths[3]);

    nn::Result nnRes = g_pClient->Ioctl(
        &result, reinterpret_cast<int*>(&error), socket, command,
        dataInSlot0, dataInSlot1, dataInSlot2, dataInSlot3,
        dataOutSlot0, dataOutSlot1, dataOutSlot2, dataOutSlot3,
        embeddedSegmentCount);
    NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes, error, result);

    // Restore user's command block
    RestoreIoctlEmbeddedPointers(command, embeddedBuffers, embeddedBufferLengths, embeddedSegmentCount);

    if (result < 0)
    {
        nn::socket::detail::SetLastError(error);
    }

    return result;
}

int Poll(PollFd* pSocketDescriptors, NfdsT numberOfDescriptors, int timeoutMilliseconds)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    int result;
    Errno error = Errno::ESuccess;

    // EInval over EFault to be more Cafe-like
    if(pSocketDescriptors == nullptr && numberOfDescriptors != 0)
    {
        nn::socket::detail::SetLastError(Errno::EInval);
        return -1;
    }
    else if (numberOfDescriptors * sizeof(PollFd) > UINT_MAX)
    {
        nn::socket::detail::SetLastError(Errno::EFault);
        return -1;
    }

    nn::sf::InBuffer descriptorsIn(
        reinterpret_cast<const char*>(pSocketDescriptors),
        numberOfDescriptors * sizeof(*pSocketDescriptors));

    nn::sf::OutBuffer descriptorsOut(
        reinterpret_cast<char*>(pSocketDescriptors),
        numberOfDescriptors * sizeof(*pSocketDescriptors));

    nn::Result nnRes = g_pClient->Poll(
        &result, reinterpret_cast<int*>(&error), descriptorsIn, descriptorsOut, numberOfDescriptors, timeoutMilliseconds);
    NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes, error, result);

    if (result < 0)
    {
        nn::socket::detail::SetLastError(error);
    }

    return result;
}

int Poll(pollfd* pSocketDescriptors, nfds_t numberOfDescriptors, int timeoutMilliseconds)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    int result;
    Errno error = Errno::ESuccess;

    // EInval over EFault to be more Cafe-like
    if(pSocketDescriptors == nullptr && numberOfDescriptors != 0)
    {
        nn::socket::detail::SetLastError(Errno::EInval);
        return -1;
    }
    else if (numberOfDescriptors * sizeof(pollfd) > UINT_MAX)
    {
        nn::socket::detail::SetLastError(Errno::EFault);
        return -1;
    }

    nn::sf::InBuffer descriptorsIn(
        reinterpret_cast<const char*>(pSocketDescriptors),
        numberOfDescriptors * sizeof(pollfd));

    nn::sf::OutBuffer descriptorsOut(
        reinterpret_cast<char*>(pSocketDescriptors),
        numberOfDescriptors * sizeof(pollfd));

    nn::Result nnRes = g_pClient->Poll(
        &result, reinterpret_cast<int*>(&error), descriptorsIn, descriptorsOut, numberOfDescriptors, timeoutMilliseconds);
    NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes, error, result);

    if (result < 0)
    {
        nn::socket::detail::SetLastError(error);
    }
    return result;
}

int Socket(Family domain, Type type, Protocol protocol)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    int result;
    Errno error = Errno::ESuccess;

    nn::Result nnRes = g_pClient->Socket(
        &result, reinterpret_cast<int*>(&error), static_cast<int>(domain), static_cast<int>(type), static_cast<int>(protocol));
    NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes, error, result);

    if (result < 0)
    {
        nn::socket::detail::SetLastError(error);
    }

    return result;
}

int Socket(int domain, int type, int protocol)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    int result;
    Errno error = Errno::ESuccess;

    nn::Result nnRes = g_pClient->Socket(
        &result, reinterpret_cast<int*>(&error), domain, type, protocol);
    NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes, error, result);

    if (result < 0)
    {
        nn::socket::detail::SetLastError(error);
    }

    return result;
}

int SocketExempt(Family domain, Type type, Protocol protocol)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    int result;
    Errno error = Errno::ESuccess;

    nn::Result nnRes = g_pClient->SocketExempt(
        &result, reinterpret_cast<int*>(&error), static_cast<int>(domain), static_cast<int>(type), static_cast<int>(protocol));
    NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes, error, result);

    if (result < 0)
    {
        nn::socket::detail::SetLastError(error);
    }

    return result;
}

int SocketExempt(int domain, int type, int protocol)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    int result;
    Errno error = Errno::ESuccess;

    nn::Result nnRes = g_pClient->SocketExempt(
        &result, reinterpret_cast<int*>(&error), domain, type, protocol);
    NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes, error, result);

    if (result < 0)
    {
        nn::socket::detail::SetLastError(error);
    }

    return result;
}

int Open(const char* devicePath, OpenFlag flags)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    int result;
    Errno error = Errno::ESuccess;

    // EInval over EFault to be more Cafe-like
    if (devicePath == nullptr)
    {
        nn::socket::detail::SetLastError(Errno::EInval);
        return -1;
    }
    size_t bufferLength = strnlen(devicePath, NetworkDevicePathLength) + 1;
    nn::sf::InBuffer path(reinterpret_cast<const char*>(devicePath), bufferLength);

    nn::Result nnRes = g_pClient->Open(
        &result, reinterpret_cast<int*>(&error), path, static_cast<int>(flags));
    NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes, error, result);

    if (result < 0)
    {
        nn::socket::detail::SetLastError(error);
    }

    return result;
}

int Open(const char* devicePath, int flags)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    int result;
    Errno error = Errno::ESuccess;

    // EInval over EFault to be more Cafe-like
    if (devicePath == nullptr)
    {
        nn::socket::detail::SetLastError(Errno::EInval);
        return -1;
    }
    size_t bufferLength = strnlen(devicePath, NetworkDevicePathLength) + 1;
    nn::sf::InBuffer path(reinterpret_cast<const char*>(devicePath), bufferLength);

    nn::Result nnRes = g_pClient->Open(
        &result, reinterpret_cast<int*>(&error), path, flags);
    NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes, error, result);

    if (result < 0)
    {
        nn::socket::detail::SetLastError(error);
    }

    return result;
}

int Close(int socket)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    int result;
    Errno error = Errno::ESuccess;

    nn::Result nnRes = g_pClient->Close(&result, reinterpret_cast<int*>(&error), socket);
    NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes, error, result);

    if (result < 0)
    {
        nn::socket::detail::SetLastError(error);
    }

    return result;
}

int Sysctl(int* pMibEntries, size_t mibEntryCount, void* pOldValue, size_t* pOldValueLength, void* pNewValue, size_t newValueLength)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    int result;
    Errno error = Errno::ESuccess;
    // EInval over EFault to be more Cafe-like
    if ((pNewValue == nullptr && newValueLength != 0) ||
        pMibEntries == nullptr)
    {
        nn::socket::detail::SetLastError(Errno::EInval);
        return -1;
    }
    else if ((pOldValueLength != nullptr && *pOldValueLength > UINT_MAX) || newValueLength > UINT_MAX)
    {
        nn::socket::detail::SetLastError(Errno::EFault);
        return -1;
    }
    unsigned int oldValueLength = static_cast<unsigned int>(
        (pOldValueLength != nullptr) ? *pOldValueLength : 0);

    nn::sf::InBuffer mib(
        reinterpret_cast<const char*>(pMibEntries),
        mibEntryCount * sizeof(int));

    nn::sf::InBuffer newValue(
        reinterpret_cast<const char*>(pNewValue),
        newValueLength);

    nn::sf::OutBuffer oldValue(
        reinterpret_cast<char*>(pOldValue),
        oldValueLength);

    nn::Result nnRes = g_pClient->Sysctl(
        &result, reinterpret_cast<int*>(&error), mib, oldValue, &oldValueLength, newValue);
    NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes, error, result);

    if (result < 0)
    {
        nn::socket::detail::SetLastError(error);
    }
    else
    {
        if (pOldValueLength != nullptr)
        {
            *pOldValueLength = oldValueLength;
        }
    }

    return result;
}

int DuplicateSocket(int socket, uint64_t ownerProcessId)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    int result;
    Errno error = Errno::ESuccess;


    nn::Result nnRes = g_pClient->DuplicateSocket(&result, reinterpret_cast<int*>(&error), socket, ownerProcessId);
    NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes, error, result);

    if (result < 0)
    {
        nn::socket::detail::SetLastError(error);
    }

    return result;
}

int GetResourceStatistics(StatisticsType type, void* outBuffer, size_t outBufferLength, uint32_t options)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(g_pClient, "socket library has not been initialized");
    int result;
    Errno error = Errno::ESuccess;
    // EInval over EFault to be more Cafe-like
    if (outBuffer == nullptr)
    {
        nn::socket::detail::SetLastError(Errno::EInval);
        return -1;
    }
    else if (outBufferLength > INT_MAX)
    {
        nn::socket::detail::SetLastError(Errno::EFault);
        return -1;
    }

    nn::sf::OutBuffer pOut(reinterpret_cast<char*>(outBuffer), outBufferLength);

    nn::Result nnRes = g_pClient->GetResourceStatistics(&result, reinterpret_cast<int*>(&error), Pid, type, pOut, options);
    NN_DETAIL_SOCKET_TRANSLATE_NN_RESULT_TO_BSD_ERROR(nnRes, error, result);

    if (result < 0)
    {
        nn::socket::detail::SetLastError(error);
    }

    return result;
}

const char* InetNtop(Family family, const void* pNetworkAddress, char* addressStringBuffer, SockLenT addressStringBufferLength)
NN_NOEXCEPT
{
    const char* socketNtop(Family, const void*, char*, SockLenT);
    return socketNtop(family, pNetworkAddress, addressStringBuffer, addressStringBufferLength);
}

const char* InetNtop(int family, const void* pNetworkAddress, char* addressStringBuffer, socklen_t addressStringBufferLength)
NN_NOEXCEPT
{
    const char* socketNtop(int, const void*, char*, socklen_t);
    return socketNtop(family, pNetworkAddress, addressStringBuffer, addressStringBufferLength);
}

int InetPton(Family family, const char* pAddressString, void* pOutAddressBuffer)
NN_NOEXCEPT
{
    int socketPton(Family, const char*, void*);
    return socketPton(family, pAddressString, pOutAddressBuffer);
}

int InetPton(int family, const char* pAddressString, void* pOutAddressBuffer)
NN_NOEXCEPT
{
    int socketPton(int, const char*, void*);
    return socketPton(family, pAddressString, pOutAddressBuffer);
}

int InetAton(const char* addressStringBuffer, InAddr* pOutNetworkAddress)
NN_NOEXCEPT
{
    int socketAton(const char*, InAddr*);
    return socketAton(addressStringBuffer, pOutNetworkAddress);
}

int InetAton(const char* addressStringBuffer, in_addr* pOutNetworkAddress)
NN_NOEXCEPT
{
    int socketAton(const char*, in_addr*);
    return socketAton(addressStringBuffer, pOutNetworkAddress);
}

char* InetNtoa(InAddr networkAddress)
NN_NOEXCEPT
{
    char* socketNtoa(InAddr);
    return socketNtoa(networkAddress);
}

char* InetNtoa(in_addr networkAddress)
NN_NOEXCEPT
{
    char* socketNtoa(in_addr);
    return socketNtoa(networkAddress);
}

#if BYTE_ORDER == BIG_ENDIAN
uint16_t InetHtons(uint16_t hostValue)
NN_NOEXCEPT
{
    return hostValue;
}
uint32_t InetHtonl(uint32_t hostValue)
NN_NOEXCEPT
{
    return hostValue;
}
uint16_t InetNtohs(uint16_t networkValue)
NN_NOEXCEPT
{
    return networkValue;
}
uint32_t InetNtohl(uint32_t networkValue)
NN_NOEXCEPT
{
    return networkValue;
}
#else
uint16_t InetHtons(uint16_t hostValue)
NN_NOEXCEPT
{
    return __builtin_bswap16(hostValue);
}
uint32_t InetHtonl(uint32_t hostValue)
NN_NOEXCEPT
{
    return __builtin_bswap32(hostValue);
}
uint16_t InetNtohs(uint16_t networkValue)
NN_NOEXCEPT
{
    return __builtin_bswap16(networkValue);
}
uint32_t InetNtohl(uint32_t networkValue)
NN_NOEXCEPT
{
    return __builtin_bswap32(networkValue);
}
#endif

}}}

