﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/sm/sm_UserApi.h>
#include <nn/sm/sm_Types.h>
#include <nn/sm/detail/sm_UserInterface.h>

#include <nn/svc/svc_Handle.h>
#include <nn/svc/svc_Result.h>
#include <nn/svc/svc_Base.h>

#include <nn/sf/sf_HipcClient.h>
#include <nn/sf/sf_ExpHeapAllocator.h>


namespace nn { namespace sm {

    const ServiceName ServiceName::InvalidName = { { 0 } };

    namespace
    {
        struct AllocatorTag
        {
        };

        typedef nn::sf::ExpHeapStaticAllocator<1024, AllocatorTag> MyAllocator;

        NN_ALIGNAS(8) Bit8 g_RefInterfaceBuffer[sizeof(sf::SharedPointer<nn::sm::detail::IUserInterface>)];
        sf::SharedPointer<nn::sm::detail::IUserInterface>* g_pRefInterface;

        os::MutexType g_Lock = NN_OS_MUTEX_INITIALIZER(false);
        int g_InitializeCount = 0;


        Result InitializeCore()
        {
            MyAllocator::Initialize(nn::lmem::CreationOption_NoOption);

            Result result;
            svc::Handle session;

            for (;;)
            {
                // TORIAEZU: なんらかの方法でこのコードが実行される時には
                //           ポートが作成済みであることが保障されるべき
                result = nn::svc::ConnectToNamedPort(&session, nn::sm::detail::PortName);
                if( result.IsSuccess() )
                {
                    break;
                }
                if (result <= nn::svc::ResultNotFound())
                {
                    nn::os::SleepThread( nn::TimeSpan::FromMicroSeconds(50000) );
                }
                else
                {
                    return result;
                }
            }

            g_pRefInterface = new(g_RefInterfaceBuffer) sf::SharedPointer<nn::sm::detail::IUserInterface>();
            (*g_pRefInterface) = nn::sf::CreateHipcProxyByHandle
                        <nn::sm::detail::IUserInterface, MyAllocator::Policy>(session);
            if( result.IsFailure() )
            {
                svc::CloseHandle(session);
                return result;
            }

            result = (*g_pRefInterface)->RegisterClient(0);
            if( result.IsFailure() )
            {
                NN_ABORT("not implemented");
                return result;
            }

            return ResultSuccess();
        }
    }

    Result Initialize() NN_NOEXCEPT
    {
        Result result;

        os::LockMutex(&g_Lock);
        if( g_InitializeCount >= 1 )
        {
            g_InitializeCount++;
            result = ResultSuccess();
        }
        else
        {
            result = InitializeCore();
            if( result.IsSuccess() )
            {
                g_InitializeCount = 1;
            }
        }
        os::UnlockMutex(&g_Lock);

        return result;
    }

    Result Finalize() NN_NOEXCEPT
    {
        return ResultSuccess();
    }

    Result GetServiceHandle(svc::Handle* pOut, const char* pName, size_t nameSize) NN_NOEXCEPT
    {
        sf::NativeHandle handle;
        auto ret = (*g_pRefInterface)->GetServiceHandle(&handle, ServiceName::Make(pName, nameSize));
        *pOut = svc::Handle(handle.GetOsHandle());
        handle.Detach();
        return ret;
    }

    Result RegisterService(svc::Handle* pOut, const char* pName, size_t nameSize, int sessionCountMax, bool isLight) NN_NOEXCEPT
    {
        sf::NativeHandle handle;
        auto name = ServiceName::Make(pName, nameSize);
        auto ret = (*g_pRefInterface)->RegisterService(&handle, name, sessionCountMax, isLight);
        *pOut = svc::Handle(handle.GetOsHandle());
        handle.Detach();
        return ret;
    }

    Result UnregisterService(const char* pName, size_t nameSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT( g_InitializeCount > 0 );
        return (*g_pRefInterface)->UnregisterService(ServiceName::Make(pName, nameSize));
    }

}}  // namespace nn::sm
