﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#if !defined(__INTELLISENSE__)

#define NN_SF_DETAIL_BINARY_VERSION_EFFECTIVE 0 // NOLINT(preprocessor/const)

#include <nn/sf/hipc/server/sf_HipcServerSessionManagerWithDomain.h>
#include <nn/sf/hipc/detail/sf_IHipcManager.sfdl.h>

#include <nn/nn_Result.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/cmif/sf_ICmifDomain.h>
#include <nn/sf/cmif/server/sf_CmifServerObjectInfo.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/hipc/sf_HipcDirectApi.h>
#include <nn/sf/hipc/server/sf_HipcServerApiModel.h>
#include <nn/sf/cmif/server/sf_CmifDomainServerObject.h>
#include <utility>

namespace nn { namespace sf { namespace hipc { namespace server {

class HipcServerSessionManagerWithDomain::HipcManagerImpl
    : private HipcServerApiModelHolder
{
private:

    HipcServerSessionManagerWithDomain* m_pManager;
    HipcServerSession* m_pSession;

    Result CloneCurrentObjectImpl(Out<sf::NativeHandle> pOut, HipcServerSessionManager* pManager) NN_NOEXCEPT
    {
        auto&& target = m_pSession->m_X.Clone();
        NN_RESULT_THROW_UNLESS(target, sf::hipc::ResultObjectNotFoundInDomain());
        HipcServerSessionHandle serverSessionHandle;
        HipcClientSessionHandle clientSessionHandle;
        // TODO: unwinding
        NN_ABORT_UNLESS_RESULT_SUCCESS(HipcServerApiModelHolder::CreateSession(&serverSessionHandle, &clientSessionHandle));
        NN_ABORT_UNLESS_RESULT_SUCCESS(pManager->Register(serverSessionHandle, std::move(target)));
        *pOut = sf::NativeHandle(reinterpret_cast<const nn::os::NativeHandle&>(clientSessionHandle), true);
        NN_RESULT_SUCCESS;
    }

public:

    explicit HipcManagerImpl(HipcServerSessionManagerWithDomain* pManager, HipcServerSession* pSession, HipcServerApiModel* pServerApiModel) NN_NOEXCEPT
        : HipcServerApiModelHolder(pServerApiModel)
        , m_pManager(pManager)
        , m_pSession(pSession)
    {
    }

    Result ConvertCurrentObjectToDomain(Out<cmif::CmifDomainObjectId> pOut) NN_NOEXCEPT
    {
        auto pDomain = m_pManager->CreateDomain();
        NN_RESULT_THROW_UNLESS(pDomain, sf::hipc::ResultOutOfHipcDomain());
        SharedPointer<cmif::server::CmifDomainServerObject> cmifDomain(pDomain, false);
        cmif::CmifDomainObjectId id;
        NN_RESULT_DO(pDomain->ReserveEntry(&id, 1));
        // 成功確定
        pDomain->RegisterObject(id, std::move(m_pSession->m_X));
        m_pSession->m_X = cmif::server::CmifServerObjectInfo(std::move(cmifDomain));
        *pOut = id;
        NN_RESULT_SUCCESS;
    }

    Result CopyFromCurrentDomain(Out<sf::NativeHandle> pOut, cmif::CmifDomainObjectId id) NN_NOEXCEPT
    {
        auto pDomain = m_pSession->m_X.GetServiceObject<cmif::server::CmifDomainServerObject>();
        NN_RESULT_THROW_UNLESS(pDomain, sf::hipc::ResultTargetIsNotDomain());
        auto&& target = pDomain->GetObject(id);
        NN_RESULT_THROW_UNLESS(target, sf::hipc::ResultObjectNotFoundInDomain());
        HipcServerSessionHandle serverSessionHandle;
        HipcClientSessionHandle clientSessionHandle;
        // TODO: unwinding
        NN_ABORT_UNLESS_RESULT_SUCCESS(HipcServerApiModelHolder::CreateSession(&serverSessionHandle, &clientSessionHandle));
        NN_ABORT_UNLESS_RESULT_SUCCESS(m_pManager->Register(serverSessionHandle, std::move(target)));
        *pOut = sf::NativeHandle(reinterpret_cast<const nn::os::NativeHandle&>(clientSessionHandle), true);
        NN_RESULT_SUCCESS;
    }

    Result CloneCurrentObject(Out<sf::NativeHandle> pOut) NN_NOEXCEPT
    {
        return CloneCurrentObjectImpl(pOut, m_pManager);
    }

    Result QueryPointerBufferSize(Out<uint16_t> pOut) NN_NOEXCEPT
    {
        *pOut = static_cast<uint16_t>(m_pSession->m_PointerBufferSize);
        NN_RESULT_SUCCESS;
    }

    Result CloneCurrentObjectEx(Out<sf::NativeHandle> pOut, uint32_t tag) NN_NOEXCEPT
    {
        auto pManager = m_pManager->GetHipcServerSessionManagerInternal(tag);
        NN_RESULT_THROW_UNLESS(pManager, sf::ResultNotSupported()); // TODO: 適切な Result
        return CloneCurrentObjectImpl(pOut, pManager);
    }
};

Result HipcServerSessionManagerWithDomain::ProcessManagerInvoke(HipcServerSession* pSession, const detail::HipcMessageBufferAccessor* pInAccessor, const detail::HipcMessageHeaderInfo* pInRequestHeaderInfo, void* outMessageBuffer, size_t outMessageBufferSize) NN_NOEXCEPT
{
    UnmanagedServiceObject<detail::IHipcManager, HipcManagerImpl> hipcManager(this, pSession, HipcServerApiModelHolder::GetHipcServerApiModel());
    return this->ProcessInvokeMethodImpl(cmif::server::CmifServerObjectInfo(hipcManager.GetShared()), pSession, pInAccessor, pInRequestHeaderInfo, outMessageBuffer, outMessageBufferSize);
}

Result HipcServerSessionManagerWithDomain::Process2ManagerInvoke(HipcServerSession* pSession, void* inMessageBuffer, size_t inMessageBufferSize, void* outMessageBuffer, size_t outMessageBufferSize) NN_NOEXCEPT
{
    UnmanagedServiceObject<detail::IHipcManager, HipcManagerImpl> hipcManager(this, pSession, HipcServerApiModelHolder::GetHipcServerApiModel());
    return this->Process2InvokeMethodImpl(cmif::server::CmifServerObjectInfo(hipcManager.GetShared()), pSession, inMessageBuffer, inMessageBufferSize, outMessageBuffer, outMessageBufferSize);
}

}}}}

#endif // !defined(__INTELLISENSE__)
