﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/sf/cmif/server/sf_CmifServerDomainManager.h>

#include <nn/nn_Result.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/cmif/server/sf_CmifServerObjectInfo.h>
#include <utility>
#include <new>
#include <mutex>

namespace nn { namespace sf { namespace cmif { namespace server {

CmifServerDomainManager::EntryAllocator::EntryAllocator(EntryStorage entryBuffer[], int entryCount) NN_NOEXCEPT
    : m_EntryCount(entryCount)
    , m_Mutex(false)
{
    auto current = static_cast<Entry*>(nullptr);
    for (int i = entryCount - 1; i >= 0; --i)
    {
        auto p = new (entryBuffer + i) Entry;
        p->pNext = current;
        p->pOwner = nullptr;
        current = p;
    }
    this->m_Entries = current;
    this->m_FreeEntryHead = current;
}

CmifServerDomainManager::EntryAllocator::~EntryAllocator() NN_NOEXCEPT
{
    for (int i = 0; i < m_EntryCount; ++i)
    {
        m_Entries[i].~Entry();
    }
}

inline CmifDomainObjectId CmifServerDomainManager::EntryAllocator::GetId(Entry* p) NN_NOEXCEPT
{
    auto i = static_cast<decltype(CmifDomainObjectId::value)>(p - m_Entries);
    CmifDomainObjectId ret = { i + 1 };
    return ret;
}

inline CmifServerDomainManager::Entry* CmifServerDomainManager::EntryAllocator::GetEntry(CmifDomainObjectId id) NN_NOEXCEPT
{
    // 以下のコードをオーバフローを考慮したコード
    // auto i = id.value - 1;
    // if (!(0 <= i && i < m_EntryCount))
    // {
    //     return nullptr;
    // }
    // return m_Entries + i;
    if (!(id.value >= 1))
    {
        return nullptr;
    }
    auto i = id.value - 1;
    if (!(i <= static_cast<uint32_t>(m_EntryCount)))
    {
        return nullptr;
    }
    return m_Entries + i;
}

CmifServerDomainManager::Domain::Domain(CmifServerDomainManager* pManager) NN_NOEXCEPT
    : m_pManager(pManager)
{
}

CmifServerDomainManager::Domain::~Domain() NN_NOEXCEPT
{
    while (!m_Entries.empty())
    {
        auto p = &m_Entries.front();
        {
            std::lock_guard<decltype(m_pManager->m_EntryOwnerMutex)> lk(m_pManager->m_EntryOwnerMutex);
            NN_SDK_ASSERT_EQUAL(this, p->pOwner);
            p->pOwner = nullptr;
        }
        p->target.Reset();
        m_Entries.pop_front();
        m_pManager->m_EntryAllocator.DeallocateEntry(p);
    }
}

void CmifServerDomainManager::Domain::DisposeImpl() NN_NOEXCEPT
{
    auto pManager = m_pManager;
    this->~Domain();
    pManager->DeallocateDomainStorage(this);
}

Result CmifServerDomainManager::Domain::ReserveEntry(CmifDomainObjectId ids[], int count) NN_NOEXCEPT
{
    for (int i = 0; i < count; ++i)
    {
        auto p = m_pManager->m_EntryAllocator.AllocateEntry();
        NN_RESULT_THROW_UNLESS(p, sf::cmif::ResultOutOfDomainEntry());
        NN_SDK_ASSERT(!p->pOwner, "[SF-Internal]");
        ids[i] = m_pManager->m_EntryAllocator.GetId(p);
    }
    NN_RESULT_SUCCESS;
}

void CmifServerDomainManager::Domain::UnReserveEntry(const CmifDomainObjectId ids[], int count) NN_NOEXCEPT
{
    for (int i = 0; i < count; ++i)
    {
        auto p = m_pManager->m_EntryAllocator.GetEntry(ids[i]);
        NN_SDK_ASSERT_NOT_NULL(p);
        NN_SDK_ASSERT(!p->pOwner, "[SF-Internal]");
        m_pManager->m_EntryAllocator.DeallocateEntry(p);
    }
}

void CmifServerDomainManager::Domain::RegisterObject(CmifDomainObjectId id, CmifServerObjectInfo&& x) NN_NOEXCEPT
{
    auto p = m_pManager->m_EntryAllocator.GetEntry(id);
    NN_SDK_ASSERT_NOT_NULL(p);
    {
        std::lock_guard<decltype(m_pManager->m_EntryOwnerMutex)> lk(m_pManager->m_EntryOwnerMutex);
        NN_SDK_ASSERT(!p->pOwner, "[SF-Internal]");
        p->pOwner = this;
        m_Entries.push_back(*p);
    }
    p->target = std::move(x);
}

CmifServerObjectInfo CmifServerDomainManager::Domain::UnregisterObject(CmifDomainObjectId id) NN_NOEXCEPT
{
    CmifServerObjectInfo ret;
    auto p = m_pManager->m_EntryAllocator.GetEntry(id);
    if (!p)
    {
        return CmifServerObjectInfo();
    }
    {
        std::lock_guard<decltype(m_pManager->m_EntryOwnerMutex)> lk(m_pManager->m_EntryOwnerMutex);
        if (!(this == p->pOwner))
        {
            return CmifServerObjectInfo();
        }
        p->pOwner = nullptr;
        ret = std::move(p->target);
        m_Entries.erase(m_Entries.iterator_to(*p));
    }
    m_pManager->m_EntryAllocator.DeallocateEntry(p);
    return ret;
}

CmifServerObjectInfo CmifServerDomainManager::Domain::GetObject(CmifDomainObjectId id) NN_NOEXCEPT
{
    auto p = m_pManager->m_EntryAllocator.GetEntry(id);
    if (!p)
    {
        return CmifServerObjectInfo();
    }
    {
        std::lock_guard<decltype(m_pManager->m_EntryOwnerMutex)> lk(m_pManager->m_EntryOwnerMutex);
        if (!(this == p->pOwner))
        {
            return CmifServerObjectInfo();
        }
    }
    return p->target.Clone();
}

CmifServerDomainManager::Entry* CmifServerDomainManager::EntryAllocator::AllocateEntry() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    if (!m_FreeEntryHead)
    {
        return nullptr;
    }
    auto ret = m_FreeEntryHead;
    this->m_FreeEntryHead = ret->pNext;
    return ret;
}

void CmifServerDomainManager::EntryAllocator::DeallocateEntry(CmifServerDomainManager::Entry* p) NN_NOEXCEPT
{
    NN_SDK_ASSERT(!p->pOwner, "[SF-Internal]");
    NN_SDK_ASSERT(!p->target, "[SF-Internal]");
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    p->pNext = m_FreeEntryHead;
    this->m_FreeEntryHead = p;
}

CmifServerDomainManager::CmifServerDomainManager(EntryStorage entryBuffer[], int entryCount) NN_NOEXCEPT
    : m_EntryOwnerMutex(false)
    , m_EntryAllocator(entryBuffer, entryCount)
{
}

CmifDomainServerObject* CmifServerDomainManager::CreateDomain() NN_NOEXCEPT
{
    auto p = this->AllocateDomainStorage();
    if (!p)
    {
        return nullptr;
    }
    return new (p) Domain(this);
}

}}}}
