﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "sdmmc_SdHostStandardController.h"
#include "sdmmc_ClockResetController.tegra.h"
#if (defined(NN_DETAIL_SDMMC_PORT_SD_CARD_0_ENABLE))
    #include <nn/pinmux/pinmux.h>
#endif
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Abort.h>

namespace nn { namespace sdmmc {
namespace detail {

bool IsSocRevisionMariko() NN_NOEXCEPT;

const nn::dd::PhysicalAddress ApbMiscRegistersPhysicalAddress = 0x70000000ull;
const size_t ApbMiscRegistersSize = 0x4000;

const size_t SdmmcRegistersSize = 0x200;
#if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
    const int PeripheralInterruptOffset = 32;
#endif

class SdmmcController : public SdHostStandardController
{
private:
    struct SdmmcRegisters
    {
        volatile SdHostStandardRegisters sdHostStandardRegisters;
        volatile uint32_t   vendorClockCntrl0;              // 0x100
        volatile uint32_t   vendorSysSwCntrl0;              // 0x104
        volatile uint32_t   vendorErrIntrStatus0;           // 0x108
        volatile uint32_t   vendorCapOverrides0;            // 0x10c
        volatile uint32_t   vendorBootCntrl0;               // 0x110
        volatile uint32_t   vendorBootActTimeout0;          // 0x014
        volatile uint32_t   bootDatTimeout0;                // 0x118
        volatile uint32_t   vendorDebounceCount0;           // 0x11c
        volatile uint32_t   vendorMiscCntrl0;               // 0x120
        volatile uint32_t   maxCurrentOverride0;            // 0x124
        volatile uint32_t   maxCurrentOverrideHi0;          // 0x128
        volatile uint8_t    reserved1[0x80];
        volatile uint32_t   vendorIoTrimCntrl0;             // 0x1ac
        volatile uint32_t   vendorDllcalCfg0;               // 0x1b0
        volatile uint32_t   vendorDllCtrl00;                // 0x1b4
        volatile uint32_t   vendorDllCtrl10;                // 0x1b8
        volatile uint32_t   vendorDllcalCfgSta0;            // 0x1bc
        volatile uint32_t   vendorTuningCntrl00;            // 0x1c0
        volatile uint32_t   vendorTuningCntrl10;            // 0x1c4
        volatile uint32_t   vendorTuningStatus00;           // 0x1c8
        volatile uint32_t   vendorTuningStatus10;           // 0x1cc
        volatile uint32_t   vendorClkGateHysteresisCount0;  // 0x1d0
        volatile uint32_t   vendorPresetVal00;              // 0x1d4
        volatile uint32_t   vendorPresetVal10;              // 0x1d8
        volatile uint32_t   vendorPresetVal20;              // 0x1dc
        volatile uint32_t   sdmemcomppadctrl0;              // 0x1e0
        volatile uint32_t   autoCalConfig0;                 // 0x1e4
        volatile uint32_t   autoCalInterval0;               // 0x1e8
        volatile uint32_t   autoCalStatus0;                 // 0x1ec
        volatile uint32_t   ioSpare0;                       // 0x1f0
        volatile uint32_t   mmcifFifoctrl0;                 // 0x1f4
        volatile uint32_t   timeoutWcalSdmmc0;              // 0x1f8
    };

    SdmmcRegisters* m_pSdmmcRegisters;

    bool m_IsShutdowned;
    bool m_IsAwake;

    SpeedMode m_CurrentSpeedMode;

    BusPower m_BusPowerBeforeSleep;
    BusWidth m_BusWidthBeforeSleep;
    SpeedMode m_SpeedModeBeforeSleep;
    uint8_t m_TapValueBeforeSleep;
    bool m_IsPowerSavingEnableBeforeSleep;

    uint8_t m_TapValueForHs400;
    bool m_IsValidTapValueForHs400;
    Result m_DriveStrengthCalibrationStatus;

    void ReleaseReset(SpeedMode speedMode) NN_NOEXCEPT;
    void AssertReset() NN_NOEXCEPT;
    Result StartupCore(BusPower busPower) NN_NOEXCEPT;
    Result SetClockTrimmer(SpeedMode speedMode, uint8_t tapValue) NN_NOEXCEPT;
    uint8_t GetCurrentTapValue() NN_NOEXCEPT;
    Result CalibrateDll() NN_NOEXCEPT;
    Result SetSpeedModeWithTapValue(SpeedMode speedMode, uint8_t tapValue) NN_NOEXCEPT;
    Result IssueTuningCommand(uint32_t commandIndex) NN_NOEXCEPT;

protected:
    void SetDriveCodeOffsets(BusPower busPower) NN_NOEXCEPT;
    void CalibrateDriveStrength(BusPower busPower) NN_NOEXCEPT;

    virtual void SetPad() NN_NOEXCEPT = 0;
    virtual detail::ClockResetController::Module GetClockResetModule() const NN_NOEXCEPT = 0;
    #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
        virtual int GetInterruptNumber() const NN_NOEXCEPT = 0;
        virtual nn::os::InterruptEventType* GetInterruptEvent() const NN_NOEXCEPT = 0;
    #endif
    virtual bool IsNeedPeriodicDriveStrengthCalibration() NN_NOEXCEPT = 0;
    virtual void ClearPadParked() NN_NOEXCEPT = 0;
    virtual Result PowerOn(BusPower busPower) NN_NOEXCEPT = 0;
    virtual void PowerOff() NN_NOEXCEPT = 0;
    virtual Result LowerBusPower() NN_NOEXCEPT = 0;
    virtual void SetSchmittTrigger(BusPower busPower) NN_NOEXCEPT = 0;
    virtual uint8_t GetOutboundTapValue() const NN_NOEXCEPT = 0;
    virtual uint8_t GetDefaultInboundTapValue() const NN_NOEXCEPT = 0;
    virtual uint8_t GetVrefSelValue() const NN_NOEXCEPT = 0;
    virtual void SetSlewCodes() NN_NOEXCEPT = 0;
    virtual void GetAutoCalOffsets(uint8_t* pOutAutoCalPdOffset, uint8_t* pOutAutoCalPuOffset, BusPower busPower) const NN_NOEXCEPT = 0;
    virtual void SetDriveStrengthToDefaultValues(BusPower busPower) NN_NOEXCEPT = 0;

public:
    explicit SdmmcController(nn::dd::PhysicalAddress registersPhysicalAddress) NN_NOEXCEPT
        : SdHostStandardController(registersPhysicalAddress, SdmmcRegistersSize)
    {
        uintptr_t registersAddress = nn::dd::QueryIoMappingAddress(registersPhysicalAddress, SdmmcRegistersSize);
        NN_ABORT_UNLESS(registersAddress != 0);
        m_pSdmmcRegisters = reinterpret_cast<SdmmcRegisters*>(registersAddress);
        m_IsShutdowned = true;
        m_IsAwake = true;
        m_IsValidTapValueForHs400 = false;
        m_DriveStrengthCalibrationStatus = ResultDriveStrengthCalibrationNotCompleted();

        // 以下は適当な値、設定した後しか参照されない
        m_TapValueForHs400 = 0;
        m_CurrentSpeedMode = SpeedMode_MmcIdentification;
        m_BusPowerBeforeSleep = BusPower_Off;
        m_BusWidthBeforeSleep = BusWidth_1Bit;
        m_SpeedModeBeforeSleep = SpeedMode_MmcIdentification;
        m_TapValueBeforeSleep = 0;
        m_IsPowerSavingEnableBeforeSleep = false;
    }

    void LogRegisters() NN_NOEXCEPT;

    virtual void Initialize() NN_NOEXCEPT NN_OVERRIDE
    {
        SetPad();

        detail::ClockResetController::Initialize(GetClockResetModule());

        #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
            nn::os::InterruptEventType* pInterruptEvent = GetInterruptEvent();
            // レベル割り込みのため、ManualClear を指定する
            nn::os::InitializeInterruptEvent(pInterruptEvent, GetInterruptNumber(), nn::os::EventClearMode_ManualClear);
            SdHostStandardController::PreSetInterruptEvent(pInterruptEvent);
        #endif

        SdHostStandardController::Initialize();
    }

    virtual void Finalize() NN_NOEXCEPT NN_OVERRIDE
    {
        SdHostStandardController::Finalize();

        #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
            nn::os::FinalizeInterruptEvent(GetInterruptEvent());
        #endif

        detail::ClockResetController::Finalize(GetClockResetModule());
    }

    virtual Result Startup(BusPower busPower, BusWidth busWidth, SpeedMode speedMode, bool isPowerSavingEnable) NN_NOEXCEPT NN_OVERRIDE;
    virtual void Shutdown() NN_NOEXCEPT NN_OVERRIDE;
    virtual void PutToSleep() NN_NOEXCEPT NN_OVERRIDE;
    virtual Result Awaken() NN_NOEXCEPT NN_OVERRIDE;
    virtual Result SwitchToSdr12() NN_NOEXCEPT NN_OVERRIDE;
    virtual Result SetSpeedMode(SpeedMode speedMode) NN_NOEXCEPT NN_OVERRIDE;

    virtual SpeedMode GetSpeedMode() const NN_NOEXCEPT NN_OVERRIDE
    {
        return m_CurrentSpeedMode;
    }

    virtual void SetPowerSaving(bool isPowerSavingEnable) NN_NOEXCEPT NN_OVERRIDE;
    virtual void EnableDeviceClock() NN_NOEXCEPT NN_OVERRIDE;
    virtual Result IssueCommand(const Command* pCommand, TransferData* pTransferData, uint32_t* pOutTransferredNumBlocks) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result IssueStopTransmissionCommand(uint32_t* pOutResponse) NN_NOEXCEPT NN_OVERRIDE;

    virtual bool IsSupportedTuning() const NN_NOEXCEPT NN_OVERRIDE
    {
        return true;
    }

    virtual Result Tuning(SpeedMode speedMode, uint32_t commandIndex) NN_NOEXCEPT NN_OVERRIDE;
    virtual void SaveTuningStatusForHs400() NN_NOEXCEPT NN_OVERRIDE;

    virtual Result GetInternalStatus() const NN_NOEXCEPT NN_OVERRIDE
    {
        // 将来、他にも保持する内部状態を追加した場合は、異常から優先して返す予定
        return m_DriveStrengthCalibrationStatus;
    }
};

#if (defined(NN_DETAIL_SDMMC_PORT_SD_CARD_0_ENABLE))
    const nn::dd::PhysicalAddress Sdmmc1RegistersPhysicalAddress = 0x700B0000ull;

    class Sdmmc1Controller : public SdmmcController
    {
    private:
        #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
            static nn::os::InterruptEventType m_InterruptEvent;
        #endif

        nn::pinmux::PinmuxSession m_PinmuxSession;
        BusPower m_BusPower;

    protected:
        virtual void SetPad() NN_NOEXCEPT NN_OVERRIDE
        {
            // pinmux により設定されている想定
        }

        virtual detail::ClockResetController::Module GetClockResetModule() const NN_NOEXCEPT NN_OVERRIDE
        {
            return detail::ClockResetController::Module_Sdmmc1;
        }

        #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
            virtual int GetInterruptNumber() const NN_NOEXCEPT NN_OVERRIDE
            {
                return PeripheralInterruptOffset + 14;
            }
        #endif

        #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
            virtual nn::os::InterruptEventType* GetInterruptEvent() const NN_NOEXCEPT NN_OVERRIDE
            {
                return &m_InterruptEvent;
            }
        #endif

        virtual bool IsNeedPeriodicDriveStrengthCalibration() NN_NOEXCEPT NN_OVERRIDE
        {
            if (IsSocRevisionMariko())
            {
                return false;
            }
            else
            {
                return true;
            }
        }

        virtual void ClearPadParked() NN_NOEXCEPT NN_OVERRIDE
        {
            // スリープ復帰時に pinmux によりクリアされている想定
        }

        virtual Result PowerOn(BusPower busPower) NN_NOEXCEPT NN_OVERRIDE;
        virtual void PowerOff() NN_NOEXCEPT NN_OVERRIDE;
        virtual Result LowerBusPower() NN_NOEXCEPT NN_OVERRIDE;

        virtual void SetSchmittTrigger(BusPower busPower) NN_NOEXCEPT NN_OVERRIDE
        {
            SdHostStandardController::EnsureControl();

            if (IsSocRevisionMariko())
            {
                // CLK, CMD, DAT0-4 の E_SCHMT Enable
                nn::pinmux::SetPinAssignment(&m_PinmuxSession, nn::pinmux::PinAssignment_Sdmmc1SchmtEnable);
            }
            else
            {
                switch (busPower)
                {
                case BusPower_1_8V:
                    // CLK, CMD, DAT0-4 の E_SCHMT Enable
                    nn::pinmux::SetPinAssignment(&m_PinmuxSession, nn::pinmux::PinAssignment_Sdmmc1SchmtEnable);
                    return;
                case BusPower_3_3V:
                    // CLK, CMD, DAT0-4 の E_SCHMT Disable
                    nn::pinmux::SetPinAssignment(&m_PinmuxSession, nn::pinmux::PinAssignment_Sdmmc1SchmtDisable);
                    return;
                case BusPower_Off:
                    NN_FALL_THROUGH;
                default:
                    NN_UNEXPECTED_DEFAULT;
                }
            }
        }

        virtual uint8_t GetOutboundTapValue() const NN_NOEXCEPT NN_OVERRIDE
        {
            if (IsSocRevisionMariko())
            {
                return 0xE;
            }
            else
            {
                return 2;
            }
        }

        virtual uint8_t GetDefaultInboundTapValue() const NN_NOEXCEPT NN_OVERRIDE
        {
            if (IsSocRevisionMariko())
            {
                return 0xB;
            }
            else
            {
                return 4;
            }
        }

        virtual uint8_t GetVrefSelValue() const NN_NOEXCEPT NN_OVERRIDE
        {
            if (IsSocRevisionMariko())
            {
                return 0x0;
            }
            else
            {
                return 0x7;
            }
        }

        virtual void SetSlewCodes() NN_NOEXCEPT NN_OVERRIDE
        {
            if (IsSocRevisionMariko())
            {
                // Reset 値のまま
            }
            else
            {
                SdHostStandardController::EnsureControl();

                uintptr_t registersAddress = nn::dd::QueryIoMappingAddress(ApbMiscRegistersPhysicalAddress, ApbMiscRegistersSize);
                volatile uint32_t* pRegApbMiscGpSdmmc1PadCfgpadctrl0 = reinterpret_cast<volatile uint32_t*>(registersAddress + 0xa98);
                uint32_t value = *pRegApbMiscGpSdmmc1PadCfgpadctrl0;
                const uint8_t REG_APB_MISC_GP_SDMMC1_PAD_CFGPADCTRL_0__CFG2TMC_SDMMC1_CLK_CFG_CAL_DRVUP_SLWF__SHIFT = 30;
                const uint32_t REG_APB_MISC_GP_SDMMC1_PAD_CFGPADCTRL_0__CFG2TMC_SDMMC1_CLK_CFG_CAL_DRVUP_SLWF__MASK = 0x3U << REG_APB_MISC_GP_SDMMC1_PAD_CFGPADCTRL_0__CFG2TMC_SDMMC1_CLK_CFG_CAL_DRVUP_SLWF__SHIFT;
                value = (value & (~REG_APB_MISC_GP_SDMMC1_PAD_CFGPADCTRL_0__CFG2TMC_SDMMC1_CLK_CFG_CAL_DRVUP_SLWF__MASK)) | (0x1 << REG_APB_MISC_GP_SDMMC1_PAD_CFGPADCTRL_0__CFG2TMC_SDMMC1_CLK_CFG_CAL_DRVUP_SLWF__SHIFT);
                const uint8_t REG_APB_MISC_GP_SDMMC1_PAD_CFGPADCTRL_0__CFG2TMC_SDMMC1_CLK_CFG_CAL_DRVDN_SLWR__SHIFT = 28;
                const uint32_t REG_APB_MISC_GP_SDMMC1_PAD_CFGPADCTRL_0__CFG2TMC_SDMMC1_CLK_CFG_CAL_DRVDN_SLWR__MASK = 0x3U << REG_APB_MISC_GP_SDMMC1_PAD_CFGPADCTRL_0__CFG2TMC_SDMMC1_CLK_CFG_CAL_DRVDN_SLWR__SHIFT;
                value = (value & (~REG_APB_MISC_GP_SDMMC1_PAD_CFGPADCTRL_0__CFG2TMC_SDMMC1_CLK_CFG_CAL_DRVDN_SLWR__MASK)) | (0x1 << REG_APB_MISC_GP_SDMMC1_PAD_CFGPADCTRL_0__CFG2TMC_SDMMC1_CLK_CFG_CAL_DRVDN_SLWR__SHIFT);
                *pRegApbMiscGpSdmmc1PadCfgpadctrl0 = value;
                (void)(*pRegApbMiscGpSdmmc1PadCfgpadctrl0); // APB_MISC レジスタ書き込み完了を確かにする
            }
        }

        virtual void GetAutoCalOffsets(uint8_t* pOutAutoCalPdOffset, uint8_t* pOutAutoCalPuOffset, BusPower busPower) const NN_NOEXCEPT NN_OVERRIDE
        {
            NN_ABORT_UNLESS_NOT_NULL(pOutAutoCalPdOffset);
            NN_ABORT_UNLESS_NOT_NULL(pOutAutoCalPuOffset);
            if (IsSocRevisionMariko())
            {
                switch (busPower)
                {
                case BusPower_1_8V:
                    *pOutAutoCalPdOffset = 6;
                    *pOutAutoCalPuOffset = 6;
                    return;
                case BusPower_3_3V:
                    *pOutAutoCalPdOffset = 0;
                    *pOutAutoCalPuOffset = 0;
                    return;
                case BusPower_Off:
                    NN_FALL_THROUGH;
                default:
                    NN_UNEXPECTED_DEFAULT;
                }
            }
            else
            {
                switch (busPower)
                {
                case BusPower_1_8V:
                    *pOutAutoCalPdOffset = 123;
                    *pOutAutoCalPuOffset = 123;
                    return;
                case BusPower_3_3V:
                    *pOutAutoCalPdOffset = 125;
                    *pOutAutoCalPuOffset = 0;
                    return;
                case BusPower_Off:
                    NN_FALL_THROUGH;
                default:
                    NN_UNEXPECTED_DEFAULT;
                }
            }
        }

        virtual void SetDriveStrengthToDefaultValues(BusPower busPower) NN_NOEXCEPT NN_OVERRIDE
        {
            SdHostStandardController::EnsureControl();

            uintptr_t registersAddress = nn::dd::QueryIoMappingAddress(ApbMiscRegistersPhysicalAddress, ApbMiscRegistersSize);
            volatile uint32_t* pRegApbMiscGpSdmmc1PadCfgpadctrl0 = reinterpret_cast<volatile uint32_t*>(registersAddress + 0xa98);
            uint8_t drvCodeUp;
            uint8_t drvCodeDn;
            if (IsSocRevisionMariko())
            {
                drvCodeUp = 0x8;
                drvCodeDn = 0x8;
            }
            else
            {
                switch (busPower)
                {
                case BusPower_1_8V:
                    drvCodeUp = 11;
                    drvCodeDn = 15;
                    break;
                case BusPower_3_3V:
                    drvCodeUp = 12;
                    drvCodeDn = 12;
                    break;
                case BusPower_Off:
                    NN_FALL_THROUGH;
                default:
                    NN_UNEXPECTED_DEFAULT;
                }
            }
            uint32_t value = *pRegApbMiscGpSdmmc1PadCfgpadctrl0;
            const uint8_t REG_APB_MISC_GP_SDMMC1_PAD_CFGPADCTRL_0__CFG2TMC_SDMMC1_PAD_CAL_DRVUP__SHIFT = 20;
            const uint32_t REG_APB_MISC_GP_SDMMC1_PAD_CFGPADCTRL_0__CFG2TMC_SDMMC1_PAD_CAL_DRVUP__MASK = 0x7FU << REG_APB_MISC_GP_SDMMC1_PAD_CFGPADCTRL_0__CFG2TMC_SDMMC1_PAD_CAL_DRVUP__SHIFT;
            value = (value & (~REG_APB_MISC_GP_SDMMC1_PAD_CFGPADCTRL_0__CFG2TMC_SDMMC1_PAD_CAL_DRVUP__MASK)) | (drvCodeUp << REG_APB_MISC_GP_SDMMC1_PAD_CFGPADCTRL_0__CFG2TMC_SDMMC1_PAD_CAL_DRVUP__SHIFT);
            const uint8_t REG_APB_MISC_GP_SDMMC1_PAD_CFGPADCTRL_0__CFG2TMC_SDMMC1_PAD_CAL_DRVDN__SHIFT = 12;
            const uint32_t REG_APB_MISC_GP_SDMMC1_PAD_CFGPADCTRL_0__CFG2TMC_SDMMC1_PAD_CAL_DRVDN__MASK = 0x7FU << REG_APB_MISC_GP_SDMMC1_PAD_CFGPADCTRL_0__CFG2TMC_SDMMC1_PAD_CAL_DRVDN__SHIFT;
            value = (value & (~REG_APB_MISC_GP_SDMMC1_PAD_CFGPADCTRL_0__CFG2TMC_SDMMC1_PAD_CAL_DRVDN__MASK)) | (drvCodeDn << REG_APB_MISC_GP_SDMMC1_PAD_CFGPADCTRL_0__CFG2TMC_SDMMC1_PAD_CAL_DRVDN__SHIFT);
            *pRegApbMiscGpSdmmc1PadCfgpadctrl0 = value;
            (void)(*pRegApbMiscGpSdmmc1PadCfgpadctrl0); // APB_MISC レジスタ書き込み完了を確かにする
        }

    public:
        Sdmmc1Controller() NN_NOEXCEPT
            : SdmmcController(Sdmmc1RegistersPhysicalAddress)
        {
            m_BusPower = BusPower_Off;
        }

        virtual void Initialize() NN_NOEXCEPT NN_OVERRIDE;
        virtual void Finalize() NN_NOEXCEPT NN_OVERRIDE;

        virtual bool IsSupportedBusPower(BusPower busPower) const NN_NOEXCEPT NN_OVERRIDE
        {
            switch (busPower)
            {
            case BusPower_Off:
                return true;
            case BusPower_1_8V:
                return true;
            case BusPower_3_3V:
                return true;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }

        virtual bool IsSupportedBusWidth(BusWidth busWidth) const NN_NOEXCEPT NN_OVERRIDE
        {
            switch (busWidth)
            {
            case BusWidth_1Bit:
                return true;
            case BusWidth_4Bit:
                return true;
            case BusWidth_8Bit:
                return false;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }
    };
#endif

#if (defined(NN_DETAIL_SDMMC_PORT_GC_ASIC_0_ENABLE) || defined(NN_DETAIL_SDMMC_PORT_MMC_0_ENABLE))
    class Sdmmc2And4Controller : public SdmmcController
    {
    protected:
        virtual bool IsNeedPeriodicDriveStrengthCalibration() NN_NOEXCEPT NN_OVERRIDE
        {
            return false;
        }

        virtual Result PowerOn(BusPower busPower) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(busPower);
            // デバイス電源も IO 電源も ON っぱなし想定
            NN_RESULT_SUCCESS;
        }

        virtual void PowerOff() NN_NOEXCEPT NN_OVERRIDE
        {
            // デバイス電源も IO 電源も ON っぱなし想定
        }

        virtual Result LowerBusPower() NN_NOEXCEPT NN_OVERRIDE
        {
            NN_ABORT("Only 1.8V is supported.\n");
            return ResultNotSupported();
        }

        virtual void SetSchmittTrigger(BusPower busPower) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(busPower);
            // SDMMC2, SDMMC4 は Reset 値のまま
        }

        virtual uint8_t GetOutboundTapValue() const NN_NOEXCEPT NN_OVERRIDE
        {
            if (IsSocRevisionMariko())
            {
                return 0xD;
            }
            else
            {
                return 8;
            }
        }

        virtual uint8_t GetDefaultInboundTapValue() const NN_NOEXCEPT NN_OVERRIDE
        {
            if (IsSocRevisionMariko())
            {
                return 0xB;
            }
            else
            {
                return 0;
            }
        }

        virtual uint8_t GetVrefSelValue() const NN_NOEXCEPT NN_OVERRIDE
        {
            return 0x7;
        }

        virtual void SetSlewCodes() NN_NOEXCEPT NN_OVERRIDE
        {
            // SDMMC2, SDMMC4 は Reset 値のまま
        }

        virtual void GetAutoCalOffsets(uint8_t* pOutAutoCalPdOffset, uint8_t* pOutAutoCalPuOffset, BusPower busPower) const NN_NOEXCEPT NN_OVERRIDE
        {
            NN_ABORT_UNLESS_NOT_NULL(pOutAutoCalPdOffset);
            NN_ABORT_UNLESS_NOT_NULL(pOutAutoCalPuOffset);

            NN_ABORT_UNLESS(busPower == BusPower_1_8V);

            *pOutAutoCalPdOffset = 5;
            *pOutAutoCalPuOffset = 5;
        }

    public:
        explicit Sdmmc2And4Controller(nn::dd::PhysicalAddress registersPhysicalAddress) NN_NOEXCEPT
            : SdmmcController(registersPhysicalAddress)
        {
        }

        virtual bool IsSupportedBusPower(BusPower busPower) const NN_NOEXCEPT NN_OVERRIDE
        {
            switch (busPower)
            {
            case BusPower_Off:
                return true;
            case BusPower_1_8V:
                return true;
            case BusPower_3_3V:
                return false;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }

        virtual bool IsSupportedBusWidth(BusWidth busWidth) const NN_NOEXCEPT NN_OVERRIDE
        {
            switch (busWidth)
            {
            case BusWidth_1Bit:
                return true;
            case BusWidth_4Bit:
                return true;
            case BusWidth_8Bit:
                return true;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }
    };
#endif

#if (defined(NN_DETAIL_SDMMC_PORT_GC_ASIC_0_ENABLE))
    const nn::dd::PhysicalAddress Sdmmc2RegistersPhysicalAddress = 0x700B0200ull;

    class Sdmmc2Controller : public Sdmmc2And4Controller
    {
    private:
        #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
            static nn::os::InterruptEventType m_InterruptEvent;
        #endif

    protected:
        virtual void SetPad() NN_NOEXCEPT NN_OVERRIDE
        {
            // Mariko では pinmux により設定されている想定、それ以外は Reset 値のまま
        }

        virtual detail::ClockResetController::Module GetClockResetModule() const NN_NOEXCEPT NN_OVERRIDE
        {
            return detail::ClockResetController::Module_Sdmmc2;
        }

        #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
            virtual int GetInterruptNumber() const NN_NOEXCEPT NN_OVERRIDE
            {
                return PeripheralInterruptOffset + 15;
            }
        #endif

        #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
            virtual nn::os::InterruptEventType* GetInterruptEvent() const NN_NOEXCEPT NN_OVERRIDE
            {
                return &m_InterruptEvent;
            }
        #endif

        virtual void ClearPadParked() NN_NOEXCEPT NN_OVERRIDE
        {
            if (IsSocRevisionMariko())
            {
                // スリープ復帰時に pinmux によりクリアされている想定
            }
            else
            {
                // SDMMC をリセット解除していないタイミングでコールするため、
                // SdHostStandardController::EnsureControl(); は不要

                uintptr_t registersAddress = nn::dd::QueryIoMappingAddress(ApbMiscRegistersPhysicalAddress, ApbMiscRegistersSize);
                volatile uint32_t* pRegApbMiscGpEmmc2PadCfgpadctrl0 = reinterpret_cast<volatile uint32_t*>(registersAddress + 0xa9c);
                const uint8_t REG_APB_MISC_GP_EMMC2_PAD_CFGPADCTRL_0__MISC2PMC_EMMC2_XXX_PARK__SHIFT = 14;
                const uint32_t REG_APB_MISC_GP_EMMC2_PAD_CFGPADCTRL_0__MISC2PMC_EMMC2_XXX_PARK__PARKED = 0x1FFFU << REG_APB_MISC_GP_EMMC2_PAD_CFGPADCTRL_0__MISC2PMC_EMMC2_XXX_PARK__SHIFT;
                *pRegApbMiscGpEmmc2PadCfgpadctrl0 &= (~REG_APB_MISC_GP_EMMC2_PAD_CFGPADCTRL_0__MISC2PMC_EMMC2_XXX_PARK__PARKED);
                (void)(*pRegApbMiscGpEmmc2PadCfgpadctrl0);  // APB_MISC レジスタ書き込み完了を確かにする
            }
        }

        virtual void SetDriveStrengthToDefaultValues(BusPower busPower) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_ABORT_UNLESS(busPower == BusPower_1_8V);

            SdHostStandardController::EnsureControl();

            uintptr_t registersAddress = nn::dd::QueryIoMappingAddress(ApbMiscRegistersPhysicalAddress, ApbMiscRegistersSize);
            if (IsSocRevisionMariko())
            {
                // http://spdlybra.nintendo.co.jp/jira/browse/NSBG-5714
                volatile uint32_t* pRegApbMiscGpSdmmc2PadCfgpadctrl0 = reinterpret_cast<volatile uint32_t*>(registersAddress + 0xa9c);
                uint32_t value = *pRegApbMiscGpSdmmc2PadCfgpadctrl0;
                const uint8_t REG_APB_MISC_GP_EMMC2_PAD_CFGPADCTRL_0__CFG2TMC_EMMC2_PAD_DRVUP_COMP__SHIFT = 20;
                const uint32_t REG_APB_MISC_GP_EMMC2_PAD_CFGPADCTRL_0__CFG2TMC_EMMC2_PAD_DRVUP_COMP__MASK = 0x7FU << REG_APB_MISC_GP_EMMC2_PAD_CFGPADCTRL_0__CFG2TMC_EMMC2_PAD_DRVUP_COMP__SHIFT;
                value = (value & (~REG_APB_MISC_GP_EMMC2_PAD_CFGPADCTRL_0__CFG2TMC_EMMC2_PAD_DRVUP_COMP__MASK)) | (0xAU << REG_APB_MISC_GP_EMMC2_PAD_CFGPADCTRL_0__CFG2TMC_EMMC2_PAD_DRVUP_COMP__SHIFT);
                const uint8_t REG_APB_MISC_GP_EMMC2_PAD_CFGPADCTRL_0__CFG2TMC_EMMC2_PAD_DRVDN_COMP__SHIFT = 12;
                const uint32_t REG_APB_MISC_GP_EMMC2_PAD_CFGPADCTRL_0__CFG2TMC_EMMC2_PAD_DRVDN_COMP__MASK = 0x7FU << REG_APB_MISC_GP_EMMC2_PAD_CFGPADCTRL_0__CFG2TMC_EMMC2_PAD_DRVDN_COMP__SHIFT;
                value = (value & (~REG_APB_MISC_GP_EMMC2_PAD_CFGPADCTRL_0__CFG2TMC_EMMC2_PAD_DRVDN_COMP__MASK)) | (0xAU << REG_APB_MISC_GP_EMMC2_PAD_CFGPADCTRL_0__CFG2TMC_EMMC2_PAD_DRVDN_COMP__SHIFT);
                *pRegApbMiscGpSdmmc2PadCfgpadctrl0 = value;
                (void)(*pRegApbMiscGpSdmmc2PadCfgpadctrl0); // APB_MISC レジスタ書き込み完了を確かにする
            }
            else
            {
                volatile uint32_t* pRegApbMiscGpEmmc2PadCfgpadctrl0 = reinterpret_cast<volatile uint32_t*>(registersAddress + 0xa9c);
                uint32_t value = *pRegApbMiscGpEmmc2PadCfgpadctrl0;
                const uint8_t REG_APB_MISC_GP_EMMC2_PAD_CFGPADCTRL_0__CFG2TMC_EMMC2_PAD_DRVUP_COMP__SHIFT = 8;
                const uint32_t REG_APB_MISC_GP_EMMC2_PAD_CFGPADCTRL_0__CFG2TMC_EMMC2_PAD_DRVUP_COMP__MASK = 0x3FU << REG_APB_MISC_GP_EMMC2_PAD_CFGPADCTRL_0__CFG2TMC_EMMC2_PAD_DRVUP_COMP__SHIFT;
                value = (value & (~REG_APB_MISC_GP_EMMC2_PAD_CFGPADCTRL_0__CFG2TMC_EMMC2_PAD_DRVUP_COMP__MASK)) | (0x10U << REG_APB_MISC_GP_EMMC2_PAD_CFGPADCTRL_0__CFG2TMC_EMMC2_PAD_DRVUP_COMP__SHIFT);
                const uint8_t REG_APB_MISC_GP_EMMC2_PAD_CFGPADCTRL_0__CFG2TMC_EMMC2_PAD_DRVDN_COMP__SHIFT = 2;
                const uint32_t REG_APB_MISC_GP_EMMC2_PAD_CFGPADCTRL_0__CFG2TMC_EMMC2_PAD_DRVDN_COMP__MASK = 0x3FU << REG_APB_MISC_GP_EMMC2_PAD_CFGPADCTRL_0__CFG2TMC_EMMC2_PAD_DRVDN_COMP__SHIFT;
                value = (value & (~REG_APB_MISC_GP_EMMC2_PAD_CFGPADCTRL_0__CFG2TMC_EMMC2_PAD_DRVDN_COMP__MASK)) | (0x10U << REG_APB_MISC_GP_EMMC2_PAD_CFGPADCTRL_0__CFG2TMC_EMMC2_PAD_DRVDN_COMP__SHIFT);
                *pRegApbMiscGpEmmc2PadCfgpadctrl0 = value;
                (void)(*pRegApbMiscGpEmmc2PadCfgpadctrl0);  // APB_MISC レジスタ書き込み完了を確かにする
            }
        }

    public:
        Sdmmc2Controller() NN_NOEXCEPT : Sdmmc2And4Controller(Sdmmc2RegistersPhysicalAddress)
        {
        }
    };
#endif

// SDMMC3 は使用予定がないため、未実装

#if (defined(NN_DETAIL_SDMMC_PORT_MMC_0_ENABLE))
    const nn::dd::PhysicalAddress Sdmmc4RegistersPhysicalAddress = 0x700B0600ull;

    class Sdmmc4Controller : public Sdmmc2And4Controller
    {
    private:
        #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
            static nn::os::InterruptEventType m_InterruptEvent;
        #endif

    protected:
        virtual void SetPad() NN_NOEXCEPT NN_OVERRIDE
        {
            if (IsSocRevisionMariko())
            {
                // SDMMC をリセット解除していないタイミングでコールするため、
                // SdHostStandardController::EnsureControl(); は不要

                uintptr_t registersAddress = nn::dd::QueryIoMappingAddress(ApbMiscRegistersPhysicalAddress, ApbMiscRegistersSize);

                volatile uint32_t* pRegApbMiscGpEmmc4PadCfgpadctrl0 = reinterpret_cast<volatile uint32_t*>(registersAddress + 0xab4);
                const uint32_t REG_APB_MISC_GP_EMMC4_PAD_CFGPADCTRL_0__CFG2TMC_EMMC4_PAD_E_SCH = 0x1U << 0;
                *pRegApbMiscGpEmmc4PadCfgpadctrl0 |= REG_APB_MISC_GP_EMMC4_PAD_CFGPADCTRL_0__CFG2TMC_EMMC4_PAD_E_SCH;

                // http://spdlybra.nintendo.co.jp/jira/browse/NSBG-7728#comment-2321356
                // http://spdlybra.nintendo.co.jp/jira/browse/SIGLO-68247#comment-2270325
                volatile uint32_t* pRegApbMiscGpEmmc4PadPupdCfgpadctrl0 = reinterpret_cast<volatile uint32_t*>(registersAddress + 0xabc);
                const uint32_t REG_APB_MISC_GP_EMMC4_PAD_PUPD_CFGPADCTRL_0__CFG2TMC_EMMC4_PAD_DQS_PUPD_PULLD = 0x1U << 22;
                const uint32_t REG_APB_MISC_GP_EMMC4_PAD_PUPD_CFGPADCTRL_0__CFG2TMC_EMMC4_PAD_CLK_PUPD_PULLD = 0x1U << 2;
                const uint32_t REG_APB_MISC_GP_EMMC4_PAD_PUPD_CFGPADCTRL_0__CFG2TMC_EMMC4_PAD_CMD_PUPD_PULLU = 0x1U << 1;
                const uint32_t RegApbMiscGpEmmc4PadPupdCfgpadctrl0ClearBits = \
                    REG_APB_MISC_GP_EMMC4_PAD_PUPD_CFGPADCTRL_0__CFG2TMC_EMMC4_PAD_DQS_PUPD_PULLD | \
                    REG_APB_MISC_GP_EMMC4_PAD_PUPD_CFGPADCTRL_0__CFG2TMC_EMMC4_PAD_CLK_PUPD_PULLD | \
                    REG_APB_MISC_GP_EMMC4_PAD_PUPD_CFGPADCTRL_0__CFG2TMC_EMMC4_PAD_CMD_PUPD_PULLU;
                *pRegApbMiscGpEmmc4PadPupdCfgpadctrl0 &= (~RegApbMiscGpEmmc4PadPupdCfgpadctrl0ClearBits);
                (void)(*pRegApbMiscGpEmmc4PadPupdCfgpadctrl0);  // APB_MISC レジスタ書き込み完了を確かにする
            }
            else
            {
                // Reset 値のまま
            }
        }

        virtual detail::ClockResetController::Module GetClockResetModule() const NN_NOEXCEPT NN_OVERRIDE
        {
            return detail::ClockResetController::Module_Sdmmc4;
        }

        #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
            virtual int GetInterruptNumber() const NN_NOEXCEPT NN_OVERRIDE
            {
                return PeripheralInterruptOffset + 31;
            }
        #endif

        #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
            virtual nn::os::InterruptEventType* GetInterruptEvent() const NN_NOEXCEPT NN_OVERRIDE
            {
                return &m_InterruptEvent;
            }
        #endif

        virtual void ClearPadParked() NN_NOEXCEPT NN_OVERRIDE
        {
            // SDMMC をリセット解除していないタイミングでコールするため、
            // SdHostStandardController::EnsureControl(); は不要

            uintptr_t registersAddress = nn::dd::QueryIoMappingAddress(ApbMiscRegistersPhysicalAddress, ApbMiscRegistersSize);
            volatile uint32_t* pRegApbMiscGpEmmc4PadCfgpadctrl0 = reinterpret_cast<volatile uint32_t*>(registersAddress + 0xab4);
            const uint8_t REG_APB_MISC_GP_EMMC4_PAD_CFGPADCTRL_0__MISC2PMC_EMMC4_XXX_PARK__SHIFT = 14;
            const uint32_t REG_APB_MISC_GP_EMMC4_PAD_CFGPADCTRL_0__MISC2PMC_EMMC4_XXX_PARK__PARKED = 0x1FFFU << REG_APB_MISC_GP_EMMC4_PAD_CFGPADCTRL_0__MISC2PMC_EMMC4_XXX_PARK__SHIFT;
            *pRegApbMiscGpEmmc4PadCfgpadctrl0 &= (~REG_APB_MISC_GP_EMMC4_PAD_CFGPADCTRL_0__MISC2PMC_EMMC4_XXX_PARK__PARKED);
            (void)(*pRegApbMiscGpEmmc4PadCfgpadctrl0);  // APB_MISC レジスタ書き込み完了を確かにする
        }

        virtual void SetDriveStrengthToDefaultValues(BusPower busPower) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_ABORT_UNLESS(busPower == BusPower_1_8V);

            SdHostStandardController::EnsureControl();

            uintptr_t registersAddress = nn::dd::QueryIoMappingAddress(ApbMiscRegistersPhysicalAddress, ApbMiscRegistersSize);
            volatile uint32_t* pRegApbMiscGpEmmc4PadCfgpadctrl0 = reinterpret_cast<volatile uint32_t*>(registersAddress + 0xab4);
            uint8_t drvCodeUp;
            uint8_t drvCodeDn;
            if (IsSocRevisionMariko())
            {
                drvCodeUp = 0xA;
                drvCodeDn = 0xA;
            }
            else
            {
                drvCodeUp = 0x10;
                drvCodeDn = 0x10;
            }
            uint32_t value = *pRegApbMiscGpEmmc4PadCfgpadctrl0;
            const uint8_t REG_APB_MISC_GP_EMMC4_PAD_CFGPADCTRL_0__CFG2TMC_EMMC4_PAD_DRVUP_COMP__SHIFT = 8;
            const uint32_t REG_APB_MISC_GP_EMMC4_PAD_CFGPADCTRL_0__CFG2TMC_EMMC4_PAD_DRVUP_COMP__MASK = 0x3FU << REG_APB_MISC_GP_EMMC4_PAD_CFGPADCTRL_0__CFG2TMC_EMMC4_PAD_DRVUP_COMP__SHIFT;
            value = (value & (~REG_APB_MISC_GP_EMMC4_PAD_CFGPADCTRL_0__CFG2TMC_EMMC4_PAD_DRVUP_COMP__MASK)) | (drvCodeUp << REG_APB_MISC_GP_EMMC4_PAD_CFGPADCTRL_0__CFG2TMC_EMMC4_PAD_DRVUP_COMP__SHIFT);
            const uint8_t REG_APB_MISC_GP_EMMC4_PAD_CFGPADCTRL_0__CFG2TMC_EMMC4_PAD_DRVDN_COMP__SHIFT = 2;
            const uint32_t REG_APB_MISC_GP_EMMC4_PAD_CFGPADCTRL_0__CFG2TMC_EMMC4_PAD_DRVDN_COMP__MASK = 0x3FU << REG_APB_MISC_GP_EMMC4_PAD_CFGPADCTRL_0__CFG2TMC_EMMC4_PAD_DRVDN_COMP__SHIFT;
            value = (value & (~REG_APB_MISC_GP_EMMC4_PAD_CFGPADCTRL_0__CFG2TMC_EMMC4_PAD_DRVDN_COMP__MASK)) | (drvCodeDn << REG_APB_MISC_GP_EMMC4_PAD_CFGPADCTRL_0__CFG2TMC_EMMC4_PAD_DRVDN_COMP__SHIFT);
            *pRegApbMiscGpEmmc4PadCfgpadctrl0 = value;
            (void)(*pRegApbMiscGpEmmc4PadCfgpadctrl0);  // APB_MISC レジスタ書き込み完了を確かにする
        }

    public:
        Sdmmc4Controller() NN_NOEXCEPT : Sdmmc2And4Controller(Sdmmc4RegistersPhysicalAddress)
        {
        }
    };
#endif

} // namespace detail {
}} // namespace nn { namespace sdmmc {
