﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/dd.h>
#include "sdmmc_IHostController.h"
#if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
    #include <nn/os/os_InterruptEvent.h>
#endif
#include "sdmmc_SdHostStandardRegisters.h"

namespace nn { namespace sdmmc {
namespace detail {

class SdHostStandardController : public IHostController
{
protected:
    SdHostStandardRegisters* m_pRegisters;

    #if (defined(NN_DETAIL_SDMMC_EXPECT_SMMU_ENABLED))
        struct BufferInfo {
            uintptr_t bufferAddress;
            size_t bufferSize;
            nn::dd::DeviceVirtualAddress bufferDeviceVirtualAddress;
        };

        static const int NumBufferInfos = 3;
        BufferInfo m_BufferInfos[NumBufferInfos];
    #endif

    #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
        nn::os::MultiWaitType m_Waiter;
        nn::os::InterruptEventType* m_pInterruptEvent;
        nn::os::MultiWaitHolderType m_InterruptEventHolder;
        nn::os::EventType* m_pRemovedEvent;
        nn::os::MultiWaitHolderType m_RemovedEventHolder;
    #endif

    #if (defined(NN_DETAIL_SDMMC_ADMA2_ENABLE))
        void* m_pADma2Descriptors;
        const size_t ADma2DescriptorsSize = 4096;   // 1 ページ分（この値を変更する場合は sdmmc_Common.h の WorkBufferSizeForHostController も変更のこと）
        uint64_t m_ADma2DescriptorsAddress;
    #else   // SDMA
        uint64_t m_NextSdmaAddress;
        uint32_t m_CheckTransferIntervalMilliSeconds;
    #endif

    uint32_t m_DeviceClockFrequencyKHz;
    bool m_IsPowerSavingEnable;
    bool m_IsDeviceClockEnable;

    #if (defined(NN_DETAIL_SDMMC_ISSUE_ABORT_COMMAND_FOR_DEBUG))
        uint32_t m_AbortCommandIndex;
        uint32_t m_AbortCommandArgument;
        int64_t m_AbortTimingNanoSeconds;
        bool m_IsAbortCommandEnable;
    #endif

    ResponseType m_LastResponseType;
    uint32_t m_LastResponse[4];                 // 最大 R2 128bit
    uint32_t m_LastStopTransmissionResponse;    // R1b 32bit

    #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
        void PreSetInterruptEvent(nn::os::InterruptEventType* pInterruptEvent) NN_NOEXCEPT
        {
            m_pInterruptEvent = pInterruptEvent;
        }
    #endif

    #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
        bool IsRemoved() NN_NOEXCEPT
        {
            if (m_pRemovedEvent != nullptr)
            {
                if (nn::os::TryWaitEvent(m_pRemovedEvent))
                {
                    return true;
                }
            }
            return false;
        }
    #endif

    void SetDeviceClockFrequencyKHz(uint32_t deviceClockFrequencyKHz)
    {
        m_DeviceClockFrequencyKHz = deviceClockFrequencyKHz;
    }

    #if (defined(NN_DETAIL_SDMMC_EXPECT_SMMU_ENABLED))
        void ResetBufferInfos() NN_NOEXCEPT;
        nn::dd::DeviceVirtualAddress GetDeviceVirtualAddress(uintptr_t bufferAddress, size_t bufferSize) NN_NOEXCEPT;
    #endif
    #if (defined(NN_DETAIL_SDMMC_ADMA2_ENABLE))
        size_t MakeADma2Descriptors(const TransferData* pTransferData) NN_NOEXCEPT;
    #endif
    void EnsureControl() NN_NOEXCEPT;
    Result EnableInternalClock() NN_NOEXCEPT;
    void SetBusPower(BusPower busPower) NN_NOEXCEPT;
    #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
        Result WaitInterrupt(uint32_t timeoutMilliSeconds) NN_NOEXCEPT;
        void ClearInterrupt() NN_NOEXCEPT;
    #endif
    void EnableInterruptStatus() NN_NOEXCEPT;
    void DisableInterruptStatus() NN_NOEXCEPT;
    void SetTransfer(uint32_t* pOutTransferredNumBlocks, const TransferData* pTransferData) NN_NOEXCEPT;
    void SetTransferForTuning() NN_NOEXCEPT;
    void SetCommand(const Command* pCommand, bool isWithTransferData) NN_NOEXCEPT;
    void SetCommandForTuning(uint32_t commandIndex) NN_NOEXCEPT;
    Result ResetCmdDatLine() NN_NOEXCEPT;
    Result AbortTransaction() NN_NOEXCEPT;
    Result WaitWhileCommandInhibit(bool isWithDatLine) NN_NOEXCEPT;
    Result CheckAndClearInterruptStatus(uint16_t* pOutNormalInterruptStatus, uint16_t waitNormalInterruptStatus) NN_NOEXCEPT;
    Result WaitCommandComplete() NN_NOEXCEPT;
    Result WaitTransferComplete() NN_NOEXCEPT;
    Result WaitWhileBusy() NN_NOEXCEPT;
    void GetResponse(uint32_t* pOutResponse, size_t responseSize, ResponseType responseType) const NN_NOEXCEPT;
    #if (defined(NN_DETAIL_SDMMC_ISSUE_ABORT_COMMAND_FOR_DEBUG))
        Result IssueAbortCommandForDebug(uint32_t commandIndex, uint32_t commandArgument, int64_t abortTimingNanoSeconds) NN_NOEXCEPT;
    #endif
    Result IssueStopTransmissionCommandWithDeviceClock(uint32_t* pOutResponse) NN_NOEXCEPT;
    Result IssueCommandWithDeviceClock(const Command* pCommand, TransferData* pTransferData, uint32_t* pOutTransferredNumBlocks) NN_NOEXCEPT;

public:
    SdHostStandardController(nn::dd::PhysicalAddress registersPhysicalAddress, size_t regisersSize) NN_NOEXCEPT;
    void LogRegisters() NN_NOEXCEPT;

    #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
        virtual void PreSetRemovedEvent(nn::os::EventType* pEvent) NN_NOEXCEPT NN_OVERRIDE
        {
            m_pRemovedEvent = pEvent;
        }
    #endif

    virtual void Initialize() NN_NOEXCEPT NN_OVERRIDE;
    virtual void Finalize() NN_NOEXCEPT NN_OVERRIDE;
    #if (defined(NN_DETAIL_SDMMC_EXPECT_SMMU_ENABLED))
        virtual void RegisterDeviceVirtualAddress(uintptr_t bufferAddress, size_t bufferSize, nn::dd::DeviceVirtualAddress bufferDeviceVirtualAddress) NN_NOEXCEPT NN_OVERRIDE;
        virtual void UnregisterDeviceVirtualAddress(uintptr_t bufferAddress, size_t bufferSize, nn::dd::DeviceVirtualAddress bufferDeviceVirtualAddress) NN_NOEXCEPT NN_OVERRIDE;
    #endif
    virtual void SetWorkBuffer(void* pWorkBuffer, size_t workBufferSize) NN_NOEXCEPT NN_OVERRIDE;

    virtual Result SwitchToSdr12() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_ABORT("SwitchToSdr12 isn't implemented.\n");
        return ResultNotImplemented();
    }

    virtual BusPower GetBusPower() const NN_NOEXCEPT NN_OVERRIDE;
    virtual void SetBusWidth(BusWidth busWidth) NN_NOEXCEPT NN_OVERRIDE;
    virtual BusWidth GetBusWidth() const NN_NOEXCEPT NN_OVERRIDE;

    virtual uint32_t GetDeviceClockFrequencyKHz() const NN_NOEXCEPT NN_OVERRIDE
    {
        return m_DeviceClockFrequencyKHz;
    }

    virtual void SetPowerSaving(bool isPowerSavingEnable) NN_NOEXCEPT NN_OVERRIDE;

    virtual bool IsPowerSavingEnable() const NN_NOEXCEPT NN_OVERRIDE
    {
        return m_IsPowerSavingEnable;
    }

    virtual void EnableDeviceClock() NN_NOEXCEPT NN_OVERRIDE;
    virtual void DisableDeviceClock() NN_NOEXCEPT NN_OVERRIDE;

    virtual bool IsDeviceClockEnable() const NN_NOEXCEPT NN_OVERRIDE
    {
        return m_IsDeviceClockEnable;
    }

    #if (defined(NN_DETAIL_SDMMC_ISSUE_ABORT_COMMAND_FOR_DEBUG))
        virtual void SetAbortCommandForDebug(uint32_t commandIndex, uint32_t commandArgument, int64_t abortTimingNanoSeconds) NN_NOEXCEPT NN_OVERRIDE
        {
            m_AbortCommandIndex = commandIndex;
            m_AbortCommandArgument = commandArgument;
            m_AbortTimingNanoSeconds = abortTimingNanoSeconds;
            m_IsAbortCommandEnable = true;
        }

        virtual void ClearAbortCommandForDebug() NN_NOEXCEPT NN_OVERRIDE
        {
            m_IsAbortCommandEnable = false;
        }
    #endif

    virtual uint32_t GetMaxTransferNumBlocks() const NN_NOEXCEPT NN_OVERRIDE
    {
        return REG_BLOCK_COUNT__MAX;
    }

    virtual void ChangeCheckTransferInterval(uint32_t milliSeconds) NN_NOEXCEPT NN_OVERRIDE;
    virtual void SetDefaultCheckTransferInterval() NN_NOEXCEPT NN_OVERRIDE;
    virtual Result IssueCommand(const Command* pCommand, TransferData* pTransferData, uint32_t* pOutTransferredNumBlocks) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result IssueStopTransmissionCommand(uint32_t* pOutResponse) NN_NOEXCEPT NN_OVERRIDE;
    virtual void GetLastResponse(uint32_t* pOutResponse, size_t responseSize, ResponseType responseType) const NN_NOEXCEPT NN_OVERRIDE;
    virtual void GetLastStopTransmissionResponse(uint32_t* pOutResponse, size_t responseSize) const NN_NOEXCEPT NN_OVERRIDE;

    virtual bool IsSupportedTuning() const NN_NOEXCEPT NN_OVERRIDE
    {
        return false;   // 未実装
    }

    virtual Result Tuning(SpeedMode speedMode, uint32_t commandIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(speedMode);
        NN_UNUSED(commandIndex);
        NN_ABORT("Tuning isn't implemented.\n");
        return ResultNotImplemented();
    }

    virtual void SaveTuningStatusForHs400() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_ABORT("Tuning isn't implemented.\n");
    }
};

} // namespace detail {
}} // namespace nn { namespace sdmmc {
