﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/sdmmc/sdmmc_SdCard.h>
#include "sdmmc_BaseDeviceAccessor.h"
#if (defined(NN_DETAIL_SDMMC_SD_CARD_DETECTOR_ENABLE))
    #include "sdmmc_DeviceDetector.h"
#endif

namespace nn { namespace sdmmc {
namespace detail {

// Switch Function Access mode 定義
enum SwitchFunctionAccessMode
{
    SwitchFunctionAccessMode_Default = 0x0,
    SwitchFunctionAccessMode_HighSpeed = 0x1,
    SwitchFunctionAccessMode_Sdr50 = 0x2,
    SwitchFunctionAccessMode_Sdr104 = 0x3,
    SwitchFunctionAccessMode_Ddr50 = 0x4
};

class SdCardDevice : public BaseDevice
{
private:
    #if (defined(NN_DETAIL_SDMMC_SD_CARD_DETECTOR_ENABLE))
        mutable nn::os::EventType m_RemovedEvent;
    #endif
    uint16_t m_Rca;
    bool m_IsValidRca;
    bool m_IsUhsIMode;

public:
    void OnDeactivate() NN_NOEXCEPT
    {
        m_IsValidRca = false;
        m_IsUhsIMode = false;
    }

    SdCardDevice() NN_NOEXCEPT
    {
        m_Rca = 0;
        OnDeactivate();
    }

    virtual void Deactivate() NN_NOEXCEPT NN_OVERRIDE
    {
        OnDeactivate();
        BaseDevice::Deactivate();
    }

    #if (defined(NN_DETAIL_SDMMC_SD_CARD_DETECTOR_ENABLE))
        virtual nn::os::EventType* GetRemovedEvent() const NN_NOEXCEPT NN_OVERRIDE
        {
            return &m_RemovedEvent;
        }
    #else
        #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
            virtual nn::os::EventType* GetRemovedEvent() const NN_NOEXCEPT NN_OVERRIDE
            {
                return nullptr; // 挿抜は検出しない
            }
        #endif
    #endif

    virtual DeviceType GetDeviceType() const NN_NOEXCEPT NN_OVERRIDE
    {
        return DeviceType_SdCard;
    }

    void SetOcrAndHighCapacity(uint32_t ocr) NN_NOEXCEPT;

    void SetRca(uint16_t rca) NN_NOEXCEPT
    {
        m_Rca = rca;
        m_IsValidRca = true;
    }

    virtual uint16_t GetRca() const NN_NOEXCEPT NN_OVERRIDE
    {
        NN_ABORT_UNLESS(m_IsValidRca);
        return m_Rca;
    }

    void SetUhsIMode(bool isUhsIMode) NN_NOEXCEPT
    {
        m_IsUhsIMode = isUhsIMode;
    }

    bool IsUhsIMode() const NN_NOEXCEPT
    {
        return m_IsUhsIMode;
    }
};

class SdCardDeviceAccessor : public BaseDeviceAccessor
{
private:
    SdCardDevice m_SdCardDevice;
    void* m_pWorkBuffer;
    size_t m_WorkBufferSize;
    #if (defined(NN_DETAIL_SDMMC_SD_CARD_DETECTOR_ENABLE))
        DeviceDetector* m_pSdCardDetector;
    #endif
    BusWidth m_MaxBusWidth;
    SpeedMode m_MaxSpeedMode;
    bool m_IsInitialized;

    #if (defined(NN_DETAIL_SDMMC_SD_CARD_DETECTOR_ENABLE))
        void RemovedCallback() NN_NOEXCEPT;

        static void RemovedCallbackEntry(void* pParameter) NN_NOEXCEPT
        {
            SdCardDeviceAccessor* pSdCardDeviceAccessor = reinterpret_cast<SdCardDeviceAccessor*>(pParameter);
            pSdCardDeviceAccessor->RemovedCallback();
        }
    #endif

    Result IssueCommandSendRelativeAddr(uint16_t* pOutRca) const NN_NOEXCEPT;
    Result IssueCommandSendIfCond() const NN_NOEXCEPT;
    Result IssueCommandCheckSupportedFunction(void* pOutSwitchFunctionStatusBuffer, size_t switchFunctionStatusBufferSize) const NN_NOEXCEPT;
    Result IssueCommandSwitchAccessMode(void* pOutSwitchFunctionStatusBuffer, size_t switchFunctionStatusBufferSize, bool isSetFunction, SwitchFunctionAccessMode accessMode) const NN_NOEXCEPT;
    Result IssueCommandVoltageSwitch() const NN_NOEXCEPT;
    Result IssueCommandAppCmd(DeviceState expectedState, uint32_t deviceStatusBitsIgnored) const NN_NOEXCEPT;

    Result IssueCommandAppCmd(DeviceState expectedState) const NN_NOEXCEPT
    {
        return IssueCommandAppCmd(expectedState, 0);
    }

    Result IssueCommandSetBusWidth4bit() const NN_NOEXCEPT;
    Result IssueCommandSdStatus(void* pOutSdStatusBuffer, size_t sdStatusBufferSize) const NN_NOEXCEPT;
    Result IssueCommandSendOpCond(uint32_t* pOutOcr, bool isLessThanSpecVersion2_00, bool isHostSupportUhsIMode) const NN_NOEXCEPT;
    Result IssueCommandClearCardDetect() const NN_NOEXCEPT;
    Result IssueCommandSendScr(void* pOutScrBuffer, size_t scrBufferSize) const NN_NOEXCEPT;
    Result EnterUhsIMode() NN_NOEXCEPT;
    Result ChangeToReadyState(bool isLessThanSpecVersion2_00, bool isHostSupportUhsIMode) NN_NOEXCEPT;
    Result ChangeToStbyStateAndGetRca() NN_NOEXCEPT;
    Result SetMemoryCapacity(void* pCsd) NN_NOEXCEPT;
    void TryDisconnectDat3PullUpResistor() const NN_NOEXCEPT;
    Result GetScr(void* pOutScrBuffer, size_t scrBufferSize) const NN_NOEXCEPT;
    Result ExtendBusWidth(BusWidth maxBusWidth, uint8_t sdBusWidths) NN_NOEXCEPT;
    Result SwitchAccessMode(SwitchFunctionAccessMode accessMode, void* pSdCardWorkBuffer, size_t sdCardWorkBufferSize) NN_NOEXCEPT;
    Result ExtendBusSpeedAtUhsIMode(SpeedMode maxSpeedMode, void* pSdCardWorkBuffer, size_t sdCardWorkBufferSize) NN_NOEXCEPT;
    Result ExtendBusSpeedAtNonUhsIMode(SpeedMode maxSpeedMode, bool isLessThanSpecVersion1_10, void* pSdCardWorkBuffer, size_t sdCardWorkBufferSize) NN_NOEXCEPT;
    Result GetSdStatus(void* pOutSdStatusBuffer, size_t sdStatusBufferSize) const NN_NOEXCEPT;
    Result StartupSdCardDevice(BusWidth maxBusWidth, SpeedMode maxSpeedMode, void* pSdCardWorkBuffer, size_t sdCardWorkBufferSize) NN_NOEXCEPT;

protected:
    virtual Result OnActivate() NN_NOEXCEPT NN_OVERRIDE;
    virtual Result OnReadWrite(uint32_t sectorIndex, uint32_t numSectors, void* pBuffer, size_t bufferSize, bool isRead) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result ReStartup() NN_NOEXCEPT NN_OVERRIDE;

public:
    #if (defined(NN_DETAIL_SDMMC_SD_CARD_DETECTOR_ENABLE))
        explicit SdCardDeviceAccessor(IHostController* pHostController, DeviceDetector* pDeviceDetector) NN_NOEXCEPT
            : BaseDeviceAccessor(pHostController), m_pSdCardDetector(pDeviceDetector)
    #else
        explicit SdCardDeviceAccessor(IHostController* pHostController) NN_NOEXCEPT
            : BaseDeviceAccessor(pHostController)
    #endif
    {
        m_pWorkBuffer = nullptr;
        m_WorkBufferSize = 0;
        m_MaxBusWidth = BusWidth_4Bit;              // 適当な値、設定した後しか参照されない
        m_MaxSpeedMode = SpeedMode_SdCardSdr104;    // 適当な値、設定した後しか参照されない
        m_IsInitialized = false;
    }

    virtual void Initialize() NN_NOEXCEPT NN_OVERRIDE;
    virtual void Finalize() NN_NOEXCEPT NN_OVERRIDE;

    void SetSdCardWorkBuffer(void* pSdCardWorkBuffer, size_t sdCardWorkBufferSize) NN_NOEXCEPT
    {
        m_pWorkBuffer = pSdCardWorkBuffer;
        m_WorkBufferSize = sdCardWorkBufferSize;
    }

    virtual Result Activate() NN_NOEXCEPT NN_OVERRIDE;
    virtual Result GetSpeedMode(SpeedMode* pOutSpeedMode) const NN_NOEXCEPT NN_OVERRIDE;
    void PutSdCardToSleep() NN_NOEXCEPT;
    void AwakenSdCard() NN_NOEXCEPT;
    Result GetSdCardProtectedAreaCapacity(uint32_t* pOutNumSectors) const NN_NOEXCEPT;
    Result GetSdCardScr(void* pOutScrBuffer, size_t scrBufferSize) const NN_NOEXCEPT;
    Result GetSdCardSwitchFunctionStatus(void* pOutSwitchFunctionStatusBuffer, size_t switchFunctionStatusBufferSize,
        SdCardSwitchFunction sdCardSwitchFunction) const NN_NOEXCEPT;
    Result GetSdCardCurrentConsumption(uint16_t* pOutCurrentConsumption, SpeedMode speedMode) const NN_NOEXCEPT;
    Result GetSdCardSdStatus(void* pOutSdStatusBuffer, size_t sdStatusBufferSize) const NN_NOEXCEPT;

    bool IsSdCardInserted() NN_NOEXCEPT
    {
        #if (defined(NN_DETAIL_SDMMC_SD_CARD_DETECTOR_ENABLE))
            return m_pSdCardDetector->IsInserted();
        #else
            NN_ABORT("NN_DETAIL_SDMMC_SD_CARD_DETECTOR_ENABLE isn't defined.\n");
            return true;    // ひとまず挿入っぱなし想定
        #endif
    }

    void RegisterSdCardDetectionEventCallback(DeviceDetectionEventCallback callback, void* pParameter) NN_NOEXCEPT
    {
        #if (defined(NN_DETAIL_SDMMC_SD_CARD_DETECTOR_ENABLE))
            m_pSdCardDetector->RegisterDetectionEventCallback(callback, pParameter);
        #else
            NN_ABORT("NN_DETAIL_SDMMC_SD_CARD_DETECTOR_ENABLE isn't defined.\n");
        #endif
    }

    void UnregisterSdCardDetectionEventCallback() NN_NOEXCEPT
    {
        #if (defined(NN_DETAIL_SDMMC_SD_CARD_DETECTOR_ENABLE))
            m_pSdCardDetector->UnregisterDetectionEventCallback();
        #else
            NN_ABORT("NN_DETAIL_SDMMC_SD_CARD_DETECTOR_ENABLE isn't defined.\n");
        #endif
    }

    bool IsSdCardRemoved() NN_NOEXCEPT
    {
        #if (defined(NN_DETAIL_SDMMC_SD_CARD_DETECTOR_ENABLE))
            return m_SdCardDevice.IsRemoved();
        #else
            NN_ABORT("NN_DETAIL_SDMMC_SD_CARD_DETECTOR_ENABLE isn't defined.\n");
            return true;    // ひとまず Deactivate, Activate が必要な状況にしておく
        #endif
    }
};

} // namespace detail {
}} // namespace nn { namespace sdmmc {
