﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/sdmmc/sdmmc_Common.h>
#if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
    #include <nn/os.h>
#endif
#if (defined(NN_DETAIL_SDMMC_EXPECT_SMMU_ENABLED))
    #include <nn/dd.h>
#endif
#include <nn/nn_Abort.h>

namespace nn { namespace sdmmc {
namespace detail {

enum ResponseType {
    ResponseType_R0,
    ResponseType_R1,
    ResponseType_R2,
    ResponseType_R3,
    ResponseType_R6,
    ResponseType_R7
};

enum TransferDirection {
    TransferDirection_ReadFromDevice,
    TransferDirection_WriteToDevice
};

struct Command
{
    uint32_t m_CommandIndex;
    uint32_t m_CommandArgument;
    ResponseType m_ResponseType;
    bool m_IsBusy;

    Command(uint32_t commandIndex, uint32_t commandArgument, ResponseType responseType, bool isBusy) NN_NOEXCEPT : \
        m_CommandIndex(commandIndex), m_CommandArgument(commandArgument), m_ResponseType(responseType), m_IsBusy(isBusy) {}
};

struct TransferData
{
    void* m_pBuffer;
    size_t m_BlockSize;
    uint32_t m_NumBlocks;
    TransferDirection m_TransferDirection;
    bool m_IsMultiBlockTransfer;
    bool m_IsStopTransmissionCommandEnable;

    TransferData(void* pBuffer, size_t blockSize, uint32_t numBlocks, TransferDirection transferDirection,
        bool isMultiBlockTransfer, bool isStopTransmissionCommandEnable) NN_NOEXCEPT : \
        m_pBuffer(pBuffer), m_BlockSize(blockSize), m_NumBlocks(numBlocks), m_TransferDirection(transferDirection),
            m_IsMultiBlockTransfer(isMultiBlockTransfer), m_IsStopTransmissionCommandEnable(isStopTransmissionCommandEnable)
    {
        if (numBlocks > 1)
        {
            NN_ABORT_UNLESS(isMultiBlockTransfer);
        }
    }

    TransferData(void* pBuffer, size_t blockSize, uint32_t numBlocks, TransferDirection transferDirection) NN_NOEXCEPT : \
        m_pBuffer(pBuffer), m_BlockSize(blockSize), m_NumBlocks(numBlocks), m_TransferDirection(transferDirection),
            m_IsMultiBlockTransfer(false), m_IsStopTransmissionCommandEnable(false)
    {
        NN_ABORT_UNLESS(numBlocks == 1);
    }
};

class IHostController
{
public:
    #if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
        virtual void PreSetRemovedEvent(nn::os::EventType* pEvent) NN_NOEXCEPT = 0;
    #endif
    virtual void Initialize() NN_NOEXCEPT = 0;
    virtual void Finalize() NN_NOEXCEPT = 0;
    #if (defined(NN_DETAIL_SDMMC_EXPECT_SMMU_ENABLED))
        virtual void RegisterDeviceVirtualAddress(uintptr_t bufferAddress, size_t bufferSize, nn::dd::DeviceVirtualAddress bufferDeviceVirtualAddress) NN_NOEXCEPT = 0;
        virtual void UnregisterDeviceVirtualAddress(uintptr_t bufferAddress, size_t bufferSize, nn::dd::DeviceVirtualAddress bufferDeviceVirtualAddress) NN_NOEXCEPT = 0;
    #endif
    virtual void SetWorkBuffer(void* pWorkBuffer, size_t workBufferSize) NN_NOEXCEPT = 0;
    virtual Result Startup(BusPower busPower, BusWidth busWidth, SpeedMode speedMode, bool isPowerSavingEnable) NN_NOEXCEPT = 0;
    virtual void Shutdown() NN_NOEXCEPT = 0;
    virtual void PutToSleep() NN_NOEXCEPT = 0;
    virtual Result Awaken() NN_NOEXCEPT = 0;
    virtual Result SwitchToSdr12() NN_NOEXCEPT = 0;
    virtual bool IsSupportedBusPower(BusPower busPower) const NN_NOEXCEPT = 0;
    virtual BusPower GetBusPower() const NN_NOEXCEPT = 0;
    virtual bool IsSupportedBusWidth(BusWidth busWidth) const NN_NOEXCEPT = 0;
    virtual void SetBusWidth(BusWidth busWidth) NN_NOEXCEPT = 0;
    virtual BusWidth GetBusWidth() const NN_NOEXCEPT = 0;
    virtual Result SetSpeedMode(SpeedMode speedMode) NN_NOEXCEPT = 0;
    virtual SpeedMode GetSpeedMode() const NN_NOEXCEPT = 0;
    virtual uint32_t GetDeviceClockFrequencyKHz() const NN_NOEXCEPT = 0;
    virtual void SetPowerSaving(bool isPowerSavingEnable) NN_NOEXCEPT = 0;
    virtual bool IsPowerSavingEnable() const NN_NOEXCEPT = 0;
    virtual void EnableDeviceClock() NN_NOEXCEPT = 0;
    virtual void DisableDeviceClock() NN_NOEXCEPT = 0;
    virtual bool IsDeviceClockEnable() const NN_NOEXCEPT = 0;
    #if (defined(NN_DETAIL_SDMMC_ISSUE_ABORT_COMMAND_FOR_DEBUG))
        virtual void SetAbortCommandForDebug(uint32_t commandIndex, uint32_t commandArgument, int64_t abortTimingNanoSeconds) NN_NOEXCEPT = 0;
        virtual void ClearAbortCommandForDebug() NN_NOEXCEPT = 0;
    #endif
    virtual uint32_t GetMaxTransferNumBlocks() const NN_NOEXCEPT = 0;
    virtual void ChangeCheckTransferInterval(uint32_t milliSeconds)  NN_NOEXCEPT = 0;
    virtual void SetDefaultCheckTransferInterval() NN_NOEXCEPT = 0;
    virtual Result IssueCommand(const Command* pCommand, TransferData* pTransferData, uint32_t* pOutTransferredNumBlocks) NN_NOEXCEPT = 0;
    virtual Result IssueStopTransmissionCommand(uint32_t* pOutResponse) NN_NOEXCEPT = 0;
    virtual void GetLastResponse(uint32_t* pOutResponse, size_t responseSize, ResponseType responseType) const NN_NOEXCEPT = 0;
    virtual void GetLastStopTransmissionResponse(uint32_t* pOutResponse, size_t responseSize) const NN_NOEXCEPT = 0;
    virtual bool IsSupportedTuning() const NN_NOEXCEPT = 0;
    virtual Result Tuning(SpeedMode speedMode, uint32_t commandIndex) NN_NOEXCEPT = 0;
    virtual void SaveTuningStatusForHs400() NN_NOEXCEPT = 0;
    virtual Result GetInternalStatus() const NN_NOEXCEPT = 0;

    Result IssueCommand(const Command* pCommand, TransferData* pTransferData) NN_NOEXCEPT
    {
        return IssueCommand(pCommand, pTransferData, nullptr);
    }

    Result IssueCommand(const Command* pCommand) NN_NOEXCEPT
    {
        return IssueCommand(pCommand, nullptr, nullptr);
    }
};

} // namespace detail {
}} // namespace nn { namespace sdmmc {
