﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "sdmmc_BaseDeviceAccessor.h"
#include "sdmmc_Log.h"
#if (defined(NN_DETAIL_SDMMC_THREAD_SAFE_ENABLE))
    #include <mutex>
#endif

namespace nn { namespace sdmmc {
namespace detail {

#if (defined(NN_DETAIL_SDMMC_THREAD_SAFE_ENABLE))
    #define NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD   std::lock_guard<nn::os::Mutex> lock(m_pBaseDevice->m_DeviceMutex)
#else
    #define NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD
#endif

#if (defined(NN_DETAIL_SDMMC_USE_NN_OS_FOR_EVENT))
    #define NN_DETAIL_SDMMC_RESULT_THROW_IF_REMOVED     if (m_pBaseDevice->IsRemoved()) { return ResultDeviceRemoved(); }
#else
    #define NN_DETAIL_SDMMC_RESULT_THROW_IF_REMOVED
#endif

void BaseDevice::GetLegacyCapacityParameters(uint8_t* pOutCSizeMult, uint8_t* pOutReadBlLen) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutCSizeMult);
    NN_ABORT_UNLESS_NOT_NULL(pOutReadBlLen);

    // m_Csd[2]: CSD[55:40]
    // CSD[49:47] C_SIZE_MULT の取得
    *pOutCSizeMult = static_cast<uint8_t>((m_Csd[2] & 0x0380U) >> 7);

    // m_Csd[4]: CSD[87:72]
    // CSD[83:80] READ_BL_LEN の取得
    *pOutReadBlLen = static_cast<uint8_t>((m_Csd[4] & 0x0F00U) >> 8);
}

Result BaseDevice::SetLegacyMemoryCapacity() NN_NOEXCEPT
{
    // m_Csd[4]: CSD[87:72], m_Csd[3]: CSD[71:56]
    // CSD[73:62]: C_SIZE の取得
    uint32_t cSize = ((m_Csd[4] & 0x0003U) << 10) | ((m_Csd[3] & 0xFFC0U) >> 6);

    uint8_t cSizeMult = 0;
    uint8_t readBlLen = 0;
    GetLegacyCapacityParameters(&cSizeMult, &readBlLen);
    NN_DETAIL_SDMMC_DEBUG_LOG("Legacy C_SIZE: %u, C_SIZE_MULT: %u, READ_BL_LEN: %u\n", cSize, static_cast<uint32_t>(cSizeMult), static_cast<uint32_t>(readBlLen));
    if (((cSizeMult + 2) + readBlLen) >= 9)
    {
        // 桁あふれしないように、べき乗をまとめて計算
        // BLOCKNR = (C_SIZE+1)^MULT;
        // MULT = 2^(C_SIZE_MULT+2);
        // BLOCK_LEN = 2^READ_BL_LEN;
        // memory capacity(byte) = BLOCKNR * BLOCK_LEN;
        // memory capacity(sector) = memory capacity(byte) / 512;
        m_MemoryCapacity = (cSize + 1) << ((cSizeMult + 2) + readBlLen - 9);
        m_IsValidMemoryCapacity = true;
        NN_RESULT_SUCCESS;
    }
    else
    {
        return ResultUnexpectedDeviceCsdValue();
    }
}

Result BaseDevice::CheckDeviceStatus(uint32_t deviceStatus) const NN_NOEXCEPT
{
    // 簡易チェック
    if ((deviceStatus & DeviceStatus_Mask_AllErrors) == 0)
    {
        NN_RESULT_SUCCESS;
    }

    // エラー Result として優先する順にチェックする
    if ((deviceStatus & DeviceStatus_Bit_ComCrcError) != 0)
    {
        return ResultDeviceStatusComCrcError();
    }
    if ((deviceStatus & DeviceStatus_Bit_DeviceEccFailed) != 0)
    {
        return ResultDeviceStatusDeviceEccFailed();
    }
    if ((deviceStatus & DeviceStatus_Bit_CcError) != 0)
    {
        return ResultDeviceStatusCcError();
    }
    if ((deviceStatus & DeviceStatus_Bit_Error) != 0)
    {
        return ResultDeviceStatusError();
    }
    if (GetDeviceType() == DeviceType_Mmc)
    {
        if ((deviceStatus & DeviceStatus_Bit_SwitchError) != 0)
        {
            return ResultDeviceStatusSwitchError();
        }
    }
    if ((deviceStatus & DeviceStatus_Bit_AddressMisalign) != 0)
    {
        return ResultDeviceStatusAddressMisalign();
    }
    if ((deviceStatus & DeviceStatus_Bit_BlockLenError) != 0)
    {
        return ResultDeviceStatusBlockLenError();
    }
    if ((deviceStatus & DeviceStatus_Bit_EraseSeqError) != 0)
    {
        return ResultDeviceStatusEraseSeqError();
    }
    if ((deviceStatus & DeviceStatus_Bit_EraseParam) != 0)
    {
        return ResultDeviceStatusEraseParam();
    }
    if ((deviceStatus & DeviceStatus_Bit_WpViolation) != 0)
    {
        return ResultDeviceStatusWpViolation();
    }
    if ((deviceStatus & DeviceStatus_Bit_LockUnlockFailed) != 0)
    {
        return ResultDeviceStatusLockUnlockFailed();
    }
    if ((deviceStatus & DeviceStatus_Bit_CidCsdOverwrite) != 0)
    {
        return ResultDeviceStatusCidCsdOverwrite();
    }
    if ((deviceStatus & DeviceStatus_Bit_WpEraseSkip) != 0)
    {
        return ResultDeviceStatusWpEraseSkip();
    }
    if ((deviceStatus & DeviceStatus_Bit_EraseReset) != 0)
    {
        return ResultDeviceStatusEraseReset();
    }
    if ((deviceStatus & DeviceStatus_Bit_IllegalCommand) != 0)
    {
        // SDA Physical Spec Version 2.00 未満への CMD8 発行で発生する場合がある
        return ResultDeviceStatusIllegalCommand();
    }
    if ((deviceStatus & DeviceStatus_Bit_AddressOutOfRange) != 0)
    {
        // 終端セクタのリードアクセスで発生する場合がある
        return ResultDeviceStatusAddressOutOfRange();
    }

    NN_RESULT_SUCCESS;
}

DeviceState BaseDevice::GetDeviceState(uint32_t r1Response) const NN_NOEXCEPT
{
    return static_cast<DeviceState>((r1Response & DeviceStatus_Mask_CurrentState) >> DeviceStatus_Pos_CurrentState);
}

Result BaseDeviceAccessor::IssueCommandAndCheckR1(uint32_t* pOutResponse, uint32_t commandIndex, uint32_t commandArgument, bool isBusy, DeviceState expectedState, uint32_t deviceStatusBitsIgnored) const NN_NOEXCEPT
{
    ResponseType responseType = ResponseType_R1;

    Command command(commandIndex, commandArgument, responseType, isBusy);
    NN_RESULT_DO(m_pHostController->IssueCommand(&command));

    NN_ABORT_UNLESS_NOT_NULL(pOutResponse);
    m_pHostController->GetLastResponse(pOutResponse, sizeof(uint32_t), responseType);
    if (deviceStatusBitsIgnored != 0)
    {
        *pOutResponse &= (~deviceStatusBitsIgnored);
    }
    NN_ABORT_UNLESS_NOT_NULL(m_pBaseDevice);
    NN_RESULT_DO(m_pBaseDevice->CheckDeviceStatus(*pOutResponse));

    if (expectedState != DeviceState_Unknown)
    {
        if (m_pBaseDevice->GetDeviceState(*pOutResponse) != expectedState)
        {
            return ResultUnexpectedDeviceState();
        }
    }

    NN_RESULT_SUCCESS;
}

Result BaseDeviceAccessor::IssueCommandGoIdleState() const NN_NOEXCEPT
{
    // argument[31:0] stuff bit
    Command command(0, 0, ResponseType_R0, false);
    return m_pHostController->IssueCommand(&command);
}

Result BaseDeviceAccessor::IssueCommandAllSendCid(void* pOutCidBuffer, size_t cidBufferSize) const NN_NOEXCEPT
{
    ResponseType responseType = ResponseType_R2;

    // argument[31:0] stuff bit
    Command command(2, 0, responseType, false);
    NN_RESULT_DO(m_pHostController->IssueCommand(&command));

    NN_ABORT_UNLESS_NOT_NULL(pOutCidBuffer);
    NN_ABORT_UNLESS((reinterpret_cast<uintptr_t>(pOutCidBuffer) % sizeof(uint32_t)) == 0);
    NN_ABORT_UNLESS(cidBufferSize >= DeviceCidSize);
    m_pHostController->GetLastResponse(reinterpret_cast<uint32_t*>(pOutCidBuffer), DeviceCidSize, responseType);

    NN_DETAIL_SDMMC_DEBUG_LOG("CID:\n");
    NN_DETAIL_SDMMC_DATA_LOG(pOutCidBuffer, DeviceCidSize);

    NN_RESULT_SUCCESS;
}

Result BaseDeviceAccessor::IssueCommandSelectCard() const NN_NOEXCEPT
{
    // argument[31:16] RCA
    // argument[15: 0] stuff bits
    // R1b, このコマンドで状態遷移するため、state チェックは除外する
    NN_ABORT_UNLESS_NOT_NULL(m_pBaseDevice);
    uint32_t argument = static_cast<uint32_t>(m_pBaseDevice->GetRca()) << 16;
    return IssueCommandAndCheckR1(7, argument, true, DeviceState_Unknown);
}

Result BaseDeviceAccessor::IssueCommandSendCsd(void* pOutCsdBuffer, size_t csdBufferSize) const NN_NOEXCEPT
{
    ResponseType responseType = ResponseType_R2;

    // argument[31:16] RCA
    // argument[15: 0] stuff bits
    NN_ABORT_UNLESS_NOT_NULL(m_pBaseDevice);
    uint32_t argument = static_cast<uint32_t>(m_pBaseDevice->GetRca()) << 16;
    Command command(9, argument, responseType, false);
    NN_RESULT_DO(m_pHostController->IssueCommand(&command));

    NN_ABORT_UNLESS_NOT_NULL(pOutCsdBuffer);
    NN_ABORT_UNLESS((reinterpret_cast<uintptr_t>(pOutCsdBuffer) % sizeof(uint32_t)) == 0);
    NN_ABORT_UNLESS(csdBufferSize >= DeviceCsdSize);
    m_pHostController->GetLastResponse(reinterpret_cast<uint32_t*>(pOutCsdBuffer), DeviceCsdSize, responseType);

    NN_DETAIL_SDMMC_DEBUG_LOG("CSD:\n");
    NN_DETAIL_SDMMC_DATA_LOG(pOutCsdBuffer, DeviceCsdSize);

    NN_RESULT_SUCCESS;
}

Result BaseDeviceAccessor::IssueCommandSendStatus(uint32_t* pOutDeviceStatus, uint32_t deviceStatusBitsIgnored) const NN_NOEXCEPT
{
    // argument[31:16] RCA
    // argument[15: 0] stuff bits
    NN_ABORT_UNLESS_NOT_NULL(m_pBaseDevice);
    uint32_t argument = static_cast<uint32_t>(m_pBaseDevice->GetRca()) << 16;
    return IssueCommandAndCheckR1(pOutDeviceStatus, 13, argument, false, DeviceState_Tran, deviceStatusBitsIgnored);
}

Result BaseDeviceAccessor::IssueCommandSetBlockLenToSectorSize() const NN_NOEXCEPT
{
    // argument[31: 0] block length, SectorSize 固定
    return IssueCommandAndCheckR1(16, SectorSize, false, DeviceState_Tran);
}

Result BaseDeviceAccessor::IssueCommandMultipleBlock(uint32_t* pOutTransferredNumBlocks, uint32_t sectorIndex, uint32_t numSectors, void* pBuffer, bool isRead) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(m_pBaseDevice);

    ResponseType responseType = ResponseType_R1;

    // argument[31:0] data address
    uint32_t commandArgument;
    if (m_pBaseDevice->IsHighCapacity())
    {
        commandArgument = sectorIndex;
    }
    else
    {
        commandArgument = sectorIndex * SectorSize;
    }
    uint32_t commandIndex;
    TransferDirection transferDirection;
    if (isRead)
    {
        commandIndex = 18;
        transferDirection = TransferDirection_ReadFromDevice;
    }
    else
    {
        commandIndex = 25;
        transferDirection = TransferDirection_WriteToDevice;
    }
    Command command(commandIndex, commandArgument, responseType, false);
    TransferData transferData(pBuffer, SectorSize, numSectors, transferDirection, true, true);
    Result result = m_pHostController->IssueCommand(&command, &transferData, pOutTransferredNumBlocks);
    if (result.IsFailure())
    {
        NN_DETAIL_SDMMC_RESULT_THROW_IF_REMOVED;

        // デバイスが検出したエラーを優先して返す
        NN_DETAIL_SDMMC_ERROR_CMD_LOG("CMD%u 0x%08X is failure(Module:%d, Description:%d)\n", commandIndex, commandArgument, result.GetModule(), result.GetDescription());
        Result overridingResult = result;

        // デバイスまで CMD12 が届いていたか不明なため、
        // ILLEGAL_COMMAND が返る可能性も想定して CMD12 を発行する
        uint32_t response = 0;
        result = m_pHostController->IssueStopTransmissionCommand(&response);
        if (result.IsSuccess())
        {
            result = m_pBaseDevice->CheckDeviceStatus(response & (~DeviceStatus_Bit_IllegalCommand));
            if (result.IsFailure())
            {
                // デバイスがエラーを検出していた場合は、それを表す Result に置き換える
                overridingResult = result;
            }
        }
        if (result.IsSuccess())
        {
            NN_DETAIL_SDMMC_ERROR_CMD_LOG("CMD12 Resp 0x%08X\n", response);
        }
        else
        {
            NN_DETAIL_SDMMC_ERROR_CMD_LOG("CMD12 Resp 0x%08X(unsure) is failure(Module:%d, Description:%d)\n", response, result.GetModule(), result.GetDescription());
        }

        NN_DETAIL_SDMMC_RESULT_THROW_IF_REMOVED;

        // ILLEGAL_COMMAND を含むデバイス内部のエラーを CMD13 発行によりクリアする
        uint32_t deviceStatus = 0;
        result = IssueCommandSendStatus(&deviceStatus, DeviceStatus_Bit_IllegalCommand);
        if ((!ResultDeviceStatusHasError::Includes(overridingResult)) && ResultDeviceStatusHasError::Includes(result))
        {
            // デバイスがエラーを検出していた場合は、それを表す Result に置き換える
            overridingResult = result;
        }
        if (result.IsSuccess())
        {
            NN_DETAIL_SDMMC_ERROR_CMD_LOG("CMD13 Resp 0x%08X\n", deviceStatus);
        }
        else
        {
            NN_DETAIL_SDMMC_ERROR_CMD_LOG("CMD13 Resp 0x%08X(unsure) is failure(Module:%d, Description:%d)\n", deviceStatus, result.GetModule(), result.GetDescription());
        }

        return overridingResult;
    }

    // ここに来たときは CMD18/CMD25 - CMD12 のコマンド通信には成功している

    uint32_t response;
    m_pHostController->GetLastResponse(&response, sizeof(response), responseType);

    uint32_t stopTransmissionResponse;
    m_pHostController->GetLastStopTransmissionResponse(&stopTransmissionResponse, sizeof(stopTransmissionResponse));

    result = m_pBaseDevice->CheckDeviceStatus(response);
    if (result.IsFailure())
    {
        NN_DETAIL_SDMMC_ERROR_CMD_LOG("CMD%u 0x%08X\nResp 0x%08X is failure(Module:%d, Description:%d)\n", commandIndex, commandArgument, response, result.GetModule(), result.GetDescription());
        NN_DETAIL_SDMMC_ERROR_CMD_LOG("CMD12 Resp 0x%08X\n", stopTransmissionResponse);

        return result;
    }

    uint32_t deviceStatusBitsIgnored = 0;
    if (isRead)
    {
        NN_ABORT_UNLESS_NOT_NULL(pOutTransferredNumBlocks);
        if ((sectorIndex + (*pOutTransferredNumBlocks)) == m_pBaseDevice->GetMemoryCapacity())
        {
            // 終端セクタを含むリードの ADDRESS_OUT_OF_RANGE を無視する（デバイスによっては発生する）
            deviceStatusBitsIgnored = DeviceStatus_Bit_AddressOutOfRange;
        }
    }
    result = m_pBaseDevice->CheckDeviceStatus(stopTransmissionResponse & (~deviceStatusBitsIgnored));
    if (result.IsFailure())
    {
        NN_DETAIL_SDMMC_ERROR_CMD_LOG("CMD%u 0x%08X Resp 0x%08X\n", commandIndex, commandArgument, response);
        NN_DETAIL_SDMMC_ERROR_CMD_LOG("CMD12 Resp 0x%08X is failure(Module:%d, Description:%d)\n", stopTransmissionResponse, result.GetModule(), result.GetDescription());
        return result;
    }

    NN_RESULT_SUCCESS;
}

Result BaseDeviceAccessor::ReadWriteSingly(uint32_t* pOutTransferredNumBlocks, uint32_t sectorIndex, uint32_t numSectors, void* pBuffer, bool isRead) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutTransferredNumBlocks);

    NN_RESULT_DO(IssueCommandMultipleBlock(pOutTransferredNumBlocks, sectorIndex, numSectors, pBuffer, isRead));

    uint32_t deviceStatusBitsIgnored = 0;
    if (isRead)
    {
        NN_ABORT_UNLESS_NOT_NULL(m_pBaseDevice);
        if ((sectorIndex + (*pOutTransferredNumBlocks)) == m_pBaseDevice->GetMemoryCapacity())
        {
            // 終端セクタを含むリードの ADDRESS_OUT_OF_RANGE を無視する（デバイスによっては発生する）
            deviceStatusBitsIgnored = DeviceStatus_Bit_AddressOutOfRange;
        }
    }
    uint32_t deviceStatus = 0;
    Result result = IssueCommandSendStatus(&deviceStatus, deviceStatusBitsIgnored);
    if (result.IsFailure())
    {
        NN_DETAIL_SDMMC_ERROR_CMD_LOG("%s sectorIndex:0x%08X numSectors:0x%08X is success\n", isRead ? "Read" : "Write", sectorIndex, (*pOutTransferredNumBlocks));
        NN_DETAIL_SDMMC_ERROR_CMD_LOG("CMD13 Resp 0x%08X(unsure) is failure(Module:%d, Description:%d)\n", deviceStatus, result.GetModule(), result.GetDescription());

        return result;
    }

    NN_RESULT_SUCCESS;
}

Result BaseDeviceAccessor::ReadWriteMultiply(uint32_t sectorIndex, uint32_t numSectors, uint32_t divisionSectorIndexAlignment, void* pBuffer, size_t bufferSize, bool isRead) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(m_pBaseDevice);

    if (numSectors == 0)
    {
        // 転送セクタ数が 0 ならば、何もしない
        NN_RESULT_SUCCESS;
    }

    // バッファサイズが十分であることを確認
    NN_ABORT_UNLESS((bufferSize / SectorSize) >= numSectors);

    uint32_t currentSectorIndex = sectorIndex;
    uint32_t restNumSectors = numSectors;
    uint8_t* pCurrentBuffer = reinterpret_cast<uint8_t*>(pBuffer);
    while (restNumSectors > 0)
    {
        uint32_t currentNumSectors = restNumSectors;
        if (divisionSectorIndexAlignment > 0)
        {
            // アラインが合わなくなるアクセスは許容しない
            NN_ABORT_UNLESS((currentSectorIndex % divisionSectorIndexAlignment) == 0);

            uint32_t maxTransferNumBlocks = m_pHostController->GetMaxTransferNumBlocks();
            if (restNumSectors > maxTransferNumBlocks)
            {
                currentNumSectors = maxTransferNumBlocks - (maxTransferNumBlocks % divisionSectorIndexAlignment);
            }
        }

        uint32_t transferredNumBlocks = 0;
        Result result = ReadWriteSingly(&transferredNumBlocks, currentSectorIndex, currentNumSectors, pCurrentBuffer, isRead);
        if (result.IsFailure())
        {
            NN_DETAIL_SDMMC_RESULT_THROW_IF_REMOVED;
            PushErrorLog(false, "%s %X %X:%X", isRead ? "R" : "W", currentSectorIndex, currentNumSectors, result.GetInnerValueForDebug());

            // リトライ（単純に同じアクセスを繰り返す）
            transferredNumBlocks = 0;
            result = ReadWriteSingly(&transferredNumBlocks, currentSectorIndex, currentNumSectors, pCurrentBuffer, isRead);
            if (result.IsFailure())
            {
                NN_DETAIL_SDMMC_RESULT_THROW_IF_REMOVED;
                PushErrorLog(false, "%s %X %X:%X", isRead ? "R" : "W", currentSectorIndex, currentNumSectors, result.GetInnerValueForDebug());

                // リトライ（デバイスを再初期化してから、同じアクセスを繰り返す）
                NN_RESULT_DO(ReStartup());
                transferredNumBlocks = 0;
                result = ReadWriteSingly(&transferredNumBlocks, currentSectorIndex, currentNumSectors, pCurrentBuffer, isRead);
                if (result.IsFailure())
                {
                    PushErrorLog(true, "%s %X %X:%X", isRead ? "R" : "W", currentSectorIndex, currentNumSectors, result.GetInnerValueForDebug());
                    return result;  // 断念する
                }
            }

            PushErrorLog(true, "%s %X %X:0", isRead ? "R" : "W", currentSectorIndex, currentNumSectors);
            m_NumReadWriteErrorCorrections++;
        }

        // restNumSectors より多く転送されることはありえない
        NN_ABORT_UNLESS(restNumSectors >= transferredNumBlocks);
        restNumSectors -= transferredNumBlocks;

        currentSectorIndex += transferredNumBlocks;
        pCurrentBuffer += (SectorSize * transferredNumBlocks);
    }

    NN_RESULT_SUCCESS;
}

#if (defined(NN_DETAIL_SDMMC_EXPECT_SMMU_ENABLED))
    void BaseDeviceAccessor::RegisterDeviceVirtualAddress(uintptr_t bufferAddress, size_t bufferSize, nn::dd::DeviceVirtualAddress bufferDeviceVirtualAddress) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(m_pBaseDevice);
        NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
        m_pHostController->RegisterDeviceVirtualAddress(bufferAddress, bufferSize, bufferDeviceVirtualAddress);
    }
#endif

#if (defined(NN_DETAIL_SDMMC_EXPECT_SMMU_ENABLED))
    void BaseDeviceAccessor::UnregisterDeviceVirtualAddress(uintptr_t bufferAddress, size_t bufferSize, nn::dd::DeviceVirtualAddress bufferDeviceVirtualAddress) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(m_pBaseDevice);
        NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
        m_pHostController->UnregisterDeviceVirtualAddress(bufferAddress, bufferSize, bufferDeviceVirtualAddress);
    }
#endif

Result BaseDeviceAccessor::Activate() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(m_pBaseDevice);
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    if (!(m_pBaseDevice->IsAwake()))
    {
        // Sleep 中
        return ResultNotAwakened();
    }
    if (m_pBaseDevice->IsActive())
    {
        // Activate 済みならば何もしない
        NN_RESULT_SUCCESS;
    }

    Result result = OnActivate();
    if (result.IsSuccess())
    {
        m_pBaseDevice->SetActive();
    }
    else
    {
        m_NumActivationFailures++;
    }
    return result;
}

void BaseDeviceAccessor::Deactivate() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(m_pBaseDevice);
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    if (m_pBaseDevice->IsActive())
    {
        m_pHostController->Shutdown();

        m_pBaseDevice->Deactivate();
    }
}

Result BaseDeviceAccessor::ReadWrite(uint32_t sectorIndex, uint32_t numSectors, void* pBuffer, size_t bufferSize, bool isRead) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(m_pBaseDevice);
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    NN_RESULT_DO(m_pBaseDevice->CheckAccessible());

    Result result = OnReadWrite(sectorIndex, numSectors, pBuffer, bufferSize, isRead);
    if (result.IsFailure())
    {
        m_NumReadWriteFailures++;
    }
    return result;
}

Result BaseDeviceAccessor::CheckConnection(SpeedMode* pOutHostSpeedMode, BusWidth* pOutHostBusWidth) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(m_pBaseDevice);
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    NN_RESULT_DO(m_pBaseDevice->CheckAccessible());

    *pOutHostSpeedMode = m_pHostController->GetSpeedMode();
    *pOutHostBusWidth = m_pHostController->GetBusWidth();

    NN_RESULT_DO(m_pHostController->GetInternalStatus());

    NN_RESULT_SUCCESS;
}

Result BaseDeviceAccessor::GetMemoryCapacity(uint32_t* pOutNumSectors) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(m_pBaseDevice);
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    NN_RESULT_DO(m_pBaseDevice->CheckAccessible());

    NN_ABORT_UNLESS_NOT_NULL(pOutNumSectors);
    *pOutNumSectors = m_pBaseDevice->GetMemoryCapacity();

    NN_RESULT_SUCCESS;
}

Result BaseDeviceAccessor::GetDeviceStatus(uint32_t* pOutDeviceStatus) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(m_pBaseDevice);
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    NN_RESULT_DO(m_pBaseDevice->CheckAccessible());

    NN_RESULT_DO(IssueCommandSendStatus(pOutDeviceStatus, 0));

    NN_RESULT_SUCCESS;
}

Result BaseDeviceAccessor::GetOcr(uint32_t* pOutOcr) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(m_pBaseDevice);
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    NN_RESULT_DO(m_pBaseDevice->CheckAccessible());

    NN_ABORT_UNLESS_NOT_NULL(pOutOcr);
    *pOutOcr = m_pBaseDevice->GetOcr();

    NN_RESULT_SUCCESS;
}

Result BaseDeviceAccessor::GetRca(uint16_t* pOutRca) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(m_pBaseDevice);
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    NN_RESULT_DO(m_pBaseDevice->CheckAccessible());

    NN_ABORT_UNLESS_NOT_NULL(pOutRca);
    *pOutRca = m_pBaseDevice->GetRca();

    NN_RESULT_SUCCESS;
}

Result BaseDeviceAccessor::GetCid(void* pOutCid, size_t cidSize) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(m_pBaseDevice);
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    NN_RESULT_DO(m_pBaseDevice->CheckAccessible());

    m_pBaseDevice->GetCid(pOutCid, cidSize);

    NN_RESULT_SUCCESS;
}

Result BaseDeviceAccessor::GetCsd(void* pOutCsd, size_t csdSize) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(m_pBaseDevice);
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;
    NN_RESULT_DO(m_pBaseDevice->CheckAccessible());

    m_pBaseDevice->GetCsd(pOutCsd, csdSize);

    NN_RESULT_SUCCESS;
}

#if (defined(NN_DETAIL_SDMMC_ISSUE_COMMAND_FOR_DEBUG))
    Result BaseDeviceAccessor::IssueCommandForDebug(uint32_t* pOutResponse, uint32_t commandIndex, uint32_t commandArgument,
        DataTransfer* pDataTransfer, bool isBusy) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(m_pBaseDevice);
        NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;

        // R1 コマンド固定
        ResponseType responseType = ResponseType_R1;
        Command command(commandIndex, commandArgument, responseType, isBusy);

        if (pDataTransfer == nullptr)
        {
            NN_RESULT_DO(m_pHostController->IssueCommand(&command));
        }
        else
        {
            // バッファサイズが十分であることを確認
            NN_ABORT_UNLESS((pDataTransfer->dataBufferSize / pDataTransfer->blockSize) >= pDataTransfer->numBlocks);

            TransferDirection transferDirection = TransferDirection_WriteToDevice;
            if (pDataTransfer->isRead)
            {
                transferDirection = TransferDirection_ReadFromDevice;
            }

            bool isMultiBlockTransfer = false;
            if ((commandIndex == 18) || (commandIndex == 25))
            {
                // CMD18, CMD25 は Multi Block 転送固定
                isMultiBlockTransfer = true;
            }

            // CMD12 自動発行なし固定
            TransferData transferData(pDataTransfer->pDataBuffer,
                pDataTransfer->blockSize, pDataTransfer->numBlocks, transferDirection, isMultiBlockTransfer, false);

            NN_RESULT_DO(m_pHostController->IssueCommand(&command, &transferData));
        }

        m_pHostController->GetLastResponse(pOutResponse, sizeof(uint32_t), responseType);
        NN_RESULT_DO(m_pBaseDevice->CheckDeviceStatus(*pOutResponse));

        NN_RESULT_SUCCESS;
    }
#endif

void BaseDeviceAccessor::GetAndClearErrorInfo(ErrorInfo* pOutErrorInfo, size_t* pOutLogSize, char* pOutLogBuffer, size_t logBufferSize) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(m_pBaseDevice);
    NN_DETAIL_SDMMC_DEVICE_LOCK_GUARD;

    NN_ABORT_UNLESS_NOT_NULL(pOutErrorInfo);
    pOutErrorInfo->numActivationFailures = m_NumActivationFailures;
    pOutErrorInfo->numActivationErrorCorrections = m_NumActivationErrorCorrections;
    pOutErrorInfo->numReadWriteFailures = m_NumReadWriteFailures;
    pOutErrorInfo->numReadWriteErrorCorrections = m_NumReadWriteErrorCorrections;
    ClearErrorInfo();

    if (pOutLogSize == nullptr)
    {
        // ログは格納しない
        return;
    }
    if ((pOutLogBuffer == nullptr) || (logBufferSize == 0))
    {
        // ログは格納しない
        *pOutLogSize = 0;
        return;
    }

    #if (defined(NN_DETAIL_SDMMC_ERROR_LOGGER_ENABLE))
        if (m_ErrorLogger.IsLogExist())
        {
            // エラーログ格納からの時間経過が分かるように取得時のタイムスタンプを残す
            PushErrorTimeStamp();

            *pOutLogSize = m_ErrorLogger.GetAndClearLogs(pOutLogBuffer, logBufferSize);
        }
        else
        {
            // ログはない
            *pOutLogSize = 0;
        }
    #else
        *pOutLogSize = 0;
    #endif
}

} // namespace detail {
}} // namespace nn { namespace sdmmc {
