﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_BitTypes.h>
#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_MemoryMapSelect.h>
#include <nn/nn_Log.h>
#include <nn/util/util_IntrusiveList.h>
#include <cstring>
#include <nn/ro/detail/ro_RoModule.h>
#include <nn/rocrt/rocrt.h>
#include <nn/util/util_TypedStorage.h>
#include "rtld.h"

#include <nn/os.h>
#include <shared_mutex>

namespace nn { namespace rtld {
namespace {

typedef nn::util::TypedStorage<nn::ro::detail::RoModuleList, sizeof(nn::ro::detail::RoModuleList), NN_ALIGNOF(nn::ro::detail::RoModuleList)> RoModuleListStorage;
RoModuleListStorage g_AutoLoad;
RoModuleListStorage g_ManualLoad;

const char* AutoLoadModuleListName = "_ZN2nn2ro6detail15g_pAutoLoadListE";
const char* ManualLoadModuleListName = "_ZN2nn2ro6detail17g_pManualLoadListE";
const char* RoDebugFlagName = "_ZN2nn2ro6detail14g_pRoDebugFlagE";
const char* LookupGlobalAutoName = "_ZN2nn2ro6detail34g_pLookupGlobalAutoFunctionPointerE";
const char* LookupGlobalManualName = "_ZN2nn2ro6detail36g_pLookupGlobalManualFunctionPointerE";

bool g_InitializedLock = false;

void Register(nn::ro::detail::RoModule* pModule)
{
    nn::ro::detail::RoModuleList& autoLoadList = nn::util::Get(g_AutoLoad);

    autoLoadList.push_back(*pModule);
}

void RelocationAutoLoadDll()
{
    nn::ro::detail::RoModuleList& autoLoadList = nn::util::Get(g_AutoLoad);
    for (nn::ro::detail::RoModuleList::iterator it = autoLoadList.begin(); it != autoLoadList.end(); it++)
    {
        // Lazy Bind
        it->Relocation(true);
    }
}

void CallInitAutoLoadDll()
{
    nn::ro::detail::RoModuleList& autoLoadList = nn::util::Get(g_AutoLoad);
    for (nn::ro::detail::RoModuleList::reverse_iterator rit = autoLoadList.rbegin(); rit != autoLoadList.rend(); rit++)
    {
        rit->CallInit();
    }
}
}

void Initialize0(uintptr_t base, const nn::ro::detail::Elf::Dyn* pDynamicSection)
{
    size_t relEnt = 0;
    size_t relaEnt = 0;
    size_t relCount = 0;
    size_t relaCount = 0;
    uintptr_t dynRel = 0;
    uintptr_t dynRela = 0;

    for (const nn::ro::detail::Elf::Dyn* pDyn = pDynamicSection; pDyn->GetTag() != nn::ro::detail::Elf::DT_NULL; pDyn++)
    {
        if (pDyn->GetTag() == nn::ro::detail::Elf::DT_REL)
        {
            dynRel = base + pDyn->GetPtr();
        }
        else if (pDyn->GetTag() == nn::ro::detail::Elf::DT_RELA)
        {
            dynRela = base + pDyn->GetPtr();
        }
        else if (pDyn->GetTag() == nn::ro::detail::Elf::DT_RELENT)
        {
            relEnt = pDyn->GetVal();
        }
        else if (pDyn->GetTag() == nn::ro::detail::Elf::DT_RELAENT)
        {
            relaEnt = pDyn->GetVal();
        }
        else if (pDyn->GetTag() == nn::ro::detail::Elf::DT_RELCOUNT)
        {
            relCount = pDyn->GetVal();
        }
        else if (pDyn->GetTag() == nn::ro::detail::Elf::DT_RELACOUNT)
        {
            relaCount = pDyn->GetVal();
        }
        else if (pDyn->GetTag() == nn::ro::detail::Elf::DT_RELENT)
        {
            while (pDyn->GetVal() != sizeof(nn::ro::detail::Elf::Rel)) {}
        }
        else if (pDyn->GetTag() == nn::ro::detail::Elf::DT_RELAENT)
        {
            while (pDyn->GetVal() != sizeof(nn::ro::detail::Elf::Rela)) {}
        }
    }

    for (size_t i = 0; i < relCount; i++)
    {
        nn::ro::detail::Elf::Rel& rel = *reinterpret_cast<nn::ro::detail::Elf::Rel*>(dynRel + relEnt * i);
        while (rel.GetType() != nn::ro::detail::Elf::R_RELATIVE) {}
        nn::ro::detail::Elf::Addr* pTarget = reinterpret_cast<nn::ro::detail::Elf::Addr*>(base + rel.GetOffset());
        *pTarget += base;
    }

    for (size_t i = 0; i < relaCount; i++)
    {
        nn::ro::detail::Elf::Rela& rela = *reinterpret_cast<nn::ro::detail::Elf::Rela*>(dynRela + relaEnt * i);
        while (rela.GetType() != nn::ro::detail::Elf::R_RELATIVE) {}
        nn::ro::detail::Elf::Addr* pTarget = reinterpret_cast<nn::ro::detail::Elf::Addr*>(base + rela.GetOffset());
        *pTarget = base + rela.GetAddend();
    }

}

void Initialize1(uintptr_t base, const nn::ro::detail::Elf::Dyn* pDynamicSection)
{
    nn::ro::detail::RoModule* pSelf = new(&nn::util::Get(nn::rocrt::g_RoModule)) nn::ro::detail::RoModule;
    new(&nn::util::Get(g_ManualLoad)) nn::ro::detail::RoModuleList;
    new(&nn::util::Get(g_AutoLoad)) nn::ro::detail::RoModuleList;

    pSelf->Initialize(base, 0, pDynamicSection); // nso はアンロードされることはないため、moduleSize は 0 を設定しておきます。
    Register(pSelf);

    // 自分以外のDLLを見つける
    for (uintptr_t v = 0;;)
    {
        nn::Result result;
        nn::svc::MemoryInfo blockInfo;
        nn::svc::PageInfo pageInfo;
        result = nn::svc::QueryMemory(&blockInfo, &pageInfo, v);
        while (result.IsFailure()) {}

        if (blockInfo.permission == nn::svc::MemoryPermission_ReadExecute &&
                blockInfo.state == nn::svc::MemoryState_Code &&
                blockInfo.baseAddress != base)
        {
            const nn::rocrt::ModuleHeaderLocation* pLocation = reinterpret_cast<const nn::rocrt::ModuleHeaderLocation*>(blockInfo.baseAddress);
            const nn::rocrt::ModuleHeader* pHeader = reinterpret_cast<const nn::rocrt::ModuleHeader*>(blockInfo.baseAddress + pLocation->headerOffset);
            while (pHeader->signature != nn::rocrt::ModuleHeaderVersion) {}
            const nn::ro::detail::Elf::Dyn* pDyn = reinterpret_cast<const nn::ro::detail::Elf::Dyn*>(blockInfo.baseAddress + pHeader->dynamicOffset + pLocation->headerOffset);
            uintptr_t bssBegin = blockInfo.baseAddress + pHeader->bssStartOffset + pLocation->headerOffset;
            uintptr_t bssEnd   = blockInfo.baseAddress + pHeader->bssEndOffset + pLocation->headerOffset;
            if (bssEnd - bssBegin)
            {
                std::memset(reinterpret_cast<void*>(bssBegin), 0, bssEnd - bssBegin);
            }
            nn::ro::detail::RoModule* pModule = new(reinterpret_cast<void *>(blockInfo.baseAddress + pHeader->moduleOffset + pLocation->headerOffset)) nn::ro::detail::RoModule;
            pModule->Initialize(blockInfo.baseAddress, 0, pDyn); // nso はアンロードされることはないため、moduleSize は 0 を設定しておきます。
            pModule->FixRelative();
            Register(pModule);
        }

        if (static_cast<uintptr_t>(blockInfo.baseAddress + blockInfo.size) <= v)
        {
            break;
        }
        v = blockInfo.baseAddress + blockInfo.size;
    }

    // ro ライブラリが rtld のインスタンスを参照できるようにする
    {
        nn::ro::detail::RoModuleList& autoLoadList = nn::util::Get(g_AutoLoad);
        for (nn::ro::detail::RoModuleList::const_iterator it = autoLoadList.cbegin(); it != autoLoadList.cend(); it++)
        {
            nn::ro::detail::Elf::Sym* pSym = it->Lookup(AutoLoadModuleListName);
            if (pSym && pSym->GetBind() != nn::ro::detail::Elf::STB_LOCAL)
            {
                nn::ro::detail::RoModuleList** pAutoLoadList = reinterpret_cast<nn::ro::detail::RoModuleList**>(it->GetBase() + pSym->GetValue());
                *pAutoLoadList = &autoLoadList;
            }

            pSym = it->Lookup(ManualLoadModuleListName);
            if (pSym && pSym->GetBind() != nn::ro::detail::Elf::STB_LOCAL)
            {
                nn::ro::detail::RoModuleList** pManualLoadList = reinterpret_cast<nn::ro::detail::RoModuleList**>(it->GetBase() + pSym->GetValue());
                *pManualLoadList = &nn::util::Get(g_ManualLoad);
            }

            pSym = it->Lookup(RoDebugFlagName);
            if (pSym && pSym->GetBind() != nn::ro::detail::Elf::STB_LOCAL)
            {
                bool** pRoDebugFlag = reinterpret_cast<bool**>(it->GetBase() + pSym->GetValue());
                *pRoDebugFlag = &ro::detail::g_EnableDebugLog;
            }

            pSym = it->Lookup(LookupGlobalAutoName);
            if (pSym && pSym->GetBind() != nn::ro::detail::Elf::STB_LOCAL)
            {
                nn::ro::detail::LookupGlobalAutoFunctionPointer* pLookupGlobalAutoFunctionPointer = reinterpret_cast<nn::ro::detail::LookupGlobalAutoFunctionPointer*>(it->GetBase() + pSym->GetValue());
                *pLookupGlobalAutoFunctionPointer = nn::rtld::LookupGlobalAuto;
            }

            pSym = it->Lookup(LookupGlobalManualName);
            if (pSym && pSym->GetBind() != nn::ro::detail::Elf::STB_LOCAL)
            {
                nn::ro::detail::LookupGlobalManualFunctionPointer** pLookupGlobalManualFunctionPointer = reinterpret_cast<nn::ro::detail::LookupGlobalManualFunctionPointer**>(it->GetBase() + pSym->GetValue());
                *pLookupGlobalManualFunctionPointer = &nn::ro::detail::g_LookupGlobalManualFunctionPointer;
            }
        }
    }

    RelocationAutoLoadDll();
} // NOLINT(impl/function_size)

void CallInit()
{
    CallInitAutoLoadDll();
    g_InitializedLock = true;
}

uintptr_t Bind(nn::ro::detail::RoModule* pModule, uint32_t index)
{
    return pModule->BindLazy(index);
}

uintptr_t LookupGlobalAuto(const char* pName)
{
    nn::ro::detail::RoModuleList& autoLoadList = nn::util::Get(g_AutoLoad);
    for (nn::ro::detail::RoModuleList::const_iterator it = autoLoadList.cbegin(); it != autoLoadList.cend(); it++)
    {
        nn::ro::detail::Elf::Sym* pSym = it->Lookup(pName);
        if (pSym && pSym->GetBind() != nn::ro::detail::Elf::STB_LOCAL)
        {
            return it->GetBase() + pSym->GetValue();
        }
    }
    return 0;
}

}}

