﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/



#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/lmem/lmem_ExpHeap.h>

#include <nn/pinmux/driver/pinmux.h>
#include <nn/pinmux/driver/pinmux_ApiForHardwareTest.h>

#include <nn/sf/sf_ObjectFactory.h>
#include <nn/sf/sf_NativeHandle.h>

#include <nn/pinmux/pinmux_IManager.sfdl.h>
#include <nn/pinmux/pinmux_ISession.sfdl.h>

#include <nn/pinmux/server/pinmux_ManagerImpl.h>

namespace nn { namespace pinmux { namespace server {

// ISession 実装クラス
class SessionImpl
{
public:
    // コンストラクタで親の SharedPointer を設定し、値によって、OpenSession / ForDev を呼び分ける
    SessionImpl(ManagerImpl* pParent, int32_t name) NN_NOEXCEPT
        : m_Parent(pParent, true),
          m_IsHardwareTest(false)
    {
        // enum の値によって ForHardwareTest と呼び分ける
        if(name < 100)
        {
            nn::pinmux::driver::OpenSession(&m_InternalSession, static_cast<nn::pinmux::AssignablePinGroupName>(name));
            m_IsHardwareTest = false;
        }
        else
        {
            nn::pinmux::driver::OpenSessionForHardwareTest(&m_InternalSession, static_cast<nn::pinmux::AssignablePinGroupNameForHardwareTest>(name));
            m_IsHardwareTest = true;
        }

    }

    // Session のデストラクタで Session のクローズを行う
    ~SessionImpl()
    {
        if(m_IsHardwareTest)
        {
            nn::pinmux::driver::CloseSessionForHardwareTest(&m_InternalSession);
        }
        else
        {
            nn::pinmux::driver::CloseSession(&m_InternalSession);
        }

    }

    // Pinmux ドライバライブラリの API
    nn::Result SetPinAssignment(std::int32_t assignment) NN_NOEXCEPT;
    nn::Result GetPinAssignment(nn::sf::Out<std::int32_t> assignment) NN_NOEXCEPT;
    nn::Result SetPinAssignmentForHardwareTest(std::int32_t assignment) NN_NOEXCEPT;
    nn::Result GetPinAssignmentForHardwareTest(nn::sf::Out<std::int32_t> assignment) NN_NOEXCEPT;

private:

    // 親への SharedPointer
    nn::sf::SharedPointer<ManagerImpl> m_Parent;

    // Pinmux ライブラリが持つ PinmuxSession 構造体のメンバ変数
    nn::pinmux::driver::PinmuxSession m_InternalSession;

    bool m_IsHardwareTest;
};



// ここから各 Pinmux API の内部実装
nn::Result SessionImpl::SetPinAssignment(std::int32_t assignment) NN_NOEXCEPT
{
    nn::pinmux::PinAssignment pinmuxAssignment = static_cast<nn::pinmux::PinAssignment>(assignment);

    nn::pinmux::driver::SetPinAssignment(&m_InternalSession, pinmuxAssignment);

    return nn::ResultSuccess();
}

nn::Result SessionImpl::GetPinAssignment(nn::sf::Out<std::int32_t> assignment) NN_NOEXCEPT
{
    *assignment = static_cast<int32_t>(nn::pinmux::driver::GetPinAssignment(&m_InternalSession));

    return nn::ResultSuccess();
}

nn::Result SessionImpl::SetPinAssignmentForHardwareTest(std::int32_t assignment) NN_NOEXCEPT
{
    nn::pinmux::PinAssignmentForHardwareTest pinmuxAssignment = static_cast<nn::pinmux::PinAssignmentForHardwareTest>(assignment);

    nn::pinmux::driver::SetPinAssignmentForHardwareTest(&m_InternalSession, pinmuxAssignment);

    return nn::ResultSuccess();
}

// マネージャーの実装
ManagerImpl::ManagerImpl() NN_NOEXCEPT
{
    // コンストラクタ内で拡張ヒープを初期化
    m_HeapHandle = nn::lmem::CreateExpHeap(&m_HeapBuffer, sizeof(m_HeapBuffer), nn::lmem::CreationOption_NoOption);
    // アロケータにヒープハンドルをアタッチ
    m_Allocator.Attach(m_HeapHandle);
}

ManagerImpl::~ManagerImpl() NN_NOEXCEPT
{
    nn::lmem::DestroyExpHeap(m_HeapHandle);
}


nn::Result ManagerImpl::OpenSession(nn::sf::Out<nn::sf::SharedPointer<nn::pinmux::ISession>> outSession, std::int32_t name) NN_NOEXCEPT
{
    // ObjectFactory の CreateSharedEmpleced で Interface 実装オブジェクトを生成し、そのオブジェクトへの共有ポインタを返す
    typedef nn::sf::ObjectFactory<MyAllocator::Policy>  Factory;
    auto p = Factory::CreateSharedEmplaced<ISession, SessionImpl>(&m_Allocator, this, name);

    // std::move で outSession に生成したオブジェクトの共有ポインタを渡す
    *outSession = std::move(p);
    return nn::ResultSuccess();
}

}}}
